#include "stm32f469xx.h"
#include "common.h"
#include "printf.h"
#include "sdUtil.h"
#include "sdHw.h"
#include <boot.h>
#include <dal.h>
#include <kal.h>

enum SdDrvMachineType {
	SdDrvMachineUnknown = 0,
	SdDrvMachineSTM32F429disco,
	SdDrvMachineSTM32F469disco,
	SdDrvMachineReSpring,
};



#define GPIO_PULL_NONE		0
#define GPIO_PULL_UP		1
#define GPIO_PULL_DOWN		2

static enum SdDrvMachineType __attribute__((const)) sdHwPrvGetMachType(void)
{
	switch (HALOEMGetHALID()) {
		case CREATE_4CC('r','e','S','p'):	return SdDrvMachineReSpring;
		case CREATE_4CC('d','4','2','9'):	return SdDrvMachineSTM32F429disco;
		case CREATE_4CC('d','4','6','9'):	return SdDrvMachineSTM32F469disco;
		default:							return SdDrvMachineUnknown;
	}
}


static void sdGpioSingleSetup(GPIO_TypeDef* gpio, uint32_t idx, bool output, uint32_t pull, uint32_t afr)
{
	gpio->MODER = (gpio->MODER &~ (3 << (idx * 2))) | ((afr ? 2 : (output ? 1 : 0)) << (idx * 2));
	gpio->OTYPER &=~ (1 << idx);
	gpio->OSPEEDR = (gpio->OSPEEDR &~ (3 << (idx * 2))) | (3 << (idx * 2));		//very high speed
	gpio->PUPDR = (gpio->PUPDR &~ (3 << (idx * 2)))  | (pull << (idx * 2));
	gpio->AFR[idx / 8] = (gpio->AFR[idx / 8] &~ (0x0F << ((idx % 8) * 4))) | afr << ((idx % 8) * 4);
}


static void sdInsertionIrqHandler(void *param)
{
	if (EXTI->PR & 0x0004) {
		
		EXTI->PR = 0x0004;
		SysNotifyBroadcastFromInterrupt(MY_CRID, MY_CRID, NULL);
	}
}

static void setupExtiIrq(void)
{
	EXTI->IMR &=~ 0x0004;	//int off
	EXTI->EMR &=~ 0x0004;	//event off
	EXTI->RTSR |= 0x0004;	//int on rising edge
	EXTI->FTSR |= 0x0004;	//int on falling edge
	EXTI->PR = 0x0004;		//clear current state
	EXTI->IMR |= 0x0004;	//int on
	
	HALInterruptSetHandler(REPALM_IRQ_NO_MANGLE(EXTI2_IRQn), sdInsertionIrqHandler, NULL);
	HALInterruptSetState(REPALM_IRQ_NO_MANGLE(EXTI2_IRQn), true);
}

static void disableExtiIrq(void)
{
	HALInterruptSetState(REPALM_IRQ_NO_MANGLE(EXTI2_IRQn), false);
	EXTI->IMR &=~ 0x0004;	//int off
	EXTI->EMR &=~ 0x0004;	//event off
	EXTI->RTSR &=~ 0x0004;	//no int on rising edge
	EXTI->FTSR &=~ 0x0004;	//no int on falling edge
	EXTI->PR = 0x0004;		//clear current state
	
	HALInterruptSetHandler(REPALM_IRQ_NO_MANGLE(EXTI2_IRQn), NULL, NULL);
}

static void sdPrvCardDetIrqSetup(void)
{
	switch (sdHwPrvGetMachType()) {
		case SdDrvMachineSTM32F429disco:
			//there is no card detect and thus nothing to do here
			break;
		
		case SdDrvMachineSTM32F469disco:
			//card detect pin (G2) is an input with a pullup
			sdGpioSingleSetup(GPIOG, 2, false, GPIO_PULL_UP, 0);
			//exti #2 uses portG
			SYSCFG->EXTICR[0] = (SYSCFG->EXTICR[0] &~ SYSCFG_EXTICR1_EXTI2) | SYSCFG_EXTICR1_EXTI2_PG;
			setupExtiIrq();
			break;
		
		case SdDrvMachineReSpring:
			//card detect pin (A2) is an input with a pullup
			sdGpioSingleSetup(GPIOA, 2, false, GPIO_PULL_UP, 0);
			//exti #2 uses portA
			SYSCFG->EXTICR[0] = (SYSCFG->EXTICR[0] &~ SYSCFG_EXTICR1_EXTI2) | SYSCFG_EXTICR1_EXTI2_PA;
			setupExtiIrq();
			//power pin is B10
			sdGpioSingleSetup(GPIOB, 10, true, GPIO_PULL_NONE, 0);
			break;
		
		default:
			//unknown
			break;
	}
}

static void sdPrvCardDetIrqUnsetup(void)
{
	switch (sdHwPrvGetMachType()) {
		case SdDrvMachineSTM32F429disco:
			//nothing to do
			break;
		
		case SdDrvMachineSTM32F469disco:
			//card detect pin (G2) is an input with a pullup
			sdGpioSingleSetup(GPIOG, 2, false, GPIO_PULL_UP, 0);
			disableExtiIrq();
			break;
		
		case SdDrvMachineReSpring:
			//card detect pin (A2) is an input with a pullup
			sdGpioSingleSetup(GPIOA, 2, false, GPIO_PULL_UP, 0);
			//exti #2 uses portA
			disableExtiIrq();
			//power pin is B10
			sdGpioSingleSetup(GPIOB, 10, true, GPIO_PULL_NONE, 0);
			break;
		
		default:
			//unknown
			break;
	}
}

static uint32_t sdPrvGetUnitBaseClock(void)	//get SDIO unit base speed in hz
{
	int32_t ret = repalmDalGetClockRate(SdioUnitClockRate);
	if (ret < 0)
		fatal("cannot go on without knowing the clock rate\n");
	
	return ret;
}

uint32_t sdHwInit(struct SdHwData *hwData)
{
	switch (sdHwPrvGetMachType()) {
		case SdDrvMachineSTM32F469disco:
		case SdDrvMachineReSpring:
			sdGpioSingleSetup(GPIOC, 9, true, GPIO_PULL_UP, 12);	//D1
			sdGpioSingleSetup(GPIOC, 10, true, GPIO_PULL_UP, 12);	//D2
			sdGpioSingleSetup(GPIOC, 11, true, GPIO_PULL_UP, 12);	//D3
			//fallthrough
		
		case SdDrvMachineSTM32F429disco:
			sdGpioSingleSetup(GPIOC, 8, true, GPIO_PULL_UP, 12);	//D0
			sdGpioSingleSetup(GPIOC, 12, true, GPIO_PULL_NONE, 12);	//CLK
			sdGpioSingleSetup(GPIOD, 2, true, GPIO_PULL_UP, 12);	//CMD
			break;
		
		default:
			//unknown
			return 0;
	}
	
	RCC->APB2ENR |= RCC_APB2ENR_SDIOEN;
	
	KALTaskDelay(1);
	SDIO->POWER = 3;		//unit is on
	SDIO->MASK = 0;			//mask everything
	KALTaskDelay(1);
	
	//hw flow control, 350khz, clock on
	SDIO->CLKCR = SDIO_CLKCR_HWFC_EN | SDIO_CLKCR_CLKEN | (((sdPrvGetUnitBaseClock() + 349000) / 350000) << SDIO_CLKCR_CLKDIV_Pos);
	
	sdPrvCardDetIrqSetup();
	
	switch (sdHwPrvGetMachType()) {
		case SdDrvMachineSTM32F469disco:
		case SdDrvMachineReSpring:
			return SD_HW_FLAG_INITED | SD_HW_FLAG_SUPPORT_4BIT | SD_HW_FLAG_SDIO_IFACE;
		
		case SdDrvMachineSTM32F429disco:
			return SD_HW_FLAG_INITED | SD_HW_FLAG_SDIO_IFACE;
		
		default:
			__builtin_unreachable();
			return 0;
	}
}

void sdHwShutdown(struct SdHwData *hwData)			//when driver unloaded. likely similar to sdHwSleep(). must be idempotent
{
	sdPrvCardDetIrqUnsetup();
	
	SDIO->MASK = 0;			//mask everything
	SDIO->CLKCR = 0;		//all is off
	KALTaskDelay(1);		//wait a bit
	SDIO->POWER = 0;		//unit is off
	KALTaskDelay(1);		//wait a bit
	RCC->APB2ENR &=~ RCC_APB2ENR_SDIOEN;
	
	switch (sdHwPrvGetMachType()) {
		case SdDrvMachineSTM32F469disco:
		case SdDrvMachineReSpring:
			sdGpioSingleSetup(GPIOC, 9, false, GPIO_PULL_UP, 0);	//D1
			sdGpioSingleSetup(GPIOC, 10, false, GPIO_PULL_UP, 0);	//D2
			sdGpioSingleSetup(GPIOC, 11, false, GPIO_PULL_UP, 0);	//D3
			//fallthrough
		
		case SdDrvMachineSTM32F429disco:
			sdGpioSingleSetup(GPIOC, 8, false, GPIO_PULL_UP, 0);	//D0
			sdGpioSingleSetup(GPIOC, 12, false, GPIO_PULL_UP, 0);	//CLK
			sdGpioSingleSetup(GPIOD, 2, false, GPIO_PULL_UP, 0);	//CMD
			break;
		
		default:
			//unknown
			break;
	}
}

void sdHwCardPower(struct SdHwData *hwData, bool on)
{
	switch (sdHwPrvGetMachType()) {
		case SdDrvMachineReSpring:
			if (on)
				GPIOB->BSRR = 1 << (10 + 16);
			else
				GPIOB->BSRR = 1 << 10;
			break;
		
		default:
			//power control not supported
			break;
	}
}

bool sdHwIsCardInserted(struct SdHwData *hwData)
{
	switch (sdHwPrvGetMachType()) {
		case SdDrvMachineSTM32F429disco:
			return true;
		
		case SdDrvMachineSTM32F469disco:
			return !(GPIOG->IDR & 4);
		
		case SdDrvMachineReSpring:
			return !(GPIOA->IDR & 4);
		
		default:
			return false;
	}
}

bool sdHwIsCardLockSwitchOn(struct SdHwData *hwData)
{
	//all STM boards use microsd which lacks the write protect switch
	return false;
}

static uint32_t sdPrvDeviceMaxCardIfaceSpeed(void)
{
	switch (sdHwPrvGetMachType()) {
		case SdDrvMachineSTM32F429disco:
			return 4000000;	//slow for long wires
		
		case SdDrvMachineReSpring:
		case SdDrvMachineSTM32F469disco:
			return 52000000;
		
		default:
			return 0;
	}
}

void sdHwSetSpeed(struct SdHwData *hwData, uint32_t maxHz)
{
	uint32_t i, maxAllowableSpeed = sdPrvDeviceMaxCardIfaceSpeed(), busSpeed = sdPrvGetUnitBaseClock();
	
	if (maxHz > maxAllowableSpeed)
		maxHz = maxAllowableSpeed;
	
	if (maxHz >= busSpeed)
		SDIO->CLKCR |= SDIO_CLKCR_BYPASS;
	else {
		for (i = 0; i < (SDIO_CLKCR_CLKDIV_Msk >> SDIO_CLKCR_CLKDIV_Pos); i++) {
			if (maxHz >= busSpeed / (2 + i))
				break;
		}
		SDIO->CLKCR = (SDIO->CLKCR &~ (SDIO_CLKCR_CLKDIV | SDIO_CLKCR_BYPASS)) | (i << SDIO_CLKCR_CLKDIV_Pos);
	}
}

bool sdHwSetBusWidth(struct SdHwData *hwData, bool useFourWide)
{
	if (useFourWide)
		SDIO->CLKCR |= SDIO_CLKCR_WIDBUS_0;
	else
		SDIO->CLKCR&=~ SDIO_CLKCR_WIDBUS_0;
	
	return true;
}

static bool sdPrvIsCardSignallingBusy(void)
{
	return !(GPIOC->IDR & 0x100);
}

uint32_t sdHwGetMaxBlocksAtOnce(struct SdHwData *hwData)
{
	return 512;
}

uint32_t sdHwGetMaxBlockSize(struct SdHwData *hwData)
{
	return 16384;
}

enum SdHwCmdResult sdHwCmd(struct SdHwData *hwData, uint_fast8_t cmd, uint32_t param, bool cmdCrcRequired, enum SdHwRespType respTyp, void *respBufOut, enum SdHwDataDir dataDir, uint_fast16_t blockSz, uint32_t numBlocks)
{
	uint32_t cmdReg, i, sta, dlen = numBlocks * blockSz, cardR1 = 0;
	uint8_t *respBufOut8 = (uint8_t*)respBufOut;
	enum SdHwCmdResult ret;
	
	SDIO->DCTRL = 0;
	
	if (dataDir == SdHwDataNone) {
		SDIO->DLEN = 0;
		SDIO->DCTRL = 0;
	}
	else {
		//verify sizes are sane
		if (!blockSz || !numBlocks || numBlocks > 512 || (blockSz & (blockSz - 1)) || blockSz > 16386)
			return SdCmdInternalError;
		
		SDIO->DTIMER = 0x100000 * numBlocks;
		SDIO->DLEN = dlen;
		SDIO->DCTRL = (SDIO->DCTRL &~ 0xfff) | SDIO_DCTRL_DTEN | (dataDir == SdHwDataRead ? SDIO_DCTRL_DTDIR : 0) | ((31 - __builtin_clz(blockSz)) << SDIO_DCTRL_DBLOCKSIZE_Pos);
	} 
	
	cmdReg = cmd << SDIO_CMD_CMDINDEX_Pos;
	cmdReg |= SDIO_CMD_CPSMEN;
	
	switch (respTyp) {
		case SdRespTypeNone:
			break;
		case SdRespTypeR1:
		case SdRespTypeR1withBusy:
		case SdRespTypeR3:
		case SdRespTypeR7:
		case SdRespTypeSdR6:
			cmdReg |= SDIO_CMD_WAITRESP_0;
			break;
		case SdRespTypeSdR2:
			cmdReg |= SDIO_CMD_WAITRESP_0 | SDIO_CMD_WAITRESP_1;
			break;
		default:
			return SdCmdInternalError;
	}

	//clearing STA using ICR takes time
	while (SDIO->STA & 0x5ff)
		SDIO->ICR = 0x5ff;
	
	SDIO->ARG = param;
	SDIO->CMD = cmdReg;		//sends it
	
	//wait for cmd sent or reply RXed
	if (respTyp == SdRespTypeNone) {
		
		while (!((sta = SDIO->STA) & (SDIO_STA_CMDSENT | SDIO_STA_CTIMEOUT)));
		
		ret = (sta & SDIO_STA_CTIMEOUT) ? SdHwCmdResultRespTimeout : SdHwCmdResultOK;
		goto out;
	}
	
	//wait for reply or crc error
	i = 0;
	while (!((sta = SDIO->STA) & (SDIO_STA_CMDREND | SDIO_STA_CCRCFAIL | SDIO_STA_CTIMEOUT))) {
		if ((++i & 32767) == 16384) {
			ret = SdHwCmdResultRespTimeout;
			goto out;
		}		
	}
	
	if (sta & SDIO_STA_CTIMEOUT) {
		logt("cmd to, sta=0x%08x, now sta=0x%08x\n", sta, SDIO->STA);
		ret = SdHwCmdResultRespTimeout;
		goto out;
	}
	
	switch (respTyp) {
		case SdRespTypeR1:
		case SdRespTypeR1withBusy:
			*respBufOut8 = sdPrvR1toSpiR1(cardR1 = SDIO->RESP1);
			break;
		
		case SdRespTypeR3:
		case SdRespTypeR7:
		case SdRespTypeSdR6:
			i = SDIO->RESP1;
			*respBufOut8++ = i >> 24;
			*respBufOut8++ = i >> 16;
			*respBufOut8++ = i >> 8;
			*respBufOut8++ = i;
			break;
		
		case SdRespTypeSdR2:
			for (i = 0; i < 4; i++) {
				
				uint32_t t = (&SDIO->RESP1)[i];
				
				respBufOut8[(3 - i) * 4 + 0] = t >> 24;
				respBufOut8[(3 - i) * 4 + 1] = t >> 16;
				respBufOut8[(3 - i) * 4 + 2] = t >> 8;
				respBufOut8[(3 - i) * 4 + 3] = t;
			}
			break;
		
		default:
			break;
	}
	
	if (sta & SDIO_STA_CCRCFAIL) {
		ret = SdCmdInternalError;
		goto out;
	}
	if (sta & SDIO_STA_CTIMEOUT) {
		ret = SdHwCmdResultRespTimeout;
		goto out;
	}
	
	//see if the error indicates that we need to report something
	if (respTyp == SdRespTypeR1 || respTyp == SdRespTypeR1withBusy) {
		if (cardR1 & 0xfff9a000) {
			loge("CARD reports error in R1 resp: 0x%08x, cur cmd was CMD%u(0x%08x)\n", cardR1, cmd, param);
			ret = SdCmdInternalError;
			goto out;
		}
		
		if (dataDir != SdHwDataNone && ((cardR1 & 0x1e00) != 0x0800)) {
			loge("CARD reports non-tran state in R1 resp for command with data: 0x%08x, cur cmd was CMD%u(0x%08x)\n", cardR1, cmd, param);
			ret = SdCmdInternalError;
			goto out;
		}
	}
	
	ret = SdHwCmdResultOK;
	
	if (dataDir != SdHwDataNone)		//skip busy wait if there is data
		return ret;
	
out:
	if (respTyp == SdRespTypeR1withBusy) {
		
		//wait for busy
		while (sdPrvIsCardSignallingBusy());
	}
	
	return ret;
}

enum SdHwReadResult sdHwReadData(struct SdHwData *hwData, uint8_t *dst, uint_fast16_t sz)
{
	uint32_t sta;
	
	while (sz) {
				
		sta = SDIO->STA;
		
		if (sta & (SDIO_STA_DTIMEOUT | SDIO_STA_DATAEND))
			return SdHwReadTimeout;
		
		if (sta & SDIO_STA_DCRCFAIL)
			return SdHwReadCrcErr;
		
		if (sta & SDIO_STA_DATAEND)
			return SdHwReadInternalError;
	
		if (sta & SDIO_STA_RXDAVL) {
		
			uint32_t t = SDIO->FIFO;
			
			if (sz >= 4) {
				
				*dst++ = t >> 0;
				*dst++ = t >> 8;
				*dst++ = t >> 16;
				*dst++ = t >> 24;
				sz -= 4;
			}
			else {
				switch (sz) {
					
					default:
						__builtin_unreachable();
						break;
					
					case 3:
						dst[2] = t >> 16;
						//fallthrough
					
					case 2:
						dst[1] = t >> 8;
						//fallthrough
					
					case 1:
						dst[0] = t >> 0;
						break;
				}
				sz = 0;
			}
		}
	}
	//wait for block end
	while (!((sta = SDIO->STA) & (SDIO_STA_DCRCFAIL | SDIO_STA_DBCKEND)));
	
	if (sta & SDIO_STA_DCRCFAIL)
		return SdHwReadCrcErr;
	
	return SdHwReadOK;
}

enum SdHwWriteReply sdHwWriteData(struct SdHwData *hwData, const uint8_t *src, uint_fast16_t sz, bool isMultiblock)
{
	uint32_t sta;
	
	while (sz) {
				
		sta = SDIO->STA;
		
		if (sta & (SDIO_STA_DTIMEOUT | SDIO_STA_DATAEND))
			return SdHwTimeout;
		
		if (sta & SDIO_STA_DCRCFAIL)
			return SdHwWriteCrcErr;
		
		if (sta & SDIO_STA_DATAEND)
			return SdHwCommErr;
	
		if (!(sta & SDIO_STA_TXFIFOF)) {
		
			uint32_t t = 0;
		
			if (sz >= 4) {
				
				t = (t << 8) + src[3];
				t = (t << 8) + src[2];
				t = (t << 8) + src[1];
				t = (t << 8) + src[0];
				src += 4;
				sz -= 4;
			}
			else {
				switch (sz) {
					case 3:
						t = (t << 8) + src[2];
						//fallthrough
					
					case 2:
						t = (t << 8) + src[1];
						//fallthrough
					
					case 1:
						t = (t << 8) + src[0];
						//fallthrough
				}
				sz = 0;
			}
			SDIO->FIFO = t;
		}
	}
	//wait for block end
	while (!((sta = SDIO->STA) & (SDIO_STA_DCRCFAIL | SDIO_STA_DBCKEND)));
	
	if (sta & SDIO_STA_DCRCFAIL)
		return SdHwWriteCrcErr;
	
	return SdHwWriteAccepted;
}

void sdHwChipDeselect(struct SdHwData *hwData)
{
	while (sdPrvIsCardSignallingBusy());
}


bool sdHwPrgBusyWait(struct SdHwData *hwData)
{
	while (sdPrvIsCardSignallingBusy());
	
	return true;
}

void sdHwGiveInitClocks(struct SdHwData *hwData)
{
	//TODO...
}

void sdHwSleep(struct SdHwData *hwData)
{
	//really should do this
}

void sdHwWake(struct SdHwData *hwData)
{
	//really should do this
}

void sdHwRxRawBytes(struct SdHwData *hwData, void *dst /* can be NULL*/, uint_fast16_t numBytes)
{
	//not used for SDIO HWs
}

void sdHwNotifyRCA(struct SdHwData *hwData, uint_fast16_t rca)
{
	//we do not need it
}

bool sdHwMultiBlockWriteSignalEnd(struct SdHwData *hwData)
{
	//not needed
	return true;
}

bool sdHwMultiBlockReadSignalEnd(struct SdHwData *hwData)
{
	//not needed
	return true;
}

void sdHwSetTimeouts(struct SdHwData *hwData, uint_fast16_t timeoutBytes, uint32_t rdTimeoutTicks, uint32_t wrTimeoutTicks)
{
	//i really should honor these...
}