#include "paceCommon.h"

//#define REPORT_EACH_INSTR	//debugging help


.section .text		//makes sure this code is at the start

.globl __entry
__entry:

//our header: the structure that PatchPace utility expects
	.hword (m68kEmuEnter - _LOC_start) / 2				//save where entry is
	.hword (_LOC_BL_Line1010exc - _LOC_start) / 2		//save where we need to insert BLs
	.hword (_LOC_BL_Line1111exc - _LOC_start) / 2
	.hword (_LOC_BL_Bkpt - _LOC_start) / 2
	.hword (_LOC_BL_DbgBreak - _LOC_start) / 2
	.hword (_LOC_BL_CallOsCall - _LOC_start) / 2
	.hword (_LOC_BL_IllegalInstr - _LOC_start) / 2
	.hword (_LOC_BL_UnimplInstr - _LOC_start) / 2
	.hword (_LOC_BL_DivZero - _LOC_start) / 2
	.hword (_LOC_BL_TraceBit - _LOC_start) / 2

//XXX: move to ccr and from ccr is a word op, logical ops on ccr are byte ops


//the actual base address of the code we'll be embedding into the PACE binary
_LOC_start:


//we do not support tracing but PatchPace needs to find ALL places to patch, so here is a dummy
//call to the tracing handler. We'll never call it
_LOC_BL_TraceBit:
	.hword 0
	.hword 0
	


////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////// ENTRY/EXIT ////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////



m68kEmuEnter:											//accepts and returns emulator state
	push				{r4-r7, lr}
	mov					r1, rNext
	mov					r2, rPC
	push				{r1-r2}
	
	//point rNext to "dispatch"
	ldr					r1, 2f
	mov					rNext, pc
	add					rNext, r1
1:	
	loadState			r0, r1
	nextInstr
	//not reached

m68kEmuExit:											//does the returning that we talked about above
	storeState			r0, r1
	pop					{r1-r2}
	mov					rNext, r1
	mov					rPC, r2
	pop					{r4-r7, pc}
	
.balign 4
2:
.word					dispatch + 1 - 1b

//we get here in many ways, but for only one reason - an instruction that is not defined to exist
instrsIllegal:
	storeState			r0, r1
_LOC_BL_IllegalInstr:
	.hword 0
	.hword 0
	//not reached: no return to here expected

//subtly different than the above. This is for instructions that exist, but are not implemented in the emulator
//these can be instructions from 68010+ or supervisor instructions that userspace apps have no reason to use.
//Also included is the fault caused by CHK.W, same as PACE does
instrsUnimpl:
	storeState			r0, r1
_LOC_BL_UnimplInstr:
	.hword 0
	.hword 0
	//not reached: no return to here expected



////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////    UTILS   ////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////

//functions to set flags post shift. on input: r0 is shift amount, r2 is shift result, and, for ASL (ONLY!) rI is initial value


//ASL V flag is a bit complex!
//V - Set if the most significant bit is changed at any time during the shift operation; cleared otherwise
//the naive way is slow, so we look for a shortcut
//we really have two cases here, either (A) the shift was by an amount > data_size, or (B) it was smaller or same
// A: V = !!initialValue (since the final value is zero and it will pass nonzero if any nonzero bits exist)
// B: T = initialValue >> (data_size - shift_amount)
//		V = (T == 0 || T == -1)
//must be called directly after the shift op, with the relevant SR bits pre-cleared

.macro setShiftFlagsAsl m68kSz, bitSz
	
	setShiftFlagsAsl\m68kSz:
		bcc				1f
		adds			rSR, SR_C + SR_X
	1:
		lsls			r2, #32 - \bitSz				//mandatory to reset flags after the potential "adds" above
		bmi				1f
		bne				2f
		adds			rSR, SR_Z
		b				2f
	1:
		adds			rSR, SR_N
	2:

		//handle the V bit, as described above
		cmp				r0, #\bitSz
		ble				2f
		//case A
		cmp				rI, #0
		beq				1f
		adds			rSR, SR_V
	1:
		nextInstr
	
	2:
		//case B
		subs			r0, #32
		negs			r0, r0
	.if \bitSz != 32
		lsls			rI, rI, #32 - \bitSz
	.endif
		asrs			rI, r0
		beq				1f
		adds			r1, #1
		beq				1f
		adds			rSR, SR_V
	1:
		nextInstr
.endm

setShiftFlagsAsl B, 8
setShiftFlagsAsl W, 16
setShiftFlagsAsl L, 32


setShiftFlagsLslB:
setShiftFlagsGenericB:
	lsls				r2, #24
	bcc					1f
	adds				rSR, SR_C + SR_X
1:
	cmp					r2, #0
	bmi					1f
	bne					2f
	adds				rSR, SR_Z
2:
	nextInstr
	
1:
	adds				rSR, SR_N
	nextInstr
	
setShiftFlagsLslW:
setShiftFlagsGenericW:
	lsls				r2, #16
	bcc					1f
	adds				rSR, SR_C + SR_X
1:
	cmp					r2, #0
	bmi					1f
	bne					2f
	adds				rSR, SR_Z
2:
	nextInstr
	
1:
	adds				rSR, SR_N
	nextInstr
	

setShiftFlagsAsrB:
setShiftFlagsAsrW:
setShiftFlagsAsrL:
setShiftFlagsLsrB:										//you'd think the "N" flag would be wrong for B & W, but for non-zero amount LSR, N will be 0 anyways and value can only decrement so Z would be right too
setShiftFlagsLsrW:
setShiftFlagsLsrL:
setShiftFlagsLslL:
setShiftFlagsGenericL:
	mrs					r0, APSR
	lsrs				r0, #28
	adr					r1, ccTabShifts
	.if ASSUME_SR_ONLY_FLAGS
		ldrb			rSR, [r0, r1]
	.else
		ldrb			r0, [r0, r1]
		orrs			rSR, r0
	.endif
	nextInstr

.balign 4
ccTabShifts:											//arm (NZCV) top 4 bits to m68k bottom 5 (XNZVC) tab for shifts (they have same meanings). xform is abcd - > cabdc
	.byte 0b00000		//arm 0000
	.byte 0b00010		//arm 0001
	.byte 0b10001		//arm 0010
	.byte 0b10011		//arm 0011
	.byte 0b00100		//arm 0100
	.byte 0b00110		//arm 0101
	.byte 0b10101		//arm 0110
	.byte 0b10111		//arm 0111
	.byte 0b01000		//arm 1000
	.byte 0b01010		//arm 1001
	.byte 0b11001		//arm 1010
	.byte 0b11011		//arm 1011
	.byte 0b01100		//arm 1100
	.byte 0b01110		//arm 1101
	.byte 0b11101		//arm 1110
	.byte 0b11111		//arm 1111


//functions to handle zero shifts for ASL/ASR/LSL/LSR/ROL/ROR (value in r2)
//On zero rotate: X unaffected, N,Z set based on initial value, V cleared, C cleared
no_shift_B:
	setCcLogical		r2, 8, 1

no_shift_W:
	setCcLogical		r2, 16, 1

no_shift_L:
	setCcLogical		r2, 32, 1

//functions to handle zero shifts for ROXL/ROXR (value in r2)
//On zero rotate/shift: X unaffected, N,Z set based on initial value, V cleared, C gets X's initial value
no_shiftX_B:
	getX				r0
	setCcLogical		r2, 8, 0						//will clear C
	adds				rSR, r0							//use the fact that the C flag is bit 0
	nextInstr

no_shiftX_W:
	getX				r0
	setCcLogical		r2, 16, 0						//will clear C
	adds				rSR, r0							//use the fact that the C flag is bit 0
	nextInstr

no_shiftX_L:
	getX				r0
	setCcLogical		r2, 32, 0						//will clear C
	adds				rSR, r0							//use the fact that the C flag is bit 0
	nextInstr

//set flags for rotateX instructions, assumes the relevant rSR bits are pre-cleared, r2 has post-rotate value, rT has APSR whose "C" is the C we need to export
setRotateXflagsB:
	setCcLogicalNC		r2, 8, 0
	//now set X and C based on rT
	mov					r0, rT
	lsrs				r0, #30							//save C into C
	bcc					1f
	adds				rSR, SR_C + SR_X
1:
	nextInstr

setRotateXflagsW:
	setCcLogicalNC		r2, 16, 0
	//now set X and C based on rT
	mov					r0, rT
	lsrs				r0, #30							//save C into C
	bcc					1f
	adds				rSR, SR_C + SR_X
1:
	nextInstr

setRotateXflagsL:
	setCcLogicalNC		r2, 32, 0
	//now set X and C based on rT
	mov					r0, rT
	lsrs				r0, #30							//save C into C
	bcc					1f
	adds				rSR, SR_C + SR_X
1:
	nextInstr

.macro smallXflags	sz
		lsls			r1, r0, #32 - \sz				//carry bit it out to carry, result's relevant bits in reg
		beq				1f
		mov				r2, rSR							//clear Z if result was nonzero rSR's all stats bits are zero adn this will not affect any flags
	1:
		bcc				1f
		bpl				2f
	//carry set, negative
		adds			rSR, SR_X + SR_C + SR_N
		b				3f
	2:
	//carry set, positive
		adds			rSR, SR_X + SR_C
		b				3f
	1:
		bpl				3f
	//carry clear, negative
		adds			rSR, SR_N

	3:
	//carry clear, positive
		orrs			rSR, r2							//XNZ_C correct, just need V
		lsrs			r1, r0, #\sz - 1
		beq				1f
		cmp				r1, #3
		beq				1f
		adds			rSR, SR_V
	1:
.endm


.macro smallXops			sz
	
	performADDX\sz:
		movs			r2, SR_Z						//get original Z
		ands			r2, rSR
		mov				r3, rSR
		clearXNZVC
		lsrs			r3, #5							//shift 68k's X into our C
		adcs			r0, r1
		smallXflags		\sz
		bx				lr

	performNEGX\sz:
		movs			r0, r1
		movs			r1, #0
		//fallthrough
	
	performSUBX\sz:
		movs			r2, SR_Z						//get original Z
		ands			r2, rSR
		mov				r3, rSR
		adds			r3, SR_X						//flip X, into r3
		clearXNZVC
		lsrs			r3, #5							//shift 68k's X into our C
		sbcs			r0, r1
		smallXflags		\sz
		bx				lr

	

.endm

smallXops	8
smallXops	16

performADDX32: //u32 (u32 lhs, u32 rhs), modifies rSR as needed
	movs				r2, SR_Z						//get original Z
	ands				r2, rSR
	lsls				r2, #2							//move it into bit #5
	lsrs				r3, rSR, #5						//shift 68k's X into our C
	adcs				r0, r1
	mrs					r1, APSR
	lsrs				r1, #28
	adds				r1, r2							//concat of 68k's original Z, and ARM APSR bits: NZCV
	adr					r2, cc_tab_addx
	
	.if ASSUME_SR_ONLY_FLAGS
		ldrb			rSR, [r1, r2]
	.else
		ldrb			r1, [r1, r2]
		clearXNZVC
		orrs			rSR, r1
	.endif
	
	bx					lr

performNEGX32: //u32 (u32 lhs, u32 rhs), modifies rSR as needed
	movs			r0, r1
	movs			r1, #0
	//fallthrough

performSUBX32: //u32 (u32 lhs, u32 rhs), modifies rSR as needed
	movs			r2, SR_Z							//get original Z
	ands			r2, rSR
	lsls			r2, #2								//move into bit #5
	adds			rSR, SR_X							//flip X
	lsrs			r3, rSR, #5							//shift 68k's X into our C
	sbcs			r0, r1
	mrs				r1, APSR
	lsrs			r1, #28
	adds			r1, r2								//concat of 68k's original Z, and ARM APSR bits: NZCV
	adr				r2, cc_tab_subx
	
	.if ASSUME_SR_ONLY_FLAGS
		ldrb		rSR, [r1, r2]
	.else
		ldrb		r1, [r1, r2]
		clearXNZVC
		orrs		rSR, r1
	.endif
	
	bx				lr

.balign 4
cc_tab_addx:											//"z" is 68k original z
	//zNZCV -> XNZVC
	.byte 0b00000		//68kZ=0, arm 0000
	.byte 0b00010		//68kZ=0, arm 0001
	.byte 0b10001		//68kZ=0, arm 0010
	.byte 0b10011		//68kZ=0, arm 0011
	.byte 0b00000		//68kZ=0, arm 0100
	.byte 0b00010		//68kZ=0, arm 0101
	.byte 0b10001		//68kZ=0, arm 0110
	.byte 0b10011		//68kZ=0, arm 0111
	.byte 0b01000		//68kZ=0, arm 1000
	.byte 0b01010		//68kZ=0, arm 1001
	.byte 0b11001		//68kZ=0, arm 1010
	.byte 0b11011		//68kZ=0, arm 1011
	.byte 0b01000		//68kZ=0, arm 1100
	.byte 0b01010		//68kZ=0, arm 1101
	.byte 0b11001		//68kZ=0, arm 1110
	.byte 0b11011		//68kZ=0, arm 1111
	.byte 0b00000		//68kZ=1, arm 0000
	.byte 0b00010		//68kZ=1, arm 0001
	.byte 0b10001		//68kZ=1, arm 0010
	.byte 0b10011		//68kZ=1, arm 0011
	.byte 0b00100		//68kZ=1, arm 0100
	.byte 0b00110		//68kZ=1, arm 0101
	.byte 0b10101		//68kZ=1, arm 0110
	.byte 0b10111		//68kZ=1, arm 0111
	.byte 0b01000		//68kZ=1, arm 1000
	.byte 0b01010		//68kZ=1, arm 1001
	.byte 0b11001		//68kZ=1, arm 1010
	.byte 0b11011		//68kZ=1, arm 1011
	.byte 0b01100		//68kZ=1, arm 1100
	.byte 0b01110		//68kZ=1, arm 1101
	.byte 0b11101		//68kZ=1, arm 1110
	.byte 0b11111		//68kZ=1, arm 1111
	
.balign 4	
cc_tab_subx:											//"z" is 68k original z
	//zNZCV -> XNZVC
	.byte 0b10001		//68kZ=0, arm 0000
	.byte 0b10011		//68kZ=0, arm 0001
	.byte 0b00000		//68kZ=0, arm 0010
	.byte 0b00010		//68kZ=0, arm 0011
	.byte 0b10001		//68kZ=0, arm 0100
	.byte 0b10011		//68kZ=0, arm 0101
	.byte 0b00000		//68kZ=0, arm 0110
	.byte 0b00010		//68kZ=0, arm 0111
	.byte 0b11001		//68kZ=0, arm 1000
	.byte 0b11011		//68kZ=0, arm 1001
	.byte 0b01000		//68kZ=0, arm 1010
	.byte 0b01010		//68kZ=0, arm 1011
	.byte 0b11001		//68kZ=0, arm 1100
	.byte 0b11011		//68kZ=0, arm 1101
	.byte 0b01000		//68kZ=0, arm 1110
	.byte 0b01010		//68kZ=0, arm 1111
	.byte 0b10001		//68kZ=0, arm 0000
	.byte 0b10011		//68kZ=0, arm 0001
	.byte 0b00000		//68kZ=0, arm 0010
	.byte 0b00010		//68kZ=0, arm 0011
	.byte 0b10101		//68kZ=0, arm 0100
	.byte 0b10111		//68kZ=0, arm 0101
	.byte 0b00100		//68kZ=0, arm 0110
	.byte 0b00110		//68kZ=0, arm 0111
	.byte 0b11001		//68kZ=0, arm 1000
	.byte 0b11011		//68kZ=0, arm 1001
	.byte 0b01000		//68kZ=0, arm 1010
	.byte 0b01010		//68kZ=0, arm 1011
	.byte 0b11101		//68kZ=0, arm 1100
	.byte 0b11111		//68kZ=0, arm 1101
	.byte 0b01100		//68kZ=0, arm 1110
	.byte 0b01110		//68kZ=0, arm 1111

//convert current NZVC value and requested CC to a boolean answer. Use 0xFF to make Scc a bit easier. input is "NZVCcccc", which is fast to make
//the table is too big to inline it where it is needed, so we take a const loading penalty to access it pc-relative. C'est la vie
.balign 4
cctabGeneric:
	.byte 0xff	//NZVC = 0000, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 0000, cccc = F  (0001), reply = NO
	.byte 0xff	//NZVC = 0000, cccc = HI (0010), reply = YES
	.byte 0x00	//NZVC = 0000, cccc = LS (0011), reply = NO
	.byte 0xff	//NZVC = 0000, cccc = CC (0100), reply = YES
	.byte 0x00	//NZVC = 0000, cccc = CS (0101), reply = NO
	.byte 0xff	//NZVC = 0000, cccc = N  (0110), reply = YES
	.byte 0x00	//NZVC = 0000, cccc = EQ (0111), reply = NO
	.byte 0xff	//NZVC = 0000, cccc = VC (1000), reply = YES
	.byte 0x00	//NZVC = 0000, cccc = VS (1001), reply = NO
	.byte 0xff	//NZVC = 0000, cccc = PL (1010), reply = YES
	.byte 0x00	//NZVC = 0000, cccc = MI (1011), reply = NO
	.byte 0xff	//NZVC = 0000, cccc = GE (1100), reply = YES
	.byte 0x00	//NZVC = 0000, cccc = LT (1101), reply = NO
	.byte 0xff	//NZVC = 0000, cccc = GT (1110), reply = YES
	.byte 0x00	//NZVC = 0000, cccc = LE (1111), reply = NO
	.byte 0xff	//NZVC = 0001, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 0001, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 0001, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 0001, cccc = LS (0011), reply = YES
	.byte 0x00	//NZVC = 0001, cccc = CC (0100), reply = NO
	.byte 0xff	//NZVC = 0001, cccc = CS (0101), reply = YES
	.byte 0xff	//NZVC = 0001, cccc = NE (0110), reply = YES
	.byte 0x00	//NZVC = 0001, cccc = EQ (0111), reply = NO
	.byte 0xff	//NZVC = 0001, cccc = VC (1000), reply = YES
	.byte 0x00	//NZVC = 0001, cccc = VS (1001), reply = NO
	.byte 0xff	//NZVC = 0001, cccc = PL (1010), reply = YES
	.byte 0x00	//NZVC = 0001, cccc = MI (1011), reply = NO
	.byte 0xff	//NZVC = 0001, cccc = GE (1100), reply = YES
	.byte 0x00	//NZVC = 0001, cccc = LT (1101), reply = NO
	.byte 0xff	//NZVC = 0001, cccc = GT (1110), reply = YES
	.byte 0x00	//NZVC = 0001, cccc = LE (1111), reply = NO
	.byte 0xff	//NZVC = 0010, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 0010, cccc = F  (0001), reply = NO
	.byte 0xff	//NZVC = 0010, cccc = HI (0010), reply = YES
	.byte 0x00	//NZVC = 0010, cccc = LS (0011), reply = NO
	.byte 0xff	//NZVC = 0010, cccc = CC (0100), reply = YES
	.byte 0x00	//NZVC = 0010, cccc = CS (0101), reply = NO
	.byte 0xff	//NZVC = 0010, cccc = NE (0110), reply = YES
	.byte 0x00	//NZVC = 0010, cccc = EQ (0111), reply = NO
	.byte 0x00	//NZVC = 0010, cccc = VC (1000), reply = NO
	.byte 0xff	//NZVC = 0010, cccc = VS (1001), reply = YES
	.byte 0xff	//NZVC = 0010, cccc = PL (1010), reply = YES
	.byte 0x00	//NZVC = 0010, cccc = MI (1011), reply = NO
	.byte 0x00	//NZVC = 0010, cccc = GE (1100), reply = NO
	.byte 0xff	//NZVC = 0010, cccc = LT (1101), reply = YES
	.byte 0x00	//NZVC = 0010, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 0010, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 0011, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 0011, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 0011, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 0011, cccc = LS (0011), reply = YES
	.byte 0x00	//NZVC = 0011, cccc = CC (0100), reply = NO
	.byte 0xff	//NZVC = 0011, cccc = CS (0101), reply = YES
	.byte 0xff	//NZVC = 0011, cccc = NE (0110), reply = YES
	.byte 0x00	//NZVC = 0011, cccc = EQ (0111), reply = NO
	.byte 0x00	//NZVC = 0011, cccc = VC (1000), reply = NO
	.byte 0xff	//NZVC = 0011, cccc = VS (1001), reply = YES
	.byte 0xff	//NZVC = 0011, cccc = PL (1010), reply = YES
	.byte 0x00	//NZVC = 0011, cccc = MI (1011), reply = NO
	.byte 0x00	//NZVC = 0011, cccc = GE (1100), reply = NO
	.byte 0xff	//NZVC = 0011, cccc = LT (1101), reply = YES
	.byte 0x00	//NZVC = 0011, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 0011, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 0100, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 0100, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 0100, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 0100, cccc = LS (0011), reply = YES
	.byte 0xff	//NZVC = 0100, cccc = CC (0100), reply = YES
	.byte 0x00	//NZVC = 0100, cccc = CS (0101), reply = NO
	.byte 0x00	//NZVC = 0100, cccc = NE (0110), reply = NO
	.byte 0xff	//NZVC = 0100, cccc = EQ (0111), reply = YES
	.byte 0xff	//NZVC = 0100, cccc = VC (1000), reply = YES
	.byte 0x00	//NZVC = 0100, cccc = VS (1001), reply = NO
	.byte 0xff	//NZVC = 0100, cccc = PL (1010), reply = YES
	.byte 0x00	//NZVC = 0100, cccc = MI (1011), reply = NO
	.byte 0xff	//NZVC = 0100, cccc = GE (1100), reply = YES
	.byte 0x00	//NZVC = 0100, cccc = LT (1101), reply = NO
	.byte 0x00	//NZVC = 0100, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 0100, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 0101, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 0101, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 0101, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 0101, cccc = LS (0011), reply = YES
	.byte 0x00	//NZVC = 0101, cccc = CC (0100), reply = NO
	.byte 0xff	//NZVC = 0101, cccc = CS (0101), reply = YES
	.byte 0x00	//NZVC = 0101, cccc = NE (0110), reply = NO
	.byte 0xff	//NZVC = 0101, cccc = EQ (0111), reply = YES
	.byte 0xff	//NZVC = 0101, cccc = VC (1000), reply = YES
	.byte 0x00	//NZVC = 0101, cccc = VS (1001), reply = NO
	.byte 0xff	//NZVC = 0101, cccc = PL (1010), reply = YES
	.byte 0x00	//NZVC = 0101, cccc = MI (1011), reply = NO
	.byte 0xff	//NZVC = 0101, cccc = GE (1100), reply = YES
	.byte 0x00	//NZVC = 0101, cccc = LT (1101), reply = NO
	.byte 0x00	//NZVC = 0101, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 0101, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 0110, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 0110, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 0110, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 0110, cccc = LS (0011), reply = YES
	.byte 0xff	//NZVC = 0110, cccc = CC (0100), reply = YES
	.byte 0x00	//NZVC = 0110, cccc = CS (0101), reply = NO
	.byte 0x00	//NZVC = 0110, cccc = NE (0110), reply = NO
	.byte 0xff	//NZVC = 0110, cccc = EQ (0111), reply = YES
	.byte 0x00	//NZVC = 0110, cccc = VC (1000), reply = NO
	.byte 0xff	//NZVC = 0110, cccc = VS (1001), reply = YES
	.byte 0xff	//NZVC = 0110, cccc = PL (1010), reply = YES
	.byte 0x00	//NZVC = 0110, cccc = MI (1011), reply = NO
	.byte 0x00	//NZVC = 0110, cccc = GE (1100), reply = NO
	.byte 0xff	//NZVC = 0110, cccc = LT (1101), reply = YES
	.byte 0x00	//NZVC = 0110, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 0110, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 0111, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 0111, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 0111, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 0111, cccc = LS (0011), reply = YES
	.byte 0x00	//NZVC = 0111, cccc = CC (0100), reply = NO
	.byte 0xff	//NZVC = 0111, cccc = CS (0101), reply = YES
	.byte 0x00	//NZVC = 0111, cccc = NE (0110), reply = NO
	.byte 0xff	//NZVC = 0111, cccc = EQ (0111), reply = YES
	.byte 0x00	//NZVC = 0111, cccc = VC (1000), reply = NO
	.byte 0xff	//NZVC = 0111, cccc = VS (1001), reply = YES
	.byte 0xff	//NZVC = 0111, cccc = PL (1010), reply = YES
	.byte 0x00	//NZVC = 0111, cccc = MI (1011), reply = NO
	.byte 0x00	//NZVC = 0111, cccc = GE (1100), reply = NO
	.byte 0xff	//NZVC = 0111, cccc = LT (1101), reply = YES
	.byte 0x00	//NZVC = 0111, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 0111, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 1000, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 1000, cccc = F  (0001), reply = NO
	.byte 0xff	//NZVC = 1000, cccc = HI (0010), reply = YES
	.byte 0x00	//NZVC = 1000, cccc = LS (0011), reply = NO
	.byte 0xff	//NZVC = 1000, cccc = CC (0100), reply = YES
	.byte 0x00	//NZVC = 1000, cccc = CS (0101), reply = NO
	.byte 0xff	//NZVC = 1000, cccc = NE (0110), reply = YES
	.byte 0x00	//NZVC = 1000, cccc = EQ (0111), reply = NO
	.byte 0xff	//NZVC = 1000, cccc = VC (1000), reply = YES
	.byte 0x00	//NZVC = 1000, cccc = VS (1001), reply = NO
	.byte 0x00	//NZVC = 1000, cccc = PL (1010), reply = NO
	.byte 0xff	//NZVC = 1000, cccc = MI (1011), reply = YES
	.byte 0x00	//NZVC = 1000, cccc = GE (1100), reply = NO
	.byte 0xff	//NZVC = 1000, cccc = LT (1101), reply = YES
	.byte 0x00	//NZVC = 1000, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 1000, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 1001, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 1001, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 1001, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 1001, cccc = LS (0011), reply = YES
	.byte 0x00	//NZVC = 1001, cccc = CC (0100), reply = NO
	.byte 0xff	//NZVC = 1001, cccc = CS (0101), reply = YES
	.byte 0xff	//NZVC = 1001, cccc = NE (0110), reply = YES
	.byte 0x00	//NZVC = 1001, cccc = EQ (0111), reply = NO
	.byte 0xff	//NZVC = 1001, cccc = VC (1000), reply = YES
	.byte 0x00	//NZVC = 1001, cccc = VS (1001), reply = NO
	.byte 0x00	//NZVC = 1001, cccc = PL (1010), reply = NO
	.byte 0xff	//NZVC = 1001, cccc = MI (1011), reply = YES
	.byte 0x00	//NZVC = 1001, cccc = GE (1100), reply = NO
	.byte 0xff	//NZVC = 1001, cccc = LT (1101), reply = YES
	.byte 0x00	//NZVC = 1001, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 1001, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 1010, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 1010, cccc = F  (0001), reply = NO
	.byte 0xff	//NZVC = 1010, cccc = HI (0010), reply = YES
	.byte 0x00	//NZVC = 1010, cccc = LS (0011), reply = NO
	.byte 0xff	//NZVC = 1010, cccc = CC (0100), reply = YES
	.byte 0x00	//NZVC = 1010, cccc = CS (0101), reply = NO
	.byte 0xff	//NZVC = 1010, cccc = NE (0110), reply = YES
	.byte 0x00	//NZVC = 1010, cccc = EQ (0111), reply = NO
	.byte 0x00	//NZVC = 1010, cccc = VC (1000), reply = NO
	.byte 0xff	//NZVC = 1010, cccc = VS (1001), reply = YES
	.byte 0x00	//NZVC = 1010, cccc = PL (1010), reply = NO
	.byte 0xff	//NZVC = 1010, cccc = MI (1011), reply = YES
	.byte 0xff	//NZVC = 1010, cccc = GE (1100), reply = YES
	.byte 0x00	//NZVC = 1010, cccc = LT (1101), reply = NO
	.byte 0xff	//NZVC = 1010, cccc = GT (1110), reply = YES
	.byte 0x00	//NZVC = 1010, cccc = LE (1111), reply = NO
	.byte 0xff	//NZVC = 1011, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 1011, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 1011, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 1011, cccc = LS (0011), reply = YES
	.byte 0x00	//NZVC = 1011, cccc = CC (0100), reply = NO
	.byte 0xff	//NZVC = 1011, cccc = CS (0101), reply = YES
	.byte 0xff	//NZVC = 1011, cccc = NE (0110), reply = YES
	.byte 0x00	//NZVC = 1011, cccc = EQ (0111), reply = NO
	.byte 0x00	//NZVC = 1011, cccc = VC (1000), reply = NO
	.byte 0xff	//NZVC = 1011, cccc = VS (1001), reply = YES
	.byte 0x00	//NZVC = 1011, cccc = PL (1010), reply = NO
	.byte 0xff	//NZVC = 1011, cccc = MI (1011), reply = YES
	.byte 0xff	//NZVC = 1011, cccc = GE (1100), reply = YES
	.byte 0x00	//NZVC = 1011, cccc = LT (1101), reply = NO
	.byte 0xff	//NZVC = 1011, cccc = GT (1110), reply = YES
	.byte 0x00	//NZVC = 1011, cccc = LE (1111), reply = NO
	.byte 0xff	//NZVC = 1100, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 1100, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 1100, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 1100, cccc = LS (0011), reply = YES
	.byte 0xff	//NZVC = 1100, cccc = CC (0100), reply = YES
	.byte 0x00	//NZVC = 1100, cccc = CS (0101), reply = NO
	.byte 0x00	//NZVC = 1100, cccc = NE (0110), reply = NO
	.byte 0xff	//NZVC = 1100, cccc = EQ (0111), reply = YES
	.byte 0xff	//NZVC = 1100, cccc = VC (1000), reply = YES
	.byte 0x00	//NZVC = 1100, cccc = VS (1001), reply = NO
	.byte 0x00	//NZVC = 1100, cccc = PL (1010), reply = NO
	.byte 0xff	//NZVC = 1100, cccc = MI (1011), reply = YES
	.byte 0x00	//NZVC = 1100, cccc = GE (1100), reply = NO
	.byte 0xff	//NZVC = 1100, cccc = LT (1101), reply = YES
	.byte 0x00	//NZVC = 1100, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 1100, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 1101, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 1101, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 1101, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 1101, cccc = LS (0011), reply = YES
	.byte 0x00	//NZVC = 1101, cccc = CC (0100), reply = NO
	.byte 0xff	//NZVC = 1101, cccc = CS (0101), reply = YES
	.byte 0x00	//NZVC = 1101, cccc = NE (0110), reply = NO
	.byte 0xff	//NZVC = 1101, cccc = EQ (0111), reply = YES
	.byte 0xff	//NZVC = 1101, cccc = VC (1000), reply = YES
	.byte 0x00	//NZVC = 1101, cccc = VS (1001), reply = NO
	.byte 0x00	//NZVC = 1101, cccc = PL (1010), reply = NO
	.byte 0xff	//NZVC = 1101, cccc = MI (1011), reply = YES
	.byte 0x00	//NZVC = 1101, cccc = GE (1100), reply = NO
	.byte 0xff	//NZVC = 1101, cccc = LT (1101), reply = YES
	.byte 0x00	//NZVC = 1101, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 1101, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 1110, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 1110, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 1110, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 1110, cccc = LS (0011), reply = YES
	.byte 0xff	//NZVC = 1110, cccc = CC (0100), reply = YES
	.byte 0x00	//NZVC = 1110, cccc = CS (0101), reply = NO
	.byte 0x00	//NZVC = 1110, cccc = NE (0110), reply = NO
	.byte 0xff	//NZVC = 1110, cccc = EQ (0111), reply = YES
	.byte 0x00	//NZVC = 1110, cccc = VC (1000), reply = NO
	.byte 0xff	//NZVC = 1110, cccc = VS (1001), reply = YES
	.byte 0x00	//NZVC = 1110, cccc = PL (1010), reply = NO
	.byte 0xff	//NZVC = 1110, cccc = MI (1011), reply = YES
	.byte 0xff	//NZVC = 1110, cccc = GE (1100), reply = YES
	.byte 0x00	//NZVC = 1110, cccc = LT (1101), reply = NO
	.byte 0x00	//NZVC = 1110, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 1110, cccc = LE (1111), reply = YES
	.byte 0xff	//NZVC = 1111, cccc = T  (0000), reply = YES
	.byte 0x00	//NZVC = 1111, cccc = F  (0001), reply = NO
	.byte 0x00	//NZVC = 1111, cccc = HI (0010), reply = NO
	.byte 0xff	//NZVC = 1111, cccc = LS (0011), reply = YES
	.byte 0x00	//NZVC = 1111, cccc = CC (0100), reply = NO
	.byte 0xff	//NZVC = 1111, cccc = CS (0101), reply = YES
	.byte 0x00	//NZVC = 1111, cccc = NE (0110), reply = NO
	.byte 0xff	//NZVC = 1111, cccc = EQ (0111), reply = YES
	.byte 0x00	//NZVC = 1111, cccc = VC (1000), reply = NO
	.byte 0xff	//NZVC = 1111, cccc = VS (1001), reply = YES
	.byte 0x00	//NZVC = 1111, cccc = PL (1010), reply = NO
	.byte 0xff	//NZVC = 1111, cccc = MI (1011), reply = YES
	.byte 0xff	//NZVC = 1111, cccc = GE (1100), reply = YES
	.byte 0x00	//NZVC = 1111, cccc = LT (1101), reply = NO
	.byte 0x00	//NZVC = 1111, cccc = GT (1110), reply = NO
	.byte 0xff	//NZVC = 1111, cccc = LE (1111), reply = YES

doBcdAdd: //u8 (u8 lhs, u8 rhs), adjusts rSR as needed
	movs				r2, #0x0f
	movs				r3, #0x0f
	ands				r2, r0
	ands				r3, r1
	subs				r0, r2
	subs				r1, r3
	mov					rT, r0
	lsrs				r0, rSR, #5						//shift out X
	mov					r0, rT
	adcs				r2, r3
	cmp					r2, #0x0a
	blt					1f
	subs				r2, #0x0a
	adds				r1, #0x10
1:
	movs				r3, SR_C + SR_X
	bics				rSR, r3
	adds				r0, r1
	cmp					r0, #0xa0
	blt					1f
	orrs				rSR, r3
	subs				r0, #0xa0
1:
	adds				r0, r2
	beq					1f
	movs				r3, SR_Z
	bics				rSR, r3
1:
	bx					lr

doBcdNeg: //u8 (u8 lhs, u8 rhs), adjusts rSR as needed
	mov					r1, r0
	movs				r0, #0
	//fallthrough

doBcdSub: //u8 (u8 lhs, u8 rhs), adjusts rSR as needed
	movs				r2, #0x0f
	movs				r3, #0x0f
	ands				r2, r0
	ands				r3, r1
	subs				r0, r2
	subs				r1, r3
	mov					rT, r0
	getX				r0
	adds				r3, r0								//add X to RHS's LSB
	mov					r0, rT
	subs				r2, r3
	bge					1f
	adds				r2, #0x0a
	adds				r1, #0x10
1:
	movs				r3, SR_C + SR_X
	bics				rSR, r3
	subs				r0, r1
	bge					1f
	orrs				rSR, r3
	adds				r0, #0xa0
1:
	adds				r0, r2
	beq					1f
	movs				r3, SR_Z
	bics				rSR, r3
1:
	bx					lr
	

doUdiv: //{u32 q, u32 r} (u32 num, u31 denom)			//32/31 divide (slightly optimized such that 32-bit denom might not work, luckily we only need to divide by 16 bits)
														//will NOT gracefully handle division by zero!
	movs				r2, #1
	movs				r3, #0
1:
	lsls				r2, #1
	lsls				r1, #1
	bpl					1b
1:
	cmp					r0, r1
	blo					2f
	subs				r0, r1
	adds				r3, r2
2:
	lsrs				r1, #1
	lsrs				r2, #1
	bne					1b
	
	mov					r1, r0
	mov					r0, r3
	bx					lr

doSdiv: //{i32 q, i32 r} (i32 num, i31 denom)			//same as above, but signed
	push				{r0, r1, r4, lr}
	movs				r4, #1
	cmp					r0, #0
	bpl					1f
	negs				r0, r0
	negs				r4, r4
1:
	cmp					r1, #0
	bpl					1f
	negs				r1, r1
	negs				r4, r4
1:	
	bl					doUdiv							//we must recalc remainder in 3/4 cases, so it is not worth trying to avoid it
	muls				r0, r4							//properly signed quotient
	pop					{r2, r3}
	muls				r3, r0
	subs				r1, r2, r3
	pop					{r4, pc}

////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////  HANDLERS  ////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////


.macro logicalOp name, m68kSz, regOp, mathOp, bitSz

	instrs\name\m68kSz\()reg:
		ubfx			r0, rI, -1, 7
		bl				readEa\m68kSz					//RHS val
		ubfx			r1, rI, 9, 3
		lsls			r1, #2							//Dn (LHS) << 2
		ldr\regOp		r2, [rD, r1]
		\mathOp			r2, r0
		str\regOp		r2, [rD, r1]
		setCcLogical	r2, \bitSz, 1					// CC
	
	instrs\name\m68kSz\()ea:
		ubfx			r0, rI, -1, 7
		bl				calcEa\m68kSz					//r0 = LHS addr
		read\bitSz		r2, r0, r1						//r2 = LHS val
		ubfx			r1, rI, 9, 3
		lsls			r1, #2							//Dn (Rhs) << 2
		ldr\regOp		r1, [rD, r1]					//RHS val
		\mathOp			r2, r1
		write\bitSz		r2, r0, rI						//store it
		setCcLogical	r2, \bitSz, 1					// CC
	
.endm

logicalOp And, B, b, ands, 8
logicalOp And, W, h, ands, 16
logicalOp And, L,  , ands, 32

.macro cmpOp m68kSz, regOp, bitSz

	instrsCmp\m68kSz:
		ubfx			r0, rI, -1, 7
		bl				readEa\m68kSz					//srcOp
		ubfx			r1, rI, 9, 3					//Dn
		lsls			r1, #2
		ldr\regOp		r1, [rD, r1]					//op
		setCcArith		r1, r0, subs, \bitSz, 0, ccTab_Cmp		// CC
		nextInstr
.endm

.macro cmpaOp m68kSz, doExtend

	instrsCmpa\m68kSz:
		ubfx			r0, rI, -1, 7
		bl				readEa\m68kSz					//srcOp
		.if \doExtend
			sxth		r0, r0
		.endif
		ubfx			r1, rI, 9, 3					//Dn
		lsls			r1, #2
		ldr				r1, [rA, r1]					//op
		setCcArith		r1, r0, subs, 32, 0, ccTab_Cmp			// CC
		nextInstr
.endm

cmpOp B, b, 8
cmpOp W, h, 16
cmpOp L,  , 32
cmpaOp W, 1
cmpaOp L, 0


.macro eorOp m68kSz, regOp, bitSz
	
	instrsEor\m68kSz\()reg:								//Dn, Dm variant
		ubfx			r0, rI, 9, 3
		lsls			r0, #2
		ldr\regOp		r0, [rD, r0]					//Dn.\m68kSz
		ubfx			r1, rI, -2, 5
		ldr\regOp		r2, [rD, r1]					//Dm.\m68kSz
		eors			r2, r0
		str\regOp		r2, [rD, r1]
		setCcLogical	r2, \bitSz, 1
	
	instrsEor\m68kSz:									//Dn, <ea> variant
		ubfx			r0, rI, -1, 7
		bl				calcEa\m68kSz
		read\bitSz		r1, r0, r2
		ubfx			r2, rI, 9, 3
		lsls			r2, #2
		ldr\regOp		r2, [rD, r2]					//Dn.\m68kSz
		eors			r1, r2
		write\bitSz		r1, r0, r2
		setCcLogical	r1, \bitSz, 1
.endm
	
eorOp B, b, 8
eorOp W, h, 16
eorOp L,  , 32

.macro cmpmOp m68kSz, bitSz
	
	instrsCmpm\m68kSz:
		ubfx			r0, rI, -2, 5
		ldr				r1, [rA, r0]					//Am
		adds			r2, r1, \bitSz / 8
		str				r2, [rA, r0]
		ubfx			r0, rI, 9, 3
		lsls			r0, #2
		ldr				r3, [rA, r0]					//An
		adds			r2, r3, \bitSz / 8
		str				r2, [rA, r0]
		read\bitSz		r0, r1, r2						//(Am)+.\m68kSz
		read\bitSz		r1, r3, r2						//(An)+.\m68kSz
		setCcArith		r1, r0, subs, \bitSz, 0, ccTab_Cmp		// CC
		nextInstr
.endm

cmpmOp B, 8
cmpmOp W, 16
cmpmOp L, 32

.balign 4
//arm (NZCV) top 4 bits to m68k bottom 4 (NZVC) tab for compare (they have same meanings except C is inverted)
ccTab_Cmp:
	.byte 0b00001		//arm 0000
	.byte 0b00011		//arm 0001
	.byte 0b00000		//arm 0010
	.byte 0b00010		//arm 0011
	.byte 0b00101		//arm 0100
	.byte 0b00111		//arm 0101
	.byte 0b00100		//arm 0110
	.byte 0b00110		//arm 0111
	.byte 0b01001		//arm 1000
	.byte 0b01011		//arm 1001
	.byte 0b01000		//arm 1010
	.byte 0b01010		//arm 1011
	.byte 0b01101		//arm 1100
	.byte 0b01111		//arm 1101
	.byte 0b01100		//arm 1110
	.byte 0b01110		//arm 1111

instrsCmpEorAndIllegal:
	bl					instrsIllegal

.macro bitopReg name, instrOnOneM, doMod, modOp
	
	instrs\name:
		ubfx			r0, rI, -1, 7					// mmmrrr << 1
		cmp				r0, #0x10						// mmm == 0 -> \name Dm,Dn
		blt				instrs\name\()Dn
		ubfx			r1, rI, 9, 3
		lsls			r1, #2
		ldr				r1, [rD, r1]					// Dm
		cmp				r0, #0x20						// mmm == 1 -> \instrOnOneM
		bge				1f
		b				instrs\instrOnOneM
	1:
		ubfx			r1, r1, 0, 3
		movs			rI, #1
		lsls			rI, r1							//1 << (Dm & 7)
		bl				calcEaB
		ldrb			r2, [r0]						//get byte
		movs			r1, SR_Z
		bics			rSR, r1
		tst				r2, rI
		bne				1f
		orrs			rSR, r1
	1:
	.if \doMod
		\modOp			r2, rI
		strb			r2, [r0]
	.endif
		nextInstr
	
	instrs\name\()Dn:									//Dn * 2 is in r0
		ubfx			r1, rI, 9, 3
		lsls			r1, #2
		ldr				r1, [rD, r1]					//Dm's value
		ubfx			r1, r1, 0, 5					//Dm's value % 32
		movs			rI, #1
		lsls			rI, r1							//1 << (Dm & 31)
		lsls			r3, r0, #1
		ldr				r2, [rD, r3]					//Dn
		movs			r1, SR_Z
		bics			rSR, r1
		tst				r2, rI
		bne				1f
		orrs			rSR, r1
	1:
	.if \doMod
		\modOp			r2, rI
		str				r2, [rD, r3]
	.endif
		nextInstr
	
	instrs\name\()Imm:
		getInstrBytex	r1, r2							// #imm
		ubfx			r0, rI, -1, 7					// mmmrrr << 1
		cmp				r0, #0x10						// mmm == 0 -> \name	#imm,Dn
		blt				instrs\name\()ImmDn
		ubfx			r1, r1, 0, 3
		movs			rI, #1
		lsls			rI, r1							//1 << (#imm & 7)
		bl				calcEaB
		ldrb			r2, [r0]						//get byte
		movs			r1, SR_Z
		bics			rSR, r1
		tst				r2, rI
		bne				1f
		orrs			rSR, r1
	1:
	.if \doMod
		\modOp			r2, rI
		strb			r2, [r0]
	.endif
		nextInstr
	
	instrs\name\()ImmDn:								//imm is in r1, Dn * 2 in r0
		lsls			r3, r0, #1
		ldr 			r2, [rD, r3]					//Dn
		ubfx			r1, r1, 0, 5
		movs			rI, #1
		lsls			rI, r1							//1 << (#imm & 31)
		movs			r1, SR_Z
		bics			rSR, r1
		tst				r2, rI
		bne				1f
		orrs			rSR, r1
	1:
	.if \doMod
		\modOp			r2, rI
		str				r2, [rD, r3]
	.endif
		nextInstr
.endm

bitopReg Btst, MovepWtoD, 0
bitopReg Bchg, MovepLtoD, 1, eors
bitopReg Bclr, MovepWfromD, 1, bics
bitopReg Bset, MovepLfromD, 1, orrs
	
.macro logicalOpImm	name, m68kSz, immReadF, regOp, bitSz, mathOp, haveSpecial, specialName

	instrs\name\m68kSz:
		ubfx			r0, rI, -1, 7					// mmmrrr << 1
		\immReadF		rI, r2							// rI = #imm
		cmp				r0, #0x10						// mmm != 0 -> \name.\m68kSz #imm,<ea>
		bge				instrs\name\m68kSz\()ea
		lsls			r0, #1							// rrr << 2
		ldr\regOp		r2, [rD, r0]					// Dn.\m68kSz
		\mathOp			r2, rI							// operate
		str\regOp		r2, [rD, r0]					// store result
		setCcLogical	r2, \bitSz, 1					// CC
	
	instrs\name\m68kSz\()ea:							//rI is #imm
	.if \haveSpecial
		cmp				r0, #0x78						// 0b111100 -> \specialName, rI has #imm
		beq				instrs\specialName
	.endif
		bl				calcEa\m68kSz
		read\bitSz		r2, r0, r1
		\mathOp			r2, rI							// operate
		write\bitSz		r2, r0, r1						// store result
		setCcLogical	r2, \bitSz, 1					// CC
.endm

instrsOriBccr:
instrsOriWsr:
	ubfx				r0, rI, 0, 5					//only the allowed bits
	orrs				rSR, r0
	nextInstr

logicalOpImm Ori, B, getInstrBytex,  b,  8, orrs, 1, OriBccr
logicalOpImm Ori, W, getInstrWordUx, h, 16, orrs, 1, OriWsr
logicalOpImm Ori, L, getInstrLong,    , 32, orrs, 0


instrsMulAndExg:
	ubfx				r0, rI, 2, 7
	add					pc, r0
	nop
	b					instrsAndBreg
	b					instrsCmpEorAndIllegal			//adress regs do not like being ANDed
	b					instrsAndBreg
	b					instrsAndBreg
	b					instrsAndBreg
	b					instrsAndBreg
	b					instrsAndBreg
	b					instrsAndBreg
	b					instrsAndWreg
	b					instrsCmpEorAndIllegal
	b					instrsAndWreg
	b					instrsAndWreg
	b					instrsAndWreg
	b					instrsAndWreg
	b					instrsAndWreg
	b					instrsAndWreg
	b					instrsAndLreg
	b					instrsCmpEorAndIllegal
	b					instrsAndLreg
	b					instrsAndLreg
	b					instrsAndLreg
	b					instrsAndLreg
	b					instrsAndLreg
	b					instrsAndLreg
	b					instrsMulu
	b					instrsCmpEorAndIllegal
	b					instrsMulu
	b					instrsMulu
	b					instrsMulu
	b					instrsMulu
	b					instrsMulu
	b					instrsMulu
	b					instrsAbcdReg
	b					instrsAbcdMem
	b					instrsAndBea
	b					instrsAndBea
	b					instrsAndBea
	b					instrsAndBea
	b					instrsAndBea
	b					instrsAndBea
	b					instrsExgDD
	b					instrsExgAA
	b					instrsAndWea
	b					instrsAndWea
	b					instrsAndWea
	b					instrsAndWea
	b					instrsAndWea
	b					instrsAndWea
	b					instrsCmpEorAndIllegal
	b					instrsExgDA
	b					instrsAndLea
	b					instrsAndLea
	b					instrsAndLea
	b					instrsAndLea
	b					instrsAndLea
	b					instrsAndLea
	b					instrsMuls
	b					instrsCmpEorAndIllegal
	b					instrsMuls
	b					instrsMuls
	b					instrsMuls
	b					instrsMuls
	b					instrsMuls
	b					instrsMuls

logicalOpImm Andi, B, getInstrBytex,  b,  8, ands, 1, AndiBccr
logicalOpImm Andi, W, getInstrWordUx, h, 16, ands, 1, AndiWsr
logicalOpImm Andi, L, getInstrLong,    , 32, ands, 0

instrsAndiBccr:
instrsAndiWsr:
	movs				r0, SR_ALL
	mvns				r0, r0
	orrs				r0, rI							//only the allowed bits
	ands				rSR, r0
	nextInstr


.macro mulOp name, isSigned 
	instrs\name:
		ubfx			r0, rI, -1, 7
		bl				readEaW							//<ea>.val
		ubfx			r1, rI, 9, 3
		lsls			r1, #2							//Dn
		.if \isSigned
			ldrsh		r2, [rD, r1]
			sxth		r0, r0
		.else
			ldrh		r2, [rD, r1]
			uxth		r0, r0
		.endif
		muls			r0, r2
		str				r0, [rD, r1]
		setCcLogical	r0, 32, 1
.endm

mulOp Mulu, 0
mulOp Muls, 1

instrsCmpEor:
	ubfx				r0, rI, 2, 7
	add					pc, r0
	nop
	b					instrsCmpB
	b					instrsCmpEorAndIllegal			//address regs do not like being compares with bytes - they are too good for that
	b					instrsCmpB
	b					instrsCmpB
	b					instrsCmpB
	b					instrsCmpB
	b					instrsCmpB
	b					instrsCmpB
	b					instrsCmpW
	b					instrsCmpW
	b					instrsCmpW
	b					instrsCmpW
	b					instrsCmpW
	b					instrsCmpW
	b					instrsCmpW
	b					instrsCmpW
	b					instrsCmpL
	b					instrsCmpL
	b					instrsCmpL
	b					instrsCmpL
	b					instrsCmpL
	b					instrsCmpL
	b					instrsCmpL
	b					instrsCmpL
	b					instrsCmpaW
	b					instrsCmpaW
	b					instrsCmpaW
	b					instrsCmpaW
	b					instrsCmpaW
	b					instrsCmpaW
	b					instrsCmpaW
	b					instrsCmpaW
	b					instrsEorBreg
	b					instrsCmpmB
	b					instrsEorB
	b					instrsEorB
	b					instrsEorB
	b					instrsEorB
	b					instrsEorB
	b					instrsEorB
	b					instrsEorWreg
	b					instrsCmpmW
	b					instrsEorW
	b					instrsEorW
	b					instrsEorW
	b					instrsEorW
	b					instrsEorW
	b					instrsEorW
	b					instrsEorLreg
	b					instrsCmpmL
	b					instrsEorL
	b					instrsEorL
	b					instrsEorL
	b					instrsEorL
	b					instrsEorL
	b					instrsEorL
	b					instrsCmpaL
	b					instrsCmpaL
	b					instrsCmpaL
	b					instrsCmpaL
	b					instrsCmpaL
	b					instrsCmpaL
	b					instrsCmpaL
	b					instrsCmpaL

instrsEoriBccr:
instrsEoriWsr:
	ubfx				r0, rI, 0, 5					//only the allowed bits
	eors				rSR, r0
	nextInstr

logicalOpImm Eori, B, getInstrBytex,  b,  8, eors, 1, EoriBccr
logicalOpImm Eori, W, getInstrWordUx, h, 16, eors, 1, EoriWsr
logicalOpImm Eori, L, getInstrLong,    , 32, eors, 0

.macro movepStart										//d0 = d + An, r1 = &Dx
	ubfx				r0, rI, -2, 5					//An << 2
	ldr					r0, [rA, r0]					//An's value
	getInstrWordSx		r1, r2
	adds				r0, r1							//d + An
	ubfx				r1, rI, 9, 3					//Dx
	lsls				r1, #2							//Dx << 2
	adds				r1, rD							//&Dx
.endm

instrsMovepWtoD:										//MOVEP.W d(An), Dx
	movepStart
	ldrb				r2, [r0, #0]
	strb				r2, [r1, #1]					//midlo8
	ldrb				r2, [r0, #2]
	strb				r2, [r1, #0]					//lo8
	nextInstr

instrsMovepLtoD:										//MOVEP.L d(An), Dx
	movepStart
	ldrb				r2, [r0, #0]
	strb				r2, [r1, #3]					//hi8
	ldrb				r2, [r0, #2]
	strb				r2, [r1, #2]					//midhi8
	ldrb				r2, [r0, #4]
	strb				r2, [r1, #1]					//midlo8
	ldrb				r2, [r0, #6]
	strb				r2, [r1, #0]					//lo8
	nextInstr

instrsMovepWfromD:										//MOVEP.W Dx, d(An)
	movepStart
	ldrb				r2, [r1, #1]					//midlo8
	strb				r2, [r0, #0]
	ldrb				r2, [r1, #0]					//lo8
	strb				r2, [r0, #2]
	nextInstr

instrsMovepLfromD:										//MOVEP.L Dx, d(An)
	movepStart
	ldrb				r2, [r1, #3]					//hi8
	strb				r2, [r0, #0]
	ldrb				r2, [r1, #2]					//midhi8
	strb				r2, [r0, #2]
	ldrb				r2, [r1, #1]					//midlo8
	strb				r2, [r0, #4]
	ldrb				r2, [r1, #0]					//lo8
	strb				r2, [r0, #6]
	nextInstr

instrsImmUnimpl:
	bl					instrsUnimpl

instrsImmIllegal:
	bl					instrsIllegal

instrsImm:
	ubfx				r0, rI, 5, 7
	add					pc, r0
	nop
	b					instrsOriB
	b					instrsOriW
	b					instrsOriL
	b					instrsImmIllegal
	b					instrsBtst
	b					instrsBchg
	b					instrsBclr
	b					instrsBset
	b					instrsAndiB
	b					instrsAndiW
	b					instrsAndiL
	b					instrsImmIllegal
	b					instrsBtst
	b					instrsBchg
	b					instrsBclr
	b					instrsBset
	b					instrsSubiB
	b					instrsSubiW
	b					instrsSubiL
	b					instrsImmIllegal
	b					instrsBtst
	b					instrsBchg
	b					instrsBclr
	b					instrsBset
	b					instrsAddiB
	b					instrsAddiW
	b					instrsAddiL
	b					instrsImmIllegal
	b					instrsBtst
	b					instrsBchg
	b					instrsBclr
	b					instrsBset
	b					instrsBtstImm
	b					instrsBchgImm
	b					instrsBclrImm
	b					instrsBsetImm
	b					instrsBtst
	b					instrsBchg
	b					instrsBclr
	b					instrsBset
	b					instrsEoriB
	b					instrsEoriW
	b					instrsEoriL
	b					instrsImmIllegal
	b					instrsBtst
	b					instrsBchg
	b					instrsBclr
	b					instrsBset
	b					instrsCmpiB
	b					instrsCmpiW
	b					instrsCmpiL
	b					instrsImmIllegal
	b					instrsBtst
	b					instrsBchg
	b					instrsBclr
	b					instrsBset
	b					instrsImmUnimpl
	b					instrsImmUnimpl
	b					instrsImmUnimpl
	b					instrsImmIllegal
	b					instrsBtst
	b					instrsBchg
	b					instrsBclr
	b					instrsBset

.macro exgOp name, xOp, yOp

	instrsExg\name:
		ubfx			r0, rI, -2, 5
		ubfx			r1, rI, 9, 3
		lsls			r1, #2
		ldr				r2, [\yOp, r0]
		ldr				r3, [\xOp, r1]
		str				r2, [\xOp, r1]
		str				r3, [\yOp, r0]
		nextInstr
.endm

exgOp DD, rD, rD
exgOp AA, rA, rA
exgOp DA, rD, rA

.macro bcdOp name, subroutine

	instrs\name\()Reg:									//ABCD Dm, Dn
		ubfx			r1, rI, -2, 5
		ldrb			r1, [rD, r1]
		ubfx			rI, rI, 9, 3
		lsls			rI, #2
		ldrb			r0, [rD, rI]
		bl				\subroutine
		strb			r0, [rD, rI]
		nextInstr
	
	instrs\name\()Mem:									//ABCD -(Ay), -(Ax)
		ubfx			r1, rI, -2, 5
		ldr				r2, [rA, r1]
		subs			r2, #1
		str				r2, [rA, r1]
		read8			r1, r2, r0
		ubfx			r2, rI, 9, 3
		lsls			r2, #2
		ldr				rI, [rA, r2]
		subs			rI, #1
		str				rI, [rA, r2]
		read8			r0, rI, r2
		bl				\subroutine
		write8			r0, rI, r2
		nextInstr
.endm

bcdOp Abcd, doBcdAdd

.macro arithOpImm name, m68kSz, immReadF, regOp, bitSz, mathOp, doStore, setX

	instrs\name\m68kSz:
		ubfx			r0, rI, -1, 7					// mmmrrr << 1
		\immReadF		rI, r2							// rI = #imm
		cmp				r0, #0x10						// mmm != 0 -> \name.\m68kSz #imm,<ea>
		bge				instrs\name\m68kSz\()ea
		lsls			r0, #1							// rrr << 2
		ldr\regOp		r2, [rD, r0]					// Dn.\m68kSz
		.if \doStore
			\mathOp		r3, r2, rI						// operate (setCcArith does not need this, so unless we store it, do not do it)
			str\regOp	r3, [rD, r0]					// store result
		.endif
		setCcArith		r2, rI, \mathOp, \bitSz, \setX, ccTab_\name		// CC
		nextInstr
	
	instrs\name\m68kSz\()ea:							//rI is #imm
		bl				calcEa\m68kSz
		read\bitSz		r2, r0, r1
		.if \doStore
			\mathOp		r3, r2, rI						// operate
			write\bitSz	r3, r0, r1						// store result
		.endif
		setCcArith		r2, rI, \mathOp, \bitSz, \setX, ccTab_\name		// CC
		nextInstr
.endm

arithOpImm Addi, B, getInstrByteSx,  b,   8, adds, 1, 1
arithOpImm Addi, W, getInstrWordSx,  h,  16, adds, 1, 1
arithOpImm Addi, L, getInstrLong,     ,  32, adds, 1, 1

arithOpImm Subi, B, getInstrByteSx,  b,   8, subs, 1, 1
arithOpImm Subi, W, getInstrWordSx,  h,  16, subs, 1, 1
arithOpImm Subi, L, getInstrLong,     ,  32, subs, 1, 1

arithOpImm Cmpi, B, getInstrByteSx,  b,  8, subs, 0, 0
arithOpImm Cmpi, W, getInstrWordSx,  h, 16, subs, 0, 0
arithOpImm Cmpi, L, getInstrLong,     , 32, subs, 0, 0

j_instrsMulAndExg:
	b					instrsMulAndExg

.balign 4
ccTab_Addi:												//arm (NZCV) top 4 bits to m68k bottom 5 (XNZVC) tab for addition (they have same meanings). xform is abcd - > cabdc
	.byte 0b00000		//arm 0000
	.byte 0b00010		//arm 0001
	.byte 0b10001		//arm 0010
	.byte 0b10011		//arm 0011
	.byte 0b00100		//arm 0100
	.byte 0b00110		//arm 0101
	.byte 0b10101		//arm 0110
	.byte 0b10111		//arm 0111
	.byte 0b01000		//arm 1000
	.byte 0b01010		//arm 1001
	.byte 0b11001		//arm 1010
	.byte 0b11011		//arm 1011
	.byte 0b01100		//arm 1100
	.byte 0b01110		//arm 1101
	.byte 0b11101		//arm 1110
	.byte 0b11111		//arm 1111
	
.balign 4
ccTab_Subi:												//arm (NZCV) top 4 bits to m68k bottom 5 (XNZVC) tab for subtraction (they have same meanings except C is inverted)
	.byte 0b10001		//arm 0000
	.byte 0b10011		//arm 0001
	.byte 0b00000		//arm 0010
	.byte 0b00010		//arm 0011
	.byte 0b10101		//arm 0100
	.byte 0b10111		//arm 0101
	.byte 0b00100		//arm 0110
	.byte 0b00110		//arm 0111
	.byte 0b11001		//arm 1000
	.byte 0b11011		//arm 1001
	.byte 0b01000		//arm 1010
	.byte 0b01010		//arm 1011
	.byte 0b11101		//arm 1100
	.byte 0b11111		//arm 1101
	.byte 0b01100		//arm 1110
	.byte 0b01110		//arm 1111

.balign 4
ccTab_Cmpi:												//arm (NZCV) top 4 bits to m68k bottom 4 (NZVC) tab for compare (they have same meanings except C is inverted)
	.byte 0b00001		//arm 0000
	.byte 0b00011		//arm 0001
	.byte 0b00000		//arm 0010
	.byte 0b00010		//arm 0011
	.byte 0b00101		//arm 0100
	.byte 0b00111		//arm 0101
	.byte 0b00100		//arm 0110
	.byte 0b00110		//arm 0111
	.byte 0b01001		//arm 1000
	.byte 0b01011		//arm 1001
	.byte 0b01000		//arm 1010
	.byte 0b01010		//arm 1011
	.byte 0b01101		//arm 1100
	.byte 0b01111		//arm 1101
	.byte 0b01100		//arm 1110
	.byte 0b01110		//arm 1111

instrsTrapLineA:
	storeState			r0, r1
	mov					rD, r0							//preserved reg - a good place to keep state
_LOC_BL_Line1010exc:
	.hword 0
	.hword 0
	loadState			rD, r1
	nextInstr

instrsTrapLineF:
	storeState			r0, r1
	mov					rD, r0							//preserved reg - a good place to keep state
_LOC_BL_Line1111exc:
	.hword 0
	.hword 0
	loadState			rD, r1
	nextInstr

instrsMoveB:
	ubfx				r0, rI, -1, 7					//r0 = ea.src << 1
	bl					readEaB							//r0 = src ea byte
	ubfx				r1, rI, 5, 7					//r1 = ea.dst << 1
	mov					rI, r0
	bl					writeEaB
	setCcLogical		rI, 8, 1

instrsMoveL:
	ubfx				r0, rI, -1, 7					//r0 = ea.src << 1
	bl					readEaL							//r0 = src ea byte
	ubfx				r1, rI, 5, 7					//r1 = ea.dst << 1
	mov					rI, r0
	bl					writeEaL
	setCcLogical		rI, 32, 1

instrsMoveW:
	ubfx				r0, rI, -1, 7					//r0 = ea.src << 1
	bl					readEaW							//r0 = src ea byte
	ubfx				r1, rI, 5, 7					//r1 = ea.dst << 1
	mov					rI, r0
	bl					writeEaW
	setCcLogical		rI, 16, 1

instrsMoveqInval:
	bl					instrsIllegal

instrsMoveq:											//threse happen often so they get special handling, we could use setCcLogical, but this is 1 cycle faster
	clearNZVC
	ubfx				r1, rI, 9, 3					//get Dn
	lsls				r0, rI, #24						//shift prohibited bit into C, data byte into top bits
	bcs					instrsMoveqInval
	lsls				r1, #2
	asrs				r0, r0, #24						//sign extend byte
	str					r0, [rD, r1]					//store
	bmi					1f
	bne					2f
	adds				rSR, SR_Z
2:
	nextInstr
1:
	adds				rSR, SR_N
	nextInstr


j_instrsOther:
	bl					instrsOther

////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////// DISPATCHER ////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////

//see "8.2 OPERATION CODE MAP" p560
dispatch:
	getInstrWordUx		rI, r0
	
	#ifdef REPORT_EACH_INSTR
	
		ldr r1, =0x70c00000 + 0x030a618
		ldr r2, =0x70c00000 + 0x030a618 + 0x128
		b lol
		.ltorg
		
	lol:
		mov				r0, rPC
		subs			r0, #2
		
		subs			r1, r0, r1
		bmi lolno
		subs			r2, r2, r0
		bmi	lolno
		
		
		mov				r1, rI
		mov				r2, rD
		mov				r3, rSR
		bl				report
	lolno:
	#endif
	
	lsrs				r0, rI, #11						//r0 =  (INSTR >> 12) << 1 + randomBit (we do not care about low bit)
	add					pc, r0
	nop
	b					instrsImm						//0x0??? - ORI,BTST,BCHG,BCLR,BSET,MOVEP,ANDI,SUBI,ADDI,EORI,CMPI
	b					instrsMoveB						//0x1??? - MOVE.B
	b					instrsMoveL						//0x2??? - MOVEA.L/MOVE.L
	b					instrsMoveW						//0x3??? - MOVEA.W/MOVE.W
	b					j_instrsOther					//0x4??? - NEGX,CHK,LEA,CLR,NEG,NOT,NBCD,SWAP,PEA,EXT,MOVEM,TST,TAS,TRAP,LINK,UNLK,RESET,NOP,STOP,RTE,RTS,TRAPV,RTR,JSR,JMP
	b					instrsQ							//0x5??? - ADDQ,SUBQ,Scc,DBcc
	b					j_instrsB						//0x6??? - Bcc,BSR,BRA
	b					instrsMoveq						//0x7??? - MOVEQ
	b					j_instrsOrDivSbcd				//0x8??? - OR,DIVU,DIVS,SBCD
	b					instrsSub						//0x9??? - SUB,SUBA,SUBX
	b					instrsTrapLineA					//0xA??? - Line A trap
	b					instrsCmpEor					//0xB??? - CMP,CMPA,CMPM,EOR
	b					j_instrsMulAndExg				//0xC??? - AND,MULU,ABCD,EXG,MULS
	b					instrsAdd						//0xD??? - ADD,ADDA,ADDX
	b					instrsShift						//0xE??? - LSL/LSR/ASR/etc
	b					instrsTrapLineF					//0xF??? - Line F trap

////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////  HANDLERS  ////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////


.macro instrQuick name, m68kSz, regOp, bitSz, mathOp, hasAmode
	instrs\name\m68kSz\()Dn:
		ubfx			r0, rI, -2, 5					//Dn << 2
		ldr\regOp		r1, [rD, r0]
		ubfxs			r2, rI, 9, 3
		bne				1f
		movs			r2, #8
	1:
		\mathOp			r3, r1, r2
		str\regOp		r3, [rD, r0]
		setCcArith		r1, r2, \mathOp, \bitSz, 1, ccTab_\name		// CC
		nextInstr
	
	.if \hasAmode
		
		instrs\name\m68kSz\()An:
			ubfx		r0, rI, -2, 5					//Dn << 2
			ldr\regOp	r1, [rA, r0]
			ubfxs		r2, rI, 9, 3
			bne			1f
			movs		r2, #8
		1:
			\mathOp		r3, r1, r2
			str\regOp	r3, [rA, r0]
			nextInstr									//no flags set for ops on address regs
		
	.endif

	instrs\name\m68kSz\()ea:
		ubfx			r0, rI, -1, 7					// mmmrrr << 1
		bl				calcEa\m68kSz
		read\bitSz		r1, r0, r2
		ubfxs			r2, rI, 9, 3
		bne				1f
		movs			r2, #8
	1:
		\mathOp			r3, r1, r2
		write\bitSz		r3, r0, rI
		setCcArith		r1, r2, \mathOp, \bitSz, 1, ccTab_\name		// CC
		nextInstr
.endm


instrQuick Addq, B, b,  8, adds, 0
instrQuick Addq, W, h, 16, adds, 0
instrQuick Addq, L,  , 32, adds, 1

instrQuick Subq, B, b,  8, subs, 0
instrQuick Subq, W, h, 16, subs, 0
instrQuick Subq, L,  , 32, subs, 1

.balign 4
ccTab_Addq:												//arm (NZCV) top 4 bits to m68k bottom 5 (XNZVC) tab for addition (they have same meanings). xform is abcd - > cabdc
	.byte 0b00000		//arm 0000
	.byte 0b00010		//arm 0001
	.byte 0b10001		//arm 0010
	.byte 0b10011		//arm 0011
	.byte 0b00100		//arm 0100
	.byte 0b00110		//arm 0101
	.byte 0b10101		//arm 0110
	.byte 0b10111		//arm 0111
	.byte 0b01000		//arm 1000
	.byte 0b01010		//arm 1001
	.byte 0b11001		//arm 1010
	.byte 0b11011		//arm 1011
	.byte 0b01100		//arm 1100
	.byte 0b01110		//arm 1101
	.byte 0b11101		//arm 1110
	.byte 0b11111		//arm 1111
	
.balign 4
ccTab_Subq:												//arm (NZCV) top 4 bits to m68k bottom 5 (XNZVC) tab for subtraction (they have same meanings except C is inverted)
	.byte 0b10001		//arm 0000
	.byte 0b10011		//arm 0001
	.byte 0b00000		//arm 0010
	.byte 0b00010		//arm 0011
	.byte 0b10101		//arm 0100
	.byte 0b10111		//arm 0101
	.byte 0b00100		//arm 0110
	.byte 0b00110		//arm 0111
	.byte 0b11001		//arm 1000
	.byte 0b11011		//arm 1001
	.byte 0b01000		//arm 1010
	.byte 0b01010		//arm 1011
	.byte 0b11101		//arm 1100
	.byte 0b11111		//arm 1101
	.byte 0b01100		//arm 1110
	.byte 0b01110		//arm 1111


instrsSccDn:
	ubfx				r0, rI, 8, 4
	ubfx				r1, rSR, -4, 8
	adds				r0, r1
	ldr					r1, 2f 
	add					r1, pc
	ldrb				r1, [r1, r0]					//reply
1:
	ubfx				r0, rI, -2, 5					//Dn << 2
	strb				r1, [rD, r0]
	nextInstr

.balign 4
2:
.word					cctabGeneric - 1b

instrsSccEa:
	ubfx				r0, rI, -1, 7					//ea << 1
	ubfx				r2, rI, 8, 4
	ubfx				r1, rSR, -4, 8
	adds				r2, r1
	ldr					r1, 2f 
	add					r1, pc
	ldrb				rI, [r1, r2]					//reply
1:
	bl					calcEaB
	strb				rI, [r0]
	nextInstr

.balign 4
2:
.word					cctabGeneric - 1b

instrsDbcc:
	ubfx				r0, rI, 8, 4
	ubfx				r1, rSR, -4, 8
	adds				r0, r1
	ldr					r1, 2f 
	add					r1, pc
	ldrb				r1, [r1, r0]					//r1 = reply
1:
	getInstrWordSx		r0, r2							//r0 = displacement (we have to read it no matter what)
	cmp					r1, #0							//true? - terminate now
	bne					3f
	ubfx				r2, rI, -2, 5					//Dn << 2
	ldrh				r3, [rD, r2]
	subs				r3, #1
	strh				r3, [rD, r2]
	adds				r3, #1
	beq					3f								//it became -1? - terminate now
	subs				r0, #2							//pc is 2 further than whence our offset is based, adjust offset accordingly
	add					rPC, r0
3:
	nextInstr

.balign 4
2:
.word					cctabGeneric - 1b

instrsIllegalQ:
	bl					instrsIllegal

instrsQ:
	ubfx				r0, rI, 2, 7					//discriminate on instr type, size, and access mode - a large table is worth it for these often-used instrs
	add					pc, r0
	nop
	b					instrsAddqBDn
	b					instrsIllegalQ					//would be addq.b #imm, An
	b					instrsAddqBea
	b					instrsAddqBea
	b					instrsAddqBea
	b					instrsAddqBea
	b					instrsAddqBea
	b					instrsAddqBea
	b					instrsAddqWDn
	b					instrsAddqLAn					//yes, L
	b					instrsAddqWea
	b					instrsAddqWea
	b					instrsAddqWea
	b					instrsAddqWea
	b					instrsAddqWea
	b					instrsAddqWea
	b					instrsAddqLDn
	b					instrsAddqLAn
	b					instrsAddqLea
	b					instrsAddqLea
	b					instrsAddqLea
	b					instrsAddqLea
	b					instrsAddqLea
	b					instrsAddqLea
	b					instrsSccDn
	b					instrsDbcc
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSubqBDn
	b					instrsIllegalQ					//would be Subq.b #imm, An
	b					instrsSubqBea
	b					instrsSubqBea
	b					instrsSubqBea
	b					instrsSubqBea
	b					instrsSubqBea
	b					instrsSubqBea
	b					instrsSubqWDn
	b					instrsSubqLAn					//yes, L
	b					instrsSubqWea
	b					instrsSubqWea
	b					instrsSubqWea
	b					instrsSubqWea
	b					instrsSubqWea
	b					instrsSubqWea
	b					instrsSubqLDn
	b					instrsSubqLAn
	b					instrsSubqLea
	b					instrsSubqLea
	b					instrsSubqLea
	b					instrsSubqLea
	b					instrsSubqLea
	b					instrsSubqLea
	b					instrsSccDn
	b					instrsDbcc
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSccEa
	b					instrsSccEa

instrsAdd:
	ubfx				r0, rI, 2, 7
	add					pc, r0
	nop
	b					instrsAddBreg
	b					instrsAddSubIllegal				//this would be ADD.B An, Dn, but A regs do not operate in byte mode
	b					instrsAddBreg
	b					instrsAddBreg
	b					instrsAddBreg
	b					instrsAddBreg
	b					instrsAddBreg
	b					instrsAddBreg
	b					instrsAddWreg
	b					instrsAddWreg
	b					instrsAddWreg
	b					instrsAddWreg
	b					instrsAddWreg
	b					instrsAddWreg
	b					instrsAddWreg
	b					instrsAddWreg
	b					instrsAddLreg
	b					instrsAddLreg
	b					instrsAddLreg
	b					instrsAddLreg
	b					instrsAddLreg
	b					instrsAddLreg
	b					instrsAddLreg
	b					instrsAddLreg
	b					instrsAddaWreg					//when An is source, it is still sign-extended to 16 bits!
	b					instrsAddaWreg
	b					instrsAddaWreg
	b					instrsAddaWreg
	b					instrsAddaWreg
	b					instrsAddaWreg
	b					instrsAddaWreg
	b					instrsAddaWreg
	b					instrsAddxBreg
	b					instrsAddxBmem
	b					instrsAddBmem
	b					instrsAddBmem
	b					instrsAddBmem
	b					instrsAddBmem
	b					instrsAddBmem
	b					instrsAddBmem
	b					instrsAddxWreg
	b					instrsAddxWmem
	b					instrsAddWmem
	b					instrsAddWmem
	b					instrsAddWmem
	b					instrsAddWmem
	b					instrsAddWmem
	b					instrsAddWmem
	b					instrsAddxLreg
	b					instrsAddxLmem
	b					instrsAddLmem
	b					instrsAddLmem
	b					instrsAddLmem
	b					instrsAddLmem
	b					instrsAddLmem
	b					instrsAddLmem
	b					instrsAddaLreg
	b					instrsAddaLreg
	b					instrsAddaLreg
	b					instrsAddaLreg
	b					instrsAddaLreg
	b					instrsAddaLreg
	b					instrsAddaLreg
	b					instrsAddaLreg

instrsAddSubIllegal:
	bl					instrsIllegal

instrsSub:
	ubfx				r0, rI, 2, 7
	add					pc, r0
	nop
	b					instrsSubBreg
	b					instrsAddSubIllegal				//this would be Sub.B An, Dn, but A regs do not operate in byte mode
	b					instrsSubBreg
	b					instrsSubBreg
	b					instrsSubBreg
	b					instrsSubBreg
	b					instrsSubBreg
	b					instrsSubBreg
	b					instrsSubWreg
	b					instrsSubWreg
	b					instrsSubWreg
	b					instrsSubWreg
	b					instrsSubWreg
	b					instrsSubWreg
	b					instrsSubWreg
	b					instrsSubWreg
	b					instrsSubLreg
	b					instrsSubLreg
	b					instrsSubLreg
	b					instrsSubLreg
	b					instrsSubLreg
	b					instrsSubLreg
	b					instrsSubLreg
	b					instrsSubLreg
	b					instrsSubaWreg					//when An is source, it is still sign-extended to 16 bits!
	b					instrsSubaWreg
	b					instrsSubaWreg
	b					instrsSubaWreg
	b					instrsSubaWreg
	b					instrsSubaWreg
	b					instrsSubaWreg
	b					instrsSubaWreg
	b					instrsSubxBreg
	b					instrsSubxBmem
	b					instrsSubBmem
	b					instrsSubBmem
	b					instrsSubBmem
	b					instrsSubBmem
	b					instrsSubBmem
	b					instrsSubBmem
	b					instrsSubxWreg
	b					instrsSubxWmem
	b					instrsSubWmem
	b					instrsSubWmem
	b					instrsSubWmem
	b					instrsSubWmem
	b					instrsSubWmem
	b					instrsSubWmem
	b					instrsSubxLreg
	b					instrsSubxLmem
	b					instrsSubLmem
	b					instrsSubLmem
	b					instrsSubLmem
	b					instrsSubLmem
	b					instrsSubLmem
	b					instrsSubLmem
	b					instrsSubaLreg
	b					instrsSubaLreg
	b					instrsSubaLreg
	b					instrsSubaLreg
	b					instrsSubaLreg
	b					instrsSubaLreg
	b					instrsSubaLreg
	b					instrsSubaLreg

.macro arithOpNonImm name, m68kSz, regOp, bitSz, mathOp, hasAmode, xMathOp, invertCforXmode

	instrs\name\m68kSz\()reg:
		ubfx			r0, rI, -1, 7
		bl				readEa\m68kSz					//RHS val
		ubfx			r1, rI, 9, 3
		lsls			r1, #2							//Dn (LHS) << 2
		ldr\regOp		r2, [rD, r1]
		\mathOp			r3, r2, r0
		str\regOp		r3, [rD, r1]
		setCcArith		r2, r0, \mathOp, \bitSz, 1, ccTab_\name		// CC
		nextInstr
	
	instrs\name\m68kSz\()mem:
		ubfx			r0, rI, -1, 7
		bl				calcEa\m68kSz					//r0 = LHS addr
		read\bitSz		r2, r0, r1						//r2 = LHS val
		ubfx			r1, rI, 9, 3
		lsls			r1, #2							//Dn (Rhs) << 2
		ldr\regOp		r1, [rD, r1]					//RHS val
		\mathOp			r3, r2, r1
		write\bitSz		r3, r0, rI						//store it
		setCcArith		r2, r1, \mathOp, \bitSz, 1, ccTab_\name		// CC
		nextInstr

	instrs\name\()a\m68kSz\()reg:
		ubfx			r0, rI, -1, 7
		bl				readEa\m68kSz					//RHS val
		.if \bitSz == 16
			sxth		r0, r0							//needed since readEaX does not sign extend and MAY return with high bits not clear
		.endif
		ubfx			r1, rI, 9, 3
		lsls			r1, #2							//Dn (LHS) << 2
		ldr				r2, [rA, r1]
		\mathOp			r3, r2, r0
		str				r3, [rA, r1]
		nextInstr										//no flags set for ops on address regs
.endm

arithOpNonImm Add, B, b,  8, adds, 0
arithOpNonImm Add, W, h, 16, adds, 1
arithOpNonImm Add, L,  , 32, adds, 1

arithOpNonImm Sub, B, b,  8, subs, 0
arithOpNonImm Sub, W, h, 16, subs, 1
arithOpNonImm Sub, L,  , 32, subs, 1

.balign 4
ccTab_Add:												//arm (NZCV) top 4 bits to m68k bottom 5 (XNZVC) tab for addition (they have same meanings). xform is abcd - > cabdc
	.byte 0b00000		//arm 0000
	.byte 0b00010		//arm 0001
	.byte 0b10001		//arm 0010
	.byte 0b10011		//arm 0011
	.byte 0b00100		//arm 0100
	.byte 0b00110		//arm 0101
	.byte 0b10101		//arm 0110
	.byte 0b10111		//arm 0111
	.byte 0b01000		//arm 1000
	.byte 0b01010		//arm 1001
	.byte 0b11001		//arm 1010
	.byte 0b11011		//arm 1011
	.byte 0b01100		//arm 1100
	.byte 0b01110		//arm 1101
	.byte 0b11101		//arm 1110
	.byte 0b11111		//arm 1111
	
	
ccTab_Sub:												//arm (NZCV) top 4 bits to m68k bottom 5 (XNZVC) tab for subtraction (they have same meanings except C is inverted)
	.byte 0b10001		//arm 0000
	.byte 0b10011		//arm 0001
	.byte 0b00000		//arm 0010
	.byte 0b00010		//arm 0011
	.byte 0b10101		//arm 0100
	.byte 0b10111		//arm 0101
	.byte 0b00100		//arm 0110
	.byte 0b00110		//arm 0111
	.byte 0b11001		//arm 1000
	.byte 0b11011		//arm 1001
	.byte 0b01000		//arm 1010
	.byte 0b01010		//arm 1011
	.byte 0b11101		//arm 1100
	.byte 0b11111		//arm 1101
	.byte 0b01100		//arm 1110
	.byte 0b01110		//arm 1111

.macro arithOpxNonImm name, m68kSz, regOp, bitSz, fCall
	
	instrs\name\m68kSz\()reg:
		ubfx			r0, rI, -2, 5					//Dm << 2
		ldr\regOp		r1, [rD, r0]					//Dm.\m68kSz value
		ubfx			rI, rI, 9, 3					//Dn
		lsls			rI, #2							//Dn << 2
		ldr\regOp		r0, [rD, rI]					//Dn.\m68kSz value
		bl				\fCall
		str\regOp		r0, [rD, rI]					//store Dn.\m68kSz value
		nextInstr
	
	instrs\name\m68kSz\()mem:
		ubfx			r0, rI, -2, 5					//Am << 2
		ldr				r0, [rA, r0]					//Am's value
		subs			r0, \bitSz / 8					//predec
		str				r0, [rA, r0]					//save it
		read\bitSz		r1, r0, r2						//value of -(Am)
		ubfx			r0, rI, 9, 3					//An
		lsls			r0, #2							//An << 2
		ldr				rI, [rA, r0]					//An's value
		subs			rI, \bitSz / 8					//predec
		str				rI, [rA, r0]					//save it
		read\bitSz		r0, rI, r3						//value of -(An)
		bl				\fCall
		write\bitSz		r0, rI, r3						//store result
		nextInstr
.endm

arithOpxNonImm Addx, B, b,  8, performADDX8
arithOpxNonImm Addx, L,  , 32, performADDX32
arithOpxNonImm Subx, B, b,  8, performSUBX8
arithOpxNonImm Subx, L,  , 32, performSUBX32
//16 bit addx & subx are below, in between shift dispatcher, for branch range reasons

j_instrsB:
	b					instrsB	

j_instrsOrDivSbcd:
	bl					instrsOrDivSbcd
	
instrsShift:
	ubfx				r0, rI, 2, 7
	add					pc, r0
	nop
	b					instrsAsrBimm
	b					instrsLsrBimm
	b					instrsRoxrBimm
	b					instrsRorBimm
	b					instrsAsrBreg
	b					instrsLsrBreg
	b					instrsRoxrBreg
	b					instrsRorBreg
	b					instrsAsrWimm
	b					instrsLsrWimm
	b					instrsRoxrWimm
	b					instrsRorWimm
	b					instrsAsrWreg
	b					instrsLsrWreg
	b					instrsRoxrWreg
	b					instrsRorWreg
	b					instrsAsrLimm
	b					instrsLsrLimm
	b					instrsRoxrLimm
	b					instrsRorLimm
	b					instrsAsrLreg
	b					instrsLsrLreg
	b					instrsRoxrLreg
	b					instrsRorLreg
	b					instrsShiftMem					//mem instrs dispatch together to save on code space - we win nothing by separating them
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsAslBimm					//yes, 68k has ASL, which is almost but not quite like an LSL
	b					instrsLslBimm
	b					instrsRoxlBimm
	b					instrsRolBimm
	b					instrsAslBreg
	b					instrsLslBreg
	b					instrsRoxlBreg
	b					instrsRolBreg
	b					instrsAslWimm
	b					instrsLslWimm
	b					instrsRoxlWimm
	b					instrsRolWimm
	b					instrsAslWreg
	b					instrsLslWreg
	b					instrsRoxlWreg
	b					instrsRolWreg
	b					instrsAslLimm
	b					instrsLslLimm
	b					instrsRoxlLimm
	b					instrsRolLimm
	b					instrsAslLreg
	b					instrsLslLreg
	b					instrsRoxlLreg
	b					instrsRolLreg
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem
	b					instrsShiftMem

//8 & 32 bit addx & subx are above, before the shift dispatcher, for branch range reasons
arithOpxNonImm Addx, W, h, 16, performADDX16
arithOpxNonImm Subx, W, h, 16, performSUBX16

.macro regShiftR name, m68kSz, regOpL, regOpS, shiftOp, needOrigValForAslFlagging

	instrs\name\m68kSz\()imm:
		ubfxs			r0, rI, 9, 3
		bne				1f
		movs			r0, #8
	1:
		ubfx			r1, rI, -2, 5					//Dn << 2
		ldr\regOpL		r2, [rD, r1]					//value
		
	do_\name\m68kSz:									//does NOT handle zero input, at all!!!
	
		.if \needOrigValForAslFlagging
			mov			rI, r2
		.endif
	
		clearXNZVC
		\shiftOp		r2, r0
		str\regOpS		r2, [rD, r1]
		bl				setShiftFlags\name\m68kSz
	
	instrs\name\m68kSz\()reg:
		ubfx			r0, rI, 9, 3					//Dm
		lsls			r0, #2
		ldr				r0, [rD, r0]
		ubfx			r1, rI, -2, 5					//Dn << 2
		ldr\regOpL		r2, [rD, r1]					//value
		ubfxs			r0, r0, 0, 6					//bit count is limited to 64, because logic...
		bne				do_\name\m68kSz
		bl				no_shift_\m68kSz
.endm


regShiftR Asr, B, sb, b, asrs, 0
regShiftR Asr, W, sh, h, asrs, 0
regShiftR Asr, L,   ,  , asrs, 0
regShiftR Lsr, B,  b, b, lsrs, 0
regShiftR Lsr, W,  h, h, lsrs, 0
regShiftR Lsr, L,   ,  , lsrs, 0
regShiftR Lsl, B,  b, b, lsls, 0
regShiftR Lsl, W,  h, h, lsls, 0
regShiftR Lsl, L,   ,  , lsls, 0
regShiftR Asl, B,  b, b, lsls, 1
regShiftR Asl, W,  h, h, lsls, 1
regShiftR Asl, L,   ,  , lsls, 1


.macro regRotate name, m68kSz, regOp, bitSz, logBitSz, core, outputBitInC

	instrs\name\m68kSz\()imm:
		ubfxs			r0, rI, 9, 3
		bne				1f
		movs			r0, #8

	1:													//r0 is #shiftAmt
		ubfx			r1, rI, -2, 5					//Dn << 2
		ldr\regOp		r2, [rD, r1]					//value
	
	instrs\name\m68kSz\()do:							//will not handle rotates amt > item_type OR amt == 0
		clearNZVC
		\core			\bitSz
		str\regOp		r2, [rD, r1]					//save result
	instrs\name\m68kSz\()setFlags:
		mov				r0, r2							//since setCcLogicalNC will corrupt r2
		setCcLogicalNC	r2, \bitSz, 0
		lsls			r0, r0, #31 - \outputBitInC
		bpl				1f
		adds			rSR, SR_C
	1:
		nextInstr
	
	instrs\name\m68kSz\()reg:
		ubfx			r0, rI, 9, 3					//Dm
		lsls			r0, #2
		ldr				r0, [rD, r0]
		ubfx			r1, rI, -2, 5					//Dn << 2
		ldr\regOp		r2, [rD, r1]					//value
		ubfxs			r0, r0, 0, 6					//bit count is limited to 64, because logic...
		beq				1f
		ubfxs			r0, r0, 0, \logBitSz
		bne				instrs\name\m68kSz\()do
		//rotate by a multiple of item size, but not zero, so set C correctly by jumping above
		clearNZVC
		b				instrs\name\m68kSz\()setFlags
	1:
		bl				no_shift_\m68kSz
.endm

.macro rotateRightCore bitSz							//r2 is val, r0 is amount, r1 is untouchable
	
	mov					r3, r2
	lsrs				r3, r0
	subs				r0, \bitSz
	negs				r0, r0
	lsls				r2, r0
	adds				r2, r3
.endm

.macro rotateLeftCore bitSz								//r2 is val, r0 is amount, r1 is untouchable
	
	mov					r3, r2
	lsls				r3, r0
	subs				r0, \bitSz
	negs				r0, r0
	lsrs				r2, r0
	adds				r2, r3
.endm

regRotate Ror, B, b,  8, 3, rotateRightCore, 7
regRotate Ror, W, h, 16, 4, rotateRightCore, 15
regRotate Ror, L,  , 32, 5, rotateRightCore, 31
regRotate Rol, B, b,  8, 3, rotateLeftCore, 0
regRotate Rol, W, h, 16, 4, rotateLeftCore, 0
regRotate Rol, L,  , 32, 5, rotateLeftCore, 0

instrsShiftMem:
	ubfx				r0, rI, 7, 4
	add					pc, r0
	nop
	b					instrsAsrWem
	b					instrsAslWem
	b					instrsLsrWem
	b					instrsLslWem
	b					instrsRoxrWmem
	b					instrsRoxlWmem
	b					instrsRorWmem
	b					instrsRolWmem

.macro regRotateX name, m68kSz, regOp, bitSz, core, outputBitInC

	instrs\name\m68kSz\()imm:
		ubfxs			r0, rI, 9, 3
		bne				1f
		movs			r0, #8
	1:													//r0 is #shiftAmt
		ubfx			r1, rI, -2, 5					//Dn << 2
		ldr\regOp		r2, [rD, r1]					//value
	
	instrs\name\m68kSz\()do:							//will not handle zero rotation
		getX			r3
		clearXNZVC
		\core			\bitSz
		str\regOp		r2, [rD, r1]					//save result
		bl				setRotateXflags\m68kSz
	
	instrs\name\m68kSz\()reg:
		ubfx			r0, rI, 9, 3					//Dm
		lsls			r0, #2
		ldr				r0, [rD, r0]
		ubfx			r1, rI, -2, 5					//Dn << 2
		ldr\regOp		r2, [rD, r1]					//value
		ubfx			r0, r0, 0, 6					//bit count is limited to 64, because logic...
		
		//modulo the requested rotation amount by the data length (9, 17, or 33)
		//due to how the instrs work, rotation that is 0 modulo data length *IS* same as rotation by zero
		
	1:
		cmp				r0, #1 + \bitSz
		blt				2f
		subs			r0, #1 + \bitSz
		bne				1b
	2:
		cmp				r0, #0
		bne				instrs\name\m68kSz\()do
		
		bl				no_shiftX_\m68kSz
.endm

.macro rotateRightXcore bitSz							//r2 is val, r0 is amount, r1 is untouchable
	
	mov				rI, r2
	lsrs			rI, r0								//shift bottom part right
	mrs				rT, apsr							//capture "C" into rT
	subs			r0, \bitSz
	negs			r0, r0								//r0 is now \bitSz - #shiftAmt
	lsls			r3, r0
	adds			rI, r3								//place input "X" properly in the result
	adds			r0, #1								//r0 is now \bitSz + 1 - #shiftAmt
	lsls			r2, r0
	adds			r2, rI								//add in top bits into result
.endm

.macro rotateLeftXcore bitSz							//r2 is val, r0 is amount, r1 is untouchable

	mov				rI, r2
	lsls			rI, r0								//shift top part left
	mrs				rT, apsr							//capture "C" into rT
	subs			r0, #1								//r0 now #shiftAmt - 1
	lsls			r3, r0
	adds			rI, r3								//place input "X" properly in the result
	subs			r0, \bitSz
	negs			r0, r0								//r0 is now \bitSz + 1 - #shiftAmt
	lsrs			r2, r0
	adds			r2, rI								//add in bottom bits into result
.endm

regRotateX Roxr, B, b,  8, rotateRightXcore
regRotateX Roxr, W, h, 16, rotateRightXcore
regRotateX Roxr, L,  , 32, rotateRightXcore
regRotateX Roxl, B, b,  8, rotateLeftXcore
regRotateX Roxl, W, h, 16, rotateLeftXcore
regRotateX Roxl, L,  , 32, rotateLeftXcore

//more shifts/rotates are below the branch dispatcher, for range reasons

//this clever dispatcher is similar to the one we use for Scc/DBcc. Why not use the same table?
//Because of how "BSR" is implemented (it is encoded where "Branch if false" would be, to complement the "Branch if true" that is BRA)

instrsB:
	ubfx				r0, rI, 8, 4
	ubfx				r1, rSR, -4, 8
	adds				r0, r1
	lsls				r0,  #2
	mov					r3, rPC							//displacement is from from here
	sbfxs				r1, rI, 0, 8					//i8 offset?
	bne					1f
	getInstrWordSx		r1, r2
1:
	adds				r1, r3							//destination
	add					pc, r0
	nop
	//we did all the work, so if branch is not taken, we just jump to next instruction
	//if it is taken, we need to copy r1 to rPC first.
	
	mov					rPC, r1
	nextInstr1											//NZVC = 0000, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 0000, cccc = F  (0001), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0000, cccc = HI (0010), reply = YES
	nextInstr1
	nop													//NZVC = 0000, cccc = LS (0011), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0000, cccc = CC (0100), reply = YES
	nextInstr1
	nop													//NZVC = 0000, cccc = CS (0101), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0000, cccc = N  (0110), reply = YES
	nextInstr1
	nop													//NZVC = 0000, cccc = EQ (0111), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0000, cccc = VC (1000), reply = YES
	nextInstr1
	nop													//NZVC = 0000, cccc = VS (1001), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0000, cccc = PL (1010), reply = YES
	nextInstr1
	nop													//NZVC = 0000, cccc = MI (1011), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0000, cccc = GE (1100), reply = YES
	nextInstr1
	nop													//NZVC = 0000, cccc = LT (1101), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0000, cccc = GT (1110), reply = YES
	nextInstr1
	nop													//NZVC = 0000, cccc = LE (1111), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0001, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 0001, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 0001, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0001, cccc = LS (0011), reply = YES
	nextInstr1
	nop													//NZVC = 0001, cccc = CC (0100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0001, cccc = CS (0101), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0001, cccc = NE (0110), reply = YES
	nextInstr1
	nop													//NZVC = 0001, cccc = EQ (0111), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0001, cccc = VC (1000), reply = YES
	nextInstr1
	nop													//NZVC = 0001, cccc = VS (1001), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0001, cccc = PL (1010), reply = YES
	nextInstr1
	nop													//NZVC = 0001, cccc = MI (1011), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0001, cccc = GE (1100), reply = YES
	nextInstr1
	nop													//NZVC = 0001, cccc = LT (1101), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0001, cccc = GT (1110), reply = YES
	nextInstr1
	nop													//NZVC = 0001, cccc = LE (1111), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0010, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 0010, cccc = F  (0001), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0010, cccc = HI (0010), reply = YES
	nextInstr1
	nop													//NZVC = 0010, cccc = LS (0011), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0010, cccc = CC (0100), reply = YES
	nextInstr1
	nop													//NZVC = 0010, cccc = CS (0101), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0010, cccc = NE (0110), reply = YES
	nextInstr1
	nop													//NZVC = 0010, cccc = EQ (0111), reply = NO
	nextInstr1
	nop													//NZVC = 0010, cccc = VC (1000), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0010, cccc = VS (1001), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0010, cccc = PL (1010), reply = YES
	nextInstr1
	nop													//NZVC = 0010, cccc = MI (1011), reply = NO
	nextInstr1
	nop													//NZVC = 0010, cccc = GE (1100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0010, cccc = LT (1101), reply = YES
	nextInstr1
	nop													//NZVC = 0010, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0010, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0011, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 0011, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 0011, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0011, cccc = LS (0011), reply = YES
	nextInstr1
	nop													//NZVC = 0011, cccc = CC (0100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0011, cccc = CS (0101), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0011, cccc = NE (0110), reply = YES
	nextInstr1
	nop													//NZVC = 0011, cccc = EQ (0111), reply = NO
	nextInstr1
	nop													//NZVC = 0011, cccc = VC (1000), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0011, cccc = VS (1001), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0011, cccc = PL (1010), reply = YES
	nextInstr1
	nop													//NZVC = 0011, cccc = MI (1011), reply = NO
	nextInstr1
	nop													//NZVC = 0011, cccc = GE (1100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0011, cccc = LT (1101), reply = YES
	nextInstr1
	nop													//NZVC = 0011, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0011, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0100, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 0100, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 0100, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0100, cccc = LS (0011), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0100, cccc = CC (0100), reply = YES
	nextInstr1
	nop													//NZVC = 0100, cccc = CS (0101), reply = NO
	nextInstr1
	nop													//NZVC = 0100, cccc = NE (0110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0100, cccc = EQ (0111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0100, cccc = VC (1000), reply = YES
	nextInstr1
	nop													//NZVC = 0100, cccc = VS (1001), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0100, cccc = PL (1010), reply = YES
	nextInstr1
	nop													//NZVC = 0100, cccc = MI (1011), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0100, cccc = GE (1100), reply = YES
	nextInstr1
	nop													//NZVC = 0100, cccc = LT (1101), reply = NO
	nextInstr1
	nop													//NZVC = 0100, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0100, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0101, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 0101, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 0101, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0101, cccc = LS (0011), reply = YES
	nextInstr1
	nop													//NZVC = 0101, cccc = CC (0100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0101, cccc = CS (0101), reply = YES
	nextInstr1
	nop													//NZVC = 0101, cccc = NE (0110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0101, cccc = EQ (0111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0101, cccc = VC (1000), reply = YES
	nextInstr1
	nop													//NZVC = 0101, cccc = VS (1001), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0101, cccc = PL (1010), reply = YES
	nextInstr1
	nop													//NZVC = 0101, cccc = MI (1011), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0101, cccc = GE (1100), reply = YES
	nextInstr1
	nop													//NZVC = 0101, cccc = LT (1101), reply = NO
	nextInstr1
	nop													//NZVC = 0101, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0101, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0110, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 0110, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 0110, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0110, cccc = LS (0011), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0110, cccc = CC (0100), reply = YES
	nextInstr1
	nop													//NZVC = 0110, cccc = CS (0101), reply = NO
	nextInstr1
	nop													//NZVC = 0110, cccc = NE (0110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0110, cccc = EQ (0111), reply = YES
	nextInstr1
	nop													//NZVC = 0110, cccc = VC (1000), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0110, cccc = VS (1001), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0110, cccc = PL (1010), reply = YES
	nextInstr1
	nop													//NZVC = 0110, cccc = MI (1011), reply = NO
	nextInstr1
	nop													//NZVC = 0110, cccc = GE (1100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0110, cccc = LT (1101), reply = YES
	nextInstr1
	nop													//NZVC = 0110, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0110, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0111, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 0111, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 0111, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0111, cccc = LS (0011), reply = YES
	nextInstr1
	nop													//NZVC = 0111, cccc = CC (0100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0111, cccc = CS (0101), reply = YES
	nextInstr1
	nop													//NZVC = 0111, cccc = NE (0110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0111, cccc = EQ (0111), reply = YES
	nextInstr1
	nop													//NZVC = 0111, cccc = VC (1000), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0111, cccc = VS (1001), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 0111, cccc = PL (1010), reply = YES
	nextInstr1
	nop													//NZVC = 0111, cccc = MI (1011), reply = NO
	nextInstr1
	nop													//NZVC = 0111, cccc = GE (1100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0111, cccc = LT (1101), reply = YES
	nextInstr1
	nop													//NZVC = 0111, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 0111, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1000, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 1000, cccc = F  (0001), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1000, cccc = HI (0010), reply = YES
	nextInstr1
	nop													//NZVC = 1000, cccc = LS (0011), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1000, cccc = CC (0100), reply = YES
	nextInstr1
	nop													//NZVC = 1000, cccc = CS (0101), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1000, cccc = NE (0110), reply = YES
	nextInstr1
	nop													//NZVC = 1000, cccc = EQ (0111), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1000, cccc = VC (1000), reply = YES
	nextInstr1
	nop													//NZVC = 1000, cccc = VS (1001), reply = NO
	nextInstr1
	nop													//NZVC = 1000, cccc = PL (1010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1000, cccc = MI (1011), reply = YES
	nextInstr1
	nop													//NZVC = 1000, cccc = GE (1100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1000, cccc = LT (1101), reply = YES
	nextInstr1
	nop													//NZVC = 1000, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1000, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1001, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 1001, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 1001, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1001, cccc = LS (0011), reply = YES
	nextInstr1
	nop													//NZVC = 1001, cccc = CC (0100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1001, cccc = CS (0101), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1001, cccc = NE (0110), reply = YES
	nextInstr1
	nop													//NZVC = 1001, cccc = EQ (0111), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1001, cccc = VC (1000), reply = YES
	nextInstr1
	nop													//NZVC = 1001, cccc = VS (1001), reply = NO
	nextInstr1
	nop													//NZVC = 1001, cccc = PL (1010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1001, cccc = MI (1011), reply = YES
	nextInstr1
	nop													//NZVC = 1001, cccc = GE (1100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1001, cccc = LT (1101), reply = YES
	nextInstr1
	nop													//NZVC = 1001, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1001, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1010, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 1010, cccc = F  (0001), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1010, cccc = HI (0010), reply = YES
	nextInstr1
	nop													//NZVC = 1010, cccc = LS (0011), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1010, cccc = CC (0100), reply = YES
	nextInstr1
	nop													//NZVC = 1010, cccc = CS (0101), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1010, cccc = NE (0110), reply = YES
	nextInstr1
	nop													//NZVC = 1010, cccc = EQ (0111), reply = NO
	nextInstr1
	nop													//NZVC = 1010, cccc = VC (1000), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1010, cccc = VS (1001), reply = YES
	nextInstr1
	nop													//NZVC = 1010, cccc = PL (1010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1010, cccc = MI (1011), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1010, cccc = GE (1100), reply = YES
	nextInstr1
	nop													//NZVC = 1010, cccc = LT (1101), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1010, cccc = GT (1110), reply = YES
	nextInstr1
	nop													//NZVC = 1010, cccc = LE (1111), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1011, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 1011, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 1011, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1011, cccc = LS (0011), reply = YES
	nextInstr1
	nop													//NZVC = 1011, cccc = CC (0100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1011, cccc = CS (0101), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1011, cccc = NE (0110), reply = YES
	nextInstr1
	nop													//NZVC = 1011, cccc = EQ (0111), reply = NO
	nextInstr1
	nop													//NZVC = 1011, cccc = VC (1000), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1011, cccc = VS (1001), reply = YES
	nextInstr1
	nop													//NZVC = 1011, cccc = PL (1010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1011, cccc = MI (1011), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1011, cccc = GE (1100), reply = YES
	nextInstr1
	nop													//NZVC = 1011, cccc = LT (1101), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1011, cccc = GT (1110), reply = YES
	nextInstr1
	nop													//NZVC = 1011, cccc = LE (1111), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1100, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 1100, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 1100, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1100, cccc = LS (0011), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1100, cccc = CC (0100), reply = YES
	nextInstr1
	nop													//NZVC = 1100, cccc = CS (0101), reply = NO
	nextInstr1
	nop													//NZVC = 1100, cccc = NE (0110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1100, cccc = EQ (0111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1100, cccc = VC (1000), reply = YES
	nextInstr1
	nop													//NZVC = 1100, cccc = VS (1001), reply = NO
	nextInstr1
	nop													//NZVC = 1100, cccc = PL (1010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1100, cccc = MI (1011), reply = YES
	nextInstr1
	nop													//NZVC = 1100, cccc = GE (1100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1100, cccc = LT (1101), reply = YES
	nextInstr1
	nop													//NZVC = 1100, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1100, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1101, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 1101, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 1101, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1101, cccc = LS (0011), reply = YES
	nextInstr1
	nop													//NZVC = 1101, cccc = CC (0100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1101, cccc = CS (0101), reply = YES
	nextInstr1
	nop													//NZVC = 1101, cccc = NE (0110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1101, cccc = EQ (0111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1101, cccc = VC (1000), reply = YES
	nextInstr1
	nop													//NZVC = 1101, cccc = VS (1001), reply = NO
	nextInstr1
	nop													//NZVC = 1101, cccc = PL (1010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1101, cccc = MI (1011), reply = YES
	nextInstr1
	nop													//NZVC = 1101, cccc = GE (1100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1101, cccc = LT (1101), reply = YES
	nextInstr1
	nop													//NZVC = 1101, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1101, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1110, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 1110, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 1110, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1110, cccc = LS (0011), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1110, cccc = CC (0100), reply = YES
	nextInstr1
	nop													//NZVC = 1110, cccc = CS (0101), reply = NO
	nextInstr1
	nop													//NZVC = 1110, cccc = NE (0110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1110, cccc = EQ (0111), reply = YES
	nextInstr1
	nop													//NZVC = 1110, cccc = VC (1000), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1110, cccc = VS (1001), reply = YES
	nextInstr1
	nop													//NZVC = 1110, cccc = PL (1010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1110, cccc = MI (1011), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1110, cccc = GE (1100), reply = YES
	nextInstr1
	nop													//NZVC = 1110, cccc = LT (1101), reply = NO
	nextInstr1
	nop													//NZVC = 1110, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1110, cccc = LE (1111), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1111, cccc = T  (0000), reply = YES
	b					instrsBSR
	nop													//NZVC = 1111, cccc = F  (0001), reply = NO
	nextInstr1
	nop													//NZVC = 1111, cccc = HI (0010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1111, cccc = LS (0011), reply = YES
	nextInstr1
	nop													//NZVC = 1111, cccc = CC (0100), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1111, cccc = CS (0101), reply = YES
	nextInstr1
	nop													//NZVC = 1111, cccc = NE (0110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1111, cccc = EQ (0111), reply = YES
	nextInstr1
	nop													//NZVC = 1111, cccc = VC (1000), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1111, cccc = VS (1001), reply = YES
	nextInstr1
	nop													//NZVC = 1111, cccc = PL (1010), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1111, cccc = MI (1011), reply = YES
	mov					rPC, r1
	nextInstr1											//NZVC == 1111, cccc = GE (1100), reply = YES
	nextInstr1
	nop													//NZVC = 1111, cccc = LT (1101), reply = NO
	nextInstr1
	nop													//NZVC = 1111, cccc = GT (1110), reply = NO
	mov					rPC, r1
	nextInstr1											//NZVC == 1111, cccc = LE (1111), reply = YES

instrsBSR:												//r1 = dst
	ldr					r2, [rA, #4 * 7]
	subs				r2, #4
	str					r2, [rA, #4 * 7]
	mov					r3, rPC
	write32				r3, r2, r0
	mov					rPC, r1
	nextInstr
	

//more shifts/rotates from above, moved here for range reasons
instrsAsrWem:
	ubfx				r0, rI, -1, 7					//ea
	bl					calcEaW
	movs				r3, r0
	movs				r0, #1
	clearXNZVC
	read16s				r2, r3, r1
	asrs				r2, #1
	write16				r2, r3, r1
	bl					setShiftFlagsAsrW

instrsLsrWem:
	ubfx				r0, rI, -1, 7					//ea
	bl					calcEaW
	movs				r3, r0
	movs				r0, #1
	clearXNZVC
	read16				r2, r3, r1
	lsrs				r2, #1
	write16				r2, r3, r1
	bl					setShiftFlagsLsrW

instrsLslWem:
	ubfx				r0, rI, -1, 7					//ea
	bl					calcEaW
	movs				r3, r0
	movs				r0, #1
	clearXNZVC
	read16				r2, r3, r1
	lsls				r2, #1
	write16				r2, r3, r1
	bl					setShiftFlagsLslW

instrsAslWem:
	ubfx				r0, rI, -1, 7					//ea
	bl					calcEaW
	movs				r3, r0
	movs				r0, #1
	clearXNZVC
	read16				rI, r3, r1
	lsls				r2, rI, #1
	write16				r2, r3, r1
	bl					setShiftFlagsAslW

instrsRorWmem:
	ubfx				r0, rI, -1, 7					//ea
	bl					calcEaW
	clearNZVC
	read16				r2, r0, r1
	lsrs				r3, r2, #1
	lsls				r2, r2, #15
	adds				r2, r3
	write16				r2, r0, r1
	bl					instrsRorWsetFlags

instrsRolWmem:
	ubfx				r0, rI, -1, 7					//ea
	bl					calcEaW
	clearNZVC
	read16				r2, r0, r1
	lsls				r3, r2, #1
	lsrs				r2, r2, #15
	adds				r2, r3
	write16				r2, r0, r1
	bl					instrsRolWsetFlags

instrsRoxrWmem:
	ubfx				r0, rI, -1, 7					//ea
	bl					calcEaW
	getX				r3
	clearXNZVC
	read16				r2, r0, r1
	lsls				r3, #16
	adds				r2, r3
	lsrs				r2, #1
	write16				r2, r0, r1
	bl					setShiftFlagsGenericW

instrsRoxlWmem:
	ubfx				r0, rI, -1, 7					//ea
	bl					calcEaW

	getX				r3
	clearXNZVC
	read16				r2, r0, r1
	adds				r2, r2
	adds				r2, r3
	write16				r2, r0, r1
	lsls				r2, #16
	bl					setShiftFlagsGenericL


div0:
	storeState			r0, r1
_LOC_BL_DivZero:
	.hword 0
	.hword 0
	//return not exected

divOverflow:
	movs				r0, SR_V
	orrs				rSR, r0
	nextInstr

instrsDivu:
	ubfx				r0, rI, -1, 7
	bl					readEaW
	uxth				r1, r0
	cmp					r1, #0
	beq					div0
	ubfx				rI, rI, 9, 3
	lsls				rI, #2							//Dn
	adds				rI, rD							//makes life easier later
	ldr					r0, [rI]
	bl					doUdiv							//r0 / r1 -> q(r0), r(r1)
	lsrs				r2, r0, #16						//verify it fits
	bne					divOverflow
	strh				r0, [rI, #0]
	strh				r1, [rI, #2]
	setCcLogical		r0, 16, 1

instrsDivs:
	ubfx				r0, rI, -1, 7
	bl					readEaW
	sxth				r1, r0
	cmp					r1, #0
	beq					div0
	ubfx				rI, rI, 9, 3
	lsls				rI, #2							//Dn
	adds				rI, rD							//makes life easier later
	ldr					r0, [rI]
	bl					doSdiv							//r0 / r1 ->q(r0), r(r1)
	sxth				r2, r0							//verify it fits
	cmp					r2, r0
	bne					divOverflow
	strh				r0, [rI, #0]
	strh				r1, [rI, #2]
	setCcLogical		r0, 16, 1

logicalOp Or, B, b, orrs, 8
logicalOp Or, W, h, orrs, 16
logicalOp Or, L,  , orrs, 32

bcdOp Sbcd, doBcdSub

instrsOrDivSbcd:
	ubfx				r0, rI, 2, 7
	add					pc, r0
	nop
	b					instrsOrBreg
	b					instrsSubOtherIllegal			//adress regs do not like neing ANDed (even though it could be of use for aligning things)
	b					instrsOrBreg
	b					instrsOrBreg
	b					instrsOrBreg
	b					instrsOrBreg
	b					instrsOrBreg
	b					instrsOrBreg
	b					instrsOrWreg
	b					instrsSubOtherIllegal
	b					instrsOrWreg
	b					instrsOrWreg
	b					instrsOrWreg
	b					instrsOrWreg
	b					instrsOrWreg
	b					instrsOrWreg
	b					instrsOrLreg
	b					instrsSubOtherIllegal
	b					instrsOrLreg
	b					instrsOrLreg
	b					instrsOrLreg
	b					instrsOrLreg
	b					instrsOrLreg
	b					instrsOrLreg
	b					instrsDivu
	b					instrsSubOtherIllegal
	b					instrsDivu
	b					instrsDivu
	b					instrsDivu
	b					instrsDivu
	b					instrsDivu
	b					instrsDivu
	b					instrsSbcdReg
	b					instrsSbcdMem
	b					instrsOrBea
	b					instrsOrBea
	b					instrsOrBea
	b					instrsOrBea
	b					instrsOrBea
	b					instrsOrBea
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsOrWea
	b					instrsOrWea
	b					instrsOrWea
	b					instrsOrWea
	b					instrsOrWea
	b					instrsOrWea
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsOrLea
	b					instrsOrLea
	b					instrsOrLea
	b					instrsOrLea
	b					instrsOrLea
	b					instrsOrLea
	b					instrsDivs
	b					instrsSubOtherIllegal
	b					instrsDivs
	b					instrsDivs
	b					instrsDivs
	b					instrsDivs
	b					instrsDivs
	b					instrsDivs

.macro leaOp reg
	
	instrsLeaA\reg:
		ubfx			r0, rI, -1, 7
		bl				calcEaGeneric
		str				r0, [rA, #4 * \reg]
		nextInstr
.endm

leaOp 0
leaOp 1
leaOp 2
leaOp 3
leaOp 4
leaOp 5
leaOp 6
leaOp 7

.macro negxOp m68kSz, regOp, bitSz, fCall
	
	instrsNegx\m68kSz:
		ubfx			r0, rI, -1, 7
		cmp				r0, #0x10
		bge				instrsNegx\m68kSz\()ea
		lsls			rI, r0, #1						//Dm << 2
		ldr\regOp		r0, [rD, rI]					//Dm.\m68kSz value
		bl				\fCall
		str\regOp		r0, [rD, rI]					//store Dn.\m68kSz value
		nextInstr
	
	instrsNegx\m68kSz\()ea:
		bl				calcEa\m68kSz
		movs			rI, r0
		read\bitSz		r0, rI, r1
		bl				\fCall
		write\bitSz		r0, rI, r1
		nextInstr
.endm

negxOp B, b, 8,  performNEGX8
negxOp W, h, 16, performNEGX16
negxOp L,  , 32, performNEGX32


.macro clrOp m68kSz, regOp, bitSz
	
	instrsClr\m68kSz:
		ubfx			r0, rI, -1, 7
		cmp				r0, #0x10
		bge				instrsClr\m68kSz\()ea
		lsls			r0, #1							//Dm << 2
		movs			r1, #0
		str\regOp		r1, [rD, r0]					//store Dn.\m68kSz value
		clearNZVC
		adds			rSR, SR_Z	
		nextInstr
	
	instrsClr\m68kSz\()ea:
		bl				calcEa\m68kSz
		movs			r1, #0
		write\bitSz		r1, r0, r2
		clearNZVC
		adds			rSR, SR_Z
		nextInstr
.endm

clrOp B, b, 8
clrOp W, h, 16
clrOp L,  , 32

.macro notOp m68kSz, regOp, bitSz
	
	instrsNot\m68kSz:
		ubfx			r0, rI, -1, 7
		cmp				r0, #0x10
		bge				instrsNot\m68kSz\()ea
		lsls			r0, #1							//Dm << 2
		ldr\regOp		r1, [rD, r0]					//Dm.\m68kSz value
		mvns			r1, r1
		str\regOp		r1, [rD, r0]					//store Dn.\m68kSz value
		setCcLogical	r1, \bitSz, 1
	
	instrsNot\m68kSz\()ea:
		bl				calcEa\m68kSz
		read\bitSz		r1, r0, r2
		mvns			r1, r1
		write\bitSz		r1, r0, r2
		setCcLogical	r1, \bitSz, 1
.endm

notOp B, b, 8
notOp W, h, 16
notOp L,  , 32

.macro tstOp m68kSz, regOp, bitSz
	
	instrsTst\m68kSz:
		ubfx			r0, rI, -1, 7
		cmp				r0, #0x10
		bge				instrsTst\m68kSz\()ea
		lsls			r0, #1							//Dm << 2
		ldr\regOp		r0, [rD, r0]					//Dm.\m68kSz value
		setCcLogical	r0, \bitSz, 1
	
	instrsTst\m68kSz\()ea:
		bl				calcEa\m68kSz
		read\bitSz		r1, r0, r2
		setCcLogical	r1, \bitSz, 1
.endm

tstOp B, b, 8
tstOp W, h, 16
tstOp L,  , 32

instrsChkW:
	ubfx				r0, rI, -1, 7
	bl					readEaW
	sxth				r0, r0
	ubfx				r1, rI, 9, 3
	lsls				r1, #2
	ldrsh				r1, [rD, r1]					//Dn.W
	cmp					r1, #0
	blt					1f								//failed
	cmp					r1, r0
	bgt					1f								//failed
	nextInstr
1:
	bl					instrsUnimpl

instrsTas:
	ubfx				r0, rI, -1, 7
	cmp					r0, #0x10
	bge					instrsTasEa
	lsls				r0, #1							//Dm << 2
	ldrb				r1, [rD, r0]					//Dm.\m68kSz value
	movs				r2, #0x80
	orrs				r2, r1
	strb				r2, [rD, r0]					//store Dn.\m68kSz value
	setCcLogical		r1, 8, 1

instrsTasEa:
	bl					calcEaB
	read8				r1, r0, r2
	movs				r2, #0x80
	orrs				r2, r1
	write8				r2, r0, r3
	setCcLogical		r1, 8, 1

instrsNbcd:
	ubfx				r0, rI, -1, 7
	cmp					r0, #0x10
	bge					instrsNbcdEa
	lsls				rI, r0, #1						//Dm << 2
	ldrb				r0, [rD, rI]					//Dm.\m68kSz value
	bl					doBcdNeg
	strb				r0, [rD, rI]					//store Dn.\m68kSz value
	nextInstr

instrsNbcdEa:
	bl					calcEaB
	movs				rI, r0
	read8				r0, rI, r1
	bl					doBcdNeg
	write8				r0, rI, r1
	nextInstr

.macro negOp m68kSz, regOp, bitSz
	
	instrsNeg\m68kSz:
		ubfx			r0, rI, -1, 7
		cmp				r0, #0x10
		bge				instrsNeg\m68kSz\()ea
		lsls			r0, #1							//Dm << 2
		ldr\regOp		r1, [rD, r0]					//Dm.\m68kSz value
		negs			r2, r1
		str\regOp		r2, [rD, r0]					//store Dn.\m68kSz value
		setCcNeg		r1, r2, \bitSz, ccTab_Neg
		nextInstr

	instrsNeg\m68kSz\()ea:
		bl				calcEa\m68kSz
		read\bitSz		r1, r0, r3
		negs			r2, r1
		write\bitSz		r2, r0, r3
		setCcNeg		r1, r2, \bitSz, ccTab_Neg
		nextInstr
.endm

negOp B, b, 8
negOp W, h, 16
negOp L,  , 32

.balign 4
ccTab_Neg:												//arm (NZCV) top 4 bits to m68k bottom 5 (XNZVC) tab for subtraction (they have same meanings except C is inverted)
	.byte 0b10001		//arm 0000
	.byte 0b10011		//arm 0001
	.byte 0b00000		//arm 0010
	.byte 0b00010		//arm 0011
	.byte 0b10101		//arm 0100
	.byte 0b10111		//arm 0101
	.byte 0b00100		//arm 0110
	.byte 0b00110		//arm 0111
	.byte 0b11001		//arm 1000
	.byte 0b11011		//arm 1001
	.byte 0b01000		//arm 1010
	.byte 0b01010		//arm 1011
	.byte 0b11101		//arm 1100
	.byte 0b11111		//arm 1101
	.byte 0b01100		//arm 1110
	.byte 0b01110		//arm 1111
	
instrsSwap:
	lsls				r0, #1
	ldr					r1, [rD, r0]
	movs				r2, #16
	rors				r1, r2
	str					r1, [rD, r0]
	setCcLogical		r1, 32, 1						//yes, SWAP sets flags, because of course it does...

instrsPea:
	ubfx				r0, rI, -1, 7
	cmp					r0, #0x10
	blt					instrsSwap
	bl					calcEaGeneric
	ldr					r1, [rA, #4 * 7]
	subs				r1, #4
	str					r1, [rA, #4 * 7]
	write32				r0, r1, r2
	nextInstr

instrsOther:
	ubfx				r0, rI, 5, 7
	add					pc, r0
	nop
	b					instrsNegxB
	b					instrsNegxW
	b					instrsNegxL
	b					instrsReadSR
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsChkW
	b					instrsLeaA0
	b					instrsClrB
	b					instrsClrW
	b					instrsClrL
	b					instrsOtherUnimpl				//move from CCR does not exist in 68000
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsChkW
	b					instrsLeaA1
	b					instrsNegB
	b					instrsNegW
	b					instrsNegL
	b					instrsWriteSRorCCR
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsChkW
	b					instrsLeaA2
	b					instrsNotB
	b					instrsNotW
	b					instrsNotL
	b					instrsWriteSRorCCR
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsChkW
	b					instrsLeaA3
	b					instrsNbcd
	b					instrsPea						//or SWAP
	b					instrsMovemStoreW				//or EXT.W
	b					instrsMovemStoreL				//or EXT.L
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsChkW
	b					instrsLeaA4
	b					instrsTstB
	b					instrsTstW
	b					instrsTstL
	b					instrsTas
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsChkW
	b					instrsLeaA5
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsMovemLoadW
	b					instrsMovemLoadL
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsChkW
	b					instrsLeaA6
	b					instrsSubOtherIllegal
	b					instrsMoreOthers
	b					instrsJsr
	b					instrsJmp
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsChkW
	b					instrsLeaA7

instrsReadSR:
	ubfx				r0, rI, -1, 7
	cmp					r0, #0x10
	bge					instrsReadSRea
	lsls				r0, #1							//Dm << 2
	strh				rSR, [rD, r0]
	nextInstr

instrsReadSRea:
	bl					calcEaW
	write16				rSR, r0, r1
	nextInstr

instrsWriteSRorCCR:
	ubfx				r0, rI, -1, 7
	bl					readEaW
	writeCCR			r0, r1
	nextInstr

//implementations of LINK and UNLK may seem to be suboptimal, but they are coded this way due to the need to handle
//such insnity as someone attempting to use them with "A7" register.
instrsLink:
	ubfx				r0, rI, -2, 5
	ldr					r2, [rA, #4 * 7]				//SP - 4 -> SP
	subs				r2, #4
	str					r2, [rA, #4 * 7]
	ldr					r1, [rA, r0]					//An -> (SP)
	write32				r1, r2, r3
	str					r2, [rA, r0]					//SP -> An
	getInstrWordSx		r0, r3							//SP + disp -> SP
	adds				r2, r0
	str					r2, [rA, #4 * 7]
	nextInstr

instrsUnlk:
	ubfx				r0, rI, -2, 5
	ldr					r1, [rA, r0]					//An -> SP
	str					r1, [rA, #4 * 7]
	read32				r2, r1, r3						//(SP) -> An
	str					r2, [rA, r0]
	ldr					r1, [rA, #4 * 7]				//SP = SP + 4 (re-read SP in case An == A7)
	adds				r1, #4
	str					r1, [rA, #4 * 7]
	nextInstr

instrsSubOtherIllegal:
	bl					instrsIllegal

instrsOtherUnimpl:
	bl					instrsUnimpl

instrsTrap0:											//breakpoint
	storeState			r0, r1
	mov					rD, r0							//preserved reg - a good place to keep state
_LOC_BL_Bkpt:
	.hword 0
	.hword 0
	loadState			rD, r1
	nextInstr

instrsTrap8:											//debug break
	storeState			r0, r1
	mov					rD, r0							//preserved reg - a good place to keep state
_LOC_BL_DbgBreak:
	.hword 0
	.hword 0
	loadState			rD, r1
	nextInstr

instrsTrap15:											//syscall
	getInstrWordUx		r1, r2							//this is expected by the os trap dispatcher
	storeState			r0, r2
	mov					rD, r0							//preserved reg - a good place to keep state
_LOC_BL_CallOsCall:
	.hword 0
	.hword 0
	loadState			rD, r1
	nextInstr

instrsTrapv:
	lsrs				r0, rSR, #2
	bcs					instrsOtherUnimpl
	nextInstr

instrsRtr:
	ldr					r1, [rA, #4 * 7]
	adds				r0, r1, #2
	str					r0, [rA, #4 * 7]
	read16				r0, r1, r2
	writeCCR			r0, r1
	//fallthrough

instrsRts:
	ldr					r1, [rA, #4 * 7]
	adds				r0, r1, #4
	str					r0, [rA, #4 * 7]
	read32				r0, r1, r2
	movs				r1, r0
	adds				r1, -EXIT_EMULATION_ADDR
	beq					instrsEmulExit
	mov					rPC, r0
	nextInstr

instrsJsr:
	ubfx				r0, rI, -1, 7
	bl					calcEaGeneric
	ldr					r1, [rA, #4 * 7]
	subs				r1, #4
	str					r1, [rA, #4 * 7]
	mov					r2, rPC
	mov					rPC, r0
	write32				r2, r1, r3
	nextInstr

instrsJmp:
	ubfx				r0, rI, -1, 7
	bl					calcEaGeneric
	movs				r1, r0
	adds				r1, -EXIT_EMULATION_ADDR
	beq					instrsEmulExit
	mov					rPC, r0
	nextInstr

instrsEmulExit:
	bl					m68kEmuExit

.macro movemStoreOp m68kSize, regOpExt, regOp, bitsSz
	
	instrsExt\m68kSize:
		lsls			r0, #1
		ldr\regOpExt	r1, [rD, r0]
		str\regOp		r1, [rD, r0]
		setCcLogical	r1, \bitsSz, 1
	
	instrsMovemStore\m68kSize:
		ubfx			r0, rI, -1, 7					//get <ea>
		lsrs			r1, r0, #3						//"mmm?"
		add				pc, r1
		nop
		b				instrsExt\m68kSize
		b				instrsSubOtherIllegal
		b				instrsMovemStoreForward\m68kSize
		b				instrsSubOtherIllegal
		b				instrsMovemStoreBackward\m68kSize
		b				instrsMovemStoreForward\m68kSize
		b				instrsMovemStoreForward\m68kSize
		b				instrsSubOtherIllegal
	
	instrsMovemStoreBackward\m68kSize:					//-(An) mode
		ubfx			r0, rI, -2, 5
		adds			rI, rA, r0						//&An
		ldr				r0, [rI]						//base addr
		getInstrWordUx	r1, r2							//mask
		movs			r2, #8 * 4
		adds			r2, rA							//point past A7
		b				3f
	2:
		ldr				r3, [r2]
		subs			r0, \bitsSz / 8
		write\bitsSz	r3, r0, r3
	3:
		subs			r2, #4
		lsrs			r1, #1
		bcs				2b
		bne				3b
		//writeback
		str				r0, [rI]
		nextInstr
	
	instrsMovemStoreForward\m68kSize:					//all other modes where we write things going forwards
		getInstrWordUx	rI, r1							//mask
		bl				calcEaGeneric					//calc the destination (no writeback this time)
		subs			r2, rD, #4						//pont to D-1 (4 bytes prior to D0)
		b				3f
	2:
		ldr				r1, [r2]
		write\bitsSz	r1, r0, r1
		adds			r0, \bitsSz / 8
	3:
		adds			r2, #4
		lsrs			rI, #1
		bcs				2b
		bne				3b
		nextInstr
.endm

movemStoreOp W, sb, h, 16
movemStoreOp L, sh,  , 32

.macro movemLoadOp m68kSize, regOp, readOp, bitsSz
	
	instrsMovemLoad\m68kSize:
		ubfx			r0, rI, -1, 7					//get <ea>
		lsrs			r1, r0, #3						//"mmm?"
		add				pc, r1
		nop
		b				instrsSubOtherIllegal
		b				instrsSubOtherIllegal
		b				instrsMovemStoreNormal\m68kSize
		b				instrsMovemStoreWbak\m68kSize
		b				instrsSubOtherIllegal
		b				instrsMovemStoreNormal\m68kSize
		b				instrsMovemStoreNormal\m68kSize
		b				instrsSubOtherIllegal
		
	instrsMovemStoreNormal\m68kSize:
		getInstrWordUx	rI, r2							//mask
		bl				calcEaGeneric					//r0 = ea
		movs			r1, #0							//no wbak
		b				instrsMovemStoreCommon\m68kSize
	
	instrsMovemStoreWbak\m68kSize:
		ubfx			r1, rI, -2, 5
		adds			r1, rA							//&An (where we'll wbak to)
		ldr				r0, [r1]						//r0 = ea
		getInstrWordUx	rI, r2							//mask
		//fallthrough
	
	instrsMovemStoreCommon\m68kSize:					//r0 = ea, r1 = wbakTo(or NULL), rI = mask
		mov				rT, r1							//writeback addr or zero
		subs			r2, rD, #4						//destination ptr = &D-1 (4 bytes before D0)
		b				3f
	2:
		\readOp			r3, r0, r1
		adds			r0, #\bitsSz / 8
		str				r3, [r2]
		
	3:
		adds			r2, #4
		lsrs			rI, #1
		bcs				2b
		bne				3b
	
		mov				r1, rT							//wbak?
		cmp				r1, #0
		beq				1f
		str				r0, [r1]
	1:
		nextInstr
.endm

movemLoadOp W, h, read16s, 16
movemLoadOp L,  , read32,  32

instrsMoreOthers:
	ubfx				r0, rI, -1, 7
	add					pc, r0
	nop
	b					instrsTrap0
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsTrap8
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsTrap15
	b					instrsLink
	b					instrsLink
	b					instrsLink
	b					instrsLink
	b					instrsLink
	b					instrsLink
	b					instrsLink
	b					instrsLink
	b					instrsUnlk
	b					instrsUnlk
	b					instrsUnlk
	b					instrsUnlk
	b					instrsUnlk
	b					instrsUnlk
	b					instrsUnlk
	b					instrsUnlk
	b					instrsOtherUnimpl				//move to USP
	b					instrsOtherUnimpl				//move to USP
	b					instrsOtherUnimpl				//move to USP
	b					instrsOtherUnimpl				//move to USP
	b					instrsOtherUnimpl				//move to USP
	b					instrsOtherUnimpl				//move to USP
	b					instrsOtherUnimpl				//move to USP
	b					instrsOtherUnimpl				//move to USP
	b					instrsOtherUnimpl				//move from USP
	b					instrsOtherUnimpl				//move from USP
	b					instrsOtherUnimpl				//move from USP
	b					instrsOtherUnimpl				//move from USP
	b					instrsOtherUnimpl				//move from USP
	b					instrsOtherUnimpl				//move from USP
	b					instrsOtherUnimpl				//move from USP
	b					instrsOtherUnimpl				//move from USP
	b					instrsOtherUnimpl				//RESET
	nextInstr1											//NOP
	b					instrsOtherUnimpl				//STOP
	b					instrsOtherUnimpl				//RTE
	b					instrsOtherUnimpl				//RTD does not exist on 68000
	b					instrsRts
	b					instrsTrapv
	b					instrsRtr						//I've never seen this used, but gotta support it
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsOtherUnimpl				//move from control register
	b					instrsOtherUnimpl				//move to control register
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal
	b					instrsSubOtherIllegal

#ifdef REPORT_EACH_INSTR
	.macro oscall name, table, call
		\name:
			push		{r0, r1}
			mov			r0, r9
			subs		r0, \table
			ldr			r0, [r0]
			ldr			r1, =\call
			ldr			r0, [r0, r1]
			mov			r12, r0
			pop			{r0, r1}
			bx			r12
		.ltorg
	.endm
	
	oscall StrIToH, 8, 0x794
	oscall StrLen 8, 0x798
	oscall StrPrintF, 8, 0x7B4
	oscall HALDbgMessage, 4, 0x008
	oscall FtrGet, 8, 0x424
	oscall FtrSet, 8, 0x43C
	
	reportRegs: //buf, regsPtr, name
		push			{r0, r4-r6, lr}
		movs			r6, r1
		movs			r1, ' '
		movs			r3, '='
		strb			r1, [r0, #0]
		strb			r1, [r0, #1]
		strb			r2, [r0, #2]
		strb			r3, [r0, #3]
		adds			r5, r0, #4
		movs			r4, #0
	1:
		ldmia			r6!, {r1}
		mov				r0, r5
		bl				StrIToH
		movs			r0, ' '
		strb			r0, [r5, #8]
		adds			r5, #9
		adds			r4, #1
		cmp				r4, #8
		bne				1b
		movs			r0, 0
		strb			r0, [r5]
		pop				{r0}
		bl				HALDbgMessage
		pop				{r4-r6, pc}	
	
	
	.macro reportFlag val
	
		strb			r2, [r0]
		lsrs			r1, #1
		bcc				1f
		movs			r3, \val
		strb			r3, [r0]
	1:
		adds			r0, #1
		
	.endm
	
	report:	//(pc, val, regs, sr)
		push			{r0-r4, lr}
		sub				sp, #128
		mov				r0, sp
		adr				r1, reportStr1
		ldr				r2, [sp, #128 + 4 * 0]
		ldr				r3, [sp, #128 + 4 * 1]
		bl				StrPrintF
		mov				r0, sp
		bl				StrLen
		add				r0, sp
		ldr				r1, [sp, #128 + 4 * 3]
		movs			r2, '-'
		reportFlag		'C'
		reportFlag		'V'
		reportFlag		'Z'
		reportFlag		'N'
		reportFlag		'X'
		movs			r3, #0
		strb			r3, [r0]
		mov				r0, sp
		bl				HALDbgMessage
		mov				r0, sp
		ldr				r1, [sp, #128 + 4 * 2]
		movs			r2, 'D'
		bl				reportRegs
		mov				r0, sp
		ldr				r1, [sp, #128 + 4 * 2]
		adds			r1, #4 * 8
		movs			r2, 'A'
		bl				reportRegs
		
		
		mov				r0, sp
		movs			r1, ' '
		movs			r2, 'S'
		movs			r3, '='
		strb			r1, [r0, #0]
		strb			r1, [r0, #1]
		strb			r2, [r0, #2]
		strb			r3, [r0, #3]
		adds			r0, #4
		ldr				r1, [sp, #128 + 4 * 2]
		ldr				r1, [r1, #4 * (8 + 7)]	//get A7
		movs			r2, #32
		adr				r4, reportHextab
	1:
		ldrb			r3, [r1]
		lsrs			r3, #4
		ldrb			r3, [r4, r3]
		strb			r3, [r0, #0]
		
		ldrb			r3, [r1]
		ubfx			r3, r3, 0, 4
		ldrb			r3, [r4, r3]
		strb			r3, [r0, #1]
		
		movs			r3, ' '
		strb			r3, [r0, #2]
		adds			r0, #3
		
		adds			r1, #1
		
		subs			r2, #1
		bne				1b
		
		movs			r3, #0
		strb			r3, [r0]
		
		mov				r0, sp
		bl				HALDbgMessage
		
		
		add				sp, #128
		pop				{r0-r4, pc}
		
		
	.balign 4
	reportStr1:
		.ascii "[%08lx] = %04x, CCR=\0"
	
	.balign 4
	reportHextab:
		.ascii "0123456789abcdef"
		
	.align 2
#endif
