#include "trivialEcc.h"


static uint_fast8_t eccCalcWordCode(uint_fast16_t src)
{
	uint_fast8_t code;
	
	code = src ^ (src >> 8);
	code ^= code >> 4;
	code &= 0x0f;
	
	src ^= src >> 2;
	src ^= src >> 1;
	
	code |= (src & 0x0001) << 4;
	code |= (src & 0x0010) << 1;
	code |= (src & 0x0100) >> 2;
	code |= (src & 0x1000) >> 5;
	
	return code;
}

static void eccEncodeWord(uint8_t *dst, uint_fast16_t src)
{
	*dst++ = src >> 8;
	*dst++ = src;
	*dst++ = eccCalcWordCode(src);
}

static int32_t eccDecodeWord(const uint8_t *src)	//negative on error
{
	static const uint8_t popcnt4[] = {0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4};
	static const uint8_t ctz4[] = {0xff, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0};
	uint8_t calcedCode, rxedCode, diffCode, diffSide, diffBottom;
	uint_fast16_t val;
	
	val = *src++;
	val <<= 8;
	val |= *src++;
	rxedCode = *src++;
	calcedCode = eccCalcWordCode(val);
	diffCode = rxedCode ^ calcedCode;
	diffSide = diffCode >> 4;
	diffBottom = diffCode & 0x0f;
	
	if (!diffCode)	//no errors?
		return (uint32_t)val;
	else if (popcnt4[diffSide] + popcnt4[diffBottom] == 1)		//error in the ecc itself - no issuese
		return (uint32_t)val;
	if (popcnt4[diffSide] == 1 && popcnt4[diffBottom] == 1) {	//correctable error in the data
		
		//side tells us which nibble needs correction, bottom tells us the correction needed
		//convert them to indices (nibble index and bit-in-nibble index)
		diffSide = ctz4[diffSide];
		diffBottom = ctz4[diffBottom];
		
		//correct
		val ^= 1 << ((diffSide * 4) + diffBottom);
		
		return val;
	}
	else														//uncorrectable
		return -1;
}

static void eccInterleave(uint8_t *dst, uint32_t dstStride, const uint8_t *src, uint32_t srcStride)	//interleave 8 bytes bitwise
{
	uint_fast8_t bit, i, v = 0;
	
	for (bit = 0; bit < 8; bit++) {
		for (i = 0; i < 8; i++) {
			v = (v >> 1) | (((src[i * srcStride] >> bit) & 1) << 7);
		}
		*dst = v;
		dst += dstStride;
	}
}

static uint_fast16_t eccCrc16(const uint8_t *src, uint_fast8_t len)
{
	uint_fast16_t crc = 0xf00f;
	
	while(len--)
		crc = (crc << 1) ^ ((crc & 0x8000) ? 0xbabe : 0) ^ *src++;
	
	return (uint16_t)crc;
}

void eccEncodeBlock(uint8_t *dst, const uint8_t *src)		//a block is precisely 16 bytes, and encodes to 39 bytes
{
	uint_fast16_t crc = eccCrc16(src, 16);
	uint8_t tmp[24];
	uint_fast8_t i;
	
	//encode each word
	for (i = 0; i < 8; i++, src += 2)
		eccEncodeWord(dst + i * 3, (((uint32_t)src[1]) << 8) + src[0]);
	
	//interleave
	for (i = 0; i < 3; i++)
		eccInterleave(tmp + 8 * i, 1, dst + i, 3);
	
	//encode each produced word
	src = tmp;
	for (i = 0; i < 12; i++, src += 2)
		eccEncodeWord(dst + i * 3, (((uint32_t)src[1]) << 8) + src[0]);
	
	//encode the crc
	eccEncodeWord(dst + i * 3, crc);
}

bool eccDecodeBlock(uint8_t *dst, const uint8_t *src)		//an encoded block is precisely 39 bytes, and encodes to 16 bytes
{
	uint8_t tmp1[24], tmp2[24];
	uint_fast16_t crcGot;
	uint_fast8_t i;
	int32_t v;
	
	//decode each outer word
	for (i = 0; i < 12; i++, src += 3) {
		v = eccDecodeWord(src);
		if (v < 0) {	//if decode fails, pass data through in case next layer can manage it
			tmp1[i * 2 + 1] = src[0];
			tmp1[i * 2 + 0] = src[1];
		}
		tmp1[i * 2 + 0] = v;
		tmp1[i * 2 + 1] = v >> 8;
	}
	
	//decode crc
	v = eccDecodeWord(src);
	if (v >= 0)
		crcGot = v;
	else {	//if decode fails, pass data through
		crcGot = *src++;
		crcGot <<= 8;
		crcGot += *src++;
	}
	
	//de-interleave
	for (i = 0; i < 3; i++)
		eccInterleave(tmp2 + i, 3, tmp1 + 8 * i, 1);
	
	//decode each inner word
	src = tmp2;
	for (i = 0; i < 8; i++, src += 3) {
		v = eccDecodeWord(src);
		if (v < 0) {	//if decode fails, pass data through in hopes that it is correct
			dst[i * 2 + 0] = src[0];
			dst[i * 2 + 1] = src[1];
		}
		dst[i * 2 + 0] = v;
		dst[i * 2 + 1] = v >> 8;
	}
	
	return eccCrc16(dst, 16) == crcGot;
}