#include <boot.h>
#include "stm32h7a3xx.h"
#include <kal.h>
#include <string.h>
#include "printf.h"
#include "nand.h"
#include "../dal/dal.h"

//PC15 = nCS (GPIO)
//MISO = PA6 (8 = SPI6.MISO)
//MOSI = PB5 (8 = SPI6.MOSI)
//SCLK = PC12 (5 = SPI6.SCLK)


static void nandPrvPowerOn(void)
{
	RCC->APB4ENR |= RCC_APB4ENR_SPI6EN;
	RCC->APB4RSTR = RCC_APB4RSTR_SPI6RST;
	RCC->APB4RSTR = 0;
	asm volatile("dsb sy");
	
	//this is super sensitive to sequencing and changes, fuck ST
	SPI6->CFG1 = SPI_CFG1_MBR_0 | SPI_CFG1_DSIZE_0 | SPI_CFG1_DSIZE_1 | SPI_CFG1_DSIZE_2;	//8 bits per xfer ,slow
	SPI6->CFG2 = SPI_CFG2_SSM;
	SPI6->CR2 = 0;	//indefinite size
	SPI6->CR1 = SPI_CR1_SSI;
	SPI6->CFG2 |= SPI_CFG2_MASTER;
	SPI6->CR1 |= SPI_CR1_SPE;
	SPI6->CR1 |= SPI_CR1_CSTART;
	
	logt("SR=0x%08x\n", SPI6->SR);
}

void nandPrvPowerOff(void)
{
	SPI6->CR1 = 0;	//off
	RCC->APB4ENR &=~ RCC_APB4ENR_SPI6EN;
}

static uint8_t spiByte(uint8_t val)
{
	*(volatile uint8_t*)&SPI6->TXDR = val;
	while (!(SPI6->SR & SPI_SR_RXP));
	val = *(volatile uint8_t*)&SPI6->RXDR;
	while (!(SPI6->SR & SPI_SR_TXC));
	
	return val;
}

static void nandPrvSelect(void)
{
	GPIOC->BSRR = 1 << (16 + 15);
}

static void nandPrvDeselect(void)
{
	GPIOC->BSRR = 1 << 15;
	(void)spiByte(0);	//delay
}

static void spiPrvDataRead(uint8_t *dst, uint32_t len)
{
	uint32_t i;
	
	if (len < 8) {
		
		while(len--)
			*dst++ = spiByte(0);
	}
	else {
	
		for (i = 0; i < 4; i++)
			*(volatile uint8_t*)&SPI6->TXDR = 0;
		for (i = 0; i < len - 4; i++) {
			*(volatile uint8_t*)&SPI6->TXDR = 0;
			while (!(SPI6->SR & SPI_SR_RXP));
			*dst++ = *(volatile uint8_t*)&SPI6->RXDR;
		}
		for (; i < len; i++) {
			while (!(SPI6->SR & SPI_SR_RXP));
			*dst++ = *(volatile uint8_t*)&SPI6->RXDR;
		}
	}
}

static void spiPrvDataWrite(const uint8_t *src, uint32_t len)
{
	uint32_t i;
	
	if (len < 8) {
		
		while (len--)
			(void)spiByte(*src++);
	}
	else {
		
		for (i = 0; i < 4; i++)
			*(volatile uint8_t*)&SPI6->TXDR = *src++;
		for (i = 0; i < len - 4; i++) {
			while (!(SPI6->SR & SPI_SR_RXP));	//means tx space avail too
			(void)*(volatile uint8_t*)&SPI6->RXDR;
			*(volatile uint8_t*)&SPI6->TXDR = *src++;
		}
		for (i = 0; i < 4; i++) {
			while (!(SPI6->SR & SPI_SR_RXP));
			(void)*(volatile uint8_t*)&SPI6->RXDR;
		}
		while (!(SPI6->SR & SPI_SR_TXC));
	}
}

void nandFillSpec(struct NandSpec *spec)
{
	static const struct NandByteRange eccedSpareBytes[] = {
		{
			.offset = 0x04,
			.len = 0x04,
		},
		{
			.offset = 0x14,
			.len = 0x04,
		},
		{
			.offset = 0x24,
			.len = 0x04,
		},
		{
			.offset = 0x34,
			.len = 0x04,
		},
		{},
	};
	static const struct NandByteRange uneccedSpareBytes[] = {
		{
			.offset = 0x01,
			.len = 0x03,
		},
		{
			.offset = 0x10,
			.len = 0x04,
		},
		{
			.offset = 0x20,
			.len = 0x04,
		},
		{
			.offset = 0x30,
			.len = 0x04,
		},
		{},
	};
	static const struct NandSpec specTemplate = {
		.nandBlocks = 1024,
		.pagesPerBlockShift = 6,
		.dataBytesPerPage = 2048,
		.spareAreaSize = 64,
		.badBlockMarker = {.offset = 0, .len = 1, },
		.maxBadBlocks = 40,
		.maxFactoryBadBlocks = 2,
		.speccedErasesPerBlock = 1000,	//100K 
	};
	
	*spec = specTemplate;
	spec->eccedSpareBytes = eccedSpareBytes;
	spec->uneccedSpareBytes = uneccedSpareBytes;
}

static void nandPrvEcc(bool enabled)
{
	nandPrvSelect();
	spiByte(0x1f);	//write status
	spiByte(0xb0);	//status 2
	spiByte(enabled ? 0x18 : 0x08);
	nandPrvDeselect();
}

bool nandInit(void)
{
	uint_fast8_t i;
	uint8_t id[3];
	
	nandPrvPowerOn();
	
	//reset
	nandPrvSelect();
	spiByte(0xff);	//write enable
	nandPrvDeselect();
	
	nandPrvSelect();
	spiByte(0x9f);
	spiByte(0x00);
	for (i = 0; i < 3; i++)
		id[i] = spiByte(0x00);
	nandPrvDeselect();
	
	logt("NAND ID %02x %02x %02x\n", id[0], id[1], id[2]);
	
	if (id[0] != 0xef || id[1] != 0xaa || id[2] != 0x21)
		return false;
	
	//unprotect device
	nandPrvSelect();
	spiByte(0x1f);	//write status
	spiByte(0xa0);	//status 1
	spiByte(0x00);
	nandPrvDeselect();
	
	//enable ECC and buffer mode
	nandPrvSelect();
	spiByte(0x1f);	//write status
	spiByte(0xb0);	//status 2
	spiByte(0x18);
	nandPrvDeselect();
	
	return true;
}

void nandDeinit(void)
{
	nandPrvPowerOff();
}

void nandSleep(void)
{
	nandDeinit();
}

void nandWake(void)
{
	(void)nandInit();
}

static int16_t nandPrvWait(uint32_t msIsh)	//we skimp on divide, thus "ish". return -1 on error, status byte on success
{
	uint64_t maxTime = repalmDalGetClockRate(TimerClockRate) / 1000 * msIsh;
	uint64_t startTime = repalmDalGetTimerVal();
	uint_fast8_t sta;
	
	do {
		
		nandPrvSelect();
		spiByte(0x0f);	//read status
		spiByte(0xc0);	//status 3
		sta = spiByte(0x00);
		nandPrvDeselect();
		
		if (repalmDalGetTimerVal() - startTime > maxTime)
			return -1;
		
	} while (sta & 1);
	
	return (uint16_t)(uint8_t)sta;
}

bool nandBlockErase(uint32_t firstPageNum)
{
	nandPrvSelect();
	spiByte(0x06);	//write enable
	nandPrvDeselect();
	
	nandPrvSelect();
	spiByte(0xd8);	//write enable
	spiByte(0x00);	//dummy
	spiByte(firstPageNum >> 8);	//page addr hi
	spiByte(firstPageNum);	//page addr lo
	nandPrvDeselect();
	
	return !(nandPrvWait(50) & 0x8004);	//erase is 10ms, success = not negative and not E-FAIL
}

bool nandReadUid(uint8_t uid[static 16])
{
	bool ret = false;
	uint_fast8_t i, j;
	
	//enable otp acces using OTP-E bit in status 2
	nandPrvSelect();
	spiByte(0x1f);	//write status
	spiByte(0xb0);	//status 2
	spiByte(0x58);
	nandPrvDeselect();
	
	nandPrvSelect();
	spiByte(0x13);	//page data read
	spiByte(0x00);	//dummy
	spiByte(0);		//page addr hi (0 = uid)
	spiByte(0);		//page addr lo (0 = uid)
	nandPrvDeselect();
	
	if (nandPrvWait(1) >= 0) {	//specced to 60 us
		
		nandPrvSelect();
		spiByte(0x03);	//read
		spiByte(0x00);	//column addr hi
		spiByte(0x00);	//column addr lo
		spiByte(0x00);	//dummy
		spiPrvDataRead(uid, 16);
		
		//verify
		ret = true;
		for (j = 0; j < 10; j++) {
			
			for (i = 0; i < 16; i++) {
				if ((spiByte(0x00) ^ uid[i]) != 0xff)
					ret = false;
			}
			
			for (i = 0; i < 16; i++) {
				if (spiByte(0x00) != uid[i])
					ret = false;
			}
		}
		
		nandPrvDeselect();
	}
	
	//disable otp acces using OTP-E bit in status 2
	nandPrvSelect();
	spiByte(0x1f);	//write status
	spiByte(0xb0);	//status 2
	spiByte(0x18);
	nandPrvDeselect();
	
	return ret;
}

int_fast8_t nandPageReadPartial(uint32_t pageNum, void *data, uint32_t ofst, uint32_t len, bool withEcc)
{
	static const int8_t ret[] = {0, 99, -1, -1};	//we cannot estimate error amount, so we err on the side of "we corrected a lot"
	int16_t sta;
	
	nandPrvEcc(withEcc);
	
	nandPrvSelect();
	spiByte(0x13);				//page data read
	spiByte(0x00);				//dummy
	spiByte(pageNum >> 8);		//page addr hi (0 = uid)
	spiByte(pageNum);			//page addr lo (0 = uid)
	nandPrvDeselect();
	
	sta = nandPrvWait(1);		//specced to 60 us
	if (sta < 0)
		return -1;
	
	nandPrvSelect();
	spiByte(0x03);				//read
	spiByte(ofst >> 8);			//column addr hi
	spiByte(ofst);				//column addr lo
	spiByte(0x00);				//dummy
	spiPrvDataRead(data, len);
	nandPrvDeselect();
	
	return ret[(sta >> 4) & 3];
}

bool nandPageWritePartial(uint32_t pageNum, const void *data, uint32_t ofst, uint32_t len, bool withEcc)
{
	nandPrvEcc(withEcc);
	
	//write enable (required)
	nandPrvSelect();
	spiByte(0x06);	//write enable
	nandPrvDeselect();
	
	//load program data
	nandPrvSelect();
	spiByte(0x02);			//load program data
	spiByte(ofst >> 8);		//column addr hi
	spiByte(ofst);			//column addr lo
	spiPrvDataWrite(data, len);
	nandPrvDeselect();
	
	//do program
	nandPrvSelect();
	spiByte(0x10);			//program execute
	spiByte(0x00);			//dummy
	spiByte(pageNum >> 8);	//page addr hi
	spiByte(pageNum);		//page addr lo
	nandPrvDeselect();
	
	return !(nandPrvWait(2) & 0x8008);	//page program is 700us, success = not negative and not P-FAIL
}

