#include "skin.h"
#include "printf.h"
#include "common.h"
#include "palmosInternal.h"
#include "../dal/halDisplay.h"



static void* sbarPrvGetResWithDefault(uint32_t type, uint32_t id, void *defaultRet)
{
	MemHandle mh = DmGet1Resource(type, id);
	void* ret;
	
	return (mh && (ret = MemHandleLock(mh))) ? ret : defaultRet;
}

static void* sbarPrvGetRes(uint32_t type, uint32_t id)
{
	return sbarPrvGetResWithDefault(type, id, NULL);
}

static bool sbarPrvSkinLoadState(struct SbarSkinState *state, uint32_t base, bool noBitmapsAllowed)
{
	state->aslk = sbarPrvGetRes('aslk', base + SKIN_RESID_OFST_ASLK);
	state->nor = sbarPrvGetRes('abmp', base + SKIN_RESID_OFST_NORMAL);
	state->act = sbarPrvGetResWithDefault('abmp', base + SKIN_RESID_OFST_ACTIVE, state->nor);
	
	return state->aslk && (noBitmapsAllowed || (state->nor && state->act));
}

static void sbarPrvSkinUnloadRes(void* ptr, bool isRamBased)
{
	if (ptr) {
		
		if (isRamBased) {
			MemHandle mh = MemPtrRecoverHandle(ptr);
			MemHandleUnlock(mh);
			DmReleaseResource(mh);
		}
		else {
			DmReleaseResource((MemHandle)ptr);	//yes, you read that right
		}
	}
}

static void sbarPrvSkinUnloadState(struct SbarSkinState *state, bool isRamBased)
{
	if (state->act == state->nor)
		state->act = NULL;
	
	sbarPrvSkinUnloadRes(state->aslk, isRamBased);
	sbarPrvSkinUnloadRes(state->act, isRamBased);
	sbarPrvSkinUnloadRes(state->nor, isRamBased);
}

static bool sbarPrvSkinLoadPinsButtonState(struct SbarSkinDiaButtonState *btn, uint32_t base)
{
	btn->ena = sbarPrvGetRes('abmp', base + SKIN_RESID_OFST_DIA_CTL_ENA);
	btn->dis = sbarPrvGetResWithDefault('abmp', base + SKIN_RESID_OFST_DIA_CTL_DIS, btn->ena);
	btn->act = sbarPrvGetResWithDefault('abmp', base + SKIN_RESID_OFST_DIA_CTL_ACT, btn->ena);
	
	return btn->ena && btn->dis && btn->act;
}

static void sbarPrvSkinUnloadPinsButtonState(struct SbarSkinDiaButtonState *btn, bool isRamBased)
{
	if (btn->dis == btn->ena)
		btn->dis = NULL;
	if (btn->act == btn->ena)
		btn->act = NULL;
	
	sbarPrvSkinUnloadRes(btn->ena, isRamBased);
	sbarPrvSkinUnloadRes(btn->dis, isRamBased);
	sbarPrvSkinUnloadRes(btn->act, isRamBased);
}

static bool sbarPrvSkinLoadInteger(uint32_t resId, uint32_t *valP)
{
	MemHandle mh;
	
	mh = DmGet1Resource('aint', resId);
	if (!mh)
		return false;
	
	//on ARM, resources are always 4-byte aligned. use that
	*valP = *(uint32_t*)MemHandleLock(mh);
	MemHandleUnlock(mh);
	DmReleaseResource(mh);
	
	return true;
}

bool sbarPrvSkinTryLoadOne(struct SbarSkin* skin, LocalID LID, bool isRamBased)
{
	uint32_t i, dispW, dispH, dispDensity;
	LocalID skinLID;
	bool ret = false;
	DmOpenRef db;
	MemHandle mh;
	Err e;
	
	//zero it out just in case
	MemSet(skin, sizeof(*skin), 0);
	
	//get display info
	e = HALDisplayGetAttributes(hwrDispHorizontal, &dispW);
	if (e) {
		loge("Cannot get screen %s\n", "width");
		return false;
	}
	e = HALDisplayGetAttributes(hwrDispVertical, &dispH);
	if (e) {
		loge("Cannot get screen %s\n", "height");
		return false;
	}
	dispDensity = BmpGetDensity(WinGetBitmap(WinGetDisplayWindow()));
	
	//open the skin databse
	db = DmOpenDatabase(LID, dmModeReadOnly);
	if (!db) {
		loge("Cannot find DIA skin\n");
		return false;
	}
	
	mh = DmGet1Resource(SKIN_RES_SCREEN_REQ_TYPE, SKIN_RESID_BASE);
	if (mh) {
		const struct SbarSkinScreenReq *sr = MemHandleLock(mh);
		for (i = 0; i < MemHandleSize(mh) / sizeof(struct SbarSkinScreenReq); i++) {
			if (sr[i].width == dispW && sr[i].height == dispH && sr[i].density == dispDensity) {
				
				ret = true;
				break;
			}
		}
		MemHandleUnlock(mh);
		DmReleaseResource(mh);
	
		ret = ret && sbarPrvSkinLoadState(&skin->states[SBAR_STATE_UP], SKIN_RESID_BASE + SKIN_RESID_ADDED_DIA, false);
		ret = ret && sbarPrvSkinLoadState(&skin->states[SBAR_STATE_JUST_SBAR], SKIN_RESID_BASE + SKIN_RESID_ADDED_SBAR, false);
		ret = ret && sbarPrvSkinLoadState(&skin->states[SBAR_STATE_FULLSCREEN], SKIN_RESID_BASE + SKIN_RESID_ADDED_FULL, true);
		
		if (!ret) {
			loge("required resource not found\n");
			return false;
		}
		
		ret = ret && sbarPrvSkinLoadPinsButtonState(&skin->diaDn, SKIN_RESID_BASE + SKIN_RESID_ADDED_DIA_CTL_DN);
		ret = ret && sbarPrvSkinLoadPinsButtonState(&skin->diaUp, SKIN_RESID_BASE + SKIN_RESID_ADDED_DIA_CTL_UP);
		
		ret = ret && sbarPrvSkinLoadInteger(SKIN_RESID_BASE + SKIN_RESID_ADDED_INK + SKIN_RESID_OFST_BW, &skin->inkBW);
		ret = ret && sbarPrvSkinLoadInteger(SKIN_RESID_BASE + SKIN_RESID_ADDED_INK + SKIN_RESID_OFST_COLOR, &skin->inkColor);
		
		ret = ret && sbarPrvSkinLoadInteger(SKIN_RESID_BASE + SKIN_RESID_ADDED_BG + SKIN_RESID_OFST_BW, &skin->backBW);
		ret = ret && sbarPrvSkinLoadInteger(SKIN_RESID_BASE + SKIN_RESID_ADDED_BG + SKIN_RESID_OFST_COLOR, &skin->backColor);
		
		ret = ret && sbarPrvSkinLoadInteger(SKIN_RESID_BASE + SKIN_RESID_ADDED_FG + SKIN_RESID_OFST_BW, &skin->foreBW);
		ret = ret && sbarPrvSkinLoadInteger(SKIN_RESID_BASE + SKIN_RESID_ADDED_FG + SKIN_RESID_OFST_COLOR, &skin->foreColor);
		
		ret = ret && sbarPrvSkinLoadInteger(SKIN_RESID_BASE + SKIN_RESID_ADDED_ACT + SKIN_RESID_OFST_BW, &skin->foreActBW);
		ret = ret && sbarPrvSkinLoadInteger(SKIN_RESID_BASE + SKIN_RESID_ADDED_ACT + SKIN_RESID_OFST_COLOR, &skin->foreActColor);
		
		
		if (ret) {
		
			DmOpenDatabaseInfo(db, &skinLID, NULL, NULL, NULL);
			DmDatabaseProtect(skinLID, true);
			DmCloseDatabase(db);
			return true;
		}
	}
//fail: cleanup

	DmCloseDatabase(db);
	
	for (i = 0; i < SBAR_STATE_NUMBER; i++)
		sbarPrvSkinUnloadState(&skin->states[i], isRamBased);
	
	sbarPrvSkinUnloadPinsButtonState(&skin->diaDn, isRamBased);
	sbarPrvSkinUnloadPinsButtonState(&skin->diaUp, isRamBased);
	
	return false;
}

bool sbarPrvSkinLoad(struct SbarSkin* skin)
{
	DmSearchStateType ss;
	uint32_t i;
	
	//first all in ram them all in rom
	for (i = 0; i < 2; i++) {
		
		bool first = true;
		LocalID LID;
		
		while (errNone == DmGetNextDatabaseByTypeCreator(first, &ss, SKIN_DB_TYPE, MODULE_CRID, false, &LID)) {
			
			bool isRamDb = errNone == DmDatabaseProtect(LID, true);	//easiest way to test if a DB is in rom
			first = false;

			if (isRamDb)		//undo the protection if we applied it
				DmDatabaseProtect(LID, false);
			
			if ((isRamDb && i != 0) || (!isRamDb && i != 1))
				continue;
			
			if (sbarPrvSkinTryLoadOne(skin, LID, isRamDb))
				return true;
		}
	}
	
	return false;
}
