#include <string.h>
#include "zodiac.h"
#include "printf.h"
#include "boot.h"
#include "kal.h"



struct TwDeviceEntry {
	
	struct TwDeviceEntry *next, *prev;
	const char *name;
	TwDeviceInitF initF;
	void *data;
};

static struct TwDeviceEntry *mDeviceEntries;
static struct TwDevice *mDevices;
static uint32_t mTwDeviceMutex;

#define TW_DEVICE_ERROR_DOESNT_EXIST		0x0002
#define TW_DEVICE_ERROR_INVALID_HANDLE		0x0009
#define TW_DEVICE_ERROR_OUT_OF_MEMORY		0x000c
#define TW_DEVICE_ERROR_INVALID_PTR			0x000e
#define TW_DEVICE_ERROR_ALREADY_EXISTS		0x0011
#define TW_DEVICE_ERROR_INVAL_REQUEST		0x0016




Err DALEXPORT impl_TwDeviceOpen(struct TwDevice **handleOutP, const char *name, const char *mode)
{
	struct TwDevice *dev = NULL;
	struct TwDeviceEntry *ent;
	Err e = errNone;
	
	logt("%s\n", __func__);
	
	if (!handleOutP)
		return TW_DEVICE_ERROR_INVALID_PTR;
	
	KALMutexReserve(mTwDeviceMutex, -1);
	
	for (ent = mDeviceEntries; ent && strcmp(name, ent->name); ent = ent->next);
	if (!ent)
		e = TW_DEVICE_ERROR_DOESNT_EXIST;
	else {
	
		dev = MemChunkNew(0, sizeof(*dev), 0x200);
		if (!ent)
			e = TW_DEVICE_ERROR_OUT_OF_MEMORY;
		else {
			memset(dev, 0, sizeof(*dev));
			
			dev->initF = ent->initF;
			dev->data[0] = ent->data;
			dev->entry = ent;
			
			e = dev->initF(dev, name, mode);
			if (e != errNone) {
				
				MemChunkFree(dev);
				dev = NULL;
			}
			else {
				
				dev->next = mDevices;
				if (dev->next)
					dev->next->prev = dev;
				mDevices = dev;
			}
		}
	}
	KALMutexRelease(mTwDeviceMutex);
	
	*handleOutP = dev;
	
	return e;
}

Err DALEXPORT impl_TwDeviceClose(struct TwDevice *handle)
{
	Err e = errNone;
	
	logt("%s\n", __func__);
	
	if (!handle || !handle->initF)
		return TW_DEVICE_ERROR_INVALID_HANDLE;
	
	if (handle->closeF)
		e = handle->closeF(handle);
	
	KALMutexReserve(mTwDeviceMutex, -1);
	
	if (handle->next)
		handle->next->prev = handle->prev;
	if (handle->prev)
		handle->prev->next = handle->next;
	else
		mDevices = handle->next;
		
	KALMutexRelease(mTwDeviceMutex);
	
	return e;
}

Err DALEXPORT impl_TwDeviceRead(struct TwDevice *handle, void* buf, int32_t lenP)
{
	logt("%s\n", __func__);
	
	if (!handle || !handle->initF)
		return TW_DEVICE_ERROR_INVALID_HANDLE;
	
	if (!handle->readF)
		return TW_DEVICE_ERROR_INVAL_REQUEST;
	
	return handle->readF(handle, buf, lenP);
}

Err DALEXPORT impl_TwDeviceWrite(struct TwDevice *handle, const void* buf, int32_t lenP)
{
	logt("%s\n", __func__);
	
	if (!handle || !handle->initF)
		return TW_DEVICE_ERROR_INVALID_HANDLE;
	
	if (!handle->writeF)
		return TW_DEVICE_ERROR_INVAL_REQUEST;
	
	return handle->writeF(handle, buf, lenP);
}

Err DALEXPORT impl_TwDeviceGetProperty(struct TwDevice *handle, int32_t prop, void* buf, int32_t *lenP)
{
	logt("%s\n", __func__);
	
	if (!handle || !handle->initF)
		return TW_DEVICE_ERROR_INVALID_HANDLE;
	
	if (!handle->getPropF)
		return TW_DEVICE_ERROR_INVAL_REQUEST;
	
	return handle->getPropF(handle, prop, buf, lenP);
}

Err DALEXPORT impl_TwDeviceSetProperty(struct TwDevice *handle, int32_t prop, const void* buf, int32_t lenP)
{
	logt("%s\n", __func__);
	
	if (!handle || !handle->initF)
		return TW_DEVICE_ERROR_INVALID_HANDLE;
	
	if (!handle->setPropF)
		return TW_DEVICE_ERROR_INVAL_REQUEST;
	
	return handle->setPropF(handle, prop, buf, lenP);
}

Err DALEXPORT impl_TwDeviceControl(struct TwDevice *handle, int32_t cmd, void* buf, int32_t lenP)
{
	logt("%s\n", __func__);
	
	if (!handle || !handle->initF)
		return TW_DEVICE_ERROR_INVALID_HANDLE;
	
	if (!handle->controlF)
		return TW_DEVICE_ERROR_INVAL_REQUEST;
	
	return handle->controlF(handle, cmd, buf, lenP);
}

Err DALEXPORT impl_TwDeviceRegister(const char *name, TwDeviceInitF initF, void* data)		//name pointer remains valid post call
{
	struct TwDeviceEntry *ent;
	Err e = errNone;
	
	logt("%s\n", __func__);
	
	KALMutexReserve(mTwDeviceMutex, -1);
	
	for (ent = mDeviceEntries; ent && strcmp(name, ent->name); ent = ent->next);
	if (ent)
		e = TW_DEVICE_ERROR_ALREADY_EXISTS;
	else {
	
		ent = MemChunkNew(0, sizeof(*ent), 0x200);
		if (!ent)
			e = TW_DEVICE_ERROR_OUT_OF_MEMORY;
		else {
			memset(ent, 0, sizeof(*ent));
			ent->name = name;
			ent->initF = initF;
			ent->data = data;
			ent->next = mDeviceEntries;
			if (ent->next)
				ent->next->prev = ent;
			mDeviceEntries = ent;
		}
	}
	KALMutexRelease(mTwDeviceMutex);
	
	return e;
}

Err DALEXPORT impl_TwDeviceUnregister(const char *name, TwDeviceInitF init, void* data)
{
	struct TwDeviceEntry *ent;
	struct TwDevice *dev;
	Err e = errNone;
	
	logt("%s\n", __func__);
	
	KALMutexReserve(mTwDeviceMutex, -1);
	
	for (ent = mDeviceEntries; ent && strcmp(name, ent->name); ent = ent->next);
	if (!ent)
		e = TW_DEVICE_ERROR_DOESNT_EXIST;
	else {
		//unlink
		if (ent->next)
			ent->next->prev = ent->prev;
		if (ent->prev)
			ent->prev->next = ent->next;
		else
			mDeviceEntries = ent->next;
		
		for (dev = mDevices; dev; dev = dev->next) {
			
			if (dev->entry == ent)
				fatal("Device '%s' still opened at unregister time\n", name);
		}
		
		MemChunkFree(ent);
	}
	KALMutexRelease(mTwDeviceMutex);
	
	return e;
}

void zodDevicesInit(void)
{
	if (errNone != KALMutexCreate(&mTwDeviceMutex, CREATE_4CC('_','T','w','D')))
		fatal("Failed to init TwDevice mtx\n");
}
