#include "printf.h"
#include "boot.h"
#include "sony.h"
#include "pace.h"
#include "dal.h"
#include <FeatureMgr.h>
#include <string.h>

#define SONY_FLAG_IS_SONY_KNOWN		0x80

#define NUM_SLEEP_HANDLERS	2
#define NUM_WAKE_HANDLERS	2

#define SONY_PORT_STATE_HOLD_IS_ON			0x01
#define SONY_PORT_STATE_HEADPHONES_IN		0x02
#define SONY_PORT_STATE_DEVICE_OPEN			0x04	//for nx and such. for some tablets always on, for others always off
#define SONY_PORT_STATE_SCREEN_FLIPPED		0x08	//for nx and such, for some tablets always on, for others always off
#define SONY_PORT_STATE_CAMERA_OPEN			0x20

#define SONY_LED_NO_POWER					1
#define SONY_LED_NO_MS						2
#define SONY_LED_NO_WIFI					4
#define SONY_LED_NO_RECORDING				8
#define SONY_LED_NO_BLUETOOTH				16
#define SONY_LED_NO_KEYBOARD				64

typedef void (*SonySleepWakeHandler)(void* param);

struct SonySleepWakeHandlerPackage {
	SonySleepWakeHandler cbk;
	void *data;
};

static struct SonySleepWakeHandlerPackage mSleepHandlers[NUM_SLEEP_HANDLERS];
static struct SonySleepWakeHandlerPackage mWakeHandlers[NUM_WAKE_HANDLERS];
static uint32_t mPortState = SONY_PORT_STATE_DEVICE_OPEN | SONY_PORT_STATE_SCREEN_FLIPPED; //must be uint32_t as we export a pointer to his
static uint8_t mRotState = 0;


static uint32_t sonyDalPatchSonyBatteryCurrentValues(uint16_t *centivoltsP, uint16_t *percentP);



static uint8_t isSonyGetFlag(void)
{
	static uint8_t flag = 0;	//0x80 bit is set if we know. lower bit is boolean
	
	if (!(flag & SONY_FLAG_IS_SONY_KNOWN)) {		//do not know? find out
		
		DmSearchStateType ss;
		char name[33] = {0,};
		LocalID LID;
		Err e;
		
		if (dalGetInitStage() >= DAL_INIT_STAGE_MEM_AND_DM) {
			
			flag = 0;
		
			e = DmGetNextDatabaseByTypeCreator(true, &ss, CREATE_4CC('l','i','b','r'), CREATE_4CC('S','d','a','l'), false, &LID);
			if (e == dmErrCantFind)
				logi("Non-sony device detected\n");
			else if (e)
				fatal("Unexpected error from DmGetNextDatabaseByTypeCreator(): 0x%04x\n", e);
			else {
				e = DmDatabaseInfo(LID, name, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
				if (e)
					fatal("Unexpected error from DmDatabaseInfo(): 0x%04x\n", e);
				if (strcmp(name, "SonyDAL"))
					logw("Unexpexted lib with SonyDAL type but name '%s'. Assuming non-sony device\n", name);
				else {
					logi("Sony device detected. Installing sony features\n");
					flag = 1;
				}	
			}
			flag |= SONY_FLAG_IS_SONY_KNOWN;
		}
	}
	
	return flag;
}

bool isSony(void)
{
	uint8_t sonyFlag = isSonyGetFlag();
	
	if (sonyFlag & SONY_FLAG_IS_SONY_KNOWN)
		return !!(sonyFlag &~ SONY_FLAG_IS_SONY_KNOWN);
	
	fatal("%s was called but the answer is not known\n", __func__);
	return false;
}

bool isSony_safe(bool whatToAssume)
{
	uint8_t sonyFlag = isSonyGetFlag();

	if (sonyFlag & SONY_FLAG_IS_SONY_KNOWN)
		return !!(sonyFlag &~ SONY_FLAG_IS_SONY_KNOWN);
	
	return whatToAssume;
}

static Err sonyFtr20031func(uint32_t selector, void* data)	//some sort of an ioctl		//0x99C4 @ TH55 DAL
{
	Err ret = errNone;
	
	switch (selector) {
		case 0x00:
		case 0x01:
		case 0x02:
		case 0x03:
		case 0x04:
		case 0x05:
		case 0x06:
		case 0x07:
		case 0x08:
		case 0x09:
		case 0x0a:
		case 0x0b:
		case 0x0c:
		case 0x0d:
		case 0x0e:
		case 0x0f:
		case 0x10:
		case 0x11:
		case 0x12:
		case 0x13:
			fatal("{G,S}etting weird sony ioctl value %u from 0x%08lx\n", selector, __builtin_return_address(0));
			break;
		
		default:
			ret = sysErrParamErr;
			break;
	}
	
	return ret;
}

static uint16_t sonyDalRead(uint32_t sel)	//likely reads an ADC given by index		//0x4DE8 @ TH55 DAL
{
	uint16_t ret = errNone;
	
	fatal("Dony dal adc read %u\n", sel);
	
	//func disables interrupts while it runs
	
	switch (sel) {
		case 0:
		case 1:
		case 2:
		case 3:
		case 4:
		case 5:
		case 6:
			fatal("{G,S}etting weird sony hw LL ioctl value %u from 0x%08lx\n", sel, __builtin_return_address(0));
			break;
		
		default:
			break;
	}
	
	return ret;
}


static uint32_t patchSonyDalMainEntrypt(uint16_t cmd, void *cmdPBP, uint16_t flags)
{
	logt("SonyDAL main entrypt called (cmd=%d)\n", cmd);
	
	return errNone;
}

static Err (*mOldFtrGet)(uint32_t crid, uint16_t id, uint32_t *valP);
static Err pFtrGet(uint32_t crid, uint16_t id, uint32_t *valP)
{
	Err e = mOldFtrGet(crid, id, valP);
	
	if (crid == CREATE_4CC('S','h','A','l'))
		loge("FtrGet("FMT_4CC", %u, &(0x%08x)) -> 0x%04x, from 0x%08lx\n", CNV_4CC(crid), id, *valP, e, __builtin_return_address(0));
	
	return e;
}

void sonyHalSetInitStage(uint32_t stage)
{
	uint32_t sonyDalModuleRef = 0, bootRefNum = 0;
	void (*sonyDalMain)(uint32_t stage);
	Err e;
		
	if (stage < DAL_INIT_STAGE_NO_UI || !isSony())
		return;
	
	if (stage == DAL_INIT_STAGE_NO_UI) {
		
		logt("Loading SonyDAL\n");
		e = SysLoadModule(CREATE_4CC('l','i','b','r'), CREATE_4CC('S','d','a','l'), 0, 0, &sonyDalModuleRef);
		if (e || !sonyDalModuleRef)
			fatal("SonyDAL failed to load: 0x%04x\n", e);
		
		e = SysLoadModule(CREATE_4CC('r','s','r','c'), CREATE_4CC('p','s','y','s'), 0, 0, &bootRefNum);
		if (e || !bootRefNum)
			fatal("Cannot find Boot\n");
		
		e = SysPatchEntry(bootRefNum, 0x109, &pFtrGet, (void**)&mOldFtrGet);
		if (e || !mOldFtrGet)
			fatal("Cannot patch FtrGet\n");
		
		loge("FtrGet patched\n");
	}
	else {
		logt("Finding SonyDAL\n");
		e = SysFindModule(CREATE_4CC('l','i','b','r'), CREATE_4CC('S','d','a','l'), 0, 0, &sonyDalModuleRef);
		if (e || !sonyDalModuleRef)
			fatal("SonyDAL failed to be found: 0x%04x\n", e);
	}
	
	logt("Finding SonyDalMain()\n");
	e = SysGetEntryAddresses(sonyDalModuleRef, 1, 1, (void**)&sonyDalMain);
	if (e || !sonyDalMain)
		fatal("Failed to find SonyDalMain: 0x%04x\n", e);
	
	if (stage == DAL_INIT_STAGE_ALL_UP) {

		logt("Setting sony features\n");
		FtrSet(CREATE_4CC('S','h','A','l'), 20020, (uint32_t)1);			//sony code points this to some globals, see TH55 DAL @ 0x87EC
		FtrSet(CREATE_4CC('S','h','A','l'), 20022, (uint32_t)&sonyDalPatchSonyBatteryCurrentValues);
		FtrSet(CREATE_4CC('S','h','A','l'), 20024, (uint32_t)&sonyDalRead);
		FtrSet(CREATE_4CC('S','h','A','l'), 20027, (uint32_t)3);			//sony code points this to some globals, see NX60 DAL @ 0x15914
		FtrSet(CREATE_4CC('S','h','A','l'), 20031, (uint32_t)&sonyFtr20031func);
	}
	
	logt("Calling SonyDalMain()\n");
	sonyDalMain(stage);
	logt("SonyDalMain() returned\n");
}


static void sonyDalPatchSonyGetGlobalsPtr(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static uint32_t sonyDalPatchSonyMain(uint32_t stage)
{
	DmOpenRef ref;
	
	logt("SonyDAL main(stage=%d)\n", stage);
	
	switch(stage) {
		case 2:
			//for new devices
			ref = DmOpenDatabaseByTypeCreator(CREATE_4CC('a','s','l','k'), CREATE_4CC('S','h','A','l'), dmModeReadOnly | dmModeLeaveOpen);
			//for old devices
			if (!ref)
				ref = DmOpenDatabaseByTypeCreator(CREATE_4CC('s','i','l','k'), CREATE_4CC('S','h','A','l'), dmModeReadOnly | dmModeLeaveOpen);
			
			if (!ref)
				fatal("Failed to open silk coord file\n");
				
			break;
		case 3:
			
			//device features ('ShAl', 20000) bits:
			//	0x2000 - silkscreen allowed
			//	0x0008 - somehow related to silk too
			
			//'ShAl' 21001 - if set, IrqNumber for Rmc (remote control)
			//'ShAl' 21003 - if set, IrqNumber for Pa1Lib (sony custom audio chip driver)
			
			//'ShAl' 20026 - if not set, defaults to 50000, some sort of a delay used by Pa1Lib, passed to HALDelay()
		
		
			FtrSet(CREATE_4CC('S','h','A','l'), 20005, 1);					//number of memory stick slots (we have 1)
			FtrSet(CREATE_4CC('S','h','A','l'), 20021, (uint32_t)&"med");	//points to current cpu config name
			
			//from th55
			FtrSet(CREATE_4CC('S','h','A','l'), 20000, 0x899A2CF);			//device features
			FtrSet(CREATE_4CC('S','h','A','l'), 20050, (uint32_t)&mPortState);
			//FtrSet(CREATE_4CC('S','h','A','l'), 21000, 0x44);
			//FtrSet(CREATE_4CC('S','h','A','l'), 21002, 0x2D);
			//FtrSet(CREATE_4CC('S','h','A','l'), 21005, 0x06);
			//FtrSet(CREATE_4CC('S','h','A','l'), 21010, 0x34);
			//FtrSet(CREATE_4CC('S','s','Y','s'), 10000, 0x07);	//som global
			
			//from ux50
			FtrSet(CREATE_4CC('S','h','A','l'), 20000, 0x9926DF);			//device features
			
			
			
			/*
			//from ux50
			FtrSet(CREATE_4CC('S','h','A','l'), 20000, 0x9926DF);			//device features
			FtrSet(CREATE_4CC('S','h','A','l'), 21000, 0x44);
			FtrSet(CREATE_4CC('S','h','A','l'), 21002, 0x2D);
			FtrSet(CREATE_4CC('S','h','A','l'), 21005, 0x06);
			FtrSet(CREATE_4CC('S','h','A','l'), 21007, 0x32);
			FtrSet(CREATE_4CC('S','h','A','l'), 21008, 0x33);
			FtrSet(CREATE_4CC('S','h','A','l'), 21009, 0x38);
			FtrSet(CREATE_4CC('S','s','Y','s'), 10000, globals + 138);
			*/
			
			/*
			//from ux40
			FtrSet(CREATE_4CC('S','h','A','l'), 20000, 0x9926DF);			//device features
			FtrSet(CREATE_4CC('S','h','A','l'), 21000, 0x44);
			FtrSet(CREATE_4CC('S','h','A','l'), 21002, 0x2D);
			FtrSet(CREATE_4CC('S','h','A','l'), 21005, 0x06);
			FtrSet(CREATE_4CC('S','h','A','l'), 21007, 0x32);
			FtrSet(CREATE_4CC('S','h','A','l'), 21008, 0x33);
			FtrSet(CREATE_4CC('S','h','A','l'), 21009, 0x34);
			FtrSet(CREATE_4CC('S','s','Y','s'), 10000, globals + 138);
			*/
			
			/*
			//from nx60
			FtrSet(CREATE_4CC('S','h','A','l'), 20000, 0x49C3FDF);			//device features
			FtrSet(CREATE_4CC('S','h','A','l'), 20026, 0xC350);
			FtrSet(CREATE_4CC('S','h','A','l'), 21000, 0x21);
			FtrSet(CREATE_4CC('S','h','A','l'), 21001, 0x22);
			FtrSet(CREATE_4CC('S','h','A','l'), 21002, 0x27);
			FtrSet(CREATE_4CC('S','h','A','l'), 21003, 0x23);
			FtrSet(CREATE_4CC('S','h','A','l'), 21004, 0x1f);
			FtrSet(CREATE_4CC('S','h','A','l'), 21005, 0x28);
			FtrSet(CREATE_4CC('S','h','A','l'), 21006, 0x29);
			FtrSet(CREATE_4CC('S','h','A','l'), 21007, 0x25);
			FtrSet(CREATE_4CC('S','h','A','l'), 21008, 0x24);
			FtrSet(CREATE_4CC('S','h','A','l'), 21009, 0x20);
			FtrSet(CREATE_4CC('S','s','Y','s'), 10000, globals + 132);
			*/
			
			/*
			//from TG50
			FtrSet(CREATE_4CC('S','h','A','l'), 20000, 0x3C919CCF);			//device features
			FtrSet(CREATE_4CC('S','h','A','l'), 20026, 0xC350);
			FtrSet(CREATE_4CC('S','h','A','l'), 21000, 0x21);
			FtrSet(CREATE_4CC('S','h','A','l'), 21001, 0x22);
			FtrSet(CREATE_4CC('S','h','A','l'), 21002, 0x27);
			FtrSet(CREATE_4CC('S','h','A','l'), 21003, 0x23);
			FtrSet(CREATE_4CC('S','h','A','l'), 21004, 0x1f);
			FtrSet(CREATE_4CC('S','h','A','l'), 21005, 0x28);
			FtrSet(CREATE_4CC('S','h','A','l'), 21006, 0x29);
			FtrSet(CREATE_4CC('S','h','A','l'), 21007, 0x25);
			FtrSet(CREATE_4CC('S','h','A','l'), 21008, 0x24);
			FtrSet(CREATE_4CC('S','h','A','l'), 21009, 0x20);
			FtrSet(CREATE_4CC('S','s','Y','s'), 10000, globals + 132);
			
			
			
			TH55:	(strict superset of TJ37, by: 0x0811A001, differs from ux50 by 0x08008410
				20000 = 0x0899A2CF
			
			NX80:
				20000 = 0x4C9CBFDF OR 0x4C9DBFDF		(dependong on some global)
			
			TG50
				200000 = 0x3C919CCF
			
			TJ35:
				20000 = 0x008000CE
				20005 = 0x00000001
				20021 = 0x008FFD2C
				20022 = 0x200A3E38
				20024 = 0x200B95FC
				20031 = 0x200A34E3
				20033 = 0x008D7CF0
				21000 = 0x00
				21002 = 0x00
			
			TJ37:	//beats tj35 by 0x80200
				20000 = 0x008802CE
				20001 = 0x00000001
				20005 = 0x00000001
				20021 = 0x008FFD00
				20022 = 0x200A1C74
				20024 = 0x200B9888
				20050 = 0x008FFE94
				21000 = 0x00
				21002 = 0x00
				21010 = 0x10
				
			VZ90:
				20000 = 0x088421CF
				20001 = 0x00000009
				20005 = 0x00000001
				20020 = 0x1ff04710
				20021 = 0x007ffb19
				20022 = 0x2009b1bc
				20024 = 0x200956e0
				20031 = 0x2009a7d8
				20034 = 0x00000040
				20050 = 0x007ffe24
				20070 = 0x007ea848
				21000 = 0x44
				21002 = 0x2D
				21005 = 0x06
			
			Ux50:	//beats tj37 by 0x00112411
				20000 = 0x009926DF
				20005 = 0x00000001
				20020 = 0x1ff04148
				20021 = 0x007FFB12
				20022 = 0x2009CF8B
				20024 = 0x200967e0
				20031 = 0x2009c1f0
				20050 = 0x007ffe30
				21000 = 0x44
				21002 = 0x2D
				21005 = 0x06
				21007 = 0x32
				21008 = 0x33
				21009 = 0x34
			
			NX60:
				20000 = 0x04943FDF
				20005 = 0x00000001
				20020 = 0x1ff03da4
				20021 = 0x004ffd98
				20022 = 0x2002335c
				20024 = 0x20038e30
				20026 = 0x0000c350
				20027 = 0x1ff061cc
				21000 = 0x21
				21001 = 0x22
				21002 = 0x27
				21003 = 0x23
				21004 = 0x1f
				21005 = 0x28
				21006 = 0x29
				21007 = 0x25
				21008 = 0x24
				21009 = 0x20
			*/
			//nothing;
			break;
	}
	
	return 0;
}

static uint32_t sonyDalPatchSonyVersion(void)
{
	return 1;
}

static Err sonyDalPatchSonySleepSetHandler(uint32_t idx /* order */, SonySleepWakeHandler cbk, void* cbkData)
{
	if (idx >= NUM_SLEEP_HANDLERS)
		fatal("Index too high for %s\n", __func__);
	if (mSleepHandlers[idx].cbk)
		fatal("Setting duplicate handler for index %u in %s\n", idx, __func__);
	
	mSleepHandlers[idx].cbk = cbk;
	mSleepHandlers[idx].data = cbkData;
	
	return 0;
}

static Err sonyDalPatchSonyWakeSetHandler(uint32_t idx /* order */, SonySleepWakeHandler cbk, void* cbkData)
{
	if (idx >= NUM_WAKE_HANDLERS)
		fatal("Index too high for %s\n", __func__);
	if (mSleepHandlers[idx].cbk)
		fatal("Setting duplicate handler for index %u in %s\n", idx, __func__);
	
	mWakeHandlers[idx].cbk = cbk;
	mWakeHandlers[idx].data = cbkData;
	
	return 0;
}

static void sonyDalPatchSonySleep(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonyWake(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonyWakeCheck(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static uint32_t sonyDalPatchSonyCpuConfig(uint32_t op, void *data)
{
	struct {
		uint16_t alwaysOne;
		uint16_t mhz;
		const char *name;
	} static const cpuStates[] = {
		{1, 100, "slow"},
		{1, 200, "med"},
		{1, 300, "fast"},
	};
	
	static const char cpuName[] = "rePalm vCPU";
	static uint16_t curCpuState = 0;
	
	logt("SonyCpuConfig(%u, ...)\n", op);
	
	switch(op) {
		case 1:		//u16 get num configs available;
			*(uint16_t*)data = sizeof(cpuStates) / sizeof(*cpuStates);
			break;
		case 2:		//get table of speed settings
			*(const void**)data = cpuStates;
			break;
		case 3:		//u16 get current mode
			*(uint16_t*)data = curCpuState;
			break;
		case 4:		//u16 set current mode
			curCpuState = *(uint16_t*)data;
			if (curCpuState >= sizeof(cpuStates) / sizeof(*cpuStates))
				curCpuState = 0;
			break;
		case 5:		//not sure. returns the alwaysOne field of cur state
			*(uint16_t*)data = cpuStates[curCpuState].alwaysOne;
			break;
		case 6:		//not sure. returns 1 on nx60, 0 on th55
			return 1;
			break;
		case 8:		//u16 name length
			*(uint16_t*)data = sizeof(cpuName);
			break;
		case 9:		//char* name
			memcpy(data, cpuName, sizeof(cpuName));
			break;
		case 10:	//u32 set cpu auto-freq low limit
			break;
		case 11:	//u32 set cpu auto-freq high limit
			break;
		case 12:	//nodata, remove cpu auto-freq limits
			break;
		case 13:	//not sure
			return 0xffff0000;
		case 14:	//nodata, save cpu limits (one-deep stack)
			break;
		case 15:	//nodata, restore cpu limit (that one stack from above)
			break;
	}
	
	return 0;
}

static void sonyDalPatchSonyTimerCreate(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonyTimerDelete(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonyTimerSet(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static Err sonyDalPatchSonyStPortGetState(uint32_t *stateP)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));

	*stateP = mPortState;
		
	return 0;
}

static void sonyDalPatchSonyHoldMaskData(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonySetHoldState(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static Err sonyDalPatchSonyGetHoldMask(uint32_t sel, uint32_t *outVal)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	*outVal = sel == 1 ? 2 : 0;	//as per nx60, tj35
	return errNone;
}

static int sonyDalPatchSonySetScrRotState(uint32_t state)		//return 1 on failure. only possible valid input values are 0 and 1
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	if (state > 2)
		return 1;
	
	mRotState = state;
	return 0;
}

static Err sonyDalPatchSonyGetScrRotState(uint32_t *stateP)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	*stateP = mRotState;
	return 0;
}

static void sonyDalPatchSonyPicIoctl(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static int sonyDalPatchSonyLedSetState(uint32_t which, bool on, uint32_t mustBeZero)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	if (mustBeZero)
		return 2;
	
	switch (which) {
		case SONY_LED_NO_POWER:
		case SONY_LED_NO_MS:
		case SONY_LED_NO_WIFI:
		case SONY_LED_NO_RECORDING:
		case SONY_LED_NO_BLUETOOTH:
			return 0;
		default:
			return 1;
	}
}

static int sonyDalPatchSonyLedGetState(uint32_t which, bool *onP, uint32_t *notSureP)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	if (!onP && !notSureP)
		return 2;
	
	switch (which) {
		case SONY_LED_NO_POWER:
		case SONY_LED_NO_MS:
		case SONY_LED_NO_WIFI:
		case SONY_LED_NO_RECORDING:
		case SONY_LED_NO_BLUETOOTH:
			break;
		default:
			return 1;
	}
	
	if (onP)
		*onP = false;
	
	if (notSureP)
		*notSureP = 0;
	
	return 0;
}

static Err sonyDalPatchSonyAdcRead(uint32_t op, uint16_t *outValP)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	*outValP = 0;
	
	return errNone;
}

static void sonyDalPatchSonyAdcWakeRead(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonyAdcWakeWrite(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static Err sonyDalPatchSonyExtCntSetPort(bool something, const uint32_t *valInOut)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	//XXX: we need to do something here, no?
	
	return errNone;
}

static void sonyDalPatchSonySndOpen(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonySndClose(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonySndPlay(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonySndStop(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonySndPause(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonySndRead(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonySndWrite(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static int sonyDalPatchSonySndIoctl(void)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	//xxx
	
	return -1;
}

static int sonyDalPatchSonySndDevIoctl(void)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	//xxx
	
	return -1;
}

static uint32_t sonyDalPatchSonyInfoBatteryGetPercent(void)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	return 99;
}

static uint32_t sonyDalPatchSonyBatteryCurrentValues(uint16_t *centivoltsP, uint16_t *percentP)			//0xA2D4 @ TH55 DAL
{
	fatal("sonyHalBatteryGetReal called from 0x%08lx\n", __builtin_return_address(0));
	
	*centivoltsP = 420;
	return *percentP = 99;
}

static int sonyDalPatchSonyBatteryThresholdInfo(bool set, uint32_t which, bool *isInPercentP, uint16_t *threshValP, uint16_t *displayThresholdP)
{
	static uint16_t thres[16] = {10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, }, displayThresh[16] = {15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, };
	static bool isInPercent[16] = {true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, };
	
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	if (!isInPercentP || !threshValP || !displayThresholdP || which >= 0x10)
		return 1;
	
	if (set) {
		isInPercent[which] = *isInPercentP;
		thres[which] = *threshValP;
		displayThresh[which] = *displayThresholdP;
	}
	else {
		*isInPercentP = isInPercent[which];
		*threshValP = thres[which];
		*displayThresholdP = displayThresh[which];
	}
	
	return 0;
}

static int sonyDalPatchSonyBatteryVoltage2Percent(uint16_t centivolts, uint16_t *percentP)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	if (centivolts < 370)
		*percentP = 0;
	else if (centivolts > 420)
		*percentP = 100;
	else
		*percentP = (centivolts - 370) * 100 / (420 - 370);
	
	return 0;
}

static int sonyDalPatchSonyBatteryPercent2Voltage(uint16_t percent, uint16_t *centivoltsP)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	*centivoltsP = 370 + (420 - 370) * 100 / percent;
	
	return 0;
}

//times can be set to 0xffff if not avail
//this func exists on all clies except NX60/NX70
static int sonyDalPatchSonyGetBatteryInfo(uint8_t *battIdxP, uint16_t *someBatteryFlagsP, uint8_t *percentP, uint16_t *batteryChargingTimeInMinutes, uint16_t *batteryUsableTimeInMinutes, uint16_t * someVal3P)
{
	uint16_t dockStatus;
	Boolean pluggedIn;
	uint8_t percent;
	
	//abou someBatteryFlagsP:
	// a complex bitfield. all we know so far is:
	//	0x80 - plugged in (but not necessarily have external power). either chargnig or full
	//	0x40 - we have external power
	//	0x20 - discharging (or full if 0x80 set)
	//	0x10 - charging
	//	0x01 - present
	
	//new from 2022, previous vals might be wrong-er. this is from detailed examination of nz90, calling the api might be more help sometime:
	//	0x00000080 - full or charging
	//	0x00000040 - NOT discharging (charging or not in use)
	//	0x00000020 - is full (if charging or full)
	//	0x00000004 - internal battery (else external)
	//	0x00000002 - usable (for external batteries batt will not be reported if this is clear)
	//	0x00000001 - present
	
	/*	logix in nz90 sonysys for one bettery present it thus:
	
		NOTE: first string is "where is power coming from" so "external charging" does not mean external battery. just "not from internal battery"
	
		if (flags & 0x04) {
			if (flags & 0x80) {
				
				if (flags & 0x20) {
					
					external, full
				}
				else {
					
					external, charging
				}
			}
			else {
				
				if (flags & 0x40) {
					
					external, not in use
				}
				else {
					
					internal, discharging
				}
			}
		}
		else {
			
			external, "----"
		}
		

		for two batteries see:  ROM:0000AAC8 in nz90 sonysys.prc

	//see more in sony_dal_battery_handling_nz90.idb and nz90-sonysys-battery-handling-amdc0000.idb

	
	*/
	
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	if (!battIdxP)
		return 2;
	
	if (*battIdxP == 0xFF) {		//if 0xFFFF then return max battery index
		*battIdxP = 1;
		return 0;
	}
	
	(void)HALDockStatus(&dockStatus);
	(void)SysBatteryInfoV40(false, NULL, NULL, NULL, NULL, &pluggedIn, &percent);
	
	switch (*battIdxP) {
		case 0:				//battery 0 - internal
		
			if (batteryChargingTimeInMinutes)
				*batteryChargingTimeInMinutes = 0xffff;
			
			if (batteryUsableTimeInMinutes)
				*batteryUsableTimeInMinutes = 0xffff;
			
			if (someVal3P)
				*someVal3P = 0xffff;
			
			if (pluggedIn) {
				
				if (percentP)
					*percentP = (dockStatus & HAL_DOCK_STATUS_CHARGING) ? 0xFF : 100;
				
				if (someBatteryFlagsP)
					*someBatteryFlagsP = (*someBatteryFlagsP &~ 0x0A) | 0xC5;
				
				if (dockStatus & HAL_DOCK_STATUS_CHARGING)
					*someBatteryFlagsP = (*someBatteryFlagsP &~ 0x20) | 0x10;
				else
					*someBatteryFlagsP = (*someBatteryFlagsP &~ 0x10) | 0x20;
			}
			else {
				if (percentP)
					*percentP = percent;
				
				if (someBatteryFlagsP)
					*someBatteryFlagsP = (*someBatteryFlagsP &~ 0xBA) | 0x05;
				
				if (dockStatus & HAL_DOCK_STATUS_EXTRNAL_PWR)
					*someBatteryFlagsP |= 0x40;
				else
					*someBatteryFlagsP &=~ 0x40;
			}
			
			break;

		case 1:				//battery 1 - ?
			
			if (percentP)
				*percentP = 0xFF;
			
			if (batteryChargingTimeInMinutes)
				*batteryChargingTimeInMinutes = 0xffff;
			
			if (batteryUsableTimeInMinutes)
				*batteryUsableTimeInMinutes = 0xffff;
			
			if (someVal3P)
				*someVal3P = 0xffff;
			
			if (someBatteryFlagsP)
				*someBatteryFlagsP = (*someBatteryFlagsP &~ 0xF6) | 9;
			
			if (dockStatus & HAL_DOCK_STATUS_EXTRNAL_PWR)
				return 1;
			
			break;

		default:
			return 2;	
	}
	
	return 0;
}

static Err sonyDalPatchSonyBatteryInfoExt(uint8_t battIdx, uint32_t selector, void* dataP)
{
	//known selectors (from NZ90)
	//sel		data type		notes
	//0x00		u16				number of charge cycles that's been done
	//0x01		u8				degratation percent
	//0x02		u16				design capacity in mAh
	//0x03		u16				??? (might be time related, 3600 is used in calculation)
	//0x04		u16				??? (might be time related, 3600 is used in calculation)
	//0x05		char[]			manufacturer name is written to provided buffer, zero-terminated
	//0x06		char[]			model ID is written to provided buffer, zero-terminated
	//0x07		u32				???
	//0x08		u16				???
	//see more in sony_dal_battery_handling_nz90.idb and nz90-sonysys-battery-handling-amdc0000.idb
	
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static void sonyDalPatchSonyHoldAlert(void)
{
	fatal("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
}

static Err sonyDalPatchSonyBackupRAM(uint32_t flags, uint32_t flags2, bool bool3, uint32_t val4, uint32_t val5)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	return 0x700D;
}

static Err sonyDalPatchSonyRestoreRAM(uint32_t noIdea, ...)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	return 0x700D;
}

static Err sonyDalPatchSonyGetBackupInfo(uint32_t *something, uint32_t *somethingElse, void* infoOutP)
{
	logt("%s called from 0x%08lx\n", __func__, __builtin_return_address(0));
	
	return 0x700D;
}

static void patchRmcExtLibEntrypt(void)
{
	fatal("RmcLib entrypt called from 0x%08lx\n", __builtin_return_address(0));
}

static void patchPa1LibLibEntrypt(void)
{
	fatal("Pa1Lib entrypt called from 0x%08lx\n", __builtin_return_address(0));
}

void sonyPatchLibEntrypts(uint32_t dbType, uint32_t dbCrid, uint32_t *entryptAddrs, uint32_t nEntries)
{
	uint32_t i;
	
	if (dalGetInitStage() < DAL_INIT_STAGE_MEM_AND_DM || !isSony())		//no sony libs load before this stage
		return;
	
	if (dbType == CREATE_4CC('l','i','b','r') && dbCrid == CREATE_4CC('S','d','a','l')) {
		
		static const void* sonyDalPatches[] = {
			sonyDalPatchSonyGetGlobalsPtr, sonyDalPatchSonyMain, sonyDalPatchSonyVersion, sonyDalPatchSonySleepSetHandler,
			sonyDalPatchSonyWakeSetHandler, sonyDalPatchSonySleep, sonyDalPatchSonyWake, sonyDalPatchSonyWakeCheck,
			sonyDalPatchSonyCpuConfig, sonyDalPatchSonyTimerCreate, sonyDalPatchSonyTimerDelete, sonyDalPatchSonyTimerSet,
			sonyDalPatchSonyStPortGetState, sonyDalPatchSonyHoldMaskData, sonyDalPatchSonySetHoldState, sonyDalPatchSonyGetHoldMask,
			sonyDalPatchSonySetScrRotState, sonyDalPatchSonyGetScrRotState, sonyDalPatchSonyPicIoctl, sonyDalPatchSonyLedSetState,
			sonyDalPatchSonyLedGetState, sonyDalPatchSonyAdcRead, sonyDalPatchSonyAdcWakeRead, sonyDalPatchSonyAdcWakeWrite,
			sonyDalPatchSonyExtCntSetPort, sonyDalPatchSonySndOpen, sonyDalPatchSonySndClose, sonyDalPatchSonySndPlay,
			sonyDalPatchSonySndStop, sonyDalPatchSonySndPause, sonyDalPatchSonySndRead, sonyDalPatchSonySndWrite,
			sonyDalPatchSonySndIoctl, sonyDalPatchSonySndDevIoctl, sonyDalPatchSonyInfoBatteryGetPercent,
			sonyDalPatchSonyBatteryThresholdInfo, sonyDalPatchSonyBatteryCurrentValues, sonyDalPatchSonyBatteryVoltage2Percent,
			sonyDalPatchSonyBatteryPercent2Voltage, sonyDalPatchSonyGetBatteryInfo, sonyDalPatchSonyBatteryInfoExt,
			sonyDalPatchSonyHoldAlert, sonyDalPatchSonyBackupRAM, sonyDalPatchSonyRestoreRAM, sonyDalPatchSonyGetBackupInfo, 
		};
		
		logt("Patching SonyDAL library entries\n");
		if (nEntries > sizeof(sonyDalPatches) / sizeof(*sonyDalPatches))
			fatal("SonyDAL has too many entrypoints\n");
		
		for (i = 0; i < nEntries; i++)
			entryptAddrs[i] = (uint32_t)sonyDalPatches[i];
	}
	else if (CREATE_4CC('a','e','x','t') && dbCrid == CREATE_4CC('S','e','R','m')) {
		logt("Patching RmcExt lib entrypts\n");
		
		for (i = 0; i < nEntries; i++)
			entryptAddrs[i] = (uint32_t)patchRmcExtLibEntrypt;
	}
	else if (CREATE_4CC('l','i','b','r') && dbCrid == CREATE_4CC('y','P','1','L')) {
		logt("Patching Pa1Lib lib entrypts\n");
		
		for (i = 0; i < nEntries; i++)
			entryptAddrs[i] = (uint32_t)patchPa1LibLibEntrypt;
	}
}

static uint32_t patchRmcExtMainEntrypt(uint16_t cmd, void *cmdPBP, uint16_t flags)
{
	logt("RmcExt main entrypt called (cmd=%d)\n", cmd);
	
	return errNone;
}

static uint32_t patchPa1LibMainEntrypt(uint16_t cmd, void *cmdPBP, uint16_t flags)
{
	logt("Pa1Lib main entrypt called (cmd=%d)\n", cmd);
	
	return errNone;
}

void sonyPatchLibMainEntry(uint32_t dbType, uint32_t dbCrid, ModuleEntryPoint* entryPtP)
{
	if (dalGetInitStage() < DAL_INIT_STAGE_MEM_AND_DM || !isSony())		//no sony libs load before this stage
		return;
	
	if (dbType == CREATE_4CC('l','i','b','r') && dbCrid == CREATE_4CC('S','d','a','l')) {
		logt("Patching SonyDAL main entrypt\n");
		*entryPtP = patchSonyDalMainEntrypt;
	}
	else if (CREATE_4CC('a','e','x','t') && dbCrid == CREATE_4CC('S','e','R','m')) {
		logt("Patching RmcExt main entrypt\n");
		
		*entryPtP = patchRmcExtMainEntrypt;
	}
	else if (CREATE_4CC('l','i','b','r') && dbCrid == CREATE_4CC('y','P','1','L')) {
		logt("Patching Pa1Lib main entrypt\n");
		
		*entryPtP = patchPa1LibMainEntrypt;
	}
}

bool sonyDenyLibLoad(uint32_t dbType, uint32_t dbCrid)	//return true to deny loading
{
	if (dbType == CREATE_4CC('a','e','x','t') && dbCrid == CREATE_4CC('S','e','S','d'))		//SoundExt
		return true;
	
	return false;
}

