#include "halPenAndKeys.h"
#include "kernel_int.h"
#include "hostCtl.h"
#include "printf.h"
#include "kernel.h"
#include "atomic.h"
#include "timers.h"
#include "input.h"
#include "entry.h"
#include "slab.h"
#include "heap.h"
#include "list.h"
#include "irqs.h"
#include "mpu.h"
#include "emu.h"
#include "rtc.h"
#include "kal.h"
#include "ral.h"
#include "dal.h"
#include "cpu.h"


/*

	TID 0 is special. it is the TID of the idle thread AND used to indicate "none" for many cases since idle thread (like Perry) dont't do much
	priorities are u16 and higher means more important (suck it, other schedulers). prio of 0xffff is reserved for the timer task, prio 0 is reserved
	for idle task
	
	reaper task shall exist to free task stacks, it will operate on a lockless linked list and will be woken anytime another stack has been added
	to that list it should have a high-enough prio to be often scheduled. we allocate prio 0xfffe to it
	
	for scheduling tasks with same prio, variable mCurRunGeneration is used. it is incremented by scheduler every time this happens, task insertion
	into heap will consider tasks that are more BEHIND mCurRunGeneration as higher prio. when descheduling a task, its runGeneration shall we set
	to mCurRunGeneration, this shall also happen when a task is created
	
*/

#define SCHED_MAX_THREADS		((1 << (SCHED_MAX_THREADS_LOG)) - 1)





struct SchedPrimitiveHead {	//MUST be first member of each struct
	struct ListNode list;	//list of all promitives of this type
	uint32_t id;			//ID of this one (for lookups)
	uint32_t tag;
};

#define MUTEX_LOCK_COUNT_NOT_RECURSIVE	0xFFFFFFFF

struct Mutex {
	struct SchedPrimitiveHead common;
	struct ListNode blockedList;
	tid_t owner;
	uint32_t lockCount;					//current lock count or MUTEX_LOCK_COUNT_NOT_RECURSIVE if not recursive. DO NOT use to check if locked. use "owner" member instead
};

struct Sem {
	struct SchedPrimitiveHead common;
	uint32_t counter;
	struct ListNode blockedList;
};

struct Mailbox {
	struct SchedPrimitiveHead common;
	struct ListNode blockedList;
	uint32_t readIdx, writeIdx, size;	//read == write means empty
	uint32_t *messages;				//we waste a slot on empty signalling, so this always has one extra slot
};

struct EvtGrp {
	struct SchedPrimitiveHead common;
	struct ListNode blockedList;
	uint32_t state;
};

struct Timer {
	struct SchedPrimitiveHead common;
	uint64_t expires;	//in our timebase
	uint32_t r9;
	KernTimerCbk cbk;
	void* cbkData;
};

struct StackFreeListItem {
	struct StackFreeListItem *next;			//must be first element
	void* actualPtr;
};

#define REG_NO_SP					13
#define REG_NO_LR					14
#define REG_NO_PC					15

#define SCHED_BLOCK_REASON_NONE				0		//not blocked
#define SCHED_BLOCK_REASON_NEW				1		//new task
#define SCHED_BLOCK_REASON_MANUAL			2		//blocked by manual request
#define SCHED_BLOCK_REASON_MUTEX			3		//blocked on mutex
#define SCHED_BLOCK_REASON_SEMA				4		//blocked on semaphore
#define SCHED_BLOCK_REASON_MBX				5		//blocked on mailbox
#define SCHED_BLOCK_REASON_EVTGRP			6		//blocked on event group
#define SCHED_BLOCK_REASON_DELAY			7		//blocked by schedTaskDelay
#define SCHED_BLOCK_REASON_WAIT				8		//blocked by schedTaskWait
#define SCHED_BLOCK_REASON_WAIT_AND_MANUAL	9		//blocked by schedTaskWait and manually too


static LIST_DECLARE(mAllTasks);						//all tasks
static struct Task * volatile mCurTask = NULL;		//current task
static struct Task * mNextTask = NULL;		//next task to run
static ktimer_t mSchedTimer = 0;
static uint8_t mSchedulerLockCnt = 0;	//number of scheduler disablements that have happened
static uint16_t mCurRunGeneration = 0;
static tid_t mTimersTaskTid, mStackReaperTid;

//the heap contains only runnable tasks
static struct Task* mTaskHeap[SCHED_MAX_THREADS] = {0, };
static uint16_t mNumRunnableTasks = 0;

static LIST_DECLARE(mMutexes);				//all mutexes
static LIST_DECLARE(mSems);					//all semaphores
static LIST_DECLARE(mMbxs);					//all mailboxes
static LIST_DECLARE(mEvtGrps);				//all event groups
static LIST_DECLARE(mTimers);				//all timers

static struct Slab *mTasksSlab, *mMutexesSlab, *mSemsSlab, *mTimersSlab, *mEvtGrpsSlab, *mMailboxesSlab;

//the TLS lives here - swapped each ctx swtch
struct TaskTls TLS;

static uint32_t mIdleTaskStack[64] __attribute__((aligned(32)));	//this is only safe since neither of these ever end
static uint32_t mTimersTaskStack[128] __attribute__((aligned(32)));
static uint32_t mReaperTaskStack[128] __attribute__((aligned(32)));

static struct StackFreeListItem *mStacksFreeList = 0;

static void schedRequestResched(void);			//safe to call from any irq AND from SVC context



static bool schedIsTaskHigherInHeap(const struct Task* lhs, const struct Task *rhs)
{
	if (lhs->prio == rhs->prio) {
		int16_t lhsRunGenerationDiff = mCurRunGeneration - lhs->runGeneration;
		int16_t rhsRunGenerationDiff = mCurRunGeneration - rhs->runGeneration;
		
		return lhsRunGenerationDiff > rhsRunGenerationDiff;
	}
	
	return lhs->prio > rhs->prio;
}

static void schedTaskRemoveFromRunHeapByIdx(uint32_t idx)	//call only in SVC or with irqs off
{
	//replace "this" index with last task in the list
	mTaskHeap[idx] = mTaskHeap[--mNumRunnableTasks];
	
	//re-heapify
	while (idx < mNumRunnableTasks)
	{
		uint32_t childL = idx * 2 + 1;
		uint32_t childR = idx * 2 + 2;
		struct Task *tmpTskPtr;
		uint32_t beterChild;
		
		//find the higher prio child of all the children
		if (childR < mNumRunnableTasks)			//have both children - pick the smaller one to replace with
			beterChild = schedIsTaskHigherInHeap(mTaskHeap[childL], mTaskHeap[childR]) ? childL : childR;
		else if (childR == mNumRunnableTasks)	//have one child (left) and thus it is the smaller child
			beterChild = childL;
		else									//no children -> re-heapifying is complete
			break;
		
		//if we're higher prio than best child, we're done
		if (schedIsTaskHigherInHeap(mTaskHeap[idx], mTaskHeap[beterChild]))
			break;
		
		//better child is better than current index. swap them and repeat the process for that new (lower) index
		tmpTskPtr = mTaskHeap[beterChild];
		mTaskHeap[beterChild] = mTaskHeap[idx];
		mTaskHeap[idx] = tmpTskPtr;
		
		idx = beterChild;
	}
}

static void schedTaskPutInRunHeap(struct Task *tsk)
{
	uint32_t idx;
	
	if (mNumRunnableTasks == SCHED_MAX_THREADS)	//unlikely
		fatal("no space in scheduler heap, sorry\n");
	
	//insert as last place in the heap
	mTaskHeap[idx = mNumRunnableTasks++] = tsk;
	
	//re-heapify
	while (idx) {
		uint32_t parentIdx = (idx - 1) / 2;
		struct Task *tmpTskPtr;
		
		//if we're lower prio than our parent, we're done
		if (schedIsTaskHigherInHeap(mTaskHeap[parentIdx], mTaskHeap[idx]))
			break;
		
		//if we're higher prio, take their place and re-loop
		tmpTskPtr = mTaskHeap[parentIdx];
		mTaskHeap[parentIdx] = mTaskHeap[idx];
		mTaskHeap[idx] = tmpTskPtr;
		
		idx = parentIdx;
	}
}

static void schedTaskRemoveFromRunHeapByPtr(struct Task *tsk)
{
	uint32_t i;
	
	schedAssertCalledInSyscallMode();
	
	for (i = 0; i < mNumRunnableTasks && mTaskHeap[i]; i++) {
		
		if (mTaskHeap[i] == tsk) {
			schedTaskRemoveFromRunHeapByIdx(i);
			break;
		}
	}
}

void schedShowTasks(void)
{
	static const char *blockReasons[] = {
		[SCHED_BLOCK_REASON_NONE] = "none",
		[SCHED_BLOCK_REASON_NEW] = "new",
		[SCHED_BLOCK_REASON_MANUAL] = "manual",
		[SCHED_BLOCK_REASON_MUTEX] = "mutex",
		[SCHED_BLOCK_REASON_SEMA] = "semaphore",
		[SCHED_BLOCK_REASON_MBX] = "mailbox",
		[SCHED_BLOCK_REASON_EVTGRP] = "event group",
		[SCHED_BLOCK_REASON_DELAY] = "delay",
		[SCHED_BLOCK_REASON_WAIT] = "wait",
		[SCHED_BLOCK_REASON_WAIT_AND_MANUAL] = "wait + manual",
	};
	
	struct ListNode *node;
	
	mpuSetStackGuard(0);	//so we do not crash. ye we do not restore it. so what? this is a debug func anywas
	
	for (node = mAllTasks.next; node != &mAllTasks; node = node->next) {
		
		struct Task *tsk = STRUCT_FROM_LIST_NODE(struct Task, tasks, node);
		uint32_t i;
		
		loge("TASK %u@0x%08x prio %u tag 0x%08x'%c%c%c%c'%s:\n", tsk->tid, tsk, tsk->prio, tsk->tag, (tsk->tag >> 24) & 0xff, (tsk->tag >> 16) & 0xff, (tsk->tag >> 8) & 0xff, (tsk->tag >> 0) & 0xff, tsk == mCurTask ? " *CURRENT*" : "");
		loge("  block reason: %s\n", blockReasons[tsk->blockReason]);
		if (tsk != mCurTask) {
			loge("  REGS:  SR 0x%08x\n", tsk->ctx.regs.sr);
			loge("   R0  = 0x%08x  R8  = 0x%08x\n", tsk->ctx.regs.r0_r3[0], tsk->ctx.regs.r4_r11[8 - 4]);
			loge("   R1  = 0x%08x  R9  = 0x%08x\n", tsk->ctx.regs.r0_r3[1], tsk->ctx.regs.r4_r11[9 - 4]);
			loge("   R2  = 0x%08x  R10 = 0x%08x\n", tsk->ctx.regs.r0_r3[2], tsk->ctx.regs.r4_r11[10 - 4]);
			loge("   R3  = 0x%08x  R11 = 0x%08x\n", tsk->ctx.regs.r0_r3[3], tsk->ctx.regs.r4_r11[11 - 4]);
			loge("   R4  = 0x%08x  R12 = 0x%08x\n", tsk->ctx.regs.r4_r11[4 - 4], tsk->ctx.regs.r12);
			loge("   R5  = 0x%08x  SP  = 0x%08x\n", tsk->ctx.regs.r4_r11[5 - 4], tsk->ctx.regs.sp);
			loge("   R6  = 0x%08x  LR  = 0x%08x\n", tsk->ctx.regs.r4_r11[6 - 4], tsk->ctx.regs.lr);
			loge("   R7  = 0x%08x  PC  = 0x%08x\n", tsk->ctx.regs.r4_r11[7 - 4], tsk->ctx.regs.pc);
			
			loge("  STACK: \n");
			for (i = 0; i < 32; i++)
				loge("   [SP, #0x%03x] = 0x%08x\n", i * 4, *(uint32_t*)(tsk->ctx.regs.sp + i * 4));
		}
	}
}

static struct Task* schedFindTaskByTid(tid_t tid)		//only call this in SVC context or with IRQs off
{
	struct ListNode *node;
	
	for (node = mAllTasks.next; node != &mAllTasks; node = node->next) {
		
		struct Task *tsk = STRUCT_FROM_LIST_NODE(struct Task, tasks, node);
		
		if (tsk->tid == tid)
			return tsk;
	}
	
	return NULL;
}

static tid_t schedFindFreeTid(void)
{
	if (listIsEmpty(&mAllTasks))
		return 0;
	else {
		tid_t tid = STRUCT_FROM_LIST_NODE(struct Task, tasks, mAllTasks.prev)->tid;
		
		do {
			if (!++tid)
				tid = 1;
			
		} while (schedFindTaskByTid(tid));
		
		return tid;
	}
}

static void __attribute__((naked)) schedTaskEndStub(void)
{
	//calls KTaskGetTid() and then KTaskDestroy() safely terminating itself
	
	asm volatile(
		"	sub  sp, #4	\n\t"
		"	mov  r1, sp	\n\t"
		"	movs r0, %1	\n\t"			
		"	svc  %0		\n\t"
		
		"	pop  {r1}	\n\t"
		"	movs r0, %2	\n\t"
		"	svc  %0		\n\t"
		:
		:"I"(KERNEL_SWI_NUM), "I"(SYSC_TASK_GET_TID), "I"(SYSC_TASK_DESTROY)
		:"memory"
	);
}

static kstatus_t schedTaskCreateInternal(uint16_t prio, uintptr_t pc, void* stackMem, uint32_t stackSz, uint32_t tag, void *exinf, bool priv, tid_t* tidP)	// -> KERN_STATUS_*
{
	uint8_t *stack = (uint8_t*)stackMem;
	uint32_t t, minStackSz;
	struct Task *tsk;

	//round stack pointer up to multiple of 8
	t = (uint32_t)stack;
	if (t & 7) {
		t = 8 - (t & 7);
		stack += t;
		stackSz -= t;
	}
	//round stack size down to multiple of 8
	stackSz &=~ 7;
	
	minStackSz = sizeof(struct CortexExcFrame);
	#ifdef MPU_STACK_GUARD_SIZE
		minStackSz += 2 * MPU_STACK_GUARD_SIZE;	//2 stack guards since alignment isnt certain
	#endif
	
	if (stackSz < minStackSz)
		stackSz = minStackSz;

	tsk = (struct Task*)slabAlloc(mTasksSlab);
	if (!tsk)
		return KERN_STATUS_MEM_ERR;
	
	//zero-init for safety
	memset(tsk, 0, sizeof(struct Task));
	
	//init regs
	tsk->ctx.regs.sp = (uintptr_t)(stack + stackSz);
	tsk->ctx.regs.lr = (uintptr_t)&schedTaskEndStub;
	tsk->ctx.regs.pc = pc &~ 1;
	tsk->ctx.regs.sr = 0;
	
	#ifdef BUILDING_FOR_BIG_ARM				//legacy
		tsk->ctx.regs.sr |= 0x1f;	//system mode
	#endif
	
	if (pc & 1) {				//thumb mode if needed
		#ifndef BUILDING_FOR_BIG_ARM		//cortex
			tsk->ctx.regs.sr |= CORTEX_SR_FLAG_T;
		#else					//legacy
			tsk->ctx.regs.sr |= LEGACY_SR_FLAG_T;
		#endif
	}
	
	asm("mov %0, r9":"=r"(tsk->ctx.regs.r4_r11[9 - 4]));		//set task's r9 to current r9
	
	//init state
	tsk->tag = tag;
	tsk->tid = schedFindFreeTid();
	tsk->stackChunk = stackMem;
	tsk->stackLimitAddr = stack;
	tsk->exinf = exinf;
	tsk->prio = prio;
	tsk->runGeneration = mCurRunGeneration;
	tsk->ctx.priv = priv;
	
	#ifdef SCHED_TASKS_ARE_ALL_PRIVILEDGED
		tsk->ctx.priv = true;
	#endif
	
	listInit(&tsk->inBlockSource);
	tsk->blockReason = SCHED_BLOCK_REASON_NEW;
	
	//link into all tasks list (at end)
	listInsertAfter(&tsk->tasks, mAllTasks.prev);
	
	//done
	*tidP = tsk->tid;
	return KERN_STATUS_OK;
}

static kstatus_t schedTaskCreate(uint16_t prio, uintptr_t pc, void* stackMem, uint32_t stackSz, uint32_t tag, void *exinf, bool priv, tid_t* tidP)
{
	if (prio < SCHED_MINUMUM_PRIO || prio > SCHED_MAXIMUM_PRIO)
		return KERN_STATUS_INVAL_REQ;

	return schedTaskCreateInternal(prio, pc, stackMem, stackSz, tag, exinf, priv, tidP);
}

static void schedTimerCbk(timer_t timer, void* timerData)
{
	schedRequestResched();
}

static void schedPickNextTask(void)						//called only in SVC context and is thus completely uninterruptible!
{
	struct Task *cur, *next = NULL;
	bool haveTie = false;
	
	schedAssertCalledInSyscallMode();
	
	next = mTaskHeap[0];	//CANNOT be NULL (at least one task is always schedulable)
	cur = mCurTask;
	
	//check sanity
	if (!next)
		fatal("No schedulable tasks found\n");
	
	//see if top runnable task has a prio tie with anyone
	//code writtenthis way to help GCC not be entirely braindead about compiling it
	if (mNumRunnableTasks >= 2 && mTaskHeap[1]->prio == next->prio)
		haveTie = true;
	if (mNumRunnableTasks > 2 && mTaskHeap[2]->prio == next->prio)
		haveTie = true;
	
	//if we do and current task is on top of heap and have a prio tie, rev generation counter and
	//reinsert outselves into the heap. after this we are 100% sure top task will no longer be us.
	if (haveTie) {
		if (next == cur) {
			cur->runGeneration = ++mCurRunGeneration;
			schedTaskRemoveFromRunHeapByIdx(0);
			schedTaskPutInRunHeap(cur);
			next = mTaskHeap[0];
			
			if (next == cur)
				fatal("After conflict resolution, current is still next, despite prio tie\n");
		}
		//set quantum timer since we do have a prio tie
		timerDelete(mSchedTimer);		//safe to do on stale handle
		mSchedTimer = timerCreate(CREATE_4CC('_','s','c','h'), schedTimerCbk, NULL, SCHED_QUANTUM_TICKS);
		if (!mSchedTimer)
			fatal("Cannot create sched timer!\n");
	}
	//ELSE, there is no conflict, schedule the top prio runnable task, which might be the current task
	
	if (next != cur) {
		
		//logi("Switching to task tid %u (tag " FMT_4CC ")\n", next->tid, CNV_4CC(next->tag)));
		
		mNextTask = next;
		schedRequestContextSwitch();
	}
}

static void schedTaskMakeRunnableEx(struct Task *tsk, bool allowResched)			//only usable in SVC mode. or with irqs off
{
	tsk->blockReason = SCHED_BLOCK_REASON_NONE;

	//if we had a timer, kill it!	
	timerDelete(tsk->timer);
	tsk->timer = 0;
	
	listRemoveItem(&tsk->inBlockSource);
	schedTaskPutInRunHeap(tsk);
	if (allowResched && (!mCurTask || tsk->prio >= mCurTask->prio))					//request  resched in case this one needs to run
		schedRequestResched();
}

static void schedTaskMakeRunnable(struct Task *tsk)									//only usable in SVC mode. or with irqs off
{
	schedTaskMakeRunnableEx(tsk, true);
}

static void schedTaskMakeRunnableWithSettingR0(struct Task *tsk, uint32_t r0val)	//only usable in SVC mode. or with irqs off
{
	tsk->ctx.regs.r0_r3[0] = r0val;
	schedTaskMakeRunnable(tsk);
}

static void schedRequestResched(void)				//safe to call from any irq AND from SVC context
{
	if (schedAmInSyscall() || irqsAreAllOff())
		schedPickNextTask();						//safe to call schedPickNextTask() directly
	else
		KTaskYield();								//perform a syscall so that it will be safe to call schedPickNextTask()
}

//callback from a timer set for a task op with a timeout (like a literal task delay or a mutex/sem/mbx/whatever syscall with a timout)
static void schedTaskOpTimeoutKernTimerCbk(ktimer_t timer, void* timerData)
{
	tid_t tid = (tid_t)timerData;
	irq_state_t irqSta;
	struct Task *tsk;
	
	irqSta = irqsAllOff();
	tsk = schedFindTaskByTid(tid);
	if (tsk && tsk->timer == timer && tsk->blockReason != SCHED_BLOCK_REASON_NONE) {
		if (tsk->blockReason == SCHED_BLOCK_REASON_WAIT_AND_MANUAL)
			tsk->blockReason = SCHED_BLOCK_REASON_MANUAL;
		else
			schedTaskMakeRunnable(tsk);
	}
	irqsRestoreState(irqSta);
}

static kstatus_t schedTaskBlockAtSource(struct Task *tsk, struct ListNode *listNodeInBlockSource, uint32_t blockReason, int32_t timeout)
{
	schedAssertCalledInSyscallMode();
	
	if (tsk->blockReason != SCHED_BLOCK_REASON_NONE)	//already blocked?
		return KERN_STATUS_INVAL_STATE;
	
	if (!timeout)										//time out right away if that is what was requested
		return KERN_STATUS_TIMEOUT;
	
	tsk->blockReason = blockReason;

	//remove from runnable list
	schedTaskRemoveFromRunHeapByPtr(tsk);

	//insert into block source at end if there is a block source
	if (listNodeInBlockSource)
		listInsertAfter(&tsk->inBlockSource, listNodeInBlockSource->prev);
	else
		listInit(&tsk->inBlockSource);

	if (mCurTask == tsk)								//blocking cur task? reschedule
		schedRequestResched();
	
	//need a timeout?
	if (timeout > 0) {
		tsk->timer = timerCreate(CREATE_4CC('_','s','c','t'), schedTaskOpTimeoutKernTimerCbk, (void*)tsk->tid, TIMER_TICKS_PER_MSEC * (uint32_t)timeout);
		if (!tsk->timer)
			fatal("Cannot create delay timer\n");
		
		return KERN_STATUS_TIMEOUT;						//so kernel stores that to r0 and at wake time it is seen
	}

	return KERN_STATUS_OK;
}

static bool schedIsUserTask(tid_t tid)
{
	return tid != 0 && tid != mTimersTaskTid && tid != mStackReaperTid;
}

static void schedStackReaperProc(void)			//higest prio in the system and not interruptible by other *TASKS*
{
	struct StackFreeListItem *memToFree;
		
	//forever
	while (1) {
		
		(void)KTaskWaitClr();
		
		//as long as there are chunk to free, free them
		do {	
			
			//atomically grab and remove first element from the linked list if it exists
			
			#if (defined(BUILD_FOR_THUMB_1) && !defined(HAVE_v8M_BASE)) || defined(BUILDING_FOR_BIG_ARM)
				
				irq_state_t sta = irqsAllOff();
				
				memToFree = mStacksFreeList;
				if (memToFree)
					mStacksFreeList = memToFree->next;
				
				irqsRestoreState(sta);
				
			#else
			
				uint32_t dummy1, dummy2;
			
				asm volatile(
					".syntax unified			\n\t"
					"1:							\n\t"
					"	ldrex %0, [%4]			\n\t"
					"	cbz   %0, 1f			\n\t"
					"	ldr   %1, [%0, %3]		\n\t"
					"	strex %2, %1, [%4]		\n\t"
					"	adds  %2, %2			\n\t"	//cmp with high regs might be 2 cycles, this is always one. STREX produces 1 or 0 so this test works properly
					"	bne   1b				\n\t"
					"1:							\n\t"
					:"=&l"(memToFree), "=&r"(dummy1), "=&r"(dummy2)
					:"I"(offsetof(struct StackFreeListItem, next)), "r"(&mStacksFreeList)
					:"memory","cc"
				);
			#endif
			
			if (memToFree) {
				ralSetSafeR9();				//set r9 to a safe value
				kheapFree(memToFree->actualPtr);
			}
			
		} while (memToFree);
		
		KTaskWait(-1);
	}
}

static kstatus_t schedTaskSuspend(tid_t tid)
{
	struct Task *tsk;
	
	if (!schedIsUserTask(tid))								//cannot suspend non-user tasks
		return KERN_STATUS_INVAL_REQ;
	
	tsk = schedFindTaskByTid(tid);
	if (!tsk)
		return KERN_STATUS_NOT_FOUND;
	
	if (tsk->blockReason == SCHED_BLOCK_REASON_WAIT) {
		tsk->blockReason = SCHED_BLOCK_REASON_WAIT_AND_MANUAL;
		return KERN_STATUS_OK;
	}
	
	if (tsk->blockReason == SCHED_BLOCK_REASON_DELAY) {		//we assume that if we're delayed and are suspended, the eventual resume overrides our delay
		timerDelete(tsk->timer);
		tsk->timer = 0;
		tsk->blockReason = SCHED_BLOCK_REASON_MANUAL;
		return KERN_STATUS_OK;
	}
	
	return schedTaskBlockAtSource(tsk, NULL, SCHED_BLOCK_REASON_MANUAL, -1);
}

static kstatus_t schedTaskStart(tid_t tid, uint32_t param)
{
	struct Task *tsk;
	
	if (!schedIsUserTask(tid))								//cannot start non-user tasks
		return KERN_STATUS_INVAL_REQ;
	
	tsk = schedFindTaskByTid(tid);
	if (!tsk)
		return KERN_STATUS_NOT_FOUND;
	
	if (tsk->blockReason != SCHED_BLOCK_REASON_NEW)
		return KERN_STATUS_INVAL_STATE;
	
	tsk->ctx.regs.r0_r3[0] = param;
	
	schedTaskMakeRunnable(tsk);
	return KERN_STATUS_OK;
}

static kstatus_t schedTaskResume(tid_t tid)
{
	struct Task *tsk;
	
	if (!schedIsUserTask(tid))								//cannot resume non-user tasks
		return KERN_STATUS_INVAL_REQ;
	
	tsk = schedFindTaskByTid(tid);
	if (!tsk)
		return KERN_STATUS_NOT_FOUND;
	
	if (tsk->blockReason != SCHED_BLOCK_REASON_MANUAL)
		return KERN_STATUS_INVAL_STATE;
	
	schedTaskMakeRunnable(tsk);
	return KERN_STATUS_OK;
}

static kstatus_t schedTaskDelay(int32_t delay)				//always acts on current task
{
	if (!mCurTask)
		fatal("a ghost called delay?\n");
	
	if (!delay)
		delay = 1;
	
	return schedTaskBlockAtSource(mCurTask, NULL, SCHED_BLOCK_REASON_DELAY, delay);
}

static kstatus_t schedTaskWaitClr(void)					//always acts on current task
{
	if (!mCurTask)
		fatal("a ghost called wait clear?\n");
	
	mCurTask->wakeCount = 0;
	
	return KERN_STATUS_OK;
}

static kstatus_t schedTaskWake(tid_t tid, bool allowKernel)
{
	struct Task *tsk;
	
	if (!allowKernel && !schedIsUserTask(tid))				//cannot wake non-user tasks
		return KERN_STATUS_INVAL_REQ;
	
	tsk = schedFindTaskByTid(tid);
	if (!tsk)
		return KERN_STATUS_NOT_FOUND;
	
	if (tsk->blockReason == SCHED_BLOCK_REASON_NONE) {		// not blocked? increment wake count
		if (!++tsk->wakeCount)	//wake overflow?
			--tsk->wakeCount;	//undo it?
		return KERN_STATUS_OK;
	}
	
	if (tsk->blockReason == SCHED_BLOCK_REASON_WAIT_AND_MANUAL) {
		tsk->blockReason = SCHED_BLOCK_REASON_MANUAL;
		return KERN_STATUS_OK;
	}
	
	if (tsk->blockReason != SCHED_BLOCK_REASON_WAIT)
		return KERN_STATUS_INVAL_STATE;

	schedTaskMakeRunnableWithSettingR0(tsk, KERN_STATUS_OK);
	return KERN_STATUS_OK;
}

static kstatus_t schedTaskWait(int32_t msec)				//always acts on current task
{
	if (!mCurTask)
		fatal("a ghost called wait?\n");
	
	if (mCurTask->wakeCount) {
		mCurTask->wakeCount--;
		return KERN_STATUS_OK;
	}

	return schedTaskBlockAtSource(mCurTask, NULL, SCHED_BLOCK_REASON_WAIT, msec);
}

static void __attribute__((naked)) schedTaskSuicideStub(void)
{
	asm volatile(
		"	movs r0, %1	\n\t"			//calls KTaskKillMe(), which is only valid to be called from this one place
		"	svc  %0		\n\t"
		:
		:"I"(KERNEL_SWI_NUM), "I"(SYSC_TASK_KILL_ME)
		:"memory"
	);
}

static void schedTaskTermFunc(void)		//runs in context of task being killed can do cleanup
{
	#ifdef THREAD_CLEANUP_PROC
		extern void THREAD_CLEANUP_PROC();
		
		THREAD_CLEANUP_PROC();
	#endif
}

static kstatus_t schedTaskDestroy(tid_t tid, struct CortexExcFrame *exc)
{
	//we need to: redirect task's pc to task term func and LR to suicide func
	uintptr_t termFuncP = (uintptr_t)&schedTaskTermFunc;
	struct Task *tsk;
	
	schedAssertCalledInSyscallMode();
	
	if (!schedIsUserTask(tid))					//cannot kill non-user tasks
		return KERN_STATUS_INVAL_REQ;
	
	tsk = schedFindTaskByTid(tid);
	
	if (!tsk)									//cannot kill a nonexistent task
		return KERN_STATUS_NOT_FOUND;
	
	//setup regs to do what we want
	if (tsk == mCurTask) {
		
		exc->lr = (uintptr_t)&schedTaskSuicideStub;
		exc->pc = termFuncP &~ 1;
		#ifndef BUILDING_FOR_BIG_ARM		//cortex
			if (termFuncP & 1)
				exc->sr |= CORTEX_SR_FLAG_T;
			else
				exc->sr &=~ CORTEX_SR_FLAG_T;
		#else								//legacy
			if (termFuncP & 1)
				exc->sr |= LEGACY_SR_FLAG_T;
			else
				exc->sr &=~ LEGACY_SR_FLAG_T;
		#endif
	}
	else {
		
		tsk->ctx.regs.lr = (uintptr_t)&schedTaskSuicideStub;
		tsk->ctx.regs.pc = termFuncP &~ 1;
		#ifndef BUILDING_FOR_BIG_ARM		//cortex
			if (termFuncP & 1)
				tsk->ctx.regs.sr |= CORTEX_SR_FLAG_T;
			else
				tsk->ctx.regs.sr &=~ CORTEX_SR_FLAG_T;
		#else								//legacy
			if (termFuncP & 1)
				tsk->ctx.regs.sr |= LEGACY_SR_FLAG_T;
			else
				tsk->ctx.regs.sr &=~ LEGACY_SR_FLAG_T;
		#endif
	}
	
	//make sure it is unblocked and remains so
	tsk->unblockable = 1;
	tsk->prio = SCHED_PRIO_SELF_CLEANUP;
	
	//run it at high prio
	listRemoveItem(&tsk->inBlockSource);
	schedTaskRemoveFromRunHeapByPtr(tsk);
	schedTaskPutInRunHeap(tsk);
	
	//make sure it runs
	schedRequestResched();
	return KERN_STATUS_OK;
}

static void schedTaskKillCurrent(uint32_t pc)
{
	struct StackFreeListItem *stack;
	uint32_t storeFail;
	kstatus_t sta;
	
	//NOTE: we CAN kill current task since after this call it will not execute even a single instr. No special handling is needed.
	//The only potential issue is the stack guard we might have set for the task. We clear it.
	
	schedAssertCalledInSyscallMode();
	
	if (!mCurTask)
		fatal("ghost request for assisted suicide?\n");

	if (!schedIsUserTask(mCurTask->tid))					//cannot kill non-user tasks
		fatal("kernel task suicide?\n");
	
	//verify the request is from the only place where it is allowed (second instr of schedTaskSuicideStub). pc will have advanced hence +4
	if (pc != ((((uintptr_t)schedTaskSuicideStub) + 4) &~ 1))
		fatal("Suicide assis syscall from an unexpected place: 0x%08x\n", pc);
	
	//block it
	if (KERN_STATUS_OK != schedTaskBlockAtSource(mCurTask, NULL, SCHED_BLOCK_REASON_MANUAL, -1))
		fatal("Cannot block suicided thread\n");

	//remove from list at block source
	listRemoveItem(&mCurTask->inBlockSource);
	
	//remove from tasks list
	listRemoveItem(&mCurTask->tasks);
	
	//remove stack guard (it was current thread so it is set
	mpuSetStackGuard(0);
	
	//add stack to our stacks-to-destroy list. no LDREX/STREX needed since we cannot be interrupted
	stack = (struct StackFreeListItem*)mCurTask->stackLimitAddr;
	stack->next = mStacksFreeList;
	stack->actualPtr = mCurTask->stackChunk;
	mStacksFreeList = stack;
	
	//free the struct
	slabFree(mTasksSlab, mCurTask);
	
	//wake up the reaper
	if (KERN_STATUS_OK != schedTaskWake(mStackReaperTid, true))
		fatal("Suicide failed\n");
}

#ifdef KERNEL_SUPPORTS_VFP

	static uint32_t schedCurTaskFpuFlagsOp(uint32_t flagBitsToSet, uint32_t flagBitsToClear)
	{
		return mCurTask->fpuCtx.flags = (mCurTask->fpuCtx.flags | flagBitsToSet) &~ flagBitsToClear;
	}

#endif

///scheduling primitives (common code)

static void* schedPrimFindById(struct ListNode* appropriateList, uint32_t id)
{
	struct ListNode *node;
	
	for (node = appropriateList->next; node != appropriateList; node = node->next) {
		
		struct SchedPrimitiveHead *prim = STRUCT_FROM_LIST_NODE(struct SchedPrimitiveHead, list, node);
		
		if (prim->id == id)
			return (void*)prim;
	}
	
	return NULL;
}

static uint32_t schedPrimFindFreeId(struct ListNode* appropriateList)
{
	static volatile uint32_t mNextId = 0;		//this method makas reuse unlikely in the near term
	int32_t curId, nextId;
	
	do {
		curId = mNextId;
		nextId = curId + 1;
		if (nextId <= 0)
			nextId = 1;
	} while (!atomicTestAndSet32(&mNextId, curId, nextId));

	while (schedPrimFindById(appropriateList, nextId)) {
		
		nextId++;
		if (nextId <= 0)
			nextId = 1;
	}
	
	return nextId;
}

static kstatus_t schedPrimConstruct(struct ListNode* appropriateList, uint32_t tag, struct Slab* appropriateSlabAllocator, void** structCreatedP)
{
	struct SchedPrimitiveHead *prim = (struct SchedPrimitiveHead*)slabAlloc(appropriateSlabAllocator);
	
	if (!prim)
		return KERN_STATUS_MEM_ERR;
	
	prim->id = schedPrimFindFreeId(appropriateList);
	prim->tag = tag;
	listInsertBefore(&prim->list, appropriateList);
	
	*structCreatedP = prim;
	return KERN_STATUS_OK;
}

static kstatus_t schedPrimDestruct(struct Slab* appropriateSlabAllocator, struct SchedPrimitiveHead* prim)
{
	listRemoveItem(&prim->list);
	slabFree(appropriateSlabAllocator, prim);
	
	return KERN_STATUS_OK;
}

static void schedPrimUnlockWithErrorTasksInList(struct ListNode *node, kstatus_t retCode)	//only works if tasks are linkid in via "inBlockSource" list which for sched primitives they always are
{
	while (!listIsEmpty(node)) {

		struct Task *tsk = STRUCT_FROM_LIST_NODE(struct Task, inBlockSource, node->next);
		
		logw("Unblocking task %u as its sync primitive was destroyed\n", tsk->tid);
		
		schedTaskMakeRunnableWithSettingR0(tsk, retCode);
	}
}

static mutex_t schedMutexCreate(uint32_t tag, bool recursive, mutex_t *mutP)
{
	struct Mutex *mut;
	kstatus_t sta;
	
	sta = schedPrimConstruct(&mMutexes, tag, mMutexesSlab, (void**)&mut);
	if (sta != KERN_STATUS_OK)
		return sta;
	
	mut->owner = 0;
	mut->lockCount = recursive ? 0 : MUTEX_LOCK_COUNT_NOT_RECURSIVE;
	listInit(&mut->blockedList);

	*mutP = mut->common.id;
	return KERN_STATUS_OK;
}

static kstatus_t schedMutexDestroy(mutex_t mid)
{
	struct Mutex *mut = (struct Mutex*)schedPrimFindById(&mMutexes, mid);

	if (!mut)
		return KERN_STATUS_NOT_FOUND;
	
	//un block all those blocked on it (this does affect scheduler)
	schedPrimUnlockWithErrorTasksInList(&mut->blockedList, KERN_STATUS_USER_FAULT);
	
	//delete it
	return schedPrimDestruct(mMutexesSlab, &mut->common);
}

static kstatus_t schedMutexLock(mutex_t mid, int32_t timeout)
{
	struct Mutex *mut = (struct Mutex*)schedPrimFindById(&mMutexes, mid);
	
	if (!mut)
		return KERN_STATUS_NOT_FOUND;
	
	if (!mut->owner || (mut->owner == mCurTask->tid && mut->lockCount != MUTEX_LOCK_COUNT_NOT_RECURSIVE)) {
		
		if (mut->lockCount != MUTEX_LOCK_COUNT_NOT_RECURSIVE)
			mut->lockCount++;

		mut->owner = mCurTask->tid;
		return KERN_STATUS_OK;
	}
	
	//else, block current task (this return value is what current task will see as return when and if it is resumed)
	return schedTaskBlockAtSource(mCurTask, &mut->blockedList, SCHED_BLOCK_REASON_MUTEX, timeout);
}

static kstatus_t schedMutexUnlock(mutex_t mid)
{
	struct Mutex *mut = (struct Mutex*)schedPrimFindById(&mMutexes, mid);
	struct Task* tsk;
	
	if (!mut)
		return KERN_STATUS_NOT_FOUND;
	
	if (mut->owner != mCurTask->tid)
		return KERN_STATUS_INVAL_REQ;
	
	if (mut->lockCount != MUTEX_LOCK_COUNT_NOT_RECURSIVE) {	//recursive mutex handler
		if (!mut->lockCount)	//underrun
			return KERN_STATUS_INVAL_REQ;
		mut->lockCount--;
		if (mut->lockCount)			//still locked?
			return KERN_STATUS_OK;
	}
	
	//no more contenders? mark as free and we're done
	if (listIsEmpty(&mut->blockedList)) {
		mut->owner = 0;
		return KERN_STATUS_OK;
	}
	
	tsk = STRUCT_FROM_LIST_NODE(struct Task, inBlockSource, mut->blockedList.next);
	mut->owner = tsk->tid;
	if (mut->lockCount != MUTEX_LOCK_COUNT_NOT_RECURSIVE)	//recursive mutex handler
		mut->lockCount = 1;
	
	schedTaskMakeRunnableWithSettingR0(tsk, KERN_STATUS_OK);
	return KERN_STATUS_OK;
}

static kstatus_t schedSemCreate(uint32_t tag, uint32_t initialVal, sema_t* sidP)
{
	struct Sem *sem;
	kstatus_t sta;
	
	sta = schedPrimConstruct(&mSems, tag, mSemsSlab, (void**)&sem);
	if (sta != KERN_STATUS_OK)
		return sta;
	
	sem->counter = initialVal;
	listInit(&sem->blockedList);
	
	*sidP = sem->common.id;
	return KERN_STATUS_OK;
}

static kstatus_t schedSemDestroy(sema_t sid)
{
	struct Sem *sem = (struct Sem*)schedPrimFindById(&mSems, sid);

	if (!sem)
		return KERN_STATUS_NOT_FOUND;
	
	//un block all those blocked on it (this does affect scheduler)
	schedPrimUnlockWithErrorTasksInList(&sem->blockedList, KERN_STATUS_USER_FAULT);
	
	//delete it
	return schedPrimDestruct(mSemsSlab, &sem->common);
}

static kstatus_t schedSemWait(sema_t sid, int32_t timeout)
{
	struct Sem *sem = (struct Sem*)schedPrimFindById(&mSems, sid);

	if (!sem)
		return KERN_STATUS_NOT_FOUND;
	
	if (sem->counter) {
		sem->counter--;
		return KERN_STATUS_OK;
	}
	
	//else, block current task (this return value is what current task will see as return when and if it is resumed)
	return schedTaskBlockAtSource(mCurTask, &sem->blockedList, SCHED_BLOCK_REASON_SEMA, timeout);
}

static kstatus_t schedSemPost(sema_t sid)
{
	struct Sem *sem = (struct Sem*)schedPrimFindById(&mSems, sid);

	if (!sem)
		return KERN_STATUS_NOT_FOUND;
	
	if (listIsEmpty(&sem->blockedList)) {
		sem->counter++;
		return KERN_STATUS_OK;
	}
	
	schedTaskMakeRunnableWithSettingR0(STRUCT_FROM_LIST_NODE(struct Task, inBlockSource, sem->blockedList.next), KERN_STATUS_OK);
	return KERN_STATUS_OK;
}

static kstatus_t schedMbxCreate(uint32_t tag, uint32_t depth, void* storage, mbx_t* midP)
{
	struct Mailbox *mbx;
	kstatus_t sta;
	
	sta = schedPrimConstruct(&mMbxs, tag, mMailboxesSlab, (void**)&mbx);
	if (sta != KERN_STATUS_OK)
		return sta;
	
	mbx->readIdx = 0;
	mbx->writeIdx = 0;
	mbx->size = depth + 1;
	mbx->messages = (uint32_t*)storage;
	listInit(&mbx->blockedList);

	*midP = mbx->common.id;
	return KERN_STATUS_OK;
}

static kstatus_t schedMbxDestroy(mbx_t mid, void** storageP)
{
	struct Mailbox *mbx = (struct Mailbox*)schedPrimFindById(&mMbxs, mid);

	if (!mbx)
		return KERN_STATUS_NOT_FOUND;
	
	//un block all those blocked on it (this does affect scheduler)
	schedPrimUnlockWithErrorTasksInList(&mbx->blockedList, KERN_STATUS_USER_FAULT);
	
	//return memory for freeing
	*storageP = mbx->messages;
	
	//delete it
	return schedPrimDestruct(mMailboxesSlab, &mbx->common);
}

static kstatus_t schedMbxWait(mbx_t mid, uint32_t* msgP, int32_t timeout)
{
	struct Mailbox *mbx = (struct Mailbox*)schedPrimFindById(&mMbxs, mid);

	if (!mbx)
		return KERN_STATUS_NOT_FOUND;
	
	if (mbx->readIdx != mbx->writeIdx) {
		if (!readUserWord(msgP, &mbx->messages[mbx->readIdx])) {	//storage space is user-provided...
			loge("MBX wait failed to read 0x%08x\n", &mbx->messages[mbx->readIdx]);
			return KERN_STATUS_USER_FAULT;
		}
		if (++mbx->readIdx == mbx->size)
			mbx->readIdx = 0;
		return KERN_STATUS_OK;
	}
	
	//else, block current task (this return value is what current task will see as return when and if it is resumed)
	return schedTaskBlockAtSource(mCurTask, &mbx->blockedList, SCHED_BLOCK_REASON_MBX, timeout);
}

static kstatus_t schedMbxSend(mbx_t mid, uint32_t msg)
{
	struct Mailbox *mbx = (struct Mailbox*)schedPrimFindById(&mMbxs, mid);
	uint32_t nextSlot;
	struct Task *tsk;
	
	if (!mbx)
		return KERN_STATUS_NOT_FOUND;
	
	while(1) {
		if (listIsEmpty(&mbx->blockedList))	{	//no waiters? write to queue
		
			nextSlot = mbx->writeIdx + 1;
			if (nextSlot == mbx->size)
				nextSlot = 0;
			
			if (nextSlot == mbx->readIdx)	//if next write would make it appear empty then it is curently full...
				return KERN_STATUS_FULL;
			
			if (!writeUserWord(&mbx->messages[mbx->writeIdx], msg)) {	//storage space is user-provided...
				loge("MBX send failed to write 0x%08x\n", &mbx->messages[mbx->writeIdx]);
				return KERN_STATUS_USER_FAULT;
			}
			mbx->writeIdx = nextSlot;
			
			return KERN_STATUS_OK;
		}
		
		//unblock a waiting task (only one since we're only sending one value)
		//this gets involved since we need to use that task's registers directly. it helps that we 100% know it is not "current"
		//also this might fault - hence the all-encompassing "while(1)"
		
		tsk = STRUCT_FROM_LIST_NODE(struct Task, inBlockSource, mbx->blockedList.next);
		if (writeUserWord((uint32_t*)tsk->ctx.regs.r0_r3[2], msg)) {
			schedTaskMakeRunnableWithSettingR0(tsk, KERN_STATUS_OK);
			return KERN_STATUS_OK;
		}
		
		//if we fault, we did NOT deliver a message, so we'll retry with another listener (or enqueue the msg) but we will wake up the faulting task with a fault error code
		schedTaskMakeRunnableWithSettingR0(tsk, KERN_STATUS_USER_FAULT);
	}
} 

static kstatus_t schedEvtGrpCreate(uint32_t tag, uint32_t state, evtgrp_t* eidP)
{
	struct EvtGrp *evtGrp;
	kstatus_t sta;
	
	sta = schedPrimConstruct(&mEvtGrps, tag, mEvtGrpsSlab, (void**)&evtGrp);
	if (sta != KERN_STATUS_OK)
		return sta;
	
	evtGrp->state = state;
	listInit(&evtGrp->blockedList);

	*eidP = evtGrp->common.id;
	return KERN_STATUS_OK;
}

static kstatus_t schedEvtGrpDestroy(evtgrp_t eid)
{
	struct EvtGrp *evtGrp = (struct EvtGrp*)schedPrimFindById(&mEvtGrps, eid);

	if (!evtGrp)
		return KERN_STATUS_NOT_FOUND;
	
	//un block all those blocked on it (this does affect scheduler)
	schedPrimUnlockWithErrorTasksInList(&evtGrp->blockedList, KERN_STATUS_USER_FAULT);
	
	//delete it
	return schedPrimDestruct(mEvtGrpsSlab, &evtGrp->common);
}

static bool schedEvtGrpIsMatch(uint32_t state, uint32_t wantedBits, bool wantAnd)
{
	if (!wantedBits)	//if we want no bits we match by default
		return true;
	else if (wantAnd) {
		if ((state & wantedBits) == wantedBits)
			return true;
	}
	else {
		if (state & wantedBits)
			return true;
	}
	
	return false;
}

static kstatus_t schedEvtGrpUpdate(evtgrp_t eid, uint32_t stateBic, uint32_t stateOrr)	//first bic then orr
{
	struct EvtGrp *evtGrp = (struct EvtGrp*)schedPrimFindById(&mEvtGrps, eid);
	uint32_t oldState, newState;
	
	if (!evtGrp)
		return KERN_STATUS_NOT_FOUND;
	
	oldState = evtGrp->state;
	newState = (oldState &~ stateBic) | stateOrr;
	evtGrp->state = newState;
	
	if (newState &~ oldState) {	//only re-evaluate blocked users if some bits got set since that is the onyl case when unblocking happens. this would not be the case if we supported "exact match" option instead of just "and" and "or"
		
		struct ListNode *node = evtGrp->blockedList.next;
		
		while (node != &evtGrp->blockedList) {

			struct Task *tsk = STRUCT_FROM_LIST_NODE(struct Task, inBlockSource, node);
			uint32_t wantedBits = tsk->ctx.regs.r0_r3[1];
			uint32_t *readBitsUserP = (uint32_t*)tsk->ctx.regs.r0_r3[2];
			bool wantAnd = !!tsk->ctx.regs.r0_r3[3];
			
			if (schedEvtGrpIsMatch(newState, wantedBits, wantAnd)) {
				//set ret val ourselves in either case
				tsk->ctx.regs.r0_r3[0] = writeUserWord(readBitsUserP, evtGrp->state) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
				node = node->next;
				schedTaskMakeRunnable(tsk);
			}
			else
				node = node->next;
		}
	}
	
	return KERN_STATUS_OK;
}

static kstatus_t schedEvtGrpWait(struct CortexExcFrame *exc, evtgrp_t eid, uint32_t wantedBits, bool wantAnd, uint32_t *returnedBitsP, uint32_t* userPtrReturnedEvtsP, int32_t timeout)
{
	struct EvtGrp *evtGrp = (struct EvtGrp*)schedPrimFindById(&mEvtGrps, eid);
	uint32_t curState;
	
	if (!evtGrp)
		return KERN_STATUS_NOT_FOUND;
	
	if (schedEvtGrpIsMatch(curState = evtGrp->state, wantedBits, wantAnd)) {
		*returnedBitsP = curState;
		return KERN_STATUS_OK;
	}
	
	//block in a way we can then understand
	//we reuse regs we can clobber to save state for later easy checking
	//keep in mind that r0 is not usable since it will be overriden by our return value from this func now
	exc->r1 = wantedBits;
	exc->r2 = (uintptr_t)userPtrReturnedEvtsP;	//this param correctly only used for blocking cases
	exc->r3 = wantAnd;
	return schedTaskBlockAtSource(mCurTask, &evtGrp->blockedList, SCHED_BLOCK_REASON_EVTGRP, timeout);
} 


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif


static void __attribute__((naked)) schedTimerCallCbk(void* cbkData, KernTimerCbk cbk, uint32_t r9)
{
	#ifdef BUILD_FOR_THUMB_1
	
		asm volatile(
			"	mov  r3, r9		\n\t"
			"	push {r3, lr}	\n\t"
			"	mov  r9, r2		\n\t"
			"	blx  r1			\n\t"
			"	pop  {r3}		\n\t"
			"	mov  r9, r3		\n\t"
			"	pop  {pc}		\n\t"
			:
			:
			:"memory","cc","r0","r1","r2","r3","r12"
		);
	#else
	
		asm volatile(
			"	push {r9, lr}	\n\t"
			"	mov  r9, r2		\n\t"
			"	blx  r1			\n\t"
			"	pop  {r9, pc}	\n\t"
			:
			:
			:"memory","cc","r0","r1","r2","r3","r12"
		);
	#endif
}

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif


static void schedTimersProc(void)			//higest prio in the system and not interruptible by other *TASKS*
{
	ralSetSafeR9();				//set r9 to a safe value
	
	while (1) {
		
		const uint64_t maxTime = 0xFFFFFFFFFFFFFFFFULL;
		uint64_t minTimeSeen, now, delay;
		struct ListNode *node;
		bool anythingDone;
		
		(void)KTaskWaitClr();
		
		do {
			minTimeSeen = maxTime;	//max uint value
			anythingDone = false;
			
			for (node = mTimers.next; node != &mTimers; node = node->next) {
	
				struct Timer *tmr = STRUCT_FROM_LIST_NODE(struct Timer, common.list, node);
				
				if (tmr->expires) {
					if (tmr->expires <= timerGetTime()) {
						
						tmr->expires = 0;
						schedTimerCallCbk(tmr->cbkData, tmr->cbk, tmr->r9);	//"tmr" might not exist after this call
						anythingDone = true;
						break;					//callback could have modified timers list, so assume nothing and restart from the begining
					}
					if (tmr->expires < minTimeSeen)
						minTimeSeen = tmr->expires;
				}
			}
			
			now = timerGetTime();
		} while (anythingDone || minTimeSeen <= now);
		
		delay = minTimeSeen - now;
		
		if (minTimeSeen == maxTime)
			(void)KTaskWait(-1);
		else if (delay < TIMER_TICKS_PER_MSEC)	//too little to wait - busy wait hre
			continue;
		else if (delay < 0x100000000ULL)		//to save us a long and complex divide in the common case
			(void)KTaskWait(((uint32_t)delay) / (uint32_t)TIMER_TICKS_PER_MSEC);
		else {
		
			delay /= TIMER_TICKS_PER_MSEC;
			
			if (delay >= 0x80000000)
				delay = 0x7FFFFFFF;
			
			(void)KTaskWait(delay);
		}
	}
}

static void schedTimersUpdate(void)
{
	schedTaskWake(mTimersTaskTid, true);
}

static kstatus_t schedTimerCreate(uint32_t tag, KernTimerCbk cbk, void* cbkData, tmr_t *tidP)
{
	struct Timer *tmr;
	kstatus_t sta;
	
	sta = schedPrimConstruct(&mTimers, tag, mTimersSlab, (void**)&tmr);
	if (sta != KERN_STATUS_OK)
		return sta;
	
	tmr->expires = 0;
	tmr->cbk = cbk;
	tmr->cbkData = cbkData;

	*tidP = tmr->common.id;
	return KERN_STATUS_OK;
}

static kstatus_t schedTimerDestroy(tmr_t tid)
{
	struct Timer *tmr = (struct Timer*)schedPrimFindById(&mTimers, tid);
	
	if (!tmr)
		return KERN_STATUS_NOT_FOUND;
	
	if (tmr->expires) {
		tmr->expires = 0;
		schedTimersUpdate();
	}
	
	return schedPrimDestruct(mTimersSlab,&tmr->common);
}

static kstatus_t schedTimerSet(tmr_t tid, uint32_t msec)
{
	struct Timer *tmr = (struct Timer*)schedPrimFindById(&mTimers, tid);
	
	if (!tmr)
		return KERN_STATUS_NOT_FOUND;
	
	asm("mov %0, r9":"=r"(tmr->r9));	//save r9
	tmr->expires = timerGetTime() + TIMER_TICKS_PER_MSEC * msec;
	schedTimersUpdate();

	return KERN_STATUS_OK;
}

static void schedIdleTaskProc(void)
{
	while(1) {
		machIdle();
	}
}

kstatus_t schedInit(void)
{
	kstatus_t sta;
	tid_t tid;
	
	listInit(&mAllTasks);
	listInit(&mSems);
	listInit(&mMutexes);
	
	logi("Initing scheduler\n");
	
	mTasksSlab = slabCreate(SCHED_MAX_THREADS, sizeof(struct Task));
	if (!mTasksSlab)
		fatal("alloc error tasks");
	mMutexesSlab = slabCreate(SCHED_MAX_MUTEXES, sizeof(struct Mutex));
	if (!mMutexesSlab)
		fatal("alloc error mutexes");
	mSemsSlab = slabCreate(SCHED_MAX_SEMAPHORES, sizeof(struct Sem));
	if (!mSemsSlab)
		fatal("alloc error sems");
	mTimersSlab = slabCreate(SCHED_MAX_TIMERS, sizeof(struct Timer));
	if (!mTimersSlab)
		fatal("alloc error timers");
	mEvtGrpsSlab = slabCreate(SCHED_MAX_EVENT_GROUPS, sizeof(struct EvtGrp));
	if (!mEvtGrpsSlab)
		fatal("alloc error event groups");
	mMailboxesSlab = slabCreate(SCHED_MAX_MAILBOXES, sizeof(struct Mailbox));
	if (!mMailboxesSlab)
		fatal("alloc error mailboxes");
	
	//create idle task (must be created first as that is what gives it zero tid)
	sta = schedTaskCreateInternal(SCHED_PRIO_IDLE, (uintptr_t)&schedIdleTaskProc, mIdleTaskStack, sizeof(mIdleTaskStack), CREATE_4CC('_','i','d','l'), NULL, false, &tid);
	if (KERN_STATUS_OK != sta) {
		fatal("Cannot create idle task\n");
		return sta;
	}
	
	if (tid)
		fatal("idle task tid not as expected\n");
	
	//make it runnable but do not schedule
	schedTaskMakeRunnableEx(schedFindTaskByTid(0), false);
	
	//create timers task
	sta = schedTaskCreateInternal(SCHED_PRIO_TIMERS, (uintptr_t)&schedTimersProc, mTimersTaskStack, sizeof(mTimersTaskStack), CREATE_4CC('_','t','m','r'), NULL, false, &mTimersTaskTid);
	if (KERN_STATUS_OK != sta) {
		fatal("Cannot create timers task\n");
		return sta;
	}
	
	//make it runnable but do not schedule
	schedTaskMakeRunnableEx(schedFindTaskByTid(mTimersTaskTid), false);
	
	sta = schedTaskCreateInternal(SCHED_PRIO_STACK_REAPER, (uintptr_t)&schedStackReaperProc, mReaperTaskStack, sizeof(mReaperTaskStack), CREATE_4CC('_','r','p','r'), NULL, false, &mStackReaperTid);
	if (KERN_STATUS_OK != sta) {
		fatal("Cannot create stack reaper task\n");
		return sta;
	}
	
	//make it runnable but do not schedule
	schedTaskMakeRunnableEx(schedFindTaskByTid(mStackReaperTid), false);
	
	logi("Scheduler init completed\n");
	
	return KERN_STATUS_OK;
}

kstatus_t schedStart(tid_t tid, void* param)
{
	struct Task *tsk = schedFindTaskByTid(tid);
	kstatus_t sta;
	
	if (mCurTask)
		return KERN_STATUS_INVAL_STATE;
	
	if (!tsk)
		return KERN_STATUS_NOT_FOUND;
	
	if (tsk->blockReason != SCHED_BLOCK_REASON_NEW)
		return KERN_STATUS_INVAL_STATE;
	
	tsk->ctx.regs.r0_r3[0] = (uintptr_t)param;
	schedTaskMakeRunnableEx(tsk, false);
		
	logi("Entering scheduled regime\n");
	
	schedSwitchToScheduledRegimeByCallingYield();
}

#ifdef EXPLICIT_EMU_CTX
	struct EmuCpuState* schedGetCurEmuContextFromFaultContext(void)
	{
		return mCurTask->emuCtx;
	}
	
	void __attribute__((used)) schedSetCurThreadEmuCtx(struct EmuCpuState* state)	//no locks needed since mCurTask is always safe to read BY MY OWN TASK
	{
		mCurTask->emuCtx = state;
	}
#endif

static void dumpmem(const void* from, uint32_t bytes, const char* label, uint32_t atAtime)
{
	const uint8_t *p = (const uint8_t*)from;
	uint32_t ofst = 0;
	
	for (ofst = 0; ofst < bytes; ofst += atAtime) {
		switch (atAtime) {
			case 1:
				loge("  [%s + 0x%03x == 0x%08x] = %02x\n", label, ofst, p + ofst, p[ofst]);
				break;
			case 2:
				loge("  [%s + 0x%03x == 0x%08x] = %04x\n", label, ofst, p + ofst, *(const uint16_t*)(p + ofst));
				break;
			case 4:
				loge("  [%s + 0x%03x == 0x%08x] = %08x\n", label, ofst, p + ofst, *(const uint32_t*)(p + ofst));
				break;
		}
	}
}

//return true if handled, sp passed in may not be quite valid
bool kernelSemihostingHandle(uint32_t *r0P, uint32_t *r1P, uint32_t *r2P, uint32_t *r3P, uint32_t r12, uint32_t sp, uint32_t lr, uint32_t pc, bool fromArm)
{
	uint32_t r0 = *r0P, r1 = *r1P;
	
	switch (r0) {
		case 0x03:			//SYS_WRITEC
			if (LOG_ERROR)
				pr("%c", *(char*)r1);
			return true;
		
		case 0x04:			//SYS_WRITE0
			if (LOG_ERROR)
				pr("%s", (const char*)r1);
			return true;
		
		case 0x18:			//angel_SWIreason_ReportException
			loge("TERMINATION REQUESTED VIA SEMIHOSTING SWI %s(code is 0x%04x)\n", fromArm? "from ARM " : "", r1);
			loge("regs: {0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x}\n", r0, r1, *r2P, *r3P, r12, lr, pc);
			loge("caution, sp data may not be valid\n");
			dumpmem((void*)sp, 0x80, "SP", 4);
			dumpmem((void*)(pc &~ 1), 0x80, "PC", 2);
			fatal("NOW HALTING\n");
			return true;
		
		case 0x150:			//kAdnSemihostNativeRegister: debug registration for a PNOlet
			logt("PNOlet registering:	\n\t");
			logt("  cookie:   0x%08x\n", *(uint32_t*)(r1 + 0x04));
			logt("  addr:     0x%08x\n", *(uint32_t*)(r1 + 0x08));
			logt("  db type:  '"FMT_4CC"' (0x%08x)\n", CNV_4CC(*(uint32_t*)(r1 + 0x10)), *(uint32_t*)(r1 + 0x10));
			logt("  db crid:  '"FMT_4CC"' (0x%08x)\n", CNV_4CC(*(uint32_t*)(r1 + 0x14)), *(uint32_t*)(r1 + 0x14));
			logt("  res type: '"FMT_4CC"' (0x%08x)\n", CNV_4CC(*(uint32_t*)(r1 + 0x18)), *(uint32_t*)(r1 + 0x18));
			logt("  res id:   %u (0x%04x)\n", *(uint16_t*)(r1 + 0x1c), *(uint16_t*)(r1 + 0x1c));
			return true;
		
		case 0x151:			//kAdnSemihostNativeUnregister: degug unregistration for a PNOlet
			logt("PNOlet unregistering:	\n\t");
			logt("  cookie:   0x%08x\n", *(uint32_t*)(r1 + 0x04));
			return true;
		
		case 0x152:			//kAdnSemihostDebugEnableSet: set debug flags
			loge("PNOlet attempted to enable debugging with flags 0x%08x. We do not support this\n", r1);
			return false;
		
		case 0x153:			//kAdnSemihostDebugEnableGet: get current debug flags
			logt("PNOlet asked if debugging is on\n");
			*r0P = 0;	//nope :)
			return true;
		
		case 0x155:			//kAdnSemihostDebugEnableGetSupported: get supported debug flags
			logt("PNOlet asked if debugging is supported\n");
			*r0P = 0;	//nope :)
			return true;
		
		default:
			logt("Unexpected semihosting call 0x%x seen at 0x%08x (lr 0x%08x), param 0x%08x\n", r0, pc, lr, r1);
			return false;
	}
}

uint8_t schedGetLockCount(void)
{
	return mSchedulerLockCnt;
}

struct Task* volatile* schedGetCurTaskP(void)
{
	return &mCurTask;
}

struct Task* schedGetNextTask(void)
{
	return mNextTask;
}

void __attribute__((used)) syscallHandle(uint8_t swiNo, struct CortexExcFrame *exc, uint32_t r4 /* only if supporting jitted semihosting swi */, uint32_t curTaskSp /* legacy arm only */)
{
	kstatus_t sta;
	uint32_t i;
	
	#ifndef BUILDING_FOR_BIG_ARM	//different method
	
		curTaskSp = (uintptr_t)(exc + 1);
	
	#endif
	
	if (swiNo == KERNEL_SWI_NUM) {
		switch (exc->r0) {
			case SYSC_TASK_GET_TID: {
				
				if (!exc->r1)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else if (mCurTask)
					exc->r0 = writeUserWord((tid_t*)exc->r1, mCurTask->tid) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
				else
					exc->r0 = KERN_STATUS_INTERNAL_ERR;
				break;
			}
			case SYSC_TASK_GET_INFO: {
				
				struct Task *tsk = schedFindTaskByTid((tid_t)exc->r1);
				struct KernTaskInfo ti;
				
				if (!tsk)
					exc->r0 = KERN_STATUS_NOT_FOUND;
				else {
				
					if (tsk == mCurTask) {
						ti.pc = (void*)exc->pc;
						ti.sp = (void*)curTaskSp;
					}
					else {
						ti.pc = (void*)tsk->ctx.regs.pc;
						ti.sp = (void*)tsk->ctx.regs.sp;
					}
					
					ti.stackLimit = tsk->stackLimitAddr;
					ti.tag = tsk->tag;
					ti.exinf = tsk->exinf;
					ti.prio = tsk->prio;
					
					exc->r0 = copyToUser((struct KernTaskInfo*)exc->r2, &ti, sizeof(struct KernTaskInfo)) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
				}
				break;
			}
			case SYSC_TASK_CREATE: {
				
				struct KernTaskCreateParams tcp;
				tid_t tid;
				
				if (!exc->r2)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else {
					if (!copyFromUser(&tcp, (const struct KernTaskCreateParams*)exc->r1, sizeof(struct KernTaskCreateParams)))
						sta = KERN_STATUS_USER_FAULT;
					else {
						sta = schedTaskCreate(tcp.prio, tcp.pc, tcp.stackMem, tcp.stackSz, tcp.tag, tcp.exinf, tcp.priv, &tid);
						if (sta == KERN_STATUS_OK)
							sta = writeUserWord((tid_t*)exc->r2, tid) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
					}
					exc->r0 = sta;
				}
				break;	
			}
			case SYSC_TASK_DESTROY: {
				exc->r0 = schedTaskDestroy(exc->r1, exc);
				break;
			}
			case SYSC_TASK_START: {
				exc->r0 = schedTaskStart(exc->r1, exc->r2);
				break;
			}
			case SYSC_TASK_SUSPEND: {
				exc->r0 = schedTaskSuspend(exc->r1);
				break;
			}
			case SYSC_TASK_RESUME: {
				exc->r0 = schedTaskResume(exc->r1);
				break;
			}
			case SYSC_TASK_WAIT: {
				exc->r0 = schedTaskWait(exc->r1);
				break;
			}
			case SYSC_TASK_WAIT_CLR: {
				exc->r0 = schedTaskWaitClr();
				break;
			}
			case SYSC_TASK_WAKE: {
				exc->r0 = schedTaskWake(exc->r1, false);
				break;
			}
			case SYSC_TASK_DELAY: {
				exc->r0 = schedTaskDelay(exc->r1);
				break;
			}
			case SYSC_TASK_SWITCHING: {
				if (!exc->r1) {
					mSchedulerLockCnt++;
					exc->r0 = KERN_STATUS_OK;
					break;
				}
				if (!mSchedulerLockCnt) {
					exc->r0 = KERN_STATUS_INVAL_STATE;	//cannot unlock what is already unlocked
					break;
				}
				
				mSchedulerLockCnt--;
				if (!mSchedulerLockCnt)
					schedRequestResched();
				
				exc->r0 = KERN_STATUS_OK;
				break;
			}
			case SYSC_TASK_YIELD: {
				schedRequestResched();
				break;
			}
			case SYSC_TASK_KILL_ME: {
				schedTaskKillCurrent(exc->pc);
				break;
			}
		#ifdef KERNEL_SUPPORTS_VFP
			case SYSC_SET_TASK_FPU_FLAGS: {
				exc->r0 = schedCurTaskFpuFlagsOp(exc->r1, exc->r2);
				break;	
			}
		#endif
			case SYSC_MUTEX_CREATE: {
	
				mutex_t mut = 0;
				if (!exc->r3)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else {
					sta = schedMutexCreate(exc->r1, exc->r2, &mut);
					if (sta == KERN_STATUS_OK)
						sta = writeUserWord((mutex_t*)exc->r3, mut) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
					exc->r0 = sta;
				}
				break;	
			}
			case SYSC_MUTEX_DESTROY: {
				exc->r0 = schedMutexDestroy(exc->r1);
				break;
			}
			case SYSC_MUTEX_RESERVE: {
				exc->r0 = schedMutexLock(exc->r1, exc->r2);
				break;
			}
			case SYSC_MUTEX_RELEASE: {
				exc->r0 = schedMutexUnlock(exc->r1);
				break;
			}
			case SYSC_SEMA_CREATE: {
	
				sema_t sem;
				if (!exc->r3)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else {
					sta = schedSemCreate(exc->r1, exc->r2, &sem);
					if (sta == KERN_STATUS_OK)
						sta = writeUserWord((sema_t*)exc->r3, sem) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
					exc->r0 = sta;
				}
				break;	
			}
			case SYSC_SEMA_DESTROY: {
				exc->r0 = schedSemDestroy(exc->r1);
				break;
			}
			case SYSC_SEMA_WAIT: {
				exc->r0 = schedSemWait(exc->r1, exc->r2);
				break;
			}
			case SYSC_SEMA_POST: {
				exc->r0 = schedSemPost(exc->r1);
				break;
			}
			case SYSC_MAILBOX_CREATE: {
	
				mbx_t mbx;
				if (!exc->r2)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else {
					struct KernMailboxCreateParams params;
					
					if (!copyFromUser(&params, (struct KernMailboxCreateParams*)exc->r1, sizeof(struct KernMailboxCreateParams)))
						sta = KERN_STATUS_USER_FAULT;
					else {
						sta = schedMbxCreate(params.tag, params.depth, params.storage, &mbx);
						if (sta == KERN_STATUS_OK)
							sta = writeUserWord((mbx_t*)exc->r2, mbx) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
					}
					exc->r0 = sta;
				}
				break;	
			}
			case SYSC_MAILBOX_DESTROY: {
				exc->r0 = schedMbxDestroy(exc->r1, (void**)exc->r2);
				break;
			}
			case SYSC_MAILBOX_SEND: {
				exc->r0 = schedMbxSend(exc->r1, exc->r2);
				break;
			}
			case SYSC_MAILBOX_WAIT: {
				
				uint32_t msg = 0;
				
				sta = schedMbxWait(exc->r1, &msg, exc->r3);
				if (sta == KERN_STATUS_OK)
					sta = writeUserWord((uint32_t*)exc->r2, msg) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
				exc->r0 = sta;
				break;
			}
			case SYSC_TIMER_CREATE: {
	
				struct KernTimerCreateParams tcp;
				tmr_t tmr;
				
				if (!exc->r2)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else {
					if (!copyFromUser(&tcp, (const struct KernTimerCreateParams*)exc->r1, sizeof(struct KernTimerCreateParams)))
						sta = KERN_STATUS_USER_FAULT;
					else {
						sta = schedTimerCreate(tcp.tag, tcp.cbk, tcp.cbkData, &tmr);
						if (sta == KERN_STATUS_OK)
							sta = writeUserWord((tmr_t*)exc->r2, tmr) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
					}
					exc->r0 = sta;
				}
				break;	
			}
			case SYSC_TIMER_DESTROY: {
				exc->r0 = schedTimerDestroy(exc->r1);
				break;
			}
			case SYSC_TIMER_SET: {
				exc->r0 = schedTimerSet(exc->r1, exc->r2);
				break;
			}
			case SYSC_EVTGRP_CREATE: {
	
				evtgrp_t evtGrp;
				
				if (!exc->r3)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else {
					sta = schedEvtGrpCreate(exc->r1, exc->r2, &evtGrp);
					if (sta == KERN_STATUS_OK)
						sta = writeUserWord((evtgrp_t*)exc->r3, evtGrp) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
					exc->r0 = sta;
				}
				break;	
			}
			case SYSC_EVTGRP_DESTROY: {
				exc->r0 = schedEvtGrpDestroy(exc->r1);
				break;
			}
			case SYSC_EVTGRP_CLEAR: {
				exc->r0 = schedEvtGrpUpdate(exc->r1, exc->r2, 0);
				break;
			}
			case SYSC_EVTGRP_SIGNAL: {
				exc->r0 = schedEvtGrpUpdate(exc->r1, 0, exc->r2);
				break;
			}
			case SYSC_EVTGRP_READ: {
				uint32_t state = 0;
				
				if (!exc->r2)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else {
					sta = schedEvtGrpWait(NULL, exc->r1, 0, false, &state, NULL, 0);	//will never take any time since we want no bits
					if (sta == KERN_STATUS_OK)
						sta = writeUserWord((uint32_t*)exc->r2, state) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
					exc->r0 = sta;
				}
				break;
			}
			case SYSC_EVTGRP_WAIT: {
				
				struct KernEvtGroupWaitParams wp;
				uint32_t state = 0;
				
				if (!copyFromUser(&wp, (const struct KernEvtGroupWaitParams*)exc->r2, sizeof(struct KernEvtGroupWaitParams)))
					exc->r0 = KERN_STATUS_USER_FAULT;
				else if (!wp.returnedEventsP)
					exc->r0 = KERN_STATUS_NULL_PARAM;
				else {
					sta = schedEvtGrpWait(exc, exc->r1, wp.wantedEvents, wp.wantAnd, &state, wp.returnedEventsP, wp.timeout);
					if (sta == KERN_STATUS_OK)
						sta = writeUserWord(wp.returnedEventsP, state) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
					exc->r0 = sta;
				}
				break;
			}
			case SYSC_RTC_GET: {
				uint32_t time;
				
				sta = drvRtcGet(&time);
				if (sta == KERN_STATUS_OK)
					sta = writeUserWord((uint32_t*)exc->r1, time) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
				exc->r0 = sta;
				break;	
			}
			case SYSC_RTC_SET: {
				exc->r0 = drvRtcSet(exc->r1);
				break;	
			}
			case SYSC_RTC_SET_ALARM: {
				exc->r0 = drvRtcSetAlarm(exc->r1);
				break;	
			}
			case SYSC_GET_UPTIME_MSEC: {
				
				uint32_t time = timerGetTime() / TIMER_TICKS_PER_MSEC;
				exc->r0 = writeUserWord((uint32_t*)exc->r1, time) ? KERN_STATUS_OK : KERN_STATUS_USER_FAULT;
				break;
			}
			case SYSC_SET_STOR_RAM_WRITABLE: {
				mpuSetStorageRamWriteable(!!exc->r1);
				exc->r0 = KERN_STATUS_OK;
				break;
			}
			default:
				fatal("unknown syscall {0x%08x from 0x%08x (from 0x%08x)\n", exc->r0, exc->pc, exc->lr);
				break;
		}
		return;
	}
	else if (swiNo == 0xAB) {		//basic semihosting support helps in debugging

		if (kernelSemihostingHandle(&exc->r0, &exc->r1, &exc->r2, &exc->r3, exc->r12, curTaskSp, exc->lr, exc->pc, false))
			return;
	}
#ifdef JIT_ENABLED
	else if (swiNo == KERNEL_ARM_SEMIHOSTING_SWI) {	//semihosting in JITted code
		
		if (kernelSemihostingHandle(&exc->r0, &exc->r1, &exc->r2, &exc->r3, exc->r12, curTaskSp, exc->lr, r4, true))
			return;
	}
#endif
	else if (swiNo == 0xAC) {	//host control lib backend
		
		hostCtl(exc);
		return;
	}
	
	fatal("SWI #0x%02x not supported (pc=0x%08x lr=0x%08x)\n", swiNo, exc->pc, exc->lr);
}

void kernelLogCurTaskForExc(void)
{
	struct Task *t = mCurTask;
	
	if (!t)
		loge("No currently known task\n");
	else {
		loge("Current task struct at 0x%08x\n", t);
		
		loge("tid: %u	prio: %u\n", t->tid, t->prio);
		loge("Task tag: '" FMT_4CC "'\n", CNV_4CC(t->tag));
	}
}
