#include "irqsCortex.h"
#include "printf.h"
#include "irqs.h"
#include "cpu.h"
#include "ral.h"

typedef uint8_t irq_state_t;

struct IrqHandlerInfo {
	HalIrqHandlerF handler;
	void* data;
};

static struct IrqHandlerInfo mIrqHandlers[CPU_NUM_IRQS] = {};


bool cpuIrqGetHandler(uint32_t irqNo, HalIrqHandlerF *curHandlerP, void** curUserDataP)
{
	irq_state_t t;
	if (irqNo >= CPU_NUM_IRQS)
		return false;
	
	t = irqsAllOff();
	if (curHandlerP)
		*curHandlerP = mIrqHandlers[irqNo].handler;
	if (curUserDataP)
		*curUserDataP = mIrqHandlers[irqNo].data;
	irqsRestoreState(t);
	
	return true;
}

bool cpuIrqSetHandler(uint32_t irqNo, HalIrqHandlerF newHandler, void* newUserData)
{
	irq_state_t t;
	if (irqNo >= CPU_NUM_IRQS)
		return false;
	
	t = irqsAllOff();
	mIrqHandlers[irqNo].handler = newHandler;
	mIrqHandlers[irqNo].data = newUserData;
	irqsRestoreState(t);
	
	return true;
}

bool cpuIrqSetState(uint32_t irqNo, bool on)
{
	irq_state_t t;
	bool wasOn;
	
	if (irqNo >= CPU_NUM_IRQS)
		return false;
	
	t = irqsAllOff();
	wasOn = NVIC_IsEnabledIRQ((IRQn_Type)irqNo);
	if (on)
		NVIC_EnableIRQ((IRQn_Type)irqNo);
	else
		NVIC_DisableIRQ((IRQn_Type)irqNo);
	irqsRestoreState(t);
	
	return wasOn;
}

void cpuIrqDefaultHandler(void)
{
	const struct IrqHandlerInfo *handler;
	uint32_t excNo, irqNo, r9state;
	
	asm volatile("mrs %0, IPSR":"=r"(excNo));
	
	if (excNo < 16 || (irqNo = excNo - 16) > CPU_NUM_IRQS) {
		fatal("Unexpected exception number %u passed to default int handler\n", excNo);
		return;
	}
	handler = &mIrqHandlers[irqNo];
	if (!handler->handler) {
		fatal("IRQ without handler caught (%u)\n", irqNo);
		return;
	}
	
	r9state = ralSetSafeR9();
	handler->handler(handler->data);
	ralRestoreR9(r9state);
}

#ifdef BUILD_FOR_THUMB_1

	irq_state_t irqsAllOff(void)
	{
		uint32_t ret;
		
		asm volatile(
			"	mrs	%0, PRIMASK		\n\t"
			"	cpsid i				\n\t"		//disable all IRQs, but keep syscalls available (a sideeffect of CPSID i is that SVC breaks)
			:"=&r"(ret)
			:
			:"memory"
		);
		
		return ret;
	}
	
	irq_state_t irqsAllOn(void)
	{
		uint32_t ret;
		
		asm volatile(
			"	mrs	%0, PRIMASK		\n\t"
			"	cpsie i				\n\t"
			:"=&r"(ret)
			:
			:"memory"
		);
		
		return ret;
	}
	
	void irqsRestoreState(irq_state_t state)
	{
		asm volatile(
			"msr PRIMASK, %0		\n\t"
			:
			:"r"(state)
			:"memory"
		);
	}
	
	bool irqsAreAllOff(void)		//on m3 they are NOT ever all off (we manpulate BASEPRI as you see)
	{
		uint32_t ret;
		
		asm volatile(
			"	mrs	%0, PRIMASK		\n\t"
			:"=&r"(ret)
			:
			:"memory"
		);
		
		return !!(ret & 1);
	}

#else
	//why this for v7M? usage faults basically. we cannot set their prio below 0, and we need to be able to take them.
	//this also keeps 

	irq_state_t irqsAllOff(void)
	{
		uint32_t ret;
		
		asm volatile(
			"	mrs	%0, BASEPRI		\n\t"
			"	msr BASEPRI, %1		\n\t"		//disable all IRQs, but keep syscalls available (a sideeffect of CPSID i is that SVC breaks)
			:"=&r"(ret)
			:"r"(4 << (8 - __NVIC_PRIO_BITS))	//prio vals 4 and higher are blocked
			:"memory"
		);
		
		return ret;
	}
	
	irq_state_t irqsAllOn(void)
	{
		uint32_t ret;
		
		asm volatile(
			"	mrs	%0, BASEPRI		\n\t"
			"	msr BASEPRI, %1		\n\t"
			:"=&r"(ret)
			:"r"(0)
			:"memory"
		);
		
		return ret;
	}
	
	void irqsRestoreState(irq_state_t state)
	{
		asm volatile(
			"msr BASEPRI, %0		\n\t"
			:
			:"r"(state)
			:"memory"
		);
	}
	
	bool irqsAreAllOff(void)		//on m3 they are NOT ever all off (we manpulate BASEPRI as you see)
	{
		return false;
	}
	
#endif


