#include "emuJitInternal.h"
#include "printf.h"
#include "emit.h"
#include "irqs.h"

//translating itself is not speed critical - let's save size
#pragma GCC push_options
#pragma GCC optimize("Os")


//due to use of jitPrvHandleSpStart() this might be suboptimal, but we live with it!
///we could teach jitPrvHandleSpStart to give us a few more temp regs, and maybe later we shall

static enum EmitStatus jitPrvEdspQaddQsub(struct EmitBuf *dest, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool add)
{
	uint32_t srReg, pushRegs;
	
	jitPrvFindTempRegs(instrAddr, (1 << rdNo) | (1 << rnNo) | (1 << rmNo), &pushRegs, NULL, false, &srReg, NULL);
	
	//PUSH {..pushRegs..}
	EMIT(HLpush, pushRegs);
	
	//MRS srReg, APSR
	EMIT(LLmrs, srReg, EMIT_SYSM_APSR);

	if (add) {
	
		//ADDS Rd, Rn, Rm
		EMIT(LLaddReg, rdNo, rnNo, rmNo, EmitShiftLsl, 0, EmitSetFlags, false);
	}
	else {
	
		//SUBS Rd, Rn, Rm
		EMIT(LLsubReg, rdNo, rnNo, rmNo, EmitShiftLsl, 0, EmitSetFlags, false);
	}
	
	//IT VS
	EMIT(LLit, EmitCcVs);

	//ORRVS srReg, CORTEX_SR_FLAG_Q
	EMIT(LLorrImm, srReg, srReg, CORTEX_SR_FLAG_Q, 0, EmitFlagsDoNotCare);
	
	//MSR APSR_nzcvq, srReg
	EMIT(LLmsr, EMIT_SYSM_APSR, EMIT_MSR_APSR_MASK_NZCVQ, srReg);
	
	//POP {..pushRegs..}
	EMIT(HLpop, pushRegs);
	
	return EmitErrNone;
}

static enum EmitStatus jitPrvEdspActOnVflag(struct EmitBuf *dest, uint32_t regDst, uint32_t regSr)
{
	//this is the fastest way i found to saturate in v7. it works, i promise. proof is up to the reader
			
	//ITTT VS
	EMIT(LLit, EmitCcVs);

	//ASR regDst, #31
	EMIT(LLmov, regDst, regDst, EmitShiftAsr, 31, EmitLeaveFlags, true);

	//RRX regDst
	EMIT(LLmov, regDst, regDst, EmitShiftRor, 0 /* ROR 0 is RRX */, EmitLeaveFlags, true);

	//ORRVS regSr, CORTEX_SR_FLAG_Q
	EMIT(LLorrImm, regSr, regSr, CORTEX_SR_FLAG_Q, 0, EmitFlagsDoNotCare);
	
	return EmitErrNone;
}

static enum EmitStatus jitPrvEdspQdaddQdsub(struct EmitBuf *dest, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool add)
{
	uint32_t srReg, pushRegs, doublingReg = rdNo;
	bool needDoublingReg = rdNo == rnNo;
	enum EmitStatus now;

	jitPrvFindTempRegs(instrAddr, (1 << rdNo) | (1 << rnNo) | (1 << rmNo), &pushRegs, NULL, false, &srReg, needDoublingReg ? &doublingReg : NULL, NULL);
	
	//PUSH {..pushRegs..}
	EMIT(HLpush, pushRegs);
	
	//MRS srReg, APSR
	EMIT(LLmrs, srReg, EMIT_SYSM_APSR);

	//ADDS doublingReg, Rm, Rm
	EMIT(LLaddReg, rdNo, rmNo, rmNo, EmitShiftLsl, 0, EmitSetFlags, false);
	
	//handle_sat Rd, srReg
	now = jitPrvEdspActOnVflag(dest, rdNo, srReg);
	if (now != EmitErrNone)
		return now;

	
	if (add) {
	
		//ADDS Rd, Rn, doublingReg
		EMIT(LLaddReg, rdNo, rnNo, doublingReg, EmitShiftLsl, 0, EmitSetFlags, false);
	}
	else {
	
		//SUBS Rd, Rn, doublingReg
		EMIT(LLsubReg, rdNo, rnNo, doublingReg, EmitShiftLsl, 0, EmitSetFlags, false);
	}
	
	//handle_sat Rd, srReg
	now = jitPrvEdspActOnVflag(dest, rdNo, srReg);
	if (now != EmitErrNone)
		return now;
	
	//MSR APSR_nzcvq, srReg
	EMIT(LLmsr, EMIT_SYSM_APSR, EMIT_MSR_APSR_MASK_NZCVQ, srReg);
	
	//POP {..pushRegs..}
	EMIT(HLpop, pushRegs);
	
	return EmitErrNone;
}

enum EmitStatus jitEmitQadd(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
{
	struct JitSimpleSpStatus sta;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdNo, NULL, &rdNo, &rmNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	now = jitPrvEdspQaddQsub(dest, instrAddr, rdNo, rnNo, rmNo, true);
	if (now != EmitErrNone)
		return now;

	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

enum EmitStatus jitEmitQsub(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
{
	struct JitSimpleSpStatus sta;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdNo, NULL, &rdNo, &rmNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	now = jitPrvEdspQaddQsub(dest, instrAddr, rdNo, rnNo, rmNo, false);
	if (now != EmitErrNone)
		return now;

	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

enum EmitStatus jitEmitQdadd(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
{
	struct JitSimpleSpStatus sta;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdNo, NULL, &rdNo, &rmNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	now = jitPrvEdspQdaddQdsub(dest, instrAddr, rdNo, rnNo, rmNo, true);
	if (now != EmitErrNone)
		return now;

	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

enum EmitStatus jitEmitQdsub(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
{
	struct JitSimpleSpStatus sta;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdNo, NULL, &rdNo, &rmNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	now = jitPrvEdspQdaddQdsub(dest, instrAddr, rdNo, rnNo, rmNo, false);
	if (now != EmitErrNone)
		return now;

	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

static enum EmitStatus jitPrvEdspIsolateRegToTop(struct EmitBuf *dest, uint32_t regDst, uint32_t regSrc, bool top, enum EmitFlagSettingPrefs flagPrefs)
{
	if (top) {
		
		//LSR regDst, regSrc, #16
		EMIT(LLmov, regDst, regSrc, EmitShiftLsr, 16, flagPrefs, false);

		//LSL regDst, regDst, #16
		EMIT(LLmov, regDst, regDst, EmitShiftLsr, 16, flagPrefs, false);
	}
	else {
		
		//LSL regDst, regSrc, #16
		EMIT(LLmov, regDst, regSrc, EmitShiftLsr, 16, flagPrefs, false);
	}
	
	return EmitErrNone;
}

static enum EmitStatus jitPrvEdspIsolateRegToBottom(struct EmitBuf *dest, uint32_t regDst, uint32_t regSrc, bool top, enum EmitFlagSettingPrefs flagPrefs)
{
	if (top) {
		
		//ASR regDst, regSrc, #16
		EMIT(LLmov, regDst, regSrc, EmitShiftAsr, 16, flagPrefs, false);
	}
	else {
		
		//SXTH regDst, regSrc
		EMIT(LLextend, regDst, regSrc, 0, false, false);
	}
	
	return EmitErrNone;
}

static enum EmitStatus jitPrvEdspSmulSmlaXY(struct EmitBuf *dest, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, int32_t raNo, bool nTop, bool mTop)
{
	uint32_t pushRegs, regN, regM, srReg;
	enum EmitStatus now;
	
	jitPrvFindTempRegs(instrAddr, (1 << rdNo) | (1 << rnNo) | (1 << rmNo) | ((raNo < 0) ? 0 : (1 << raNo)), &pushRegs, NULL, false, &regN, &regM, (raNo >= 0) ? &srReg : NULL, NULL);
	
	//PUSH {..pushRegs..}
	EMIT(HLpush, pushRegs);	
	
	if (raNo >= 0) {
		
		//MRS srReg, APSR
		EMIT(LLmrs, srReg, EMIT_SYSM_APSR);
	}
	
	//mov_to_bottom_16 regN, Rn, as_per_nTop
	now = jitPrvEdspIsolateRegToBottom(dest, regN, rnNo, nTop, raNo >= 0 ? EmitFlagsDoNotCare : EmitLeaveFlags);
	if (now != EmitErrNone)
		return now;
	
	//mov_to_bottom_16 regM, Rm, as_per_nTop
	now = jitPrvEdspIsolateRegToBottom(dest, regM, rmNo, mTop, raNo >= 0 ? EmitFlagsDoNotCare : EmitLeaveFlags);
	if (now != EmitErrNone)
		return now;
	
	if (raNo < 0) {		//no accumulate
		
		//MUL Rd, regN, regM
		EMIT(LLmulReg, rdNo, regN, regM, EmitFlagsDoNotCare, false);
	}
	else {
	
		//MUL regN, regN, regM
		EMIT(LLmulReg, regN, regN, regM, EmitFlagsDoNotCare, false);
		
		//ADDS Rd, regN, Ra
		EMIT(LLaddReg, rdNo, regN, raNo, EmitShiftLsl, 0, EmitSetFlags, false);
		
		//handle_sat Rd, srReg
		now = jitPrvEdspActOnVflag(dest, rdNo, srReg);
		if (now != EmitErrNone)
			return now;
		
		//MSR APSR_nzcvq, srReg
		EMIT(LLmsr, EMIT_SYSM_APSR, EMIT_MSR_APSR_MASK_NZCVQ, srReg);
	}
	
	//POP {..pushRegs..}
	EMIT(HLpop, pushRegs);
	
	return EmitErrNone;
}

enum EmitStatus jitEmitSmulxy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool nTop, bool mTop)
{
	struct JitSimpleSpStatus sta;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdNo, NULL, &rdNo, &rmNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	now = jitPrvEdspSmulSmlaXY(dest, instrAddr, rdNo, rnNo, rmNo, -1, nTop, mTop);
	if (now != EmitErrNone)
		return now;
	
	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

enum EmitStatus jitEmitSmlaxy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool nTop, bool mTop)
{
	struct JitSimpleSpStatus sta;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdNo, NULL, &rdNo, &rmNo, &raNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	now = jitPrvEdspSmulSmlaXY(dest, instrAddr, rdNo, rnNo, rmNo, raNo, nTop, mTop);
	if (now != EmitErrNone)
		return now;
	
	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

enum EmitStatus jitEmitSmulwy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool mTop)
{
	struct JitSimpleSpStatus sta;
	uint32_t regM, pushRegs;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdNo, NULL, &rdNo, &rmNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	jitPrvFindTempRegs(instrAddr, (1 << rdNo) | (1 << rnNo) | (1 << rmNo), &pushRegs, NULL, false, &regM, NULL);
	
	//PUSH {..pushRegs..}
	EMIT(HLpush, pushRegs);	
	
	//mov_to_top_16 regM, Rm, as_per_nTop
	now = jitPrvEdspIsolateRegToTop(dest, regM, rmNo, mTop, EmitLeaveFlags);
	if (now != EmitErrNone)
		return now;
	
	//SMULL regM, Rd, Rn, regM		//lo is irrelevant, high word is wanted result
	EMIT(LLsmull, regM, rdNo, rnNo, regM);
	
	//POP {..pushRegs..}
	EMIT(HLpop, pushRegs);

	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

enum EmitStatus jitEmitSmlawy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool mTop)
{
	uint32_t regM, pushRegs, srReg, raCopy = raNo;
	struct JitSimpleSpStatus sta;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdNo, NULL, &rdNo, &rmNo, &raNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	jitPrvFindTempRegs(instrAddr, (1 << rdNo) | (1 << rnNo) | (1 << rmNo) | (1 << raNo), &pushRegs, NULL, false, &regM, &srReg, (raNo == rdNo) ? &raCopy : NULL, NULL);
	
	//PUSH {..pushRegs..}
	EMIT(HLpush, pushRegs);	
	
	//MRS srReg, APSR
	EMIT(LLmrs, srReg, EMIT_SYSM_APSR);

	//mov_to_top_16 regM, Rm, as_per_nTop
	now = jitPrvEdspIsolateRegToTop(dest, regM, rmNo, mTop, EmitFlagsDoNotCare);
	if (now != EmitErrNone)
		return now;
	
	if (raCopy != raNo) {		//we need to copy Ra out
		
		//MOV raCopy, Ra
		EMIT(LLmov, raCopy, raNo, EmitShiftLsl, 0, EmitFlagsDoNotCare, false);
	}
	
	//SMULL regM, Rd, Rn, regM		//lo is irrelevant, high word is wanted result
	EMIT(LLsmull, regM, rdNo, rnNo, regM);
	
	//ADDS Rd, Rd, raCopy
	EMIT(LLaddReg, rdNo, rdNo, raCopy, EmitShiftLsl, 0, EmitSetFlags, false);
	
	//handle_sat Rd, srReg
	now = jitPrvEdspActOnVflag(dest, rdNo, srReg);
	if (now != EmitErrNone)
		return now;
	
	//MSR APSR_nzcvq, srReg
	EMIT(LLmsr, EMIT_SYSM_APSR, EMIT_MSR_APSR_MASK_NZCVQ, srReg);
	
	//POP {..pushRegs..}
	EMIT(HLpop, pushRegs);
	
	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

enum EmitStatus jitEmitSmlalxy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo, bool nTop, bool mTop)
{
	uint32_t regN, regM, pushRegs;
	struct JitSimpleSpStatus sta;
	struct EmitBuf ccSkip;
	enum EmitStatus now;
	
	if (cc != EmitCcAl)
		EMIT(SaveSpace, &ccSkip, 1);
		
	now = jitPrvHandleSpStart(dest, instrAddr, &sta, &rdLoNo, &rdHiNo, &rnNo, &rmNo, NULL);
	if (now != EmitErrNone)
		return now;
	
	jitPrvFindTempRegs(instrAddr, (1 << rdLoNo) | (1 << rdHiNo) | (1 << rnNo) | (1 << rmNo), &pushRegs, NULL, false, &regN, &regM, NULL);
	
	//PUSH {..pushRegs..}
	EMIT(HLpush, pushRegs);	
	
	//mov_to_bottom_16 regN, Rn, as_per_nTop
	now = jitPrvEdspIsolateRegToBottom(dest, regN, rnNo, nTop, EmitLeaveFlags);
	if (now != EmitErrNone)
		return now;
	
	//mov_to_bottom_16 regM, Rm, as_per_nTop
	now = jitPrvEdspIsolateRegToBottom(dest, regM, rmNo, mTop, EmitLeaveFlags);
	if (now != EmitErrNone)
		return now;
	
	//SMLAL RdLo, RdHi, regN, regM
	EMIT(LLsmlal, rdLoNo, rdHiNo, regN, regM);
	
	//POP {..pushRegs..}
	EMIT(HLpop, pushRegs);

	now = jitPrvHandleSpEnd(dest, &sta);
	if (now != EmitErrNone)
		return now;
	
	if (cc != EmitCcAl)
		EMIT_TO(LLbranch, &ccSkip, emitGetPtrToJumpHere(dest), emitCcInvert(cc));
	
	return EmitErrNone;
}

