#include <string.h>
#include "kernel.h"
#include "entry.h"
#include "printf.h"
#include "heap.h"
#include "dal.h"


static int32_t mSecondsOffset = 2082844800;			//palmos wants seconds since start of 1904, unix since 1970

void kernelInit(void (*entryFunc)(void), void* hyperFuncIfAny)
{
	hyperInit(hyperFuncIfAny);
	machInit(STAGE_SETUP_HEAPS, NULL);
	entryFunc();
}

kstatus_t kernelInitLate(void)
{
	return KERN_STATUS_OK;
}

kstatus_t kernelDriversInit(void)
{
	return KERN_STATUS_OK;
}

kstatus_t KTaskCreate(uint32_t tag, void* pc, void* stackMem, uint32_t stackSz, void* exinf, uint16_t prio, bool priv, tid_t* tidOutP)
{
	struct HyTaskCreate hy = {
		.prio = prio,
		.stackSz = stackSz,
		.func = pc,
		.tag = tag,
		.exinf = exinf,
		.spVal = (uint32_t)(uintptr_t)(stackMem + stackSz),
	};
	
	asm("str r9, %0"::"m"(hy.r9));		//new tasks inherit our r9
	*tidOutP = hyperCall(HYPER_TASK_CREATE, &hy);

	return *tidOutP ? KERN_STATUS_OK : KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KTaskGetTid(tid_t* tidOutP)
{
	*tidOutP = hyperCall(HYPER_TASK_GET_TID, NULL);
	return *tidOutP ? KERN_STATUS_OK : KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KTaskGetInfo(tid_t tid, struct KernTaskInfo* ti)
{
	struct HyTaskInfo hy = {.tid = tid,};
	
	if (hyperCall(HYPER_TASK_GET_INFO, &hy))
		return KERN_STATUS_INTERNAL_ERR;
	
	memset(ti, 0, sizeof(*ti));
	ti->pc = hy.func;
	ti->exinf = hy.exinf;
	ti->tag = hy.tag;
	ti->sp = (void*)(uintptr_t)(hy.initialSp - hy.stackSz + 4);	//"using" all but 4 bytes of stack
	ti->stackLimit = (void*)hy.initialSp;
	ti->prio = hy.prio;

	return KERN_STATUS_OK;
}

kstatus_t KTaskDestroy(tid_t tid)
{
	struct HyTaskInfo hy = {.tid = tid,};
	logt("getting info of deleted task\n");
	if (hyperCall(HYPER_TASK_GET_INFO, &hy))
		return KERN_STATUS_INTERNAL_ERR;
	
	logt("deleting task\n");
	if (hyperCall(HYPER_TASK_DEL, (void*)(uintptr_t)tid))
		return KERN_STATUS_INTERNAL_ERR;
		
	logt("freeing stack of deleted task\n");
	kheapFree((void*)(hy.initialSp - hy.stackSz));
	return KERN_STATUS_OK;
}

kstatus_t KTaskStart(tid_t tid, void* param)
{
	struct HyTaskStart hy = {
		.tid = tid,
		.param = param,
	};
	
	return hyperCall(HYPER_TASK_START, &hy) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KTaskSuspend(tid_t tid)
{
	fatal("KTaskSuspend() not possible on posix, Go NT!\n");
	return KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KTaskResume(tid_t tid)
{
	fatal("KTaskResume() not possible on posix, Go NT!\n");
	return KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KTaskWait(int32_t msec)
{
	//XXX: this is technically wrong
	// really param is signed and negative means forever, so this icode will sometimes end when it shouldn't.
	// we do not worry about such long uptimes on the virtual platform
	switch(hyperCall(HYPER_TASK_WAIT, (void*)(uintptr_t)(uint32_t)msec)) {
		case 0:
			return KERN_STATUS_OK;
		case 1:
			return KERN_STATUS_TIMEOUT;
		default:
			return KERN_STATUS_INTERNAL_ERR;
	}
}

kstatus_t KTaskWaitClr(void)
{
	kstatus_t sta;
	
	logt("%s() my tid is %u\n", __func__, (uint32_t)hyperCall(HYPER_TASK_GET_TID, NULL));

	do {
		sta = KTaskWait(0);
	} while (sta == KERN_STATUS_OK);
	
	if (sta == KERN_STATUS_TIMEOUT)
		sta = KERN_STATUS_OK;
	
	return sta;
}

kstatus_t KTaskWake(tid_t tid)
{
	return hyperCall(HYPER_TASK_WAKE, (void*)(uintptr_t)tid) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KTaskDelay(int32_t msec)
{
	hyperCall(HYPER_TASK_DELAY, (void*)(uintptr_t)msec);
	
	return KERN_STATUS_OK;
}

kstatus_t KTaskSwitching(bool on)
{
	kstatus_t sta;
	static mutex_t mMutex = 0;
	
	//this is not at all right, but posix won'y let us do this right
	//we just pray that whatever this wants to block will also try to lock out other tasks using this same abi
	// and would thus block here
	
	
	if (!mMutex) {
		logt("%s: creating mutex\n", __func__);
		sta = KMutexCreate(CREATE_4CC('_','t','s','m'), true, &mMutex);
		if (sta != KERN_STATUS_OK)
			return sta;
	}
	
	if (on)
		return KMutexRelease(mMutex);
	else
		return KMutexReserve(mMutex, -1);
}

kstatus_t KMutexCreate(uint32_t tag, bool recursive, mutex_t* mutHandleP)
{
	*mutHandleP = hyperCall(HYPER_MUTEX_NEW, (void*)(uintptr_t)tag);
	return *mutHandleP ? KERN_STATUS_OK : KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KMutexDestroy(mutex_t mut)
{
	return hyperCall(HYPER_MUTEX_DEL, (void*)(uintptr_t)mut) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KMutexReserve(mutex_t mut, int32_t timeoutMsec)
{
	struct HyMutexLock hy = {.mut = mut, .timeoutMsec = timeoutMsec,} ;
	int64_t rawRet = hyperCall(HYPER_MUTEX_LOCK, &hy);
	
	switch(rawRet) {
		case 0:
			return KERN_STATUS_OK;
		case 1:
			return KERN_STATUS_TIMEOUT;
		default:
			return KERN_STATUS_INTERNAL_ERR;
	}
}

kstatus_t KMutexRelease(mutex_t mut)
{
	return hyperCall(HYPER_MUTEX_UNLOCK, (void*)(uintptr_t)mut) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KSemaphoreCreate(uint32_t tag, uint32_t initialVal, sema_t* semHandleP)
{
	*semHandleP = hyperCall(HYPER_SEM_NEW, (void*)(uintptr_t)initialVal);
	return *semHandleP ? KERN_STATUS_OK : KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KSemaphoreDestroy(sema_t semHandle)
{
	return hyperCall(HYPER_SEM_DEL, (void*)(uintptr_t)semHandle) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KSemaphoreWait(sema_t semHandle, int32_t timeout)
{
	struct hyperSemWait op = {.semHandle = semHandle, .timeoutMsec = timeout, };
	switch(hyperCall(HYPER_SEM_WAIT, &op)) {
		case 0:
			return KERN_STATUS_OK;
		case 1:
			return KERN_STATUS_TIMEOUT;
		default:
			return KERN_STATUS_INTERNAL_ERR;
	}
}

kstatus_t KSemaphorePost(sema_t semHandle)
{
	return hyperCall(HYPER_SEM_POST, (void*)(uintptr_t)semHandle) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KMailboxCreate(uint32_t tag, uint32_t depth, void* storage, mbx_t* mbxHandleOutP)
{
	//we do not use the user-allocated space but we also do not track th epointer so we free it here now
	
	*mbxHandleOutP = hyperCall(HYPER_MBX_NEW, (void*)(uintptr_t)depth);
	
	if (*mbxHandleOutP) {
		kheapFree(storage);
		return KERN_STATUS_OK;
	}
	
	return KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KMailboxDestroy(mbx_t mailboxID, void** storageP)
{
	//since we never kept track of the storage, we do not have any to give back - pass back a new allocation so caller can free it as it expects
	
	if (hyperCall(HYPER_MBX_DEL, (void*)(uintptr_t)mailboxID))
		return KERN_STATUS_INTERNAL_ERR;
	
	*storageP = kheapAlloc(4);
	
	return KERN_STATUS_OK;
}

kstatus_t KMailboxSend(mbx_t mailboxID, uint32_t message)
{
	struct HyMbxSend hy = {.mailbox = mailboxID, .msg = message,};
	
	return hyperCall(HYPER_MBX_SEND, &hy) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KMailboxWait(mbx_t mailboxID, uint32_t* msgP, int32_t timeout)
{
	struct HyMbxRecv hy = {.mailbox = mailboxID, .timeout = timeout,};

	switch(hyperCall(HYPER_MBX_RECV, &hy)) {
		case 0:
			*msgP = hy.msg;
			return KERN_STATUS_OK;
		case 1:
			return KERN_STATUS_TIMEOUT;
		default:
			return KERN_STATUS_INTERNAL_ERR;
	}
}

kstatus_t KTimerCreate(uint32_t tag, KernTimerCbk cbk, void* cbkData, tmr_t *timerIDP)
{
	struct hyperTimerCreate hy = {.fn = cbk, .data = cbkData,};
	*timerIDP = hyperCall(HYPER_TIMER_NEW, &hy);
	return *timerIDP ? KERN_STATUS_OK : KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KTimerDestroy(tmr_t timerID)
{
	return hyperCall(HYPER_TIMER_DEL, (void*)(uintptr_t)timerID) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KTimerSet(tmr_t timerID, uint32_t msec)
{
	struct hyperTimerSet hy = {.timer = timerID, .msec = msec, };
	return hyperCall(HYPER_TIMER_SET, &hy) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KEventGroupCreate(uint32_t tag, uint32_t initialState, evtgrp_t* evtGrpHandleOut)
{
	*evtGrpHandleOut = hyperCall(HYPER_EVTGRP_NEW, (void*)(uintptr_t)initialState);
	return *evtGrpHandleOut ? KERN_STATUS_OK : KERN_STATUS_INTERNAL_ERR;
}

kstatus_t KEventGroupDestroy(evtgrp_t evtGrpHandle)
{
	return hyperCall(HYPER_EVTGRP_DEL, (void*)(uintptr_t)evtGrpHandle) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KEventGroupClear(evtgrp_t evtGrpHandle, uint32_t evts)
{
	struct hyEvtGrpSimpleOp op = {.evtGrpHandle = evtGrpHandle, .val = evts,};
	return hyperCall(HYPER_EVTGRP_CLR_BITS, &op) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KEventGroupSignal(evtgrp_t evtGrpHandle, uint32_t evts)
{
	struct hyEvtGrpSimpleOp op = {.evtGrpHandle = evtGrpHandle, .val = evts,};
	return hyperCall(HYPER_EVTGRP_SET_BITS, &op) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KEventGroupRead(evtgrp_t evtGrpHandle, uint32_t* curstateOutP)
{
	uint64_t readVal = hyperCall(HYPER_EVTGRP_READ, (void*)(uintptr_t)evtGrpHandle);
	
	if (readVal >> 32)
		return KERN_STATUS_INTERNAL_ERR;
	else {
		*curstateOutP = readVal;
		return KERN_STATUS_OK;
	}
}

kstatus_t KEventGroupWait(evtgrp_t evtGrpHandle, uint32_t wantedEvents, uint32_t *returnedEventsP, int32_t timeout, bool wantAnd)
{
	struct hyEvtGrpWait op = {
			.evtGrpHandle = evtGrpHandle,
			.desiredBits = wantedEvents,
			.timeoutMsec = timeout,
			.needAllBits = wantAnd, 
	};
	uint64_t rawRet;
	rawRet = hyperCall(HYPER_EVTGRP_WAIT, &op);
	switch (rawRet >> 32) {
		case 0:		//success
			*returnedEventsP = rawRet;
			return KERN_STATUS_OK;
		case 1:		//timeout
			return KERN_STATUS_TIMEOUT;
		default:	//error of some sort
			return KERN_STATUS_INTERNAL_ERR;
	}
}

kstatus_t KGetUptimeMsec(uint32_t *uptimeSecsP)
{
	*uptimeSecsP = hyperCall(HYPER_GET_MSEC_SINCE_BOOT, NULL);
	
	return KERN_STATUS_OK;
}

kstatus_t KRtcGet(uint32_t *rtcP)
{
	*rtcP = mSecondsOffset + hyperCall(HYPER_GET_SECONDS_EPOCH, NULL);
	return KERN_STATUS_OK;
}

kstatus_t KRtcSet(uint32_t seconds)
{
	uint32_t real = hyperCall(HYPER_GET_SECONDS_EPOCH, NULL);
	mSecondsOffset = seconds - real;
	
	return KERN_STATUS_OK;
}

kstatus_t KRtcSetAlarm(uint32_t seconds)
{
	return hyperCall(HYPER_SET_RTC_ALARM, (void*)(uintptr_t)(seconds ? seconds - mSecondsOffset : 0)) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}

kstatus_t KSetStorageRamWriteable(bool allowStorageRamWrites)
{
	return hyperCall(HYPER_SET_STOR_HEAP_PROT, (void*)(uintptr_t)allowStorageRamWrites) ? KERN_STATUS_INTERNAL_ERR : KERN_STATUS_OK;
}




