#define WEAK __attribute__ ((weak))
#define ALIAS(f) __attribute__ ((weak, alias (#f)))

#define NON_PORTABLE
#include <HwrMiscFlags.h>
#undef NON_PORTABLE

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "stm32f429xx.h"
#include "memmap.h"
#include "printf.h"
#include "timers.h"
#include "entry.h"
#include "heap.h"
#include "irqs.h"
#include "boot.h"
#include "mpu.h"


void __attribute__((used)) IntDefaultHandler(void)
{
	cpuIrqDefaultHandler();
}

#define VEC_(nm, pfx)	void nm##pfx(void) __attribute__ ((weak, alias ("IntDefaultHandler"))) 
#define VEC(nm)		VEC_(nm, Handler)
#define VECI(nm)	VEC_(nm, IRQHandler)


VEC(NMI_);
VEC(HardFault_);
VEC(MemManage_);
VEC(BusFault_);
VEC(UsageFault_);
VEC(SVC_);
VEC(PendSV_);
VEC(SysTick_);

VECI(WWDG_);
VECI(PVD_);
VECI(TAMP_STAMP_);
VECI(RTC_WKUP_);
VECI(FLASH_);
VECI(RCC_);
VECI(EXTI0_);
VECI(EXTI1_);
VECI(EXTI2_);
VECI(EXTI3_);
VECI(EXTI4_);
VECI(DMA1_Stream0_);
VECI(DMA1_Stream1_);
VECI(DMA1_Stream2_);
VECI(DMA1_Stream3_);
VECI(DMA1_Stream4_);
VECI(DMA1_Stream5_);
VECI(DMA1_Stream6_);
VECI(ADC_);
VECI(CAN1_TX_);
VECI(CAN1_RX0_);
VECI(CAN1_RX1_);
VECI(CAN1_SCE_);
VECI(EXTI9_5_);
VECI(TIM1_BRK_TIM9_);
VECI(TIM1_UP_TIM10_);
VECI(TIM1_TRG_COM_TIM11_);
VECI(TIM1_CC_);
VECI(TIM2_);
VECI(TIM3_);
VECI(TIM4_);
VECI(I2C1_EV_);
VECI(I2C1_ER_);
VECI(I2C2_EV_);
VECI(I2C2_ER_);
VECI(SPI1_);
VECI(SPI2_);
VECI(USART1_);
VECI(USART2_);
VECI(USART3_);
VECI(EXTI15_10_);
VECI(RTC_Alarm_);
VECI(OTG_FS_WKUP_);
VECI(TIM8_BRK_TIM12_);
VECI(TIM8_UP_TIM13_);
VECI(TIM8_TRG_COM_TIM14_);
VECI(TIM8_CC_);
VECI(DMA1_Stream7_);
VECI(FMC_);
VECI(SDIO_);
VECI(TIM5_);
VECI(SPI3_);
VECI(UART4_);
VECI(UART5_);
VECI(TIM6_DAC_);
VECI(TIM7_);
VECI(DMA2_Stream0_);
VECI(DMA2_Stream1_);
VECI(DMA2_Stream2_);
VECI(DMA2_Stream3_);
VECI(DMA2_Stream4_);
VECI(ETH_);
VECI(ETH_WKUP_);
VECI(CAN2_TX_);
VECI(CAN2_RX0_);
VECI(CAN2_RX1_);
VECI(CAN2_SCE_);
VECI(OTG_FS_);
VECI(DMA2_Stream5_);
VECI(DMA2_Stream6_);
VECI(DMA2_Stream7_);
VECI(USART6_);
VECI(I2C3_EV_);
VECI(I2C3_ER_);
VECI(OTG_HS_EP1_OUT_);
VECI(OTG_HS_EP1_IN_);
VECI(OTG_HS_WKUP_);
VECI(OTG_HS_);
VECI(DCMI_);
VECI(HASH_RNG_);
VECI(FPU_);
VECI(UART7_);
VECI(UART8_);
VECI(SPI4_);
VECI(SPI5_);
VECI(SPI6_);
VECI(SAI1_);
VECI(LTDC_);
VECI(LTDC_ER_);
VECI(DMA2D_);


//aligned by linker script as needed
__attribute__ ((section(".ramvecs"))) void (*__ISR_VECTORS[]) (void) =
{
	0,		// unused: initial sp
	0,		// unused: reset handler
	NMI_Handler,
	HardFault_Handler,
	MemManage_Handler,
	BusFault_Handler,
	UsageFault_Handler,
	0,
	0,
	0,
	0,
	SVC_Handler,		// SVCall handler
	0,					// Reserved
	0,					// Reserved
	PendSV_Handler,		// The PendSV handler
	SysTick_Handler,	// The SysTick handler
	
	// Chip Level - STM32F429
	WWDG_IRQHandler,
	PVD_IRQHandler,
	TAMP_STAMP_IRQHandler,
	RTC_WKUP_IRQHandler,
	FLASH_IRQHandler,
	RCC_IRQHandler,
	EXTI0_IRQHandler,
	EXTI1_IRQHandler,
	EXTI2_IRQHandler,
	EXTI3_IRQHandler,
	EXTI4_IRQHandler,
	DMA1_Stream0_IRQHandler,
	DMA1_Stream1_IRQHandler,
	DMA1_Stream2_IRQHandler,
	DMA1_Stream3_IRQHandler,
	DMA1_Stream4_IRQHandler,
	DMA1_Stream5_IRQHandler,
	DMA1_Stream6_IRQHandler,
	ADC_IRQHandler,
	CAN1_TX_IRQHandler,
	CAN1_RX0_IRQHandler,
	CAN1_RX1_IRQHandler,
	CAN1_SCE_IRQHandler,
	EXTI9_5_IRQHandler,
	TIM1_BRK_TIM9_IRQHandler,
	TIM1_UP_TIM10_IRQHandler,
	TIM1_TRG_COM_TIM11_IRQHandler,
	TIM1_CC_IRQHandler,
	TIM2_IRQHandler,
	TIM3_IRQHandler,
	TIM4_IRQHandler,
	I2C1_EV_IRQHandler,
	I2C1_ER_IRQHandler,
	I2C2_EV_IRQHandler,
	I2C2_ER_IRQHandler,
	SPI1_IRQHandler,
	SPI2_IRQHandler,
	USART1_IRQHandler,
	USART2_IRQHandler,
	USART3_IRQHandler,
	EXTI15_10_IRQHandler,
	RTC_Alarm_IRQHandler,
	OTG_FS_WKUP_IRQHandler,
	TIM8_BRK_TIM12_IRQHandler,
	TIM8_UP_TIM13_IRQHandler,
	TIM8_TRG_COM_TIM14_IRQHandler,
	TIM8_CC_IRQHandler,
	DMA1_Stream7_IRQHandler,
	FMC_IRQHandler,
	SDIO_IRQHandler,
	TIM5_IRQHandler,
	SPI3_IRQHandler,
	UART4_IRQHandler,
	UART5_IRQHandler,
	TIM6_DAC_IRQHandler,
	TIM7_IRQHandler,
	DMA2_Stream0_IRQHandler,
	DMA2_Stream1_IRQHandler,
	DMA2_Stream2_IRQHandler,
	DMA2_Stream3_IRQHandler,
	DMA2_Stream4_IRQHandler,
	ETH_IRQHandler,
	ETH_WKUP_IRQHandler,
	CAN2_TX_IRQHandler,
	CAN2_RX0_IRQHandler,
	CAN2_RX1_IRQHandler,
	CAN2_SCE_IRQHandler,
	OTG_FS_IRQHandler,
	DMA2_Stream5_IRQHandler,
	DMA2_Stream6_IRQHandler,
	DMA2_Stream7_IRQHandler,
	USART6_IRQHandler,
	I2C3_EV_IRQHandler,
	I2C3_ER_IRQHandler,
	OTG_HS_EP1_OUT_IRQHandler,
	OTG_HS_EP1_IN_IRQHandler,
	OTG_HS_WKUP_IRQHandler,
	OTG_HS_IRQHandler,
	DCMI_IRQHandler,
	0,
	HASH_RNG_IRQHandler,
	FPU_IRQHandler,
	UART7_IRQHandler,
	UART8_IRQHandler,
	SPI4_IRQHandler,
	SPI5_IRQHandler,
	SPI6_IRQHandler,
	SAI1_IRQHandler,
	LTDC_IRQHandler,
	LTDC_ER_IRQHandler,
	DMA2D_IRQHandler,
};


static void clockUpPeriphs(void)
{
	//enable periphs we care about
	RCC->AHB1ENR = RCC_AHB1ENR_CCMDATARAMEN | RCC_AHB1ENR_DMA2DEN | RCC_AHB1ENR_GPIOAEN | RCC_AHB1ENR_GPIOBEN | RCC_AHB1ENR_GPIOCEN | RCC_AHB1ENR_GPIODEN | RCC_AHB1ENR_GPIOEEN | RCC_AHB1ENR_GPIOFEN | RCC_AHB1ENR_GPIOGEN | RCC_AHB1ENR_DMA2DEN;
	RCC->AHB3ENR = RCC_AHB3ENR_FMCEN;
	RCC->APB2ENR = RCC_APB2ENR_LTDCEN | RCC_APB2ENR_SPI5EN | RCC_APB2ENR_SYSCFGEN | RCC_APB2ENR_TIM8EN;
	RCC->APB1ENR = RCC_APB1ENR_TIM2EN | RCC_APB1ENR_TIM5EN | RCC_APB1ENR_PWREN;
}

static void gpiosInit(void)
{
	//SDRAM = AF12,push-pull, high speed, no pullup
	//LCD_14 = AF14,push-pull, high speed, no pullup
	//LCD_9 = AF9,push-pull, high speed, no pullup
	//LCD_gpio = GPIO,out,fast,push-pull, no pullup
	//LCD_spi = AF5,push-pull, med speed, pulldown
	
	/*-- GPIOs Configuration -----------------------------------------------------*/
	/*
	 +-------------------+--------------------+--------------------+--------------------+
	 +                       SDRAM pins assignment                                      +
	 +-------------------+--------------------+--------------------+--------------------+
	 | PD0  <-> FMC_D2   | PE0  <-> FMC_NBL0  | PF0  <-> FMC_A0    | PG0  <-> FMC_A10   |
	 | PD1  <-> FMC_D3   | PE1  <-> FMC_NBL1  | PF1  <-> FMC_A1    | PG1  <-> FMC_A11   |
	 | PD8  <-> FMC_D13  | PE7  <-> FMC_D4    | PF2  <-> FMC_A2    | PG8  <-> FMC_SDCLK |
	 | PD9  <-> FMC_D14  | PE8  <-> FMC_D5    | PF3  <-> FMC_A3    | PG15 <-> FMC_NCAS  |
	 | PD10 <-> FMC_D15  | PE9  <-> FMC_D6    | PF4  <-> FMC_A4    |--------------------+ 
	 | PD14 <-> FMC_D0   | PE10 <-> FMC_D7    | PF5  <-> FMC_A5    |   
	 | PD15 <-> FMC_D1   | PE11 <-> FMC_D8    | PF11 <-> FMC_NRAS  | 
	 +-------------------| PE12 <-> FMC_D9    | PF12 <-> FMC_A6    | 
	                     | PE13 <-> FMC_D10   | PF13 <-> FMC_A7    |    
	                     | PE14 <-> FMC_D11   | PF14 <-> FMC_A8    |
	                     | PE15 <-> FMC_D12   | PF15 <-> FMC_A9    |
	 +-------------------+--------------------+--------------------+
	 | PB5 <-> FMC_SDCKE1| 
	 | PB6 <-> FMC_SDNE1 | 
	 | PC0 <-> FMC_SDNWE |
	 +-------------------+  

	+------------------------+-----------------------+----------------------------+
	+                       LCD pins assignment                                   +
	+------------------------+-----------------------+----------------------------+
	|  LCD_TFT R2 <-> PC.10  |  LCD_TFT G2 <-> PA.06 |  LCD_TFT B2 <-> PD.06      |
	|  LCD_TFT R3 <-> PB.00  |  LCD_TFT G3 <-> PG.10 |  LCD_TFT B3 <-> PG.11      |
	|  LCD_TFT R4 <-> PA.11  |  LCD_TFT G4 <-> PB.10 |  LCD_TFT B4 <-> PG.12      |
	|  LCD_TFT R5 <-> PA.12  |  LCD_TFT G5 <-> PB.11 |  LCD_TFT B5 <-> PA.03      |
	|  LCD_TFT R6 <-> PB.01  |  LCD_TFT G6 <-> PC.07 |  LCD_TFT B6 <-> PB.08      |
	|  LCD_TFT R7 <-> PG.06  |  LCD_TFT G7 <-> PD.03 |  LCD_TFT B7 <-> PB.09      |
	-------------------------------------------------------------------------------
	        |  LCD_TFT HSYNC <-> PC.06  | LCDTFT VSYNC <->  PA.04 |
	        |  LCD_TFT CLK   <-> PG.07  | LCD_TFT DE   <->  PF.10 |
	         -----------------------------------------------------

		LCD's GPIOs (LCD_gpio)
			C2 - NCS
			D12 - RDX
			D13 - WRX
		
		LCD's SPI
			F7 - SCK
			F8 - MISO
			F9 - MOSI

	
		SWD:	AF0, very high speed
			A13 - SWDIO (pullup)
			A14 - SWCLK (pulldown)

		i2c:	high speed, open drain, no pull, gpio
			A8 - SCL
			C9 - SDA
		
		i2c_nit:
			A15 - input, gpio
		
		Alarm LED:
			G13 - output gpio
		
		Audio (single ended)
			A5 - the output
		
		Button:
			A0 - input, active high, can wake us

	*/
	
	
	GPIOA->MODER	= 0x2A812A80;	//A3,A4,A6,A11,A12 - LCD_14; A13,A14 - SWD; A8 - i2c, A5 - audio
	GPIOA->OTYPER	= 0x00000100;
	GPIOA->OSPEEDR	= 0x3D822680;
	GPIOA->PUPDR	= 0x64000000;
	GPIOA->AFR[0]	= 0x0E3EE000;
	GPIOA->AFR[1]	= 0x000EE000;
	
	GPIOB->MODER	= 0x00AA280A;	//B5,B6 - SDRAM; B0,B1 - LCD_9, B8,B9,B10,B11 - LCD_14
	GPIOB->OTYPER	= 0x00000000;
	GPIOB->OSPEEDR	= 0x00AA280A;
	GPIOB->PUPDR	= 0x00000000;
	GPIOB->AFR[0]	= 0x0CC00099;
	GPIOB->AFR[1]	= 0x0000EEEE;
	
	GPIOC->MODER	= 0x0024A012;	//C0 - SDRAM; C6,C7,C10 - LCD_14; C2 - LCD_gpio; C9 - i2c
	GPIOC->OTYPER	= 0x00000200;
	GPIOC->OSPEEDR	= 0x0020A022;
	GPIOC->PUPDR	= 0x00000000;
	GPIOC->AFR[0]	= 0xEE00000C;
	GPIOC->AFR[1]	= 0x00000000;
	
	GPIOD->MODER	= 0xA52A208A;	//D0,D1,D8,D9,D10,D14,D15 - SDRAM; D3,D6 - LCD_14; D12,D13 - LCD_gpio
	GPIOD->OTYPER	= 0x00000000;
	GPIOD->OSPEEDR	= 0xAA2A208A;
	GPIOD->PUPDR	= 0x00000000;
	GPIOD->AFR[0]	= 0x0E00E0CC;
	GPIOD->AFR[1]	= 0xCC000CCC;
	
	GPIOE->MODER	= 0xAAAA800A;	//E0,E1,E7,E8,E9,E10,E11,E12,E13,E14,E15 - SDRAM
	GPIOE->OTYPER	= 0x00000000;
	GPIOE->OSPEEDR	= 0xAAAA800A;
	GPIOE->PUPDR	= 0x00000000;
	GPIOE->AFR[0]	= 0xC00000CC;
	GPIOE->AFR[1]	= 0xCCCCCCCC;
	
	GPIOF->MODER	= 0xAAAA8AAA;	//F0,F1,F2,F3,F4,F5,F11,F12,F13,F14,F15 - SDRAM; F10 - LCD_14; F7,F8,F9 - LCD_spi
	GPIOF->OTYPER	= 0x00000000;
	GPIOF->OSPEEDR	= 0xAAAA8AAA;
	GPIOF->PUPDR	= 0x000A8000;
	GPIOF->AFR[0]	= 0x50CCCCCC;
	GPIOF->AFR[1]	= 0xCCCCCE55;
	
	GPIOG->MODER	= 0x86A2AA0A;	//G0,G1,G4,G5,G8,G15 = SDRAM; G6,G7,G11 - LCD_14, G10,G12 - LCD_9, G13 - AlarmLED
	GPIOG->OTYPER	= 0x00000000;
	GPIOG->OSPEEDR	= 0x82A2AA0A;
	GPIOG->PUPDR	= 0x00000000;
	GPIOG->AFR[0]	= 0xEECC00CC;
	GPIOG->AFR[1]	= 0xC009E90C;
	
	//setup which GPIOs are used for external ints
	SYSCFG->EXTICR[0] = 0x00000000;
	SYSCFG->EXTICR[1] = 0x00000000;
	SYSCFG->EXTICR[2] = 0x00000000;
	SYSCFG->EXTICR[3] = 0x00000000;
}

void machIdle(void)
{
	//WFI causes issues on stm32f429, so we do nothing instead :(
}

static void mpuRegCfg(uint32_t idx, uint32_t addr, uint32_t cfg)
{
	MPU->RBAR = addr | 0x10 | idx;
	MPU->RASR = cfg;
}

//configures mmu for storage ram
void machSetStorageAreaWriteable(bool writeable)
{
	mpuRegCfg(1, 0xd0000000, (writeable ? MPU_PERM_U_RW_S_RW : MPU_PERM_U_RO_S_RO) | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (21 << 1) | 0x0000);
}

void __attribute__((used)) machInit(uint32_t stage, const void* data)
{
	if (stage == STAGE_INIT_EARLY) {
		
		//memset((void*)0xd0000000, 0, 4*1024*1024);	//zero ram
		clockUpPeriphs();	//minumal init to do after decompressor since it does NOT do it all for us
		gpiosInit();
	}
	else if (stage == STAGE_INIT_SET_VTOR) {
		SCB->VTOR = (uint32_t)__ISR_VECTORS;
	}
	else if (stage == STAGE_SETUP_HEAPS) {
		
		//register CCM first so ti is checked first
		kheapRegisterHeap(HAL_CCM_MEM_BASE, HAL_CCM_MEM_SIZE, MEM_USABLE_AS_STACK | MEM_FAST);
		kheapRegisterHeap(HAL_STATIC_MEM_BASE, HAL_STATIC_MEM_SIZE, MEM_USABLE_AS_STACK | MEM_USABLE_FOR_DMA | MEM_USABLE_FOR_EXEC | MEM_FAST);
	}
	else if (stage == STAGE_INIT_MPU) {
		//4MB region for ROM
		mpuRegCfg(0, 0xd0400000, MPU_PERM_U_RO_S_RO | MPU_MEM_TYPE_ROM | MPU_FLAG_ENABLED | (21 << 1) | 0x0000);
		
		//4MB region for storage ram (dyn ram overlays it so we do not need SRD here)
		machSetStorageAreaWriteable(false);

		//2MB region for dynamic ram, SRD makes it 1.5MB
		mpuRegCfg(2, 0xd0000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (20 << 1) | 0xc000);
		
		//SCB (0xE0000000 + 0x10000000, device, nx)
		mpuRegCfg(3, 0xE0000000, MPU_PERM_U_XX_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | (27 << 1) | 0x0000);
		
		//Periphs (0x40000000 + 0x10000000, device, nx)
		mpuRegCfg(4, 0x40000000, MPU_PERM_U_XX_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | (27 << 1) | 0x0000);
		
		//internal ram is jit TC (112K) and heap 80K
		mpuRegCfg(5, 0x20000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (17 << 1) | 0xc000);
		
		//CCM (use TBD) 0x10000000 + 0x00010000. cannot be DMAd or executed from
		mpuRegCfg(6, 0x10000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | MPU_PERM_NX | (15 << 1) | 0x0000);
		
		//mpu on
		MPU->CTRL = MPU_CTRL_ENABLE_Msk | MPU_CTRL_HFNMIENA_Msk | MPU_CTRL_PRIVDEFENA_Msk;	//PRIVDEFENA is for devices, without HFNMIENA we cnanot execute form RAM on real STM hw
	}
	else if (stage == STAGE_INIT_INTERRUPTS) {
		
		const struct MachInitDataInterrupts *info = (const struct MachInitDataInterrupts*)data;
		uint32_t mediumPrio = (info->lowestAllowablePrio + info->highestAllowablePrio) / 2;
		
		//scheduler timer interrupt is high prio too so nobody else can interrupt it (and more importantly - it will not interrupt syscalls and vise-versa)
		NVIC_SetPriority(TIM2_IRQn, info->schedulingTimerPrio);
		
		//audio is realtim
		NVIC_SetPriority(DMA2_Stream1_IRQn, info->lowestAllowablePrio);
		
		//lcd copying int is high prio as well. safe since it doesnt touch any structs
		NVIC_SetPriority(TIM5_IRQn, info->lowestAllowablePrio + 1);
		
		//set all HW ints to medium prio
		NVIC_SetPriority(EXTI15_10_IRQn, mediumPrio);
		NVIC_SetPriority(RTC_WKUP_IRQn, mediumPrio);
		NVIC_SetPriority(RTC_Alarm_IRQn, mediumPrio);
	}
}

static void machBusyWaitDelay(uint64_t ticks)
{
	uint64_t start = timerGetTime();
	
	while (timerGetTime() - start < ticks);
}

void machBusyWaitDelayMsec(uint32_t msec)
{
	machBusyWaitDelay((uint64_t)msec * TIMER_TICKS_PER_MSEC);
}

void machBusyWaitDelayUsec(uint32_t usec)
{
	machBusyWaitDelay((uint64_t)usec * (TIMER_TICKS_PER_MSEC / 1000));
}

bool hwMaybeGetRomToken(uint32_t name, const void **dataP, uint16_t *szP)
{
	if (name == CREATE_4CC('s','n','u','m')) {
		
		static char snumStr[12];
		
		if (!snumStr[0]) {
			static const char *base32 = "23456789ABCDEFGHJKLMNPQRSTUVWXYZ";
			volatile uint32_t *uid = (volatile uint32_t*)UID_BASE;
			uint32_t snumLo = uid[0] ^ uid[1];
			uint32_t snumHi = uid[1] ^ uid[2];
			uint64_t snum = (((uint64_t)snumHi) << 32) + snumLo;
			uint32_t i;
			
			for (i = 0; i < sizeof(snumStr); i++, snum >>= 5)
				snumStr[i] = base32[snum & 31];
		}
		if (dataP)
			*dataP = snumStr;
		if(szP)
			*szP = sizeof(snumStr);
		
		return true;
	}
	
	return false;
}

void hwGetMiscFlags(uint16_t *miscFlagsP, uint16_t *extMiscFlagsP)
{
	if (miscFlagsP)
		*miscFlagsP = hwrMiscFlagHasMiscFlagExt | hwrMiscFlagHasCradleDetect | hwrMiscFlagNoRTCBug | hwrMiscFlagHasMbdIrDA;
	//we do not set hwrMiscFlagHasBacklight since it is not switchable!
	
	if (extMiscFlagsP)
		*extMiscFlagsP = hwrMiscFlagExt115KIrOK | hwrMiscFlagExtHasLiIon;
}

int32_t cpuGetClockRate(enum ClockRateDevice dev)
{
	switch (dev) {
		case CpuClockRate:
		case TimerClockRate:
			return CPU_CLOCK_RATE;
		
		case SdioUnitClockRate:
			return CPU_CLOCK_RATE / 4;
		
		case UartUnitClockRate:
			return CPU_CLOCK_RATE / 2;
		
		default:
			return-1;
	}
}

bool hwPwrCtl(uint32_t selector, const uint32_t *newValP, uint32_t *oldValP)
{
	return false;
}

void machSleep(void)
{
	logi("pretending to sleep\n");
	SysTaskDelay(10000);
	logi("waking up\n");
	dalModifyWakeFlags(DAL_WAKE_FLAG_GENERAL, 0);
	//nothing yet
}

void deviceReset(bool doHardReset)
{
	if (doHardReset) {
		
		machSetStorageAreaWriteable(true);
		asm volatile("cpsid i");
		memset((void*)CPU_STORAGE_RAM_BASE, 0, CPU_STORAGE_RAM_SIZE);
	}
	NVIC_SystemReset();
}

Err machinePaceDispatch(EmulStateRef ref, uint16_t call, Err *ret68kP)
{
	return sysErrNotAllowed;
}




