#include "disp.h"
#include "printf.h"
#include "machSpecific.h"
#include "remoteioComms.h"
#include <MemoryMgr.h>
#include <string.h>
#include "kernel.h"
#include "memmap.h"
#include "timers.h"
#include "boot.h"
#include "heap.h"
#include "irqs.h"
#include "mpu.h"
#include "cpu.h"
#include "dal.h"
#include "kal.h"

#define NON_PORTABLE
#include <HwrMiscFlags.h>
#undef NON_PORTABLE


#define STANDARD_DENSITY			72

#ifdef FAKE_HI_RES
	#define DISP_DENSITY			(STANDARD_DENSITY * 2)
#else
	#define DISP_DENSITY			(STANDARD_DENSITY)
#endif


static uint8_t mCurDepth, mVisorDepth;
static uint16_t mDispW, mDispH, mSupportedDepthMap;

static uint8_t *mLastFb;
static uint8_t mLastFbDepth;
static int8_t mBacklightOn = -1;

struct PalmClutEntry mClut[256];

void dispSetBri(uint8_t bri)
{
	remoteioCommsSetBrightness(bri);
}

void dispSetContrast(uint8_t bri)
{
	remoteioCommsSetContrast(bri);
}

bool dispSetBacklight(bool on)
{
	const struct MsgContinueBoot *cbm = remoteioCommsWaitForContinueBooot();
	
	if (mBacklightOn == -1)
		mBacklightOn = (cbm->miscBits & MISC_BIT_BACKLIGHT_ON) ? 1 : 0;
	
	if (!(cbm->hwrMiscFlags & hwrMiscFlagHasBacklight))			//no backlight? no backlight control
		return false;
	if (cbm->hwrMiscExtFlags & hwrMiscFlagExtHasSWBright)		//devices with SW brightness control do not expose backlight on/off control
		return false;
	
	remoteioCommsSetBacklight(on);
	mBacklightOn = on;

	return true;
}

bool dispGetBacklight(void)
{
	const struct MsgContinueBoot *cbm = remoteioCommsWaitForContinueBooot();
	
	if (mBacklightOn == -1)
		mBacklightOn = (cbm->miscBits & MISC_BIT_BACKLIGHT_ON) ? 1 : 0;
	
	if (!(cbm->hwrMiscFlags & hwrMiscFlagHasBacklight))			//no backlight? it is off
		return false;
	
	if (cbm->hwrMiscExtFlags & hwrMiscFlagExtHasSWBright)		//devices with SW brightness ack like backlight is always off
		return false;
	
	return mBacklightOn;
}

static void dispAllowWrites(bool on)
{
	uint32_t fbMpuCfg;
	
	#ifdef FAKE_HI_RES
		//SRD sets size to 0x38000 (320x320x2 = 0x32000 in case you were curious)
		fbMpuCfg = (17 << 1) | 0x8000;
	#else
		//SRD sets size to 0xe000 (160x160x2 = 0xC800 in case you were curious)
		fbMpuCfg = (15 << 1) | 0x8000;
	#endif
	
	if (on) {
		
		mpuRegCfg(DISP_MPU_REG, 0, 0);	//allow all accesses
	}
	else {
		//allow only reads
		mpuRegCfg(DISP_MPU_REG, CPU_HARDWIRED_VRAM_ADDR, MPU_PERM_U_RO_S_RO | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | fbMpuCfg);
	}
}

bool __attribute__((used)) dispMmuFaultHandler(struct CortexExcFrame *exc, uint32_t addr)
{
	if (addr >= CPU_HARDWIRED_VRAM_ADDR && addr < CPU_HARDWIRED_VRAM_ADDR + CPU_HARDWIRED_VRAM_SIZE) {
		
		TIM5->CR1 &=~ TIM_CR1_CEN;	//so it doesnt interrupt us
		dispAllowWrites(true);
		TIM5->CR1 |= TIM_CR1_CEN;
		return true;
	}
	
	return false;
}

bool dispDrvInit(uint16_t *wP, uint16_t* hP, uint16_t* densityP, uint16_t *realDpiP, uint16_t *supportedDepthMapP, void** framebufferP, bool *indexedFmtIsLEP)
{
	const struct MsgContinueBoot *ccb = remoteioCommsWaitForContinueBooot();
	uint_fast8_t maxDepth = 32 - __builtin_clz(ccb->supportedDepths);
	uint32_t dispW = ccb->dispW * DISP_DENSITY / STANDARD_DENSITY;
	uint32_t dispH = ccb->dispH * DISP_DENSITY / STANDARD_DENSITY;
	uint32_t vramSz = dispW * dispH * maxDepth / 8;
	
	#ifdef CPU_HARDWIRED_VRAM_ADDR
		if (CPU_HARDWIRED_VRAM_SIZE < vramSz)
			fatal("VRAM is too small %u < %u\n", CPU_HARDWIRED_VRAM_SIZE, vramSz);
	#else
		#error "our mmu trickery requires a preallocated aligned buffer\n");
	#endif
	
	*wP = mDispW = dispW;
	*hP = mDispH = dispH;
	*densityP = DISP_DENSITY;
	*supportedDepthMapP = mSupportedDepthMap = ccb->supportedDepths;
	*framebufferP = (void*)CPU_HARDWIRED_VRAM_ADDR;
	*realDpiP = 75 * DISP_DENSITY / STANDARD_DENSITY;
	*indexedFmtIsLEP = false;
	
	mLastFb = (uint8_t*)CPU_HARDWIRED_VTMP2_SPACE;
	if (CPU_HARDWIRED_VTMP2_SIZE < vramSz)
		fatal("Failed to allocate screen backup memory\n");
	
	dispAllowWrites(true);	//for now
	
	mCurDepth = mVisorDepth = ccb->curDepth;
	
	//start timer to update LCD constantly
	TIM5->PSC = 0;
	TIM5->DIER = TIM_DIER_UIE;
	TIM5->SR = -1;
	TIM5->CNT = TIMER_TICKS_PER_MSEC * 1000 / 60;
	TIM5->ARR = TIMER_TICKS_PER_MSEC * 1000 / 60;
	TIM5->CR1 = TIM_CR1_DIR | TIM_CR1_URS | TIM_CR1_CEN;
	NVIC_EnableIRQ(TIM5_IRQn);
	
	return true;
}

void dispSetClut(int32_t firstIdx, uint32_t numEntries, const struct PalmClutEntry *entries)
{
	uint32_t i;
	
	if (firstIdx == -1) {
		
		if (numEntries > 256)
			return;
	}
	else if (firstIdx < 0)
		return;
	else if (firstIdx >= 256 || numEntries + firstIdx > 256)
		return;

	for (i = 0; i < numEntries; i++) {
		
		uint32_t where = (firstIdx == -1) ? entries[i].idx : i + firstIdx;
		
		mClut[where] = entries[i];
		mClut[where].idx = where;
	}
	
	remoteioCommsSetScreenClut(mCurDepth >=8 ? 256 : (1 << mCurDepth), mClut);
}

void dispSetDepth(uint32_t depth)
{
	mCurDepth = mVisorDepth = depth;
	
	#ifdef FAKE_HI_RES
		
		//havnig higher depth allows etter scaling
	
		if (depth == 8 && (mSupportedDepthMap & (1 << (16 - 1))))
			mVisorDepth = 16;
		if (depth == 2 && (mSupportedDepthMap & (1 << (4 - 1))))
			mVisorDepth = 4;
		if (depth == 1 && (mSupportedDepthMap & (1 << (4 - 1))))
			mVisorDepth = 4;
	#endif
	
	remoteioCommsSetScreenDepth(mVisorDepth);
}

void dispManualUpdate(void)
{
	//nothing
}

void dispRequestUpdate(void)
{
	//nothing
}

#ifdef FAKE_HI_RES
	
	static void resize8(uint8_t *dst, const uint8_t *src, uint32_t w, uint32_t h)
	{
		uint32_t r, c;
		
		for (r = 0; r < h; r += 2) {
			for (c = 0; c < w; c += 2) {
				*dst++ = *src;
				src += 2;
			}
			src += w;
		}
	}
	
	//assumes source is 4-byte aligned and dest is 2-byte aligned. assumes w and h are even
	static void __attribute__((naked)) resize16(uint16_t *dst, const uint16_t *src1, uint32_t w, uint32_t h)
	{
		asm volatile(
			"	push    {r4-r8, r10-r11, lr}	\n\t"
			"	mov     r12, r2					\n\t"	//r12 will be width
			"	mov     lr, r3, lsr #1			\n\t"	//lr will be height / 2
			"	add     r2, r1, r12, lsl #1		\n\t"	//r2 will be src2
			"	mov		r10, 0xf81f				\n\t"	//single red and green mask
			"	ldr     r11, =0x07e007e0		\n\t"	//double green mask
			"per_row:							\n\t"
			"	mov     r8, r2					\n\t"	//save start of row2
			"looptacular:						\n\t"
			"	ldmia   r1!, {r4}				\n\t"	//grab two pixels from src1
			"	ldmia   r2!, {r5}				\n\t"	//grab two pixels from src2
			"	bic     r6, r4, r11				\n\t"	//get red and blue from row 1
			"	bic     r7, r5, r11				\n\t"	//get red and blue from row 2
			"	uhadd8  r6, r6, r7				\n\t"	//average top row and bottom
			"	add     r6, r6, r6, lsr #16		\n\t"	//add left and right blue and red, now r6 has red and blue components, except shifted left by one
			"	and     r6, r10, r6, lsr #1		\n\t"	//shift them properly and mask off extras
			"	and     r4, r11					\n\t"	//get row1 greens
			"	and     r5, r11					\n\t"	//get row2 greens
			"	add     r4, r5					\n\t"	//add rows
			"	add     r4, r4, r4, lsr #16		\n\t"	//add left and right
			"	lsrs    r4, #7					\n\t"	//shift off extra bits
			"	bfi     r6, r4, #5, #6			\n\t"	//insert into r6
			"	strh    r6, [r0], #2			\n\t"	//store it
			"	cmp     r1, r8					\n\t"	//if row1 is now where row2 started, we are done with the row
			"	bne     looptacular				\n\t"
			"	mov     r1, r2					\n\t"	//src1 is where src2 ended up
			"	add     r2, r1, r12, lsl #1		\n\t"	//src2 is a row lower
			"	subs    lr, #1					\n\t"
			"	bne		per_row					\n\t"
			"	pop		{r4-r8, r10-r11, pc}	\n\t"
			:
			:
			: "memory", "cc"
		);
	}
	
	static void __attribute__((naked)) resize8_to_16(uint16_t *dst, const uint8_t *src1, uint32_t w, uint32_t h, struct PalmClutEntry *clut)
	{
		asm volatile(
			"	ldr     r12, [sp]				\n\t"	//get clut
			"	push    {r4-r10, lr}			\n\t"
			"	mov     r10, r2					\n\t"	//w in r10
			"per_row_8:							\n\t"
			"	add     r2, r1, r10				\n\t"	//r2 is src2
			"	mov     r9, r2					\n\t"	//save starting src2 in r9
			"per_2_pixels_8:					\n\t"
			
			"	ldrb    r4, [r1, #1]			\n\t"
			"	ldrb    r5, [r1], #2			\n\t"
			"	ldrb    r6, [r2, #1]			\n\t"
			"	ldrb    r7, [r2], #2			\n\t"
			"	ldr     r4, [r12, r4, lsl #2]	\n\t"
			"	ldr     r5, [r12, r5, lsl #2]	\n\t"
			"	ldr     r6, [r12, r6, lsl #2]	\n\t"
			"	ldr     r7, [r12, r7, lsl #2]	\n\t"
			"	uhadd8  r4, r4, r5				\n\t"
			"	uhadd8  r6, r6, r7				\n\t"
			"	uhadd8  r4, r4, r6				\n\t"	//{idx[0..7], r[8..15], g[16..23], b[24..31]} in r4
			"	ubfx	r5, r4, #11, #5			\n\t"	//get r
			"	ubfx	r6, r4, #18, #6			\n\t"	//get g
			"	ubfx	r7, r4, #27, #5			\n\t"	//get b
			"	bfi     r7, r6, #5, #6			\n\t"	//insert g
			"	bfi     r7, r5, #11, #5			\n\t"	//insert b
			
			"	ldrb    r4, [r1, #1]			\n\t"
			"	ldrb    r5, [r1], #2			\n\t"
			"	ldrb    r6, [r2, #1]			\n\t"
			"	ldrb    r8, [r2], #2			\n\t"
			"	ldr     r4, [r12, r4, lsl #2]	\n\t"
			"	ldr     r5, [r12, r5, lsl #2]	\n\t"
			"	ldr     r6, [r12, r6, lsl #2]	\n\t"
			"	ldr     r8, [r12, r8, lsl #2]	\n\t"
			"	uhadd8  r4, r4, r5				\n\t"
			"	uhadd8  r6, r6, r8				\n\t"
			"	uhadd8  r4, r4, r6				\n\t"	//{idx[0..7], r[8..15], g[16..23], b[24..31]} in r4
			"	ubfx	r5, r4, #11, #5			\n\t"	//get r
			"	ubfx	r6, r4, #18, #6			\n\t"	//get g
			"	ubfx	r8, r4, #27, #5			\n\t"	//get b
			"	bfi     r8, r6, #5, #6			\n\t"	//insert g
			"	bfi     r8, r5, #11, #5			\n\t"	//insert b
			
			"	pkhbt	r7, r7, r8, lsl #16		\n\t"	//pack two pixels into a word
			
			"	stmia   r0!, {r7}				\n\t"
			"	cmp     r1, r9					\n\t"
			"	bne     per_2_pixels_8			\n\t"
			"	mov     r1, r2					\n\t"
			"	subs    r3, #2					\n\t"
			"	bne     per_row_8				\n\t"
			"	pop		{r4-r10, pc}			\n\t"
			:
			:
			: "memory", "cc"
		);
	}
	
	static void __attribute__((naked)) resize4(uint8_t *dst, const uint8_t *src1, uint32_t w, uint32_t h, const uint8_t *avgtab)
	{
		asm volatile(
			"	push    {r4-r8, lr}				\n\t"
			"	mov     r12, r2					\n\t"	//w is r12
			"	ldr     r7, [sp, #0x18]			\n\t"	//get lookuptab to r7
			"per_row_4:							\n\t"
			"	add     r2, r1, r12, lsr #1		\n\t"	//r2 is src2
			"	mov     r8, r2					\n\t"	//save starting src2 in r8
			"per_2_pixels_4:					\n\t"
			
			"	ldrb    r4, [r1, #1]			\n\t"	//do second pixel first (easier)
			"	ldrb    r5, [r2, #1]			\n\t"
			"	ldrb    r4, [r7, r4]			\n\t"
			"	ldrb    r5, [r7, r5]			\n\t"
			"	adds    r6, r4, r5				\n\t"
			"	lsrs    r6, #1					\n\t"	//low nibble of result is ready
			
			"	ldrb    r4, [r1], #2			\n\t"	//do second pixel first (easier)
			"	ldrb    r5, [r2], #2			\n\t"
			"	ldrb    r4, [r7, r4]			\n\t"
			"	ldrb    r5, [r7, r5]			\n\t"
			"	adds    r4, r4, r5				\n\t"
			"	lsls    r4, #3					\n\t"	//high nibble of result is ready, but bit 3 is trash
			
			"	bfi     r4, r6, #0, #4			\n\t"	//insert low nibble
			
			"	strb    r4, [r0], #1			\n\t"
			"	cmp     r1, r8					\n\t"
			"	bne     per_2_pixels_4			\n\t"
			"	mov     r1, r2					\n\t"
			"	subs    r3, #2					\n\t"
			"	bne     per_row_4				\n\t"
			"	pop		{r4-r8, pc}				\n\t"
			:
			:
			: "memory", "cc"
		);
	}
	
	static void __attribute__((naked)) resize2_to_4(uint8_t *dst, const uint8_t *src1, uint32_t w, uint32_t h, const uint8_t *expandtab2)
	{
		asm volatile(
			"	push    {r4-r7, lr}				\n\t"
			"	mov     r12, r2					\n\t"	//w is r12
			"	ldr     r7, [sp, #0x14]			\n\t"	//get lookuptab to r7
			"per_row_2:							\n\t"
			"	add     r2, r1, r12, lsr #2		\n\t"	//r2 is src2
			"	mov     r6, r2					\n\t"	//save starting src2 in r6
			"per_4_pixels_2:					\n\t"
			
			"	ldrb    r4, [r1], #1			\n\t"
			"	ldrb    r5, [r2], #1			\n\t"
			"	ldrb    r4, [r7, r4]			\n\t"
			"	ldrb    r5, [r7, r5]			\n\t"
			"	adds    r4, r4, r5				\n\t"
			
			"	strb    r4, [r0], #1			\n\t"
			"	cmp     r1, r6					\n\t"
			"	bne     per_4_pixels_2			\n\t"
			"	mov     r1, r2					\n\t"
			"	subs    r3, #2					\n\t"
			"	bne     per_row_2				\n\t"
			"	pop		{r4-r7, pc}				\n\t"
			:
			:
			: "memory", "cc"
		);
	}
	
	static void __attribute__((naked)) resize1_to_4(uint8_t *dst, const uint8_t *src1, uint32_t w, uint32_t h, const uint16_t *expandtab1)
	{
		asm volatile(
			"	push    {r4-r7, lr}				\n\t"
			"	mov     r12, r2					\n\t"	//w is r12
			"	ldr     r7, [sp, #0x14]			\n\t"	//get lookuptab to r7
			"per_row_1:							\n\t"
			"	add     r2, r1, r12, lsr #3		\n\t"	//r2 is src2
			"	mov     r6, r2					\n\t"	//save starting src2 in r6
			"per_8_pixels_1:					\n\t"
			
			"	ldrb    r4, [r1], #1			\n\t"
			"	ldrb    r5, [r2], #1			\n\t"
			"	ldrh    r4, [r7, r4, lsl #1]	\n\t"
			"	ldrh    r5, [r7, r5, lsl #1]	\n\t"
			"	adds    r4, r4, r5				\n\t"
			
			"	strh    r4, [r0], #2			\n\t"
			"	cmp     r1, r6					\n\t"
			"	bne     per_8_pixels_1			\n\t"
			"	mov     r1, r2					\n\t"
			"	subs    r3, #2					\n\t"
			"	bne     per_row_1				\n\t"
			"	pop		{r4-r7, pc}				\n\t"
			:
			:
			: "memory", "cc"
		);
	}
	
	///downsizing buffer is at 0x38000 + CPU_HARDWIRED_VRAM_ADDR and has space for 160x160x2
	static void resizeAndSendScreen(const void* src, uint32_t w, uint32_t h, uint32_t d, uint32_t visorD)
	{
		static const uint8_t avgtab4[] = {0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13, 14, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13, 14, 14, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13, 14, 14, 15, };
		static const uint8_t expandtab2[] = {0x00, 0x01, 0x02, 0x03, 0x01, 0x02, 0x03, 0x04, 0x02, 0x03, 0x04, 0x05, 0x03, 0x04, 0x05, 0x07, 0x10, 0x11, 0x12, 0x13, 0x11, 0x12, 0x13, 0x14, 0x12, 0x13, 0x14, 0x15, 0x13, 0x14, 0x15, 0x17, 0x20, 0x21, 0x22, 0x23, 0x21, 0x22, 0x23, 0x24, 0x22, 0x23, 0x24, 0x25, 0x23, 0x24, 0x25, 0x27, 0x30, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x34, 0x32, 0x33, 0x34, 0x35, 0x33, 0x34, 0x35, 0x37, 0x10, 0x11, 0x12, 0x13, 0x11, 0x12, 0x13, 0x14, 0x12, 0x13, 0x14, 0x15, 0x13, 0x14, 0x15, 0x17, 0x20, 0x21, 0x22, 0x23, 0x21, 0x22, 0x23, 0x24, 0x22, 0x23, 0x24, 0x25, 0x23, 0x24, 0x25, 0x27, 0x30, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x34, 0x32, 0x33, 0x34, 0x35, 0x33, 0x34, 0x35, 0x37, 0x40, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x44, 0x42, 0x43, 0x44, 0x45, 0x43, 0x44, 0x45, 0x47, 0x20, 0x21, 0x22, 0x23, 0x21, 0x22, 0x23, 0x24, 0x22, 0x23, 0x24, 0x25, 0x23, 0x24, 0x25, 0x27, 0x30, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x34, 0x32, 0x33, 0x34, 0x35, 0x33, 0x34, 0x35, 0x37, 0x40, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x44, 0x42, 0x43, 0x44, 0x45, 0x43, 0x44, 0x45, 0x47, 0x50, 0x51, 0x52, 0x53, 0x51, 0x52, 0x53, 0x54, 0x52, 0x53, 0x54, 0x55, 0x53, 0x54, 0x55, 0x57, 0x30, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x34, 0x32, 0x33, 0x34, 0x35, 0x33, 0x34, 0x35, 0x37, 0x40, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x44, 0x42, 0x43, 0x44, 0x45, 0x43, 0x44, 0x45, 0x47, 0x50, 0x51, 0x52, 0x53, 0x51, 0x52, 0x53, 0x54, 0x52, 0x53, 0x54, 0x55, 0x53, 0x54, 0x55, 0x57, 0x70, 0x71, 0x72, 0x73, 0x71, 0x72, 0x73, 0x74, 0x72, 0x73, 0x74, 0x75, 0x73, 0x74, 0x75, 0x77,  };
		static const uint16_t expandtab1[] = {0x0000, 0x0300, 0x0300, 0x0700, 0x3000, 0x3300, 0x3300, 0x3700, 0x3000, 0x3300, 0x3300, 0x3700, 0x7000, 0x7300, 0x7300, 0x7700, 0x0003, 0x0303, 0x0303, 0x0703, 0x3003, 0x3303, 0x3303, 0x3703, 0x3003, 0x3303, 0x3303, 0x3703, 0x7003, 0x7303, 0x7303, 0x7703, 0x0003, 0x0303, 0x0303, 0x0703, 0x3003, 0x3303, 0x3303, 0x3703, 0x3003, 0x3303, 0x3303, 0x3703, 0x7003, 0x7303, 0x7303, 0x7703, 0x0007, 0x0307, 0x0307, 0x0707, 0x3007, 0x3307, 0x3307, 0x3707, 0x3007, 0x3307, 0x3307, 0x3707, 0x7007, 0x7307, 0x7307, 0x7707, 0x0030, 0x0330, 0x0330, 0x0730, 0x3030, 0x3330, 0x3330, 0x3730, 0x3030, 0x3330, 0x3330, 0x3730, 0x7030, 0x7330, 0x7330, 0x7730, 0x0033, 0x0333, 0x0333, 0x0733, 0x3033, 0x3333, 0x3333, 0x3733, 0x3033, 0x3333, 0x3333, 0x3733, 0x7033, 0x7333, 0x7333, 0x7733, 0x0033, 0x0333, 0x0333, 0x0733, 0x3033, 0x3333, 0x3333, 0x3733, 0x3033, 0x3333, 0x3333, 0x3733, 0x7033, 0x7333, 0x7333, 0x7733, 0x0037, 0x0337, 0x0337, 0x0737, 0x3037, 0x3337, 0x3337, 0x3737, 0x3037, 0x3337, 0x3337, 0x3737, 0x7037, 0x7337, 0x7337, 0x7737, 0x0030, 0x0330, 0x0330, 0x0730, 0x3030, 0x3330, 0x3330, 0x3730, 0x3030, 0x3330, 0x3330, 0x3730, 0x7030, 0x7330, 0x7330, 0x7730, 0x0033, 0x0333, 0x0333, 0x0733, 0x3033, 0x3333, 0x3333, 0x3733, 0x3033, 0x3333, 0x3333, 0x3733, 0x7033, 0x7333, 0x7333, 0x7733, 0x0033, 0x0333, 0x0333, 0x0733, 0x3033, 0x3333, 0x3333, 0x3733, 0x3033, 0x3333, 0x3333, 0x3733, 0x7033, 0x7333, 0x7333, 0x7733, 0x0037, 0x0337, 0x0337, 0x0737, 0x3037, 0x3337, 0x3337, 0x3737, 0x3037, 0x3337, 0x3337, 0x3737, 0x7037, 0x7337, 0x7337, 0x7737, 0x0070, 0x0370, 0x0370, 0x0770, 0x3070, 0x3370, 0x3370, 0x3770, 0x3070, 0x3370, 0x3370, 0x3770, 0x7070, 0x7370, 0x7370, 0x7770, 0x0073, 0x0373, 0x0373, 0x0773, 0x3073, 0x3373, 0x3373, 0x3773, 0x3073, 0x3373, 0x3373, 0x3773, 0x7073, 0x7373, 0x7373, 0x7773, 0x0073, 0x0373, 0x0373, 0x0773, 0x3073, 0x3373, 0x3373, 0x3773, 0x3073, 0x3373, 0x3373, 0x3773, 0x7073, 0x7373, 0x7373, 0x7773, 0x0077, 0x0377, 0x0377, 0x0777, 0x3077, 0x3377, 0x3377, 0x3777, 0x3077, 0x3377, 0x3377, 0x3777, 0x7077, 0x7377, 0x7377, 0x7777, };
		
		void *dst = (void*)(CPU_HARDWIRED_VRAM_ADDR + 0x38000);
		
		switch (d) {
			
			case 16:
				resize16(dst, src, w, h);
				break;
			
			case 8:
				if (mVisorDepth == 16)
					resize8_to_16(dst, src, w, h, mClut);
				else
					resize8(dst, src, w, h);		//shouldnt happen
				break;
			
			case 4:
				resize4(dst, src, w, h, avgtab4);
				break;
			
			case 2:
				if (mVisorDepth == 4)
					resize2_to_4(dst, src, w, h, expandtab2);
				else
					fatal("unsupported\n");
				break;
			
			
			case 1:
				if (mVisorDepth == 4)
					resize1_to_4(dst, src, w, h, expandtab1);
				else
					fatal("unsupported\n");
				break;
		}
		
		remoteioCommsRequestScreenRedraw(dst, (w / 2) * (h / 2) * visorD / 8, visorD == 16);
	}
#endif

void __attribute__((used)) TIM5_IRQHandler(void)	//high bits come first
{
	uint32_t fbLen = (uint32_t)mDispW * mDispH * mCurDepth / 8;
	
	TIM5->SR = 0;		//ack the interrupt
	
	//wekeep trying every 16.6ms till we succeed
	
	if ((mLastFbDepth != mCurDepth || memcmp(mLastFb, (void*)CPU_HARDWIRED_VRAM_ADDR, fbLen)) && remoteioCommsIsScreenRedrawDone()) {
	
		dispAllowWrites(false);
		TIM5->CR1 &=~ TIM_CR1_CEN;
		memcpy(mLastFb, (void*)CPU_HARDWIRED_VRAM_ADDR, fbLen);
		mLastFbDepth = mCurDepth;
		
		#ifdef FAKE_HI_RES
			
			resizeAndSendScreen(mLastFb, mDispW, mDispH, mLastFbDepth, mVisorDepth);
		#else
			
			remoteioCommsRequestScreenRedraw(mLastFb, fbLen, mLastFbDepth == 16);
		#endif
	}
	
	asm volatile("DSB 0x0f");		//c-m4f erratum
}

void dispSleep(void)
{
	//todo
}

void dispWake(void)
{
	//todo
}
