#include "ac97codec.h"
#include "printf.h"


static enum AC97deviceSelection mWhere;


bool ac97codecInit(enum AC97deviceSelection where, uint16_t *vid1P, uint16_t *vid2P)
{
	static const char *names3dEnh[] = {
		"NONE", "Analog Devices", "Creative Tech", "National Semi", "Yamaha", "BBE Sound", "Crystal Semi", "Qsound Labs",
		"Spatializer Audio Labs", "SRS Labs", "Platform Tech", "AKM Semi", "Aureal", "Aztech Labs", "Binaura", "ESS Tech",
		"Harman Int", "Nvidia", "Philips", "TI", "VLSI Tech", "TriTech", "Realtek", "Samsung",
		"Wolfson Micro", "Delata Integration", "SigmaTel", "KS Waves", "Rockwell", "?(29)", "?(30)", "?(31)",
	};
	uint16_t val;
	
	mWhere = where;
	
	for (val = 0; val < 1000 && !ac97IsCodecReady(where); val++);
	
	if (!ac97IsCodecReady(where)) {
		
		loge("Codec not ready\n");
		return false;
	}
	
	if (!ac97regRead(where, AC97_REG_RESET, &val)) {
		
		loge("Cannot read reset register\n");
		return false;
	}
	if (vid1P && !ac97regRead(where, AC97_REG_VENDOR_ID_1, vid1P)) {
		
		loge("Cannot read VID1 reg\n");
		return false;
	}
	if (vid2P && !ac97regRead(where, AC97_REG_VENDOR_ID_2, vid2P)) {
		
		loge("Cannot read VID2 reg\n");
		return false;
	}
	
	logi("AC97 codec features: \n");
	if (val & 0x0001)
		logi(" * Dedicated PIC PCM-in channel\n");
	if (val & 0x0004)
		logi(" * Bass & Trebl control\n");
	if (val & 0x0008)
		logi(" * Simulated stereo\n");
	if (val & 0x0010)
		logi(" * Headphone-out support\n");
	if (val & 0x0020)
		logi(" * Bass-boost support\n");
	if (val & 0x0040)
		logi(" * 18-bit DAC samples\n");
	if (val & 0x0080)
		logi(" * 20-bit DAC samples\n");
	if (val & 0x0100)
		logi(" * 18-bit ADC samples\n");
	if (val & 0x0200)
		logi(" * 20-bit ADC samples\n");
	if (val & 0x7c00)
		logi(" * 3D enhancement: %s\n", names3dEnh[(val >> 10) & 0x1f]);

	return true;
}

static uint32_t ac97codecPrvVolKnobToReg(enum AC97codecVolumeKnob knob)
{
	return 2 * ((uint32_t)knob);
}

bool ac97setVolume(enum AC97codecVolumeKnob which, uint8_t volL, uint8_t volR)
{
	uint32_t regNo = ac97codecPrvVolKnobToReg(which);
	uint16_t val;
	
	if (!ac97regRead(mWhere, regNo, &val))
		return false;
	
	if (volL >= 32)	//we support just 5 bits as mandatory in the spec
		volL = 31;
	if (volR >= 32)
		volR = 31;
	
	val &=~ (AC97_VOLREG_RIGHT_MASK | AC97_VOLREG_LEFT_MASK);
	val |= ((uint32_t)volL) << AC97_VOLREG_LEFT_SHIFT;
	val |= ((uint32_t)volR) << AC97_VOLREG_RIGHT_SHIFT;

	return ac97regWrite(mWhere, regNo, val);
}

bool ac97getVolume(enum AC97codecVolumeKnob which, uint8_t *volLP, uint8_t *volRP)
{
	uint32_t regNo = ac97codecPrvVolKnobToReg(which);
	uint16_t val;
	
	if (!ac97regRead(mWhere, regNo, &val))
		return false;
	
	if (volLP)
		*volLP = ((val & AC97_VOLREG_LEFT_MASK) >> AC97_VOLREG_LEFT_SHIFT);
	if (volRP)
		*volRP = ((val & AC97_VOLREG_RIGHT_MASK) >> AC97_VOLREG_RIGHT_SHIFT);
	
	return true;
}

bool ac97setMute(enum AC97codecVolumeKnob which, bool mute)
{
	uint32_t regNo = ac97codecPrvVolKnobToReg(which);
	uint16_t val;
	
	if (!ac97regRead(mWhere, regNo, &val))
		return false;
	
	if (mute)
		val |= AC97_VOLREG_MUTE;
	else
		val &=~ AC97_VOLREG_MUTE;

	return ac97regWrite(mWhere, regNo, val);
}

bool ac97getMute(enum AC97codecVolumeKnob which, bool *muteP)
{
	uint32_t regNo = ac97codecPrvVolKnobToReg(which);
	uint16_t val;
	
	if (!ac97regRead(mWhere, regNo, &val))
		return false;
	
	if (muteP)
		*muteP = !!(val & AC97_VOLREG_MUTE);
	
	return true;
}

bool ac97setPwrReg(uint16_t val)
{
	return ac97regWrite(mWhere, AC97_REG_POWERDOWN, val);
}

bool ac97getPwrReg(uint16_t *valP)
{
	return ac97regRead(mWhere, AC97_REG_POWERDOWN, valP);
}

bool ac97codecRegRead(uint8_t reg, uint16_t *regValP)
{
	return ac97regRead(mWhere, reg, regValP);
}

bool ac97codecRegWrite(uint8_t reg, uint16_t regVal)
{
	return ac97regWrite(mWhere, reg, regVal);
}