#include <stdbool.h>
#include <string.h>
#include <stdint.h>
#include "msioSimpleMemory.h"
#include "msioCommsProto.h"
#include "screenCompress.h"
#include "ral_export.h"
#include "msioComms.h"
#include "memmap.h"
#include "printf.h"
#include "timers.h"
#include "irqs.h"
#include "heap.h"
#include "msio.h"
#include "ral.h"
#include "kal.h"
#include "cpu.h"

/*
code runings at irq prio HWSVC cannot make syscalls or affect OS state in any way (it steals cycles from the OS, but the OS is not aware of it)
the best way for it to affect the OS is to trigger another irq, that is assigned a proper priority by th eOs
this can be done with EXTI, since EXTI has ability to trigger an IRQ via software
this irq can then make oscalls, etc
*/


#define OUR_CONSUMPTION_IN_CA			10		// centiAmps...yup

//priority from high bits to low
#define HAVE_BIT_NEW_AUDIO_DATA			31
#define HAVE_BIT_NEW_BRI_VAL			30
#define HAVE_BIT_NEW_BACKLITE_VAL		29
#define HAVE_BIT_NEW_CONTRAST_VAL		28
#define HAVE_BIT_NEW_DEPTH				27
#define HAVE_BIT_NEW_CLUT				26
#define HAVE_BIT_NEW_LED_CMD			25
#define HAVE_BIT_NEW_VIB_CMD			24
#define HAVE_BIT_NEW_SERIAL_OP			23
#define HAVE_BIT_NEW_SERIAL_DATA		22
#define HAVE_BIT_NEW_DISP_DATA			21
#define HAVE_BIT_RTC_SET_REQ			6
#define HAVE_BIT_RTC_ALM_REQ			5

//our state
static struct MsioPktBootParams mBootPkt = {};
static struct MsioPktBattInfo mBattPkt = {};
static bool mIsInIoMode = false;
static MsioCommsButtonStateCbk mBtnCbkF;
static MsioCommsPenStateCbk mPenCbkF;
static MsioCommsRtcSecondCbkF mRtcCbkF;

//new data
static volatile uint32_t mPendingDataFlags;			//HAVE_NEW_*
static uint8_t mNewBri, mNewBacklite, mNewContrast, mNewDepth;
static struct MsioVibLedControl __attribute__((aligned(4))) mLedCmd;
static struct MsioVibLedControl __attribute__((aligned(4))) mVibCmd;
static uint16_t mNewClutNumEntries;
static struct PalmClutEntry __attribute__((aligned(4))) mNewClutEntries[256];
static struct MsioSerialData __attribute__((aligned(4))) mNewSerialData;
static struct MsioAudioData __attribute__((aligned(4))) mNewAudioData;		//rememebr to bswap data before storting it in here
static struct MsioPktSerOp __attribute__((aligned(4))) mNewSerOp;
static uint32_t mNewDispDataLen, mNewRtcNewVal, mNewRtcAlmVal;

//data xfer state
static const uint8_t *mCurTxingDataPtr;
static uint32_t mCurTxingDataLenLeft;
static uint8_t mCurTxClrBitNoAfter;

//predeclaration
static void msioCommsPrvUpdatePendingData(uint32_t clearBits, uint32_t setBits);





static void __attribute__((section(".ramcode"))) msioCommsPrvCopyWordwise(void* dst, const void *src, uint32_t len)	//assumes over-read and over-write to match word size is ok, will copy at least one word!
{
	uint32_t dummy;
	
	asm volatile(
		"1:						\n\t"
		"	ldmia %1!, {%3}		\n\t"
		"	stmia %0!, {%3}		\n\t"
		"	subs  %2, #4		\n\t"
		"	bhi   1b			\n\t"
		:"+l"(dst), "+l"(src), "+l"(len), "=l"(dummy)
		:
		:"cc", "memory"
	);
}

static void __attribute__((section(".ramcode"))) msioCommsPrvPrepareHlMessageForTx(void)
{
	void *buf = msioGetOutBuffer(1);
	struct MsioScreenImgHdr *simg = (struct MsioScreenImgHdr*)buf;
	struct MsioShortMessage *sm = (struct MsioShortMessage*)buf;
	struct MsioRtcSetData *rtc = (struct MsioRtcSetData*)buf;
	struct MsioClutHdr *clut = (struct MsioClutHdr*)buf;
	uint32_t bit;

	bit = mPendingDataFlags ? (31 - __builtin_clz(mPendingDataFlags)) : 32;
	switch (bit) {
		case HAVE_BIT_NEW_AUDIO_DATA:
			msioCommsPrvCopyWordwise(buf, &mNewAudioData, sizeof(mNewAudioData));
			break;
		
		case HAVE_BIT_NEW_BRI_VAL:
			sm->hdr.pktTyp = MSIO_PKT_SET_BRI;
			sm->value = mNewBri;
			break;
		
		case HAVE_BIT_NEW_BACKLITE_VAL:
			sm->hdr.pktTyp = MSIO_PKT_SET_BACKLIGHT;
			sm->value = mNewBacklite;
			break;
		
		case HAVE_BIT_NEW_CONTRAST_VAL:
			sm->hdr.pktTyp = MSIO_PKT_SET_CONTRAST;
			sm->value = mNewContrast;
			break;
		
		case HAVE_BIT_NEW_DEPTH:
			sm->hdr.pktTyp = MSIO_PKT_SET_DEPTH;
			sm->value = mNewDepth;
			break;
		
		case HAVE_BIT_NEW_CLUT:
			clut->hdr.pktTyp = MSIO_PKT_CLUT_HDR;
			clut->numEntries = __builtin_bswap16(mNewClutNumEntries);
			mCurTxingDataPtr = (const void*)mNewClutEntries;
			mCurTxingDataLenLeft = mNewClutNumEntries * sizeof(*mNewClutEntries);
			mCurTxClrBitNoAfter = bit;
			bit = 32;	//do not clear yet
			break;
		
		case HAVE_BIT_NEW_LED_CMD:
			msioCommsPrvCopyWordwise(buf, &mLedCmd, sizeof(mLedCmd));
			break;
		
		case HAVE_BIT_NEW_VIB_CMD:
			msioCommsPrvCopyWordwise(buf, &mVibCmd, sizeof(mVibCmd));
			break;
		
		case HAVE_BIT_NEW_SERIAL_OP:
			msioCommsPrvCopyWordwise(buf, &mNewSerOp, sizeof(mNewSerOp));
			break;
		
		case HAVE_BIT_NEW_SERIAL_DATA:
			msioCommsPrvCopyWordwise(buf, &mNewSerialData, sizeof(mNewSerialData));
			break;
		
		case HAVE_BIT_NEW_DISP_DATA:
			simg->hdr.pktTyp = MSIO_PKT_DISPLAY_HDR;
			simg->totalSzHi = mNewDispDataLen >> 16;
			simg->totalSzLo = __builtin_bswap16(mNewDispDataLen);
			if (mNewDispDataLen <= sizeof(simg->data))
				msioCommsPrvCopyWordwise(simg->data, (const void*)CPU_HARDWIRED_VTMP1_SPACE, mNewDispDataLen);
			else {
				
				msioCommsPrvCopyWordwise(simg->data, (const void*)CPU_HARDWIRED_VTMP1_SPACE, sizeof(simg->data));
				mCurTxingDataPtr = (const void*)(CPU_HARDWIRED_VTMP1_SPACE + sizeof(simg->data));
				mCurTxingDataLenLeft = mNewDispDataLen - sizeof(simg->data);
				mCurTxClrBitNoAfter = bit;
				bit = 32;	//do not clear yet
			}
			break;
		
		case HAVE_BIT_RTC_SET_REQ:
			rtc->hdr.pktTyp = MSIO_PKT_SET_RTC;
			rtc->value = __builtin_bswap32(mNewRtcNewVal);
			break;
		
		case HAVE_BIT_RTC_ALM_REQ:
			rtc->hdr.pktTyp = MSIO_PKT_SET_ALARM;
			rtc->value = __builtin_bswap32(mNewRtcAlmVal);
			break;
		
		default:
			return;
	}
	
	(void)msioProvideLongReadData(buf, MSIO_SHORT_PKT_SZ);
	dtx(0xcc);
	dtx(*(char*)buf);
//	logi("prep: have %u, bits %04x -> giving\n", msioHaveLongDataToTx(), mPendingDataFlags);

	if (bit < 32)
		msioCommsPrvUpdatePendingData(1 << bit, 0);
}

//if we got here, reader is assumed to know how much data to expect next
//we send MSIO_MAX_PACKET_LEN or leftover (whichever is smaller)
static void __attribute__((section(".ramcode"))) msioCommsPrvPreparePacketForTx(void)
{
	//do not overwrite existing data	
	if (msioHaveLongDataToTx())
		return;
	
	if (!mCurTxingDataPtr)	//no more to send for this buffer
		msioCommsPrvPrepareHlMessageForTx();
	else {
		
		uint32_t now = mCurTxingDataLenLeft;
		void *buf = msioGetOutBuffer(1);
		
		if (now > MSIO_MAX_PACKET_LEN)
			now = MSIO_MAX_PACKET_LEN;
		
		msioCommsPrvCopyWordwise(buf, mCurTxingDataPtr, now);
		msioProvideLongReadData(buf, now);
		mCurTxingDataLenLeft -= now;
		if (mCurTxingDataLenLeft)
			mCurTxingDataPtr += now;
		else {
			mCurTxingDataPtr = NULL;
			msioCommsPrvUpdatePendingData(1 << mCurTxClrBitNoAfter, 0);
		}
		
		dtx(mCurTxingDataLenLeft ? 0xcd : 0xce);
	}
}

static void __attribute__((section(".ramcode"))) msioCommsPrvUpdatePendingData(uint32_t clearBits, uint32_t setBits)
{
	volatile uint8_t *regs = msioGetRegs();
	uint32_t newVal, oldVal, fail;
	
	do{
		asm volatile(
			"	ldrex  %0, [%3]		\n\t"
			"	bic    %1, %0, %4	\n\t"
			"	orrs   %1, %5		\n\t"
			"	strex  %2, %1, [%3]	\n\t"
			:"=&r"(oldVal), "=&r"(newVal), "=r"(fail)
			:"r"(&mPendingDataFlags), "r"(clearBits), "r"(setBits)
			:"cc", "memory"
		);
	} while (fail);
	
	dtx(0xdd);
	dtx(newVal >> 24);
	dtx(newVal >> 16);
	dtx(newVal >> 8);
	dtx(newVal >> 0);
	
	regs[MS_REG_NO_INT] = !!newVal;
	msioSignalIrq(!!newVal);
}

void __attribute__((used)) __attribute__((section(".ramcode"))) MSIO_IRQHandler(void)
{
	const void *data;
	uint16_t len;
	uint8_t tpc;

	dtx('x');
	if (!msioPoll(&tpc, &data, &len)) {
		dtx('z');
		fatal("MSIO gave us nothing\n");
	}
	
	dtx(tpc);

	if (tpc == MS_SET_CMD) {
		
		if (mIsInIoMode)
			msioSignalIrq(false);	//we treat all commands as irq ack :)
		else
			msioSimpleMemoryCommand(*(uint8_t*)data);
	}
	else if (tpc == MS_RD_LDATA) {
		
		if (mIsInIoMode)
			msioCommsPrvPreparePacketForTx();
		else
			msioSimpleMemoryLongReadCompleted();
	}
	else if (tpc == MS_RD_REG || tpc == MS_GET_INT) {
		
		if (mIsInIoMode)
			msioCommsPrvPreparePacketForTx();
		else
			msioSimpleMemoryRegsReadCompleted();
	}
	else if (tpc == MS_WR_REG) {
		
		volatile uint8_t *regs = msioGetRegs();
		uint_fast8_t rv;
		
		if (mIsInIoMode && (rv = regs[MSIO_REG_NO_POWER]) != OUR_CONSUMPTION_IN_CA) {
			
			regs[MSIO_REG_NO_POWER] = OUR_CONSUMPTION_IN_CA;
			
			dtx(0x88);
			dtx(0x77);
			dtx(rv);
			
			//0xc0 = power on
			//0xc1 = sleep
			
			//we do not do much about this, we are always powered on...
			//if this changes, "driverless" ioctl code in the driver will need to change
		}
	}
	else if (tpc == MS_WR_LDATA) {
		
		if (mIsInIoMode && len >= 3) {
			
			const struct MsioPktHeader *hdr = (const struct MsioPktHeader*)data;
			len -= 2;	//crc
			
			switch (hdr->pktTyp) {
				
				case MSIO_PKT_SET_BOOT_PARAMS: {
					
					const struct MsioPktBootParams *bp = (const struct MsioPktBootParams*)data;
					
					if (len != sizeof(struct MsioPktBootParams)) {
						loge("boot packet size wrong\n");
						break;
					}
					mBootPkt.dispW = __builtin_bswap16(bp->dispW);
					mBootPkt.dispH = __builtin_bswap16(bp->dispH);
					mBootPkt.supportedDepths = __builtin_bswap16(bp->supportedDepths);
					mBootPkt.hwrMiscFlags = __builtin_bswap16(bp->hwrMiscFlags);
					mBootPkt.hwrMiscExtFlags = __builtin_bswap16(bp->hwrMiscExtFlags);
					mBootPkt.curDepth = bp->curDepth;
					mBootPkt.miscBits = bp->miscBits;
					mBootPkt.curRtc = __builtin_bswap32(bp->curRtc);
					mBootPkt.hdr.pktTyp = MSIO_PKT_SET_BOOT_PARAMS;
					
					mBattPkt.voltageInCv =  __builtin_bswap16(bp->voltageInCv);
					mBattPkt.percent = bp->percent;
					mBattPkt.kind = bp->kind;
					mBattPkt.isPluggedIn = bp->isPluggedIn;
					
					mBattPkt.hdr.pktTyp = MSIO_PKT_BATT_INFO;
					
					//if callback has already been registered, set the RTC value correctly as we finally know it
					if (mRtcCbkF)
						mRtcCbkF(mBootPkt.curRtc, false);
					
					break;
				}
				
				case MSIO_PKT_PEN: {
					
					const struct MsioPktPen *pp = (const struct MsioPktPen*)data;
					int_fast16_t x = (int16_t)__builtin_bswap16(pp->x);
					int_fast16_t y = (int16_t)__builtin_bswap16(pp->y);
					
					if (len != sizeof(struct MsioPktPen)) {
						loge("pen packet size wrong\n");
						break;
					}
					
					if (mPenCbkF) {
						
						mPenCbkF(x, y);
				//		logi("(%d, %d)\n", x, y);
					}
					break;
				}
				
				case MSK_PKT_BUTTONS: {
					
					const struct MsioPktButtons *pb = (const struct MsioPktButtons*)data;
					
					if (len != sizeof(struct MsioPktButtons)) {
						loge("btns packet size wrong\n");
						break;
					}
					
					if (mBtnCbkF) {
						
						uint16_t keyStates[sizeof(pb->keyboardBits) / sizeof(*pb->keyboardBits)];
						uint_fast8_t i;
						
						for (i = 0; i < sizeof(pb->keyboardBits) / sizeof(*pb->keyboardBits); i++)
							keyStates[i] = __builtin_bswap16(pb->keyboardBits[i]);
						
						mBtnCbkF(__builtin_bswap32(pb->palmOsBitMask), keyStates);
				//		logi("(%d, %d)\n", x, y);
					}
					break;
				}
				
				case MSIO_PKT_RTC_REPORT: {
					const struct MsioRtcData *rtc = (const struct MsioRtcData*)data;
					
					if (len != sizeof(struct MsioRtcData)) {
						loge("rtc packet size wrong\n");
						break;
					}
					
					if (mRtcCbkF)
						mRtcCbkF(__builtin_bswap32(rtc->curTime), rtc->alarmTriggered);
					
					break;
				}
				
			}
		}
	}
	msioReleaseBuf();
}

bool __attribute__((section(".ramcode"))) msioCategoryChanged(uint_fast8_t newCat)
{
	volatile uint8_t *regs = msioGetRegs();
	
	regs[MS_REG_NO_CLASS] = 0xff;
	
	if (newCat == MS_CATEGORY_STORAGE) {
		
		regs[MS_REG_NO_TYPE] = 0xff;
		regs[MS_REG_NO_TYPE2] = 0xff;
		
		msioSimpleMemoryEnable(true);
		mIsInIoMode = false;
		regs[MS_REG_NO_INT] = 0;
		return true;
	}
	else {	//non-storage all go to re-palm category (makes DSP devices happy)
		
		regs[MS_REG_NO_TYPE] = 0;
		regs[MS_REG_NO_TYPE2] = 0;
		regs[MS_REG_NO_CATEGORY] = MS_CATEGORY_REPALM;
		msioSimpleMemoryEnable(false);
		mIsInIoMode = true;
		regs[MSIO_REG_NO_POWER] = OUR_CONSUMPTION_IN_CA;	//important that this is not zero and not >= 0x80
		regs[MS_REG_NO_INT] = 0;
		return true;
	}
}

const struct MsioPktBootParams* msioCommsWaitForContinueBoot(void)
{
	if (mBootPkt.hdr.pktTyp != MSIO_PKT_SET_BOOT_PARAMS) {
		
		logi("Waiting for boot...\n");
		
		do {
			asm volatile("":::"memory");
		} while (mBootPkt.hdr.pktTyp != MSIO_PKT_SET_BOOT_PARAMS);
		
		logi("Booting with %ux%u screen @ %u bpp (supported %04xh)\n",
			mBootPkt.dispW, mBootPkt.dispH, mBootPkt.curDepth, mBootPkt.supportedDepths);
	}
	
	return &mBootPkt;
}

void msioCommsInit(void)
{
	msioInit();
	NVIC_ClearPendingIRQ(MSIO_IRQn);
	NVIC_EnableIRQ(MSIO_IRQn);
	msioSimpleMemoryInit();
	msioSimpleMemoryEnable(true);
	msioEnable();
	logi("MSIO on\n");
/*
	//export funcs
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_UPDATE_START, visorCommsUpdateStart))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_COPY_TO_SRAM, visorCommsMsgCopyToSram))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_COPY_FROM_SRAM, visorCommsCopyMsgFromSram))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_SER_INIT, visorCommsSerInit))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_SER_OP, visorCommsSerOp))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_SER_TX, visorCommsSerTx))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_SER_DEINIT, visorCommsSerDeinit))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_GET_BUF_SZS, visorCommsGetBufSzs))
		fatal("Failed to export func\n")
*/
}

void msioCommsSetBrightness(uint8_t val)
{
	mNewBri = val;
	msioCommsPrvUpdatePendingData(0, 1 << HAVE_BIT_NEW_BRI_VAL);
}

void msioCommsSetBacklight(bool on)
{
	mNewBacklite = on;
	msioCommsPrvUpdatePendingData(0, 1 << HAVE_BIT_NEW_BACKLITE_VAL);
}

void msioCommsSetContrast(uint8_t val)
{
	mNewContrast = val;
	msioCommsPrvUpdatePendingData(0, 1 << HAVE_BIT_NEW_CONTRAST_VAL);
}

void msioCommsSetScreenDepth(uint8_t val)
{
	mNewDepth = val;
	msioCommsPrvUpdatePendingData(0, 1 << HAVE_BIT_NEW_DEPTH);
}

void msioCommsSetScreenClut(const struct PalmClutEntry *entries, uint_fast16_t nEntries)
{
	uint_fast16_t i;
	
	while (mPendingDataFlags & (1 << HAVE_BIT_NEW_CLUT));
	
	memcpy(mNewClutEntries, entries, sizeof(*mNewClutEntries) * nEntries);
	mNewClutNumEntries = nEntries;
	
	//make them look good for the target
	for (i = 0; i < nEntries; i++)
		mNewClutEntries[i].idx = i;
	
	msioCommsPrvUpdatePendingData(0, 1 << HAVE_BIT_NEW_CLUT);
}

void msioCommsVibAndLedControl(bool led, uint32_t pattern, uint16_t csecPerPiece, uint16_t csecBetween, uint16_t numTimes)
{
	struct MsioVibLedControl *ctl = led ? &mLedCmd : &mVibCmd;
	uint_fast16_t flag = led ? (1 << HAVE_BIT_NEW_LED_CMD) : (1 << HAVE_BIT_NEW_VIB_CMD);
	
	while (mPendingDataFlags & flag);
	
	ctl->hdr.pktTyp = led ? MSIO_PKT_LED_CONTROL : MSIO_PKT_VIB_CONTROL;	//might as well have a complete and ready packet
	ctl->numTimes = __builtin_bswap16(numTimes);
	ctl->pattern = __builtin_bswap32(pattern);
	ctl->csecPerPiece = __builtin_bswap16(csecPerPiece);
	ctl->csecBetween = __builtin_bswap16(csecBetween);
	
	msioCommsPrvUpdatePendingData(0, flag);
}

bool msioCommsIsScreenRedrawDone(void)
{
	return !(mPendingDataFlags & (1 << HAVE_BIT_NEW_DISP_DATA));
}

//XXX: use hash unit to check for screen diffs, it processes 64 bytes per 66 cycles and can be DMAd to
//we cna start it in parallel with compression, and abort it if hash matches prev screen
void msioCommsRequestScreenRedraw(const void *data, uint32_t len, uint_fast8_t thisDepth)
{
	if (!msioCommsIsScreenRedrawDone())
		return;
	
	mNewDispDataLen = screenCompress((uint8_t*)CPU_HARDWIRED_VTMP1_SPACE, data, len, thisDepth == 16);
	//logi("DISP %u -> %u\n", len, mNewDispDataLen);
	msioCommsPrvUpdatePendingData(0, 1 << HAVE_BIT_NEW_DISP_DATA);
}

const struct MsioPktBattInfo* msioCommsGetBatteryInfo(void)
{
	return &mBattPkt;
}

void msioCommsSetInputHandlers(MsioCommsButtonStateCbk btnF, MsioCommsPenStateCbk penF)
{
	mBtnCbkF = btnF;
	mPenCbkF = penF;
}

void msioCommsRtcSetSecondCbk(MsioCommsRtcSecondCbkF cbkF)
{
	mRtcCbkF = cbkF;
	
	//if callback is being registered after we were told to boot, set the RTC value correctly as we already know it
	if (mBootPkt.hdr.pktTyp == MSIO_PKT_SET_BOOT_PARAMS)
		cbkF(mBootPkt.curRtc, false);
}

void msioCommsRtcSetAlarm(uint32_t at)
{
	if (mNewRtcAlmVal != at) {
	
		mNewRtcAlmVal = at;
		msioCommsPrvUpdatePendingData(0, 1 << HAVE_BIT_RTC_ALM_REQ);
	}
}

void msioCommsRtcSetRtc(uint32_t val)
{
	mNewRtcNewVal = val;
	msioCommsPrvUpdatePendingData(0, 1 << HAVE_BIT_RTC_SET_REQ);
}


void msioCommsBreakLocks(void)
{
	//void
}