#include "disp.h"
#include "printf.h"
#include "machSpecific.h"
#include "msioComms.h"
#include <MemoryMgr.h>
#include <string.h>
#include "kernel.h"
#include "memmap.h"
#include "timers.h"
#include "boot.h"
#include "heap.h"
#include "irqs.h"
#include "mpu.h"
#include "cpu.h"
#include "dal.h"
#include "kal.h"

#define NON_PORTABLE
#include <HwrMiscFlags.h>
#undef NON_PORTABLE

extern void dtx(uint_fast8_t);

#define STANDARD_DENSITY			72

static int16_t mBri = -1, mContrast = -1;
static uint8_t mCurDepth;
static uint16_t mDispW, mDispH;
static int8_t mBacklightOn = -1;
static uint32_t mFpuCfg;
static struct PalmClutEntry mClut[256];		//.idx member not used and not valid

void dispSetBri(uint8_t bri)
{
	int16_t briReq = (int16_t)(uint16_t)bri;
	
	if (briReq != mBri) {
		
		mBri = briReq;
		msioCommsSetBrightness(briReq);
	}
}

void dispSetContrast(uint8_t bri)
{
	int16_t contReq = (int16_t)(uint16_t)bri;
	
	if (contReq != mContrast) {
		
		mContrast = contReq;
		msioCommsSetContrast(contReq);
	}
}

bool dispSetBacklight(bool on)
{
	const struct MsioPktBootParams *bp = msioCommsWaitForContinueBoot();
	
	if (mBacklightOn == -1)
		mBacklightOn = (bp->miscBits & MISC_BIT_BACKLIGHT_ON) ? 1 : 0;
	
	if (!(bp->hwrMiscFlags & hwrMiscFlagHasBacklight))			//no backlight? no backlight control
		return false;
	
	if (!mBacklightOn != !on) {
		
		msioCommsSetBacklight(on);
		mBacklightOn = on;
	}
	
	return true;
}

bool dispGetBacklight(void)
{
	const struct MsioPktBootParams *bp = msioCommsWaitForContinueBoot();
	
	if (mBacklightOn == -1)
		mBacklightOn = (bp->miscBits & MISC_BIT_BACKLIGHT_ON) ? 1 : 0;
	
	if (!(bp->hwrMiscFlags & hwrMiscFlagHasBacklight))			//no backlight? it is off
		return false;
	
	return mBacklightOn;
}

static void dispAllowWrites(bool on)
{
	mpuRegCfgPermOnly(DISP_MPU_REG, mFpuCfg | (on ? MPU_PERM_U_RW_S_RW : MPU_PERM_U_RO_S_RO));
}

bool __attribute__((used)) dispMmuFaultHandler(struct CortexExcFrame *exc, uint32_t addr)
{
	if (addr >= CPU_HARDWIRED_VRAM_ADDR && addr - CPU_HARDWIRED_VRAM_ADDR < CPU_HARDWIRED_VRAM_SIZE) {
		
		dtx(0xd0);
		dtx(addr >> 24);
		dtx(addr >> 16);
		dtx(addr >> 8);
		dtx(addr >> 0);
		dtx(exc->pc >> 24);
		dtx(exc->pc >> 16);
		dtx(exc->pc >> 8);
		dtx(exc->pc >> 0);
		
		
		TIM5->CR1 &=~ TIM_CR1_CEN;	//so it doesnt interrupt us
		dispAllowWrites(true);
		TIM5->CR1 |= TIM_CR1_CEN;
		return true;
	}
	
	return false;
}

static uint32_t dispMpuInit(void)
{
	uint32_t start = CPU_HARDWIRED_VRAM_ADDR, end = CPU_HARDWIRED_VRAM_ADDR + CPU_HARDWIRED_VRAM_SIZE - 1, diff = start ^ end, diffBits = 32 - __builtin_clz(diff);
	uint32_t regSzLg2, subregSzLg2, subregSzM1, srd = 0;
	
	//round such that we end up in a self-aligned block
	start >>= diffBits;
	start <<= diffBits;
	end += (1 << diffBits) - 1;
	end >>= diffBits;
	end <<= diffBits;
	
	//sort out our region and subregion size
	regSzLg2 = 31 - __builtin_clz(end - start);
	subregSzLg2 = regSzLg2 - 3;
	subregSzM1 = (1 << subregSzLg2) - 1;
	if (regSzLg2 < 8)
		fatal("vram region too small\n");
	
	//see if we can match start and end
	if ((CPU_HARDWIRED_VRAM_ADDR - start) & subregSzM1)
		fatal("vram region start cannot be matched\n");
	
	if (CPU_HARDWIRED_VRAM_SIZE & subregSzM1)
		fatal("vram region size cannot be matched\n");
	
	//calculate SRD
	srd = 0xff &~ (((1 << (CPU_HARDWIRED_VRAM_SIZE >> subregSzLg2)) - 1) << ((CPU_HARDWIRED_VRAM_ADDR - start) >> subregSzLg2));
	
	logt("vram of 0x%08x + 0x%08x will be configged as 0x%08x + 0x%08x, SRD 0x%02x\n",	
		CPU_HARDWIRED_VRAM_ADDR, CPU_HARDWIRED_VRAM_SIZE, start, 1 << regSzLg2, srd);
	
	mpuRegCfg(DISP_MPU_REG, start, 0 /* not enabled yet, but start set */);
	
	return ((regSzLg2 - 1) << 1) | (srd << 8) | MPU_FLAG_ENABLED | MPU_MEM_TYPE_RAM;
}

bool dispDrvInit(uint16_t *wP, uint16_t* hP, uint16_t* densityP, uint16_t *realDpiP, uint16_t *supportedDepthMapP, void** framebufferP, bool *indexedFmtIsLEP)
{
	const struct MsioPktBootParams *bp = msioCommsWaitForContinueBoot();
	uint_fast8_t maxDepth = 32 - __builtin_clz(bp->supportedDepths);
	uint32_t dispW = bp->dispW, dispH = bp->dispH;
	uint32_t vramSz = dispW * dispH * maxDepth / 8;
	uint_fast8_t densityMultiple;
	
	#if !defined(CPU_HARDWIRED_VRAM_ADDR) || !CPU_HARDWIRED_VRAM_ADDR
		#error "our mmu trickery requires a preallocated aligned buffer\n");
	#endif
		
	densityMultiple = (bp->miscBits & MISC_BIT_HIGH_DENSITY) ? 2 : 1;
	
	*wP = mDispW = dispW;
	*hP = mDispH = dispH;
	*densityP = STANDARD_DENSITY * densityMultiple;
	*supportedDepthMapP = bp->supportedDepths;
	*framebufferP = (void*)CPU_HARDWIRED_VRAM_ADDR;
	*realDpiP = 75 * densityMultiple;
	*indexedFmtIsLEP = false;
	
	mFpuCfg = dispMpuInit();
	dispAllowWrites(false);		//allow for now - timer will kick off a draw
	
	mCurDepth = bp->curDepth;
	
	//start timer to update LCD constantly
	TIM5->PSC = 0;
	TIM5->DIER = TIM_DIER_UIE;
	TIM5->SR = -1;
	TIM5->CNT = TIMER_TICKS_PER_MSEC * 1000 / 30;
	TIM5->ARR = TIMER_TICKS_PER_MSEC * 1000 / 30;
	TIM5->CR1 = TIM_CR1_DIR | TIM_CR1_URS;
	NVIC_EnableIRQ(TIM5_IRQn);
	
	return true;
}

void dispSetClut(int32_t firstIdx, uint32_t numEntries, const struct PalmClutEntry *entries)
{
	bool changed = false;
	uint32_t i;
	
	if (firstIdx == -1) {
		
		if (numEntries > 256)
			return;
	}
	else if (firstIdx < 0)
		return;
	else if (firstIdx >= 256 || numEntries + firstIdx > 256)
		return;
	
	for (i = 0; i < numEntries; i++) {
		
		uint32_t where = (firstIdx == -1) ? entries[i].idx : i + firstIdx;
		uint32_t r = entries[i].r;
		uint32_t g = entries[i].g;
		uint32_t b = entries[i].b;
		
		if (mClut[where].r != r || mClut[where].g != g || mClut[where].b != b)
			changed = true;
		
		mClut[where] = entries[i];
	}
	
	if (changed)
		msioCommsSetScreenClut(mClut, (mCurDepth < 8) ? (1 << mCurDepth) : 256);
}

void dispSetDepth(uint32_t depth)
{
	if (mCurDepth != depth) {
	
		mCurDepth = depth;
		msioCommsSetScreenDepth(mCurDepth);
	}
}

void dispManualUpdate(void)
{
	//nothing
}

void dispRequestUpdate(void)
{
	//nothing
}

void __attribute__((used)) TIM5_IRQHandler(void)	//high bits come first
{
	uint32_t fbLen = (uint32_t)mDispW * mDispH * mCurDepth / 8;
	
	dtx(0xd1);
	
	TIM5->SR = 0;		//ack the interrupt
	
	//we keep trying every 16.6ms till we succeed
	
	if (msioCommsIsScreenRedrawDone()) {
	
		dtx(0xd2);
		dispAllowWrites(false);
		TIM5->CR1 &=~ TIM_CR1_CEN;
		dtx(0xd3);
		msioCommsRequestScreenRedraw((void*)CPU_HARDWIRED_VRAM_ADDR, fbLen, mCurDepth);
		dtx(0xd4);
	}
}

void dispSleep(void)
{
	//todo
}

void dispWake(void)
{
	//todo
}
