#include "../dma_driver/DmaDriver.h"
#include "aximCpld.h"
#include "audiohw.h"
#include <string.h>
#include "printf.h"
#include "xscale.h"
#include "wm9705.h"
#include "input.h"

#define PEN_BUFFER_SIZE				8		//in samples (X and Y are one sample each)
#define PEN_TRASH_SAMPLES_START		5
#define PEN_TRASH_SAMPLES_END		4


static volatile enum WM9705penSampleType mWaitingSampleType;
static volatile uint32_t *mWaitingSamplePtr;
static volatile uint8_t mNumSamplesDesired;

static DrvInputKeyCbk mKeyCbk = NULL;
static DrvInputBtnCbk mBtnCbk = NULL;
static DrvInputPenCbk mPenCbk = NULL;
static DmaStream mPenDmaStream;
static uint32_t *mPenBuffer;


struct GpioToButtonMap {
	uint8_t gpioNum;
	uint8_t buttonBit 	: 6;
	uint8_t activeHigh	: 1;
};

static const struct GpioToButtonMap mBtnMap[] = {
	{0,  0,  0},	//power button (Active low)
	{2,  4,  1},	//hard #2: contacts (active high)
	{3,  3,  1},	//hard #1: calendar (active high)
	{4,  5,  1},	//hard #3: inbox (active high)
	//gpio 5 is battery door detect (high when closed)
	//gpio 9 is right tiny button (wireless), active high
	{13, 13, 1},	//left tiny button (voice rec), active high
	{11, 6,  1},	//hard #4: home (active high)
	{16, 28, 1},	//jog wheel up (active high). bit as per sony
	{22, 10, 1},	//jog wheel select (active high). bit as per sony
	{23, 29, 1},	//jog wheel down (active high). bit as per sony
	{81, 25, 1},	//navpad - right (active high)
	{82, 1,  1},	//navpad - up (active high)
	{83, 26, 1},	//navpad - select (active high)
	{84, 2,  1}, 	//navpad - down (active high)
	{85, 24, 1},	//navpad - left (active high)
};

static void inputPrvPenReport(int16_t x, int16_t y)
{
	if (mPenCbk)
		mPenCbk(x, y);
}

static void inputPrvPenSample(bool down, uint16_t x, uint16_t y)
{
	static uint8_t startSamplesDone = 0, endSamplesDone = 0, samplesBufP = 0;
	static uint16_t samplesBufX[PEN_TRASH_SAMPLES_END];
	static uint16_t samplesBufY[PEN_TRASH_SAMPLES_END];
	
	if (!down) {
		
		if (!mWaitingSamplePtr)
			wm9705penSetPden(true);	//only tell me when there is pen down
		
		if (startSamplesDone) {		//was the pen ever down?
			
			if (startSamplesDone < PEN_TRASH_SAMPLES_START) {	//we never reported a down (start discard), report last point as down and an immedaite up
				
				inputPrvPenReport(samplesBufX[samplesBufP], samplesBufY[samplesBufP]);
			}
			else if (endSamplesDone < PEN_TRASH_SAMPLES_END) {	//we never reported a down (end discard), report last point as down and an immedaite up
				
				uint32_t idx = (samplesBufP + PEN_TRASH_SAMPLES_END - 1) % PEN_TRASH_SAMPLES_END;
				
				inputPrvPenReport(samplesBufX[idx], samplesBufY[idx]);
			}
			//else we had reported a down
			
			//retport an up
			inputPrvPenReport(-1, -1);
		}
		
		startSamplesDone = 0;
		endSamplesDone = 0;
	}
	else {
		
		wm9705penSetPden(false);	//tell me even if pen is up (that is how we'll know)
		
		if (startSamplesDone < PEN_TRASH_SAMPLES_START) {
			startSamplesDone++;
			
			samplesBufX[samplesBufP] = x;	//record anyways
			samplesBufY[samplesBufP] = y;
		}
		else {
			
			if (endSamplesDone < PEN_TRASH_SAMPLES_END)
				endSamplesDone++;
			else
				inputPrvPenReport(samplesBufX[samplesBufP], samplesBufY[samplesBufP]);
			samplesBufX[samplesBufP] = x;
			samplesBufY[samplesBufP] = y;
			
			if (++samplesBufP == PEN_TRASH_SAMPLES_END)
				samplesBufP = 0;
		}
	}
}

static int32_t adcPrvGetSample(enum WM9705penSampleType samp, uint32_t numToAverage)
{
	volatile uint32_t sampVal = 0;
	uint32_t time;
	
	//wait for any other threads to be done
	time = 20971520;
	while (mWaitingSamplePtr) {	//wait
		if (!--time)
			return DAL_ADC_VAL_INVALID;
		asm volatile("":::"memory");
	}
	
	//race-y but we'll live
	mNumSamplesDesired = numToAverage;
	mWaitingSampleType = samp;
	asm volatile("":::"memory");
	mWaitingSamplePtr = &sampVal;	//must be last
	asm volatile("":::"memory");
	wm9705penSetExtraSamp(samp);
	wm9705penSetPden(false);		//will self-reset
	asm volatile("":::"memory");

	//wait for completion
	time = 20971520;
	while (mWaitingSamplePtr) {
		if (!--time)
			return DAL_ADC_VAL_INVALID;
		asm volatile("":::"memory");
	}			
	
	return (sampVal + numToAverage / 2) / numToAverage;
}

static int32_t aximBattTempCalc(uint32_t adc)	//input: PHONE adc measurement. out: temp in units of 0.1 degree C
{
	static uint16_t tempLookupTab[] = {138, 1339, 2020, 2360, 2600};
	static uint16_t tempOutputTab[] = {1000, 450, 250, 100, 0};
	const uint32_t minAdcVal = 138;
	uint32_t i;
	
	//first check for edges
	if (adc < tempLookupTab[0])
		return tempOutputTab[0];
	
	if (adc >= tempLookupTab[sizeof(tempLookupTab) / sizeof(*tempLookupTab) - 1])
		return -32768;
	
	//interpolate
	for (i = 0; i < sizeof(tempLookupTab) / sizeof(*tempLookupTab) - 1; i++) {
		
		if (adc >= tempLookupTab[i] && adc < tempLookupTab[i + 1]) {
			
			return tempOutputTab[i] - (tempOutputTab[i] - tempOutputTab[i + 1]) * (adc - tempLookupTab[i]) / (tempLookupTab[i + 1] - tempLookupTab[i]);
		}
	}
	
	//not reached
	return DAL_ADC_VAL_INVALID;
}

int32_t adcGetValue(enum AdcValueIdx which)
{
	int32_t samp, samp2;

	switch (which) {
		case AdcValBattery:
			samp = adcPrvGetSample(WM9705penSampleBMON, 8);
			if (samp == DAL_ADC_VAL_INVALID)
				return DAL_ADC_VAL_INVALID;
			return samp * 4200 / 1770;
		
		case AdcValBattTemp:
			samp = adcPrvGetSample(WM9705penSamplePHONE, 8);
			if (samp == DAL_ADC_VAL_INVALID || samp == 4095)
				return DAL_ADC_VAL_INVALID;
			return aximBattTempCalc(samp) * 10;
		
		case AdcValBackup:
			samp = adcPrvGetSample(WM9705penSampleAUXADC, 8);
			if (samp == DAL_ADC_VAL_INVALID)
				return DAL_ADC_VAL_INVALID;
			return samp * 1393 / 1755;
		
		case AdcValCurrent:
			samp = adcPrvGetSample(WM9705penSampleBMON, 8);
			if (samp == DAL_ADC_VAL_INVALID)
				return DAL_ADC_VAL_INVALID;
			aximCpldSetBits(AXIM_CPLD_BIT_CURRENT_MEAS, false);
			KALTaskDelay(30);
			samp2 = adcPrvGetSample(WM9705penSampleBMON, 8);
			aximCpldSetBits(AXIM_CPLD_BIT_CURRENT_MEAS, true);
			if (samp2 == DAL_ADC_VAL_INVALID)
				return DAL_ADC_VAL_INVALID;
			
			return (samp - samp2) * 50 * 4200 / 1770;
		
		default:
			return DAL_ADC_VAL_INVALID;
	}
}

static void inputPrvPenProcess(uint32_t *data, uint32_t len)
{
	enum WM9705penSampleType samp;
	static int32_t xVal = -1;
	uint32_t val;
	
	while (len--) {
		
		val = *data++;
		samp = (enum WM9705penSampleType)((val >> 12) & 7);
		
		switch (samp) {
			case WM9705penSampleX:
				xVal = val & 0x0fff;
				break;
			
			case WM9705penSampleY:
				if (xVal >= 0)		//have X?
					inputPrvPenSample(!!(val & 0x8000), xVal, val & 0xfff);
				xVal = -1;
				break;
						
			default:
				if (mWaitingSamplePtr && mWaitingSampleType == samp) {
					
					(*mWaitingSamplePtr) += val & 0x0fff;
					if (!--mNumSamplesDesired) {
						
						mWaitingSamplePtr = NULL;
						mWaitingSampleType = WM9705penSampleNone;
					}
				}
				break;
		}
	}
}

static void inputPrvPenDmaIrqHandler(void* userData, uint32_t strmSta)
{
	if (strmSta & DMA_STRM_IRQ_HALF)
		inputPrvPenProcess(mPenBuffer + 0 * PEN_BUFFER_SIZE / 2, PEN_BUFFER_SIZE / 2);
	else if (strmSta & DMA_STRM_IRQ_DONE)
		inputPrvPenProcess(mPenBuffer + 1 * PEN_BUFFER_SIZE / 2, PEN_BUFFER_SIZE / 2);
}

static void inputPrvButtonIrqh(void *buttonIdxAsP)
{
	uint32_t buttonIdx = (uint32_t)buttonIdxAsP;
	
	platGpioClearEdgeDetected(mBtnMap[buttonIdx].gpioNum);
	
	if (!mBtnCbk)
		return;
	
	mBtnCbk(1UL << mBtnMap[buttonIdx].buttonBit, !platGpioGetVal(mBtnMap[buttonIdx].gpioNum) == !mBtnMap[buttonIdx].activeHigh);
}

static void inputPrvGpiosConfig(bool withIrqs)		//if called with false, must be idempotent
{
	uint_fast8_t i;
	
	for (i = 0; i < sizeof(mBtnMap) / sizeof(*mBtnMap); i++) {
		
		uint32_t gpioNo = mBtnMap[i].gpioNum;
		bool enableable = true;
		uint32_t irqNo;
		
		switch (gpioNo) {
			case 0:
				irqNo = XSCALE_IRQ_NO_GPIO0;
				break;
			
			case 1:
				irqNo = XSCALE_IRQ_NO_GPIO1;
				break;
			
			default:
				irqNo = XSCALE_IRQ_NO_GPIO(gpioNo);
				enableable = false;
				break;
		}
		
		if (withIrqs) {
			
			HALInterruptSetHandler(REPALM_IRQ_NO_MANGLE(irqNo), inputPrvButtonIrqh, (void*)i);
			if (enableable)
				HALInterruptSetState(REPALM_IRQ_NO_MANGLE(irqNo), true);
		}
		
		platGpioSetFunc(gpioNo, 0);
		platGpioSetDir(gpioNo, false);
		platGpioClearEdgeDetected(gpioNo);
		
		if (withIrqs)
			platGpioSetEdgeDetect(gpioNo, true, true);
	}
	
	//reset button
	platGpioSetDir(1, false);
	platGpioSetFunc(1, 1);
}

kstatus_t drvInputInit(DrvInputKeyCbk keyCbk, DrvInputBtnCbk btnCbk, DrvInputPenCbk penCbk)
{
	mKeyCbk = keyCbk;
	mBtnCbk = btnCbk;
	mPenCbk = penCbk;
	
	inputPrvGpiosConfig(true);
	
	return KERN_STATUS_OK;
}

kstatus_t drvInputInitLate(void)
{
	static const struct DmaStreamUserCfg penDmaCfg = {
		.magic = CFG_STRUCT_MAGIX,
		.chan = XSCALE_DMA_CH_AC97_MDM_RX,
		.circBuf = 1,
		.perSz = __builtin_ctz(sizeof(uint32_t)),
		.memSz = __builtin_ctz(sizeof(uint32_t)),
		.perControlsFlow = 1,
		.memIncr = true,
		.perBurstSz = 32,
		.memBurstSz = 32,
		.toMem = 1,
		.numItems = PEN_BUFFER_SIZE,
	};
	struct PxaAC97 *ac97 = platPeriphP2V(PXA_BASE_AC97);
	uintptr_t pa;

	logi("initing input\n");
	
	if (!platDmaBufAlloc(PEN_BUFFER_SIZE * sizeof(uint32_t), &pa, (void**)&mPenBuffer))
		fatal("Cannot alloc pen DMA buffer\n");
	
	memset(mPenBuffer, 0xaa, PEN_BUFFER_SIZE * sizeof(uint32_t));
	
	mPenDmaStream = DmaLibStreamReserve(0, 15);
	if (!mPenDmaStream)
		fatal("Cannot grab input DMA stream\n");
		
	if (!DmaLibStreamConfigure(mPenDmaStream, &penDmaCfg))
		fatal("Cannot configure pen DMA stream\n");
	
	if (!DmaLibStreamSetIrqHandler(mPenDmaStream, inputPrvPenDmaIrqHandler, NULL))
		fatal("Cannot configure pen DMA irq handler\n");
	
	//addr should be PA, hence this weird construct
	if (!DmaLibStreamSetPeriphAddr(mPenDmaStream, (uintptr_t)&((struct PxaAC97*)PXA_BASE_AC97)->MODR))
		fatal("Cannot configure pen DMA PADDR\n");

	if (!DmaLibStreamSetMemAddr(mPenDmaStream, 0, pa))
		fatal("Cannot configure pen DMA MADDR\n");
	
	if (!DmaLibStreamSetIrqState(mPenDmaStream, DMA_STRM_IRQ_HALF | DMA_STRM_IRQ_DONE))
		fatal("Cannot enable pen DMA irqs\n");
	
	if (!DmaLibStreamSetEnabled(mPenDmaStream, true))
		fatal("Cannot enable pen DMA\n");
	
	if (!wm9705penInit(true, 6, false, false, false, false, false, false, true, 1, 14))
		fatal("Cannot init pen\n");
	
	return KERN_STATUS_OK;
}

kstatus_t drvInputPreSleep(void)
{
	//todo
	
	return KERN_STATUS_OK;
}

kstatus_t drvInputPostWake(void)
{
	//todo
	
	return KERN_STATUS_OK;
}

uint32_t drvInputReadKeysEarly(void)
{
	uint32_t ret = 0;
	uint_fast8_t i;
	
	inputPrvGpiosConfig(false);
	
	for (i = 0; i < sizeof(mBtnMap) / sizeof(*mBtnMap); i++) {
		
		if (platGpioGetVal(mBtnMap[i].gpioNum) == mBtnMap[i].activeHigh)
			ret |= 1UL << mBtnMap[i].buttonBit;
	}
		
	return ret;
}

