#include "halMemory.h"
#include "aximSleep.h"
#include "platform.h"
#include "memmap.h"
#include <stddef.h>
#include <stdint.h>
#include "printf.h"
#include "xscale.h"
#include "mpu.h"
#include "dal.h"

struct ExtraBackup {
	uint32_t svcRegs[2];
	uint32_t fiqRegs[7];
	uint32_t irqRegs[2];
	uint32_t abtRegs[2];
	uint32_t udfRegs[2];
	uint32_t usrRegs[11];
};

struct AximSleepStateRelevant {			//restored by bootloader
	uint32_t pc;
	uint32_t ctrlReg;		//p15 c1 c0
	uint32_t pxaCtrlReg;	//p15 c1 c1
	uint32_t ttbr;			//p15 c2 c0
	uint32_t dacr;			//p15 c3 c0
	uint32_t cpacr;			//p15 c13 c0
};

union AximSleepState {
	struct {
		uint32_t rfu1[8];						//untouched by bootloader
		struct AximSleepStateRelevant byBl;		//restored by BL
		struct ExtraBackup extra;				//unused by bootloader
	};
	uint32_t forChecksum[0x59];
};


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif



///XXX: see GO_TO_SLEEP     (ROM:0000271C) in bootloader

static void __attribute__((noinline, naked)) aximSleepPrvGoSleep(volatile uint32_t *mdrefr, volatile uint32_t *pspr)
{
	asm volatile(
		"	ldr		r2, =%7					\n\t"
	
		//save regs from each mode. remember that we should be in SYS state now, and we WAKE up in SVC state. SYS mode also cannot write to this region...
		"	add		r3, r2, %0				\n\t"
		"	msr		cpsr, #0xD3				\n\t"	//svc mode
		"	nop								\n\t"
		"	stmia	r3, {r13-r14}			\n\t"
		"	nop								\n\t"
		
		"	msr		cpsr, #0xDF				\n\t"	//sys mode (is hard)
		"	nop								\n\t"
		"	mov		r3, r13					\n\t"
		"	mov		r12, r14				\n\t"
		"	nop								\n\t"
		"	msr		cpsr, #0xD3				\n\t"	//svc mode
		"	nop								\n\t"
		"	mov		r13, r3					\n\t"
		"	mov		r14, r12				\n\t"
		"	add		r3, r2, %5				\n\t"
		"	stmia	r3, {r4-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %1				\n\t"
		"	msr		cpsr, #0xD1				\n\t"	//fiq mode
		"	nop								\n\t"
		"	stmia	r3, {r8-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %2				\n\t"
		"	msr		cpsr, #0xD2				\n\t"	//irq mode
		"	nop								\n\t"
		"	stmia	r3, {r13-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %3				\n\t"
		"	msr		cpsr, #0xD7				\n\t"	//abt mode
		"	nop								\n\t"
		"	stmia	r3, {r13-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %4				\n\t"
		"	msr		cpsr, #0xDB				\n\t"	//udf mode
		"	nop								\n\t"
		"	stmia	r3, {r13-r14}			\n\t"
		"	nop								\n\t"
	
		//save CPU state
		"	add		r3, r2, %6				\n\t"
		"	adr		r4, do_resume			\n\t"
		"	mrc		p15, 0, r5, c1, c0, 0	\n\t"
		"	mrc		p15, 0, r6, c1, c1, 0	\n\t"
		"	mrc		p15, 0, r7, c2, c0, 0	\n\t"
		"	mrc		p15, 0, r8, c3, c0, 0	\n\t"
		"	mrc		p15, 0, r9, c13, c0, 0	\n\t"
		"	stmia	r3, {r4-r9}				\n\t"
		
		//checksum everything and store sum into PSPR
		"	ldr		r4, =0x5a72				\n\t"
		"	mov		r5, r2					\n\t"
		"	mov		r6, #0x59				\n\t"
		"1:									\n\t"
		"	ldr		r7, [r5], #4			\n\t"
		"	add		r4, r7					\n\t"
		"	mov		r4, r4, ror #31			\n\t"
		"	subs	r6, #1					\n\t"
		"	bne		1b						\n\t"
		"	str		r4, [r1]				\n\t"
		
		//put memory into self refresh and go to sleep. we count on icache to make this loop possible
		"	movs   r3, #8					\n\t"
		"2:									\n\t"
		"	ldreq  r1, [r0]					\n\t"	//MDREFR.SLFRSH = 1		//put RAM into self refresh
		"	orreq  r1, #1 << 22				\n\t"
		"	streq  r1, [r0]					\n\t"
		"	nopeq							\n\t"	//this is quite existential. does conditionally executing a NO-OP change anything if
		"									\n\t"	// condition is false. and if it is true???
		"	moveq  r1, #3					\n\t"
		"	mcreq  p14, 0, r1, c7, c0, 0	\n\t"	//sleep mode
		"	nopeq							\n\t"
		"	nopeq							\n\t"
		"3:									\n\t"	//not reached when sleeping
		"	beq    3b						\n\t"
		"	subs   r3, #1					\n\t"
		"	b      2b						\n\t" 
		
		//we wake up here (in SVC mode), groggy but alive
		"do_resume:							\n\t"
		"	ldr		r2, =%7					\n\t"
		
		//restore each modes's state
		"	add		r3, r2, %0				\n\t"
		"	msr		cpsr, #0xD3				\n\t"	//svc mode
		"	nop								\n\t"
		"	ldmia	r3, {r13-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %1				\n\t"
		"	msr		cpsr, #0xD1				\n\t"	//fiq mode
		"	nop								\n\t"
		"	ldmia	r3, {r8-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %2				\n\t"
		"	msr		cpsr, #0xD2				\n\t"	//irq mode
		"	nop								\n\t"
		"	ldmia	r3, {r13-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %3				\n\t"
		"	msr		cpsr, #0xD7				\n\t"	//abt mode
		"	nop								\n\t"
		"	ldmia	r3, {r13-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %4				\n\t"
		"	msr		cpsr, #0xDB				\n\t"	//udf mode
		"	nop								\n\t"
		"	ldmia	r3, {r13-r14}			\n\t"
		"	nop								\n\t"
		
		"	add		r3, r2, %5				\n\t"
		"	msr		cpsr, #0xDF				\n\t"	//sys mode
		"	nop								\n\t"
		"	ldmia	r3, {r4-r14}			\n\t"
		"	nop								\n\t"
		
		//we are now in sys mode and regs are up
		"	bx		lr						\n\t"
	
		::
			/* 0 */	"i"(offsetof(union AximSleepState, extra.svcRegs)),
			/* 1 */	"i"(offsetof(union AximSleepState, extra.fiqRegs)),
			/* 2 */	"i"(offsetof(union AximSleepState, extra.irqRegs)),
			/* 3 */	"i"(offsetof(union AximSleepState, extra.abtRegs)),
			/* 4 */	"i"(offsetof(union AximSleepState, extra.udfRegs)),
			/* 5 */	"i"(offsetof(union AximSleepState, extra.usrRegs)),
			/* 6 */	"i"(offsetof(union AximSleepState, byBl)),
			/* 7 */	"i"(CPU_SLEEP_STATE_BASE)
		:"memory", "cc", "r0", "r1", "r2", "r3", "r12"
	);
}

void __attribute__((noinline))  aximSleepEnter(void)	//to the caller it will appear that this returned. noinline to avoid being inlined into thumb code
{
	struct PxaPwrMgr *pwr = platPeriphP2V(PXA_BASE_PWR_MGR);
	uint32_t irqState, dummy;
	
	//irqs and fiqs off
	asm volatile(
		"	mrs %0, cpsr		\n\t"
		"	orr %1, %0, #0xc0	\n\t"	//irq and fiq off
		"	msr cpsr, %1		\n\t"
		:"=r"(irqState), "=r"(dummy)
		:
		:"memory", "cc"
	);
	
	//pwr state setup
	pwr->PWER = 0x8000c58f;	//wake srcs: rtc, reset button, power button, hard buttons 1-6, sd chip
	pwr->PFER = 0xc48c;		//falling edge detect on active high buttons
	pwr->PRER = 0x0103;		//rising edge detect on active low buttons/irws
	pwr->RCSR = 0x0f;
	pwr->PCFR |= 1;
	pwr->PMCR &=~ 1;
	
	//before we go to sleep, we need to identity-map the bootloader, else it'll turn on the MMU and crash
	mmuPrvSleepMapping(true);
	HALInvalidateICache(NULL, -1);	//we'll not be writing to cacheable regions form now on...
	aximSleepPrvGoSleep(&((struct PxaMemCtrl*)platPeriphP2V(PXA_BASE_MEM_CTRL))->MDREFR, &pwr->PSPR);
	
	//wake here
	mmuPrvSleepMapping(false);
	
	//restore more periphs
	
	//restore irq states
	asm volatile(
		"	msr cpsr, %0		\n\t"
		:
		:"r"(irqState)
		:"memory", "cc"
	);
}

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif


void machSleep(void)
{
	logi("goig to sleep\n");
	aximSleepEnter();
	logi("waking up\n");
	dalModifyWakeFlags(DAL_WAKE_FLAG_GENERAL, 0);
}
