#include "disp.h"
#include "printf.h"
#include "memmap.h"
#include "machSpecific.h"
#include "palmcardProto.h"
#include "palmcardComms.h"
#include <MemoryMgr.h>
#include <string.h>
#include "pinout.h"
#include "input.h"
#include "heap.h"
#include "cpu.h"
#include "dal.h"
#include "kal.h"


#define MAX_SUPPORTED_BPP						4
#define DISP_WIDTH								160
#define DISP_HEIGHT								160
#define DISP_PALM_DENSITY						72
#define DISP_REAL_DENSITY						72	//actually is 72.75


static DrvInputKeyCbk mKeyCbk = NULL;
static DrvInputBtnCbk mBtnCbk = NULL;
static DrvInputPenCbk mPenCbk = NULL;

static uint8_t mCurDepth, mLastBtnState = 0;
static bool mDispOn = false, mBacklightOn;
static uint16_t mCurBatteryMv = 3000;
static void* mFb;



static void disPrvPenReportReal(int16_t x, int16_t y)
{
	if (mPenCbk)
		mPenCbk(x, y);
}

static void disPrvSort(int16_t *arr, uint_fast8_t num)
{
	uint_fast8_t i, j, k;
	
	for (i = 0; i < num - 1; i++) {
		
		int16_t t;
		
		for (k = i, j = i + 1; j < num; j++) {
			
			if (arr[k] > arr[j])
				k = j;
		}
		t = arr[i];
		arr[i] = arr[k];
		arr[k] = t;
	}
}

static void disPrvPenReport(int16_t xv, int16_t yv)
{
	#define MEDIAN_OF		3
	
	static int16_t x[MEDIAN_OF], y[MEDIAN_OF];
	static uint8_t counter;
	
	if (xv < 0 || yv < 0) {
		
		disPrvPenReportReal(xv, yv);
		counter = 0;
		return;
	}
	
	x[counter] = xv;
	y[counter] = yv;
	
	if (++counter == MEDIAN_OF) {
		
		counter = 0;
		disPrvSort(x, MEDIAN_OF);
		disPrvSort(y, MEDIAN_OF);
		disPrvPenReportReal(x[MEDIAN_OF / 2], y[MEDIAN_OF / 2]);
	}
}

void palmcardCommsExtBtnReport(uint_fast8_t state)
{
	uint_fast8_t prevState = mLastBtnState;
	
	mLastBtnState = state;	
	
	if (prevState != state) {
		
		uint32_t diff = prevState ^ state;
		uint32_t newlyPressed = state & diff;
		uint32_t newlyReleased = prevState & diff;
		
		while (newlyPressed) {
			uint32_t bit = newlyPressed - (newlyPressed & (newlyPressed - 1));
			newlyPressed &=~ bit;
			if (mBtnCbk)
				mBtnCbk(bit, true);
		}
		while (newlyReleased) {
			uint32_t bit = newlyReleased - (newlyReleased & (newlyReleased - 1));
			newlyReleased &=~ bit;
			if (mBtnCbk)
				mBtnCbk(bit, false);
		}
	}
}

void palmcardCommsExtPenReport(int16_t x, int16_t y)
{
	#define FIRST_TOSS		3	//first this many oints are tossed
	#define LAST_TOSS		1	//last this many are averaged (and dropped at end)
	
	static uint8_t mFirstCollected, mLastCollected, mArrPtr;
	static uint16_t buf[LAST_TOSS][2];
	static uint32_t avgX, avgY;
	
	if (x < 0 || y < 0) {	//pen is up
		
		if (mLastCollected == LAST_TOSS)	//we were already reporting data, just report pen up
			disPrvPenReport(-1, -1);
		else if (mLastCollected) {			//we did not collect enough to report. Report last point and an up
			mArrPtr = (mArrPtr + LAST_TOSS - 1) % LAST_TOSS;
			disPrvPenReport(buf[mArrPtr][0], buf[mArrPtr][1]);
			disPrvPenReport(-1, -1);
		}
		//else we did not collect enough for FIRST toss - and had reported nothing
		
		mFirstCollected = 0;
		mLastCollected = 0;
		avgX = 0;
		avgY = 0;
	}
	else if (mFirstCollected < FIRST_TOSS) {	//start toss
		
		mFirstCollected++;
	}
	else {
		if (mLastCollected == LAST_TOSS) {
			
			disPrvPenReport(avgX / LAST_TOSS, avgY / LAST_TOSS);
			avgX -= buf[mArrPtr][0];
			avgY -= buf[mArrPtr][1];
		}
		else
			mLastCollected++;
		buf[mArrPtr][0] = x;
		buf[mArrPtr][1] = y;
		avgX += x;
		avgY += y;
		mArrPtr = (mArrPtr + 1) % LAST_TOSS;
	}
}

static void dispPrvSendNewConfigAndWait(uint_fast16_t config)
{
	palmcardCommsSetDeviceState((palmcardCommsGetDeviceState() &~ PCC_DEVSTATE_MASK_SCREEN) | (config << PCC_DEVSTATE_SHIFT_SCREEN));
	palmcardCommsUpdateIrqSta(PCC_IRQ_BIT_HAVE_NEW_DEVICE_CONFIG, 0);
	while (palmcardCommsGetPendingIrqs() & PCC_IRQ_BIT_HAVE_NEW_DEVICE_CONFIG);
}

static void dispPrvTurnOff(void)
{
	uint16_t t;
	
	if (!mDispOn)
		return;
	mDispOn = false;
	
	logi("DISP: display off start\n");
	palmcardCommsSetScreenDataPtr(NULL, 0, 0);
	
	logi("DISP: data set to NULL\n");
	dispPrvSendNewConfigAndWait(PCC_DEVSTATE_VAL_SCREEN_OFF);
	
	logi("DISP: display off end\n");
}

static void dispPrvTurnOn(uint_fast8_t depth, bool firstTime)
{
	uint_fast16_t config;
	uint32_t i;
	
	if (mDispOn && depth == mCurDepth)
		return;
	else if (mDispOn)
		fatal("depth change with no off\n");
	
	logi("DISP: depth %u\n", depth);
	
	switch (depth) {
		default:	//fallthrough
		case 1: config = PCC_DEVSTATE_VAL_SCREEN_1_BPP;	break;
		case 2:	config = PCC_DEVSTATE_VAL_SCREEN_2_BPP; break;
		case 4: config = PCC_DEVSTATE_VAL_SCREEN_4_BPP; break;
	}
	
	palmcardCommsSetScreenDataPtr(mFb, 160 * 160, depth);
	dispPrvSendNewConfigAndWait(config);
	logi("DISP now on\n");
	
	mDispOn = true;
}

static void dispPrvPower(uint_fast8_t depth, bool firstTime)
{
	if (depth && mDispOn)
		fatal("cannot turn on a display that is on\n");
	else if (!depth && !mDispOn)
		fatal("cannot turn off a display that is off\n");
	else if (!depth)
		dispPrvTurnOff();
	else
		dispPrvTurnOn(depth, firstTime);
}

void dispSetDepth(uint32_t depth)
{
	if (mCurDepth != depth) {
		
		if (mDispOn) {
			dispPrvPower(0, false);
			dispPrvPower(depth, false);
		}
		mCurDepth = depth;
	}
}

void dispSetBri(uint8_t bri)
{
	//not supported
}

void dispManualUpdate(void)
{
	//screen is always auto-updating
}

void dispSetContrast(uint8_t bri)
{
	//we have no contrast settings
}

bool dispSetBacklight(bool on)
{
	uint_fast16_t devState;
		
	mBacklightOn = on;
	
	devState = palmcardCommsGetDeviceState();
	if (on)
		devState |= PCC_DEVSTATE_BIT_BACKLIGHT_ON;
	else
		devState &=~ PCC_DEVSTATE_BIT_BACKLIGHT_ON;
	palmcardCommsSetDeviceState(devState);
	palmcardCommsUpdateIrqSta(PCC_IRQ_BIT_HAVE_NEW_DEVICE_CONFIG, 0);
	
	return true;
}

bool dispGetBacklight(void)
{
	return mBacklightOn;
}

void dispRequestUpdate(void)
{
	//screen is always auto-updating
}

void dispSleep(void)
{
	dispPrvPower(0, false);
	//backlight off?
}

void dispWake(void)
{
	dispPrvPower(mCurDepth, true);
}

void dispSetClut(int32_t firstIdx, uint32_t numEntries, const struct PalmClutEntry *entries)
{
	//not needed
}

bool dispDrvInit(uint16_t *wP, uint16_t* hP, uint16_t* densityP, uint16_t *realDpiP, uint16_t *supportedDepthMapP, void** framebufferP, bool *indexedFmtIsLEP)
{
	uint32_t vramSz, supportedBppMap;
	
	logi("DISP is %u x %u, %u ppi\n", DISP_WIDTH, DISP_HEIGHT, DISP_PALM_DENSITY);
	
	vramSz = DISP_WIDTH * DISP_HEIGHT * MAX_SUPPORTED_BPP / 8;
	
	#if CPU_HARDWIRED_VRAM_ADDR
		if (CPU_HARDWIRED_VRAM_SIZE >= vramSz)
			mFb = (void*)CPU_HARDWIRED_VRAM_ADDR;
		else
			fatal("VRAM is too small %u < %u\n", CPU_HARDWIRED_VRAM_SIZE, vramSz);
	#else

		mFb = kheapAlloc(vramSz);
		if (!mFb) {
			loge("Cannot alloc fb\n");
			return false;
		}
	#endif
	
	memset(mFb, 0, vramSz);
	
	supportedBppMap = ((1 << 16) | (1 << 8) | (1 << 4) | (1 << 2) | (1 << 1)) >> 1;	//all depths palmos supports
	supportedBppMap &= (1 << MAX_SUPPORTED_BPP) - 1;								//limit to what we support
	*supportedDepthMapP = supportedBppMap;
	
	mCurDepth = 32 - __builtin_clz(supportedBppMap);
	dispPrvPower(mCurDepth, true);
	
	*wP = DISP_WIDTH;
	*hP = DISP_HEIGHT;
	*densityP = DISP_PALM_DENSITY;
	*framebufferP = mFb;
	*indexedFmtIsLEP = false;
	*realDpiP = DISP_REAL_DENSITY;
	
	return true;
}

kstatus_t drvInputInit(DrvInputKeyCbk keyCbk, DrvInputBtnCbk btnCbk, DrvInputPenCbk penCbk)
{
	mKeyCbk = keyCbk;
	mBtnCbk = btnCbk;
	mPenCbk = penCbk;
	
	return KERN_STATUS_OK;
}

uint32_t drvInputReadKeysEarly(void)
{
	return mLastBtnState;
}

void dockHwrInit(void)
{
	halDockStatusChanged(0);
}

static uint_fast8_t batteryPrvCalcPercentWithTable(uint16_t mv, const uint16_t *table)
{
	uint_fast8_t i;
	
	if (mv <= table[0])
		return 0;
	if (mv >= table[10])
		return 100;
	
	for (i = 1; i < 10 && table[i] < mv; i++);	//we end when i is the table entry that is the TOP of our range
	
	return ((table[i] - mv) * (i - 1) * 10 + (mv - table[i - 1]) * i * 10 + (table[i] - table[i - 1]) / 2) / (table[i] - table[i - 1]);
}

static uint_fast8_t batteryPrvCalcPercentWithKind(uint16_t mv, SysBatteryKind kind)
{
	static const uint16_t tabRechargeableAlkaline[] = {2240, 2350, 2400, 2440, 2490, 2560, 2630, 2700, 2800, 2900, 3000, };
	static const uint16_t tabAlkaline[] = {1800, 2100, 2250, 2330, 2380, 2430, 2480, 2540, 2620, 2750, 3000, };
	static const uint16_t tabNiCad[] = {2100, 2250, 2350, 2360, 2380, 2390, 2400, 2410, 2440, 2470, 2500, };
	static const uint16_t tabNimh[] = {2200, 2320, 2380, 2420, 2460, 2480, 2490, 2500, 2510, 2570, 2650, };

	static const uint16_t* tabs[] = {
		[sysBatteryKindRechAlk] = tabRechargeableAlkaline,
		[sysBatteryKindAlkaline] = tabAlkaline,
		[sysBatteryKindNiCad] = tabNiCad,
		[sysBatteryKindNiMH] = tabNimh,
	};
	
	if ((uint32_t)kind >= sizeof(tabs) / sizeof(*tabs) || !tabs[kind]) {
		
		logw("no battery table for this kind: %u\n", kind);
		return 50;	//always
	}
	
	return batteryPrvCalcPercentWithTable(mv, tabs[kind]);
}

void batteryInfo(SysBatteryKind *kindP, bool *haveMainsP, uint8_t *percentP)
{
	SysBatteryKind myBatteryKind = sysBatteryKindAlkaline;
	
	if (kindP)
		*kindP = myBatteryKind;
	
	if (haveMainsP)
		*haveMainsP = false;
	
	if (percentP)
		*percentP = batteryPrvCalcPercentWithKind(mCurBatteryMv, myBatteryKind);
}

void palmcardCommsExtBattReport(uint16_t sample, uint16_t Vref)
{
	mCurBatteryMv = (sample * Vref + 32767) >> 16;
}