#include "machSpecific.h"
#include "pinout.h"
#include "printf.h"
#include "i2cIO.h"
#include "cpu.h"

#define IO_EXPANDER_ADDR		0x20		//7-bit addr
#define REG_INPUT				0x00
#define REG_OUTPUT				0x01
#define REG_POL_INVERT			0x02		//for input only
#define REG_DIRECTION			0x03		//1 is in, 0 is out

#define OUR_I2C_HZ				100000



static uint8_t mCurOutVal = (1 << XPIN_RSTCTL) | (0 << XPIN_RSTPALM) | (0 << XPIN_LED) | (1 << XPIN_IRDA_nEN) | (1 << XPIN_SD_nEN) | (1 << XPIN_RAMnROM);
static uint8_t mCurDirVal = (1 << XPIN_RSTCTL) | (1 << XPIN_RSTPALM) | (0 << XPIN_LED) | (0 << XPIN_IRDA_nEN) | (0 << XPIN_SD_nEN) | (0 << XPIN_RAMnROM);


static void i2cRegWrite(uint8_t reg, uint8_t val)
{
	i2c1_hw->data_cmd = reg;
	i2c1_hw->data_cmd = I2C_IC_DATA_CMD_STOP_BITS + val;
	while (!(i2c1_hw->status & I2C_IC_STATUS_TFE_BITS));
	while (i2c1_hw->status & I2C_IC_STATUS_MST_ACTIVITY_BITS);
	asm volatile("dsb sy");
}

void i2cIoInit(void)
{
	//disable
	i2c1_hw->enable &=~ (I2C_IC_ENABLE_TX_CMD_BLOCK_BITS | I2C_IC_ENABLE_ABORT_BITS | I2C_IC_ENABLE_ENABLE_BITS);

	//configure
	i2c1_hw->con = (i2c1_hw->con &~ (I2C_IC_CON_IC_10BITADDR_MASTER_BITS | I2C_IC_CON_SPEED_BITS)) | I2C_IC_CON_RX_FIFO_FULL_HLD_CTRL_BITS | I2C_IC_CON_IC_SLAVE_DISABLE_BITS | I2C_IC_CON_IC_RESTART_EN_BITS | (I2C_IC_CON_SPEED_VALUE_FAST << I2C_IC_CON_SPEED_LSB) | I2C_IC_CON_MASTER_MODE_BITS;
	i2c1_hw->tar = (i2c1_hw->tar &~ (I2C_IC_TAR_SPECIAL_BITS | I2C_IC_TAR_GC_OR_START_BITS | I2C_IC_TAR_IC_TAR_BITS)) | (IO_EXPANDER_ADDR << I2C_IC_TAR_IC_TAR_LSB);
	i2c1_hw->fs_scl_hcnt = (i2c1_hw->fs_scl_hcnt &~ I2C_IC_FS_SCL_HCNT_IC_FS_SCL_HCNT_BITS) | ((CPU_CLOCK_RATE / OUR_I2C_HZ * 3 / 5) << I2C_IC_FS_SCL_HCNT_IC_FS_SCL_HCNT_LSB);
	i2c1_hw->fs_scl_lcnt = (i2c1_hw->fs_scl_lcnt &~ I2C_IC_FS_SCL_LCNT_IC_FS_SCL_LCNT_BITS) | ((CPU_CLOCK_RATE / OUR_I2C_HZ * 2 / 5) << I2C_IC_FS_SCL_LCNT_IC_FS_SCL_LCNT_LSB);
	i2c1_hw->fs_spklen = (i2c1_hw->fs_spklen &~ I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_BITS) | ((CPU_CLOCK_RATE / OUR_I2C_HZ / 20) << I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_LSB);
	i2c1_hw->intr_mask &=~ I2C_IC_INTR_MASK_BITS;
	i2c1_hw->enable |= I2C_IC_ENABLE_ENABLE_BITS;
	
	asm volatile("dsb sy");
		
	i2cRegWrite(REG_DIRECTION, 0xff &~ (1 << XPIN_RAMnROM));			//all inputs except XPIN_RAMnROM which we cannot touch (safe)
	i2cRegWrite(REG_OUTPUT, mCurOutVal);		//proper values
	i2cRegWrite(REG_DIRECTION, mCurDirVal);		//proper direction
}

void i2cIoPalmReset(void)
{
	//first insulate us from the reset by enabling our pullup to reset
	i2cRegWrite(REG_OUTPUT, mCurOutVal &~ (1 << XPIN_RSTCTL));
	
	//then pull down the reset pin by enabling it as output
	i2cRegWrite(REG_OUTPUT, mCurOutVal &~ ((1 << XPIN_RSTCTL) | (1 << XPIN_RSTPALM)));
	
	//then wait
	machBusyWaitDelayMsec(10);
	
	//then undo it in reverse order
	i2cRegWrite(REG_OUTPUT, mCurOutVal &~ (1 << XPIN_RSTCTL));
	i2cRegWrite(REG_OUTPUT, mCurOutVal);
}

static bool i2cIoPrvBitEnable(bool on, uint8_t mask)	//for active low outputs
{
	uint8_t oldVal = mCurOutVal, newVal = oldVal;
		
	if (on)
		newVal &=~ mask;
	else
		newVal |= mask;
	
	if (newVal != mCurOutVal)
		i2cRegWrite(REG_OUTPUT, mCurOutVal = newVal);
	
	return !(oldVal & mask);
}

bool i2cIoLED(bool on)
{
	return !i2cIoPrvBitEnable(!on, 1 << XPIN_LED);
}

bool i2cIoIrdaEnable(bool on)
{
	return i2cIoPrvBitEnable(on, 1 << XPIN_IRDA_nEN);
}

bool i2cIoSdCardEnable(bool on)
{
	return i2cIoPrvBitEnable(on, 1 << XPIN_SD_nEN);
}

void __attribute__((section(".ramcode"), used, noinline)) i2cIoRamAccessEnable(uint32_t on)		//secial case of i2cRegWrite
{
	uint_fast8_t newVal = (mCurOutVal &~ (1 << XPIN_RAMnROM)) | (on << XPIN_RAMnROM);
	
	mCurOutVal = newVal;
	i2c1_hw->data_cmd = REG_OUTPUT;
	i2c1_hw->data_cmd = I2C_IC_DATA_CMD_STOP_BITS + newVal;
	while (!(i2c1_hw->status & I2C_IC_STATUS_TFE_BITS));
	while (i2c1_hw->status & I2C_IC_STATUS_MST_ACTIVITY_BITS);
	asm volatile("dsb sy");
}
