#define WEAK __attribute__ ((weak))
#define ALIAS(f) __attribute__ ((weak, alias (#f)))

#define NON_PORTABLE
#include <HwrMiscFlags.h>
#undef NON_PORTABLE

#include <stdint.h>
#include <stdio.h>
#include "CortexEmuCpu.h"
#include "printf.h"
#include "memmap.h"
#include "entry.h"
#include "heap.h"
#include "irqs.h"
#include "boot.h"
#include "mpu.h"


void __attribute__((used)) IntDefaultHandler(void)
{
	cpuIrqDefaultHandler();
}

WEAK void Input_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void RtcHz_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void RtcAlarm_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void OsTimer_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void AudioOut_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void AudioIn_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void Ethernet_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void Joystick_IRQHandler(void) ALIAS(IntDefaultHandler);


__attribute__ ((aligned (128))) void (*const __ISR_VECTORS[]) (void) =
{
	0,							// unused: initial sp
	0,							// unused: reset handler

	NMI_Handler,				// The NMI handler
	HardFault_Handler,			// The hard fault handler

#ifdef BUILD_FOR_THUMB_1
	0,
	0,
	0,
#else
	MemManage_Handler,
	BusFault_Handler,
	UsageFault_Handler,
#endif
	0,
	0,
	0,
	0,
	
	SVC_Handler,				// SVCall handler
#ifdef BUILD_FOR_THUMB_1
	0,
#else
	DebugMonitor_Handler,		// Reserved
#endif
	0,
	PendSV_Handler,				// The PendSV handler
	SysTick_Handler,			// The SysTick handler
	
	// Chip Level interrupts here
	Input_IRQHandler,
	RtcHz_IRQHandler,
	RtcAlarm_IRQHandler,
	OsTimer_IRQHandler,
	AudioOut_IRQHandler,
	AudioIn_IRQHandler,
	Ethernet_IRQHandler,
	Joystick_IRQHandler,
};

void machIdle(void)
{
	asm volatile("wfi");
}

static void mpuRegCfg(uint32_t idx, uint32_t addr, uint32_t cfg)
{
	#ifdef __MPU_PRESENT
		if (!(MPU->TYPE & 1) && (MPU->TYPE & MPU_TYPE_DREGION_Msk)) {	//we do what we can
			
			MPU->RBAR = addr | 0x10 | idx;
			MPU->RASR = cfg;
		}
	#endif
}

//configures mmu for storage ram
void machSetStorageAreaWriteable(bool writeable)
{
	//12MB region using SRD (region is 16mb)
	mpuRegCfg(1, ALL_RAM_BASE, (writeable ? MPU_PERM_U_RW_S_RW : MPU_PERM_U_RO_S_RO) | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (23 << 1) | 0xc000);
}

void __attribute__((used)) machInit(uint32_t stage, const void* data)
{
	if (stage == STAGE_INIT_SET_VTOR) {
		SCB->VTOR = (uint32_t)__ISR_VECTORS;
	}
	else if (stage == STAGE_SETUP_HEAPS) {
		
		kheapRegisterHeap(HAL_STATIC_MEM_BASE, HAL_STATIC_MEM_SIZE, MEM_USABLE_AS_STACK | MEM_USABLE_FOR_DMA | MEM_USABLE_FOR_EXEC | MEM_FAST);
	}
	else if (stage == STAGE_INIT_MPU) {
		
		#ifdef __MPU_PRESENT
		
			//4MB region for ROM
			mpuRegCfg(0, ALL_RAM_BASE + 0x00c00000, MPU_PERM_U_RO_S_RO | MPU_MEM_TYPE_ROM | MPU_FLAG_ENABLED | (21 << 1) | 0x0000);
			
			//12MB region for storage ram (dyn ram overlays it)
			machSetStorageAreaWriteable(false);
			
			#ifdef HUGE_DYN_RAM
			
				//16MB region for dynamic ram & vram, srd makes it 10
				mpuRegCfg(2, ALL_RAM_BASE, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (23 << 1) | 0xe000);
			
			#else
			
				//4MB region for dynamic ram & vram, SRD makes it 2.5MB
				mpuRegCfg(2, ALL_RAM_BASE, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (21 << 1) | 0xe000);
				
			#endif
			
			//SCB (0xE0000000 + 0x10000000, device, nx)
			mpuRegCfg(3, 0xE0000000, MPU_PERM_U_XX_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | (27 << 1) | 0x0000);
			
			//Periphs (0x40000000 + 0x10000000, device, nx)
			mpuRegCfg(4, 0x40000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | (27 << 1) | 0x0000);
			
			//Built in SRAM (0x20000000 + 0x00180000. configged size is 0x00200000, SRD 0xc0, RAM)
			mpuRegCfg(5, 0x20000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (20 << 1) | 0xc000);
			
			//Built in CCM 64K, NX (since we cannot execute from there anyways)
			mpuRegCfg(6, 0x10000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | MPU_PERM_NX | (15 << 1) | 0x0000);
			
			//mpu on
			MPU->CTRL = MPU_CTRL_ENABLE_Msk | MPU_CTRL_HFNMIENA_Msk | MPU_CTRL_PRIVDEFENA_Msk;	//PRIVDEFENA is for devices, without HFNMIENA we cnanot execute form RAM on real STM hw
		
		#endif
	}
	else if (stage == STAGE_INIT_INTERRUPTS) {
		
		const struct MachInitDataInterrupts *info = (const struct MachInitDataInterrupts*)data;
		uint32_t mediumPrio = (info->lowestAllowablePrio + info->highestAllowablePrio) / 2;
		
		//set all HW ints to medium prio
		NVIC_SetPriority(Input_IRQn, mediumPrio);
		NVIC_SetPriority(RtcHz_IRQn, mediumPrio);
		NVIC_SetPriority(RtcAlarm_IRQn, mediumPrio);
		NVIC_SetPriority(AudioOut_IRQn, mediumPrio);
		NVIC_SetPriority(AudioIn_IRQn, mediumPrio);
	
		//scheduler timer interrupt is high prio too so nobody else can interrupt it (and more importantly - it will not interrupt syscalls and vise-versa)
		NVIC_SetPriority(OsTimer_IRQn, info->schedulingTimerPrio);
	}
}

bool hwMaybeGetRomToken(uint32_t name, const void **dataP, uint16_t *szP)
{
	return false;
}

void hwGetMiscFlags(uint16_t *miscFlagsP, uint16_t *extMiscFlagsP)
{
	if (miscFlagsP)
		*miscFlagsP = hwrMiscFlagHasMiscFlagExt | hwrMiscFlagHasCradleDetect | hwrMiscFlagNoRTCBug | hwrMiscFlagHasMbdIrDA;
	
	if (extMiscFlagsP)
		*extMiscFlagsP = hwrMiscFlagExt115KIrOK | hwrMiscFlagExtHasLiIon;
}

int32_t cpuGetClockRate(enum ClockRateDevice dev)
{
	switch (dev) {
		case CpuClockRate:
		case TimerClockRate:
			return CPU_CLOCK_RATE;
		
		default:
			return-1;
	}
}

bool hwPwrCtl(uint32_t selector, const uint32_t *newValP, uint32_t *oldValP)
{
	return false;
}

void machSleep(void)
{
	logi("pretending to sleep\n");
	SysTaskDelay(10000);
	logi("waking up\n");
	dalModifyWakeFlags(DAL_WAKE_FLAG_GENERAL, 0);
	//nothing yet
}

void deviceReset(bool doHardReset)
{
	fatal("reset not implemented\n");
}

Err machinePaceDispatch(EmulStateRef ref, uint16_t call, Err *ret68kP)
{
	return sysErrNotAllowed;
}

