#include <string.h>
#include "bitfield.h"
#include "atomic.h"
#include "heap.h"
#include "irqs.h"


struct Bitfield {
	uint32_t nwords;
	uint32_t store[];
};


struct Bitfield* bitfieldCreate(uint32_t nbits)
{
	uint32_t i, nwords = (nbits + 31) / 32;
	struct Bitfield *ret;
	
	if (!nwords)
		return NULL;
	
	ret = (struct Bitfield*)kheapAlloc(sizeof(struct Bitfield) + nwords * sizeof(uint32_t));
	if (ret) {
		
		ret->nwords = nwords;
		for (i = 0; i < nwords; i++)
			ret->store[i] = 0;
			
		if (nbits & 31)		//last is special - block off those that arent ever allowed
			ret->store[nwords - 1] = 0xffffffff << (nbits & 31);
	}
	
	return ret;
}

void bitfieldDestroy(struct Bitfield* bf)
{
	kheapFree(bf);
}

int32_t bitfieldAtomicFindZeroAndSet(struct Bitfield* bf)
{
	uint32_t i, nwords = bf->nwords;
	
	for (i = 0; i < nwords; i++) {
		
		uint32_t t, pos;
		
		do {
			pos = 32;
			t = bf->store[i];
			if (t == 0xffffffff)
				break;
			pos = 31 - __builtin_clz(t ^ (t + 1));
			
		} while (!atomicTestAndSet32(&bf->store[i], t, t | (1 << pos)));

		if (pos != 32)
			return pos + i * 32;
	}
	
	return -1;
}

void bitfieldAtomicClearBit(struct Bitfield* bf, uint32_t idx)
{
	atomicLogicOp32(&bf->store[idx / 32], ~(1UL << (idx % 32)), 0);
}

