#include <Chars.h>
#include "halPenAndKeys.h"
#include "halDisplay.h"
#include "kernel.h"
#include "printf.h"
#include "input.h"
#include "kal.h"
#include "dal.h"


#define MAX_RELEASE_TIME_FOR_DUBLE_TAP	600
#define MAX_PRESS_TIME_FOR_DOUBLE_TAP	750

#define KEYS_NO_AUTO_REPEAT_MASK		0x00000001		//which keys never send autorepeat events

static uint32_t mKeyMask = 0xFFFFFFFF;
static uint32_t mKeyState = 0;
static uint32_t mKeyDoubleTapLastTime = 0;
static uint16_t mKeyRatesInit = 0;
static uint16_t mKeyRatesPeriod = 0;
static bool mKeyRatesQueueAhead = true;
static bool mDoubleTapHadOne = false;
static uint32_t mAutorepeatTimer;

static void halHardKeysPrvAutorepeatTimerSet(void);

enum CalibStage {
	CalibNone,
	CalibDefaults,	//not usable but will not cause calculation issues
	CalibProper,	//calibrated
};

struct CalibDirection {
	int32_t slope;
	int16_t ofst;
};

struct Calib {
	struct CalibDirection forward, reverse;
};

struct CalibPrefs {
	struct Calib x, y;
};

#define CUR_CALIB_PREF_VER	0

static enum CalibStage mCalibStage = CalibNone;
static struct CalibPrefs mCalib;


static void calibDefault(struct Calib *cx, struct Calib *cy)
{
	#if defined(TOUCH_SLOPE_X) && defined(TOUCH_YINT_X) && defined(TOUCH_SLOPE_Y) && defined(TOUCH_YINT_Y)
	
		static const struct CalibDirection dfltFx = {.slope = 65536 * TOUCH_SLOPE_X, .ofst = TOUCH_YINT_X};
		static const struct CalibDirection dfltFy = {.slope = 65536 * TOUCH_SLOPE_Y, .ofst = TOUCH_YINT_Y};
		static const struct CalibDirection dfltRx = {.slope = 65536 / TOUCH_SLOPE_X, .ofst = -TOUCH_YINT_X / TOUCH_SLOPE_X};
		static const struct CalibDirection dfltRy = {.slope = 65536 / TOUCH_SLOPE_Y, .ofst = -TOUCH_YINT_Y / TOUCH_SLOPE_Y};
	
	#elif defined(TOUCH_SLOPE_X) || defined(TOUCH_YINT_X) || defined(TOUCH_SLOPE_Y) || defined(TOUCH_YINT_Y)
	
		#ifndef TOUCH_SLOPE_X
			#error "did not define TOUCH_SLOPE_X"
		#endif
		
		#ifndef TOUCH_YINT_X
			#error "did not define TOUCH_YINT_X"
		#endif
		
		#ifndef TOUCH_SLOPE_Y
			#error "did not define TOUCH_SLOPE_Y"
		#endif
		
		#ifndef TOUCH_YINT_Y
			#error "did not define TOUCH_YINT_Y"
		#endif
		
	
	#else
			
		static const struct CalibDirection dfltFx = {.slope = 65536, .ofst = 0};
		static const struct CalibDirection dfltFy = {.slope = 65536, .ofst = 0};
		static const struct CalibDirection dfltRx = {.slope = 65536, .ofst = 0};
		static const struct CalibDirection dfltRy = {.slope = 65536, .ofst = 0};
	
	#endif
	
	cx->forward = dfltFx;
	cy->forward = dfltFy;
	cx->reverse = dfltRx;
	cy->reverse = dfltRy;
}

static int16_t calibPerform(const struct CalibDirection* c, int16_t val)
{
	return (c->slope * val) / 65536 + c->ofst;
}

static void calibCheck(void)
{
	if (mCalibStage == CalibNone) {
		
		mCalibStage = CalibDefaults;
		calibDefault(&mCalib.x, &mCalib.y);
	}
	
	if (mCalibStage == CalibDefaults && dalGetInitStage() >= DAL_INIT_STAGE_ALL_UP) {
		
		uint16_t sz = sizeof(struct CalibPrefs);
		struct CalibPrefs got;
		int16_t ver;
		
		if (CUR_CALIB_PREF_VER == PrefGetAppPreferences(CREATE_4CC('p','s','y','s'), 0x9900, &got, &sz, false)) {
			
			mCalib = got;
			mCalibStage = CalibProper;
		}
	}
}

static void calibSave(void)
{
	if (dalGetInitStage() >= DAL_INIT_STAGE_ALL_UP) {
	
		PrefSetAppPreferences(CREATE_4CC('p','s','y','s'), 0x9900, CUR_CALIB_PREF_VER, &mCalib, sizeof(struct CalibPrefs), false);
		mCalibStage = CalibDefaults;
	}
}

static bool calibCalc(struct CalibDirection* dst, int32_t fromA, int32_t fromB, int32_t toA, int32_t toB)
{
	if (fromA == fromB || toA == toB)
		return false;
	
	//system conversions round down, so here we'll round up.
	dst->slope = ((int32_t)(toB - toA) * 65536 + (fromB - fromA) -1) / (fromB - fromA);
	dst->ofst = toA - dst->slope * fromA / 65536;
	
	return true;
}

uint32_t DALEXPORT impl_HALKeySetMask(uint32_t keyMask)	//bits that are set are keys that hsoudl generate events
{
	uint32_t ret;
	
	logt("%s\n", __func__);
	
	ret = mKeyMask;
	mKeyMask = keyMask;
	
	return ret;
}

uint32_t DALEXPORT impl_HALKeyGetState(void)
{
	uint32_t ret = dalGetInitStage() >= DAL_INIT_STAGE_MEM_AND_DM ? mKeyState : drvInputReadKeysEarly();
	
	logt("%s -> 0x%08lx\n", __func__, ret);
	return ret;
}

Err DALEXPORT impl_HALKeyResetDoubleTap(UInt16 *doubleTapDelayP)
{
	logst("%s\n", __func__);
	mKeyDoubleTapLastTime = 0;			//we accept this race as unlikely and the costs of a mutex here as too high
	mDoubleTapHadOne = false;
	return errNone;
}

Err DALEXPORT impl_HALPenResetCalibration(void)
{
	logt("%s\n", __func__);
	
	calibDefault(&mCalib.x, &mCalib.y);
	calibSave();
	
	return errNone;
}

/*
	the pen api kind of suck in terms of what density they use. there is basically no way to understand them without colleciting data
	do we do:
	
	LR:
	
		bottom right raw to screen:
			3120, 2591	=>  127 144
		top left raw to screen
			703, 484	=>	14 14
		calibrate (raw coords translated to screen coords using current calibration -> expected screen coords)
			14 14	=> 14 14
			127 144 => 146 146
		center:
			1942 1513	=>	82 79
	
	
	HR:
		bottom right raw to screen:
			3560, 2419	=> 306 270
		top left raw to screen
			615, 359	=>	33, -8
		calibrate (raw coords translated to screen coords using current calibration -> expected screen coords)
			16, -4		=> 14, 14
			153 135		=> 146, 146
		center:
			2123 1365	=> 164, 157

	so as we see, HALPenRawToScreen & HALPenScreenToRaw always operate at native density
	and HALPenCalibrate operates at low density always. WTF???

*/

Err DALEXPORT impl_HALPenRawToScreen(struct PointType *pt)
{
	logst("%s in (%d, %d)\n", __func__, pt->x, pt->y);
	
	calibCheck();
	
	pt->x = calibPerform(&mCalib.x.forward, pt->x);
	pt->y = calibPerform(&mCalib.y.forward, pt->y);
	
	logst("%s out (%d, %d)\n", __func__, pt->x, pt->y);
	
	
	return errNone;
}

Err DALEXPORT impl_HALPenScreenToRaw(struct PointType *pt)
{
	logst("%s in (%d, %d)\n", __func__, pt->x, pt->y);
	
	calibCheck();
	pt->x = calibPerform(&mCalib.x.reverse, pt->x);
	pt->y = calibPerform(&mCalib.y.reverse, pt->y);
	
	logst("%s out (%d, %d)\n", __func__, pt->x, pt->y);
	
	return errNone;
}

Err DALEXPORT impl_HALPenCalibrate(struct PointType *digTopLeftP, struct PointType *digBotRightP, struct PointType *scrTopLeftP, struct PointType *scrBotRightP)
{
	struct PointType scrTopLeft = *scrTopLeftP, scrBotRight = *scrBotRightP;
	struct PointType digTopLeft = *digTopLeftP, digBotRight = *digBotRightP;
	uint32_t displayDensity = halScreenGetCurBitmap()->density;
	
	logt("%s\n", __func__);
	
	//convert coords to native coordinate system (all incoming coords are screen data not raw data)
	scrTopLeft.x = (displayDensity * scrTopLeft.x + 36) / 72;
	scrTopLeft.y = (displayDensity * scrTopLeft.y + 36) / 72;
	scrBotRight.x = (displayDensity * scrBotRight.x + 36) / 72;
	scrBotRight.y = (displayDensity * scrBotRight.y + 36) / 72;
	digTopLeft.x = (displayDensity * digTopLeft.x + 36) / 72;
	digTopLeft.y = (displayDensity * digTopLeft.y + 36) / 72;
	digBotRight.x = (displayDensity * digBotRight.x + 36) / 72;
	digBotRight.y = (displayDensity * digBotRight.y + 36) / 72;

	//convert sensed digi coords that had been converted to screen coords to raw coords at current calibration settings
	HALPenScreenToRaw(&digTopLeft);
	HALPenScreenToRaw(&digBotRight);

	if (!calibCalc(&mCalib.x.forward, digTopLeft.x, digBotRight.x, scrTopLeft.x, scrBotRight.x))
		return 0xFFFF;

	if (!calibCalc(&mCalib.y.forward, digTopLeft.y, digBotRight.y, scrTopLeft.y, scrBotRight.y))
		return 0xFFFF;

	if (!calibCalc(&mCalib.x.reverse, scrTopLeft.x, scrBotRight.x, digTopLeft.x, digBotRight.x))
		return 0xFFFF;

	if (!calibCalc(&mCalib.y.reverse, scrTopLeft.y, scrBotRight.y, digTopLeft.y, digBotRight.y))
		return 0xFFFF;

	calibSave();

	return errNone;
}

Err DALEXPORT impl_HALKeyGetRates(uint16_t *initDelayP, uint16_t *periodP, Boolean *queueAheadP)
{
	logt("%s\n", __func__);
	
	if (initDelayP)
		*initDelayP = mKeyRatesInit;
	if (periodP)
		*periodP = mKeyRatesPeriod;
	if (queueAheadP)
		*queueAheadP = mKeyRatesQueueAhead;
	
	return errNone;
}

Err DALEXPORT impl_HALKeySetRates(uint16_t initDelay, uint16_t period, Boolean queueAheadP)
{
	int64_t rawRet = 0;
	
	logt("%s\n", __func__);
	
	#ifdef MACH_TYPE_VIRTUAL
		rawRet = hyperCall(HYPER_SEY_KEY_REP_RATES, (void*)((((uint32_t)period) << 16) + initDelay));
	#else
		//TODO: hw
	#endif
	
	mKeyRatesInit = initDelay;
	mKeyRatesPeriod = period;
	mKeyRatesQueueAhead = !!queueAheadP;
	
	return rawRet ? 0xffff : errNone;
}


static void halHardKeyDoEvents(uint32_t oldState, uint32_t newState, uint32_t changedBit)
{
	struct HalEvtKey evtKey = {};
	
	//check for double taps (only if only one key is in play)

	if (!(oldState ^ newState ^ changedBit)) {
	
		uint32_t last = mKeyDoubleTapLastTime, now = HALTimeGetSystemTime(), dur = now - last;
		
		if (newState == changedBit) {						// key pressed
			
			if (!last)										//key pressed and no timer running. maybe first press of a double tap. time it;
				mKeyDoubleTapLastTime = now;
			else if (dur > MAX_RELEASE_TIME_FOR_DUBLE_TAP)	//key pressed not proper time since last time
				mKeyDoubleTapLastTime = 0;			//not happening
			else {											//key pressed and timer is running. maybe second press of a double tap
				mKeyDoubleTapLastTime = now;
				mDoubleTapHadOne = true;
			}
		}
		else {												//key released
			if (dur > MAX_PRESS_TIME_FOR_DOUBLE_TAP) {		//too late
				mKeyDoubleTapLastTime = 0;			//not happening
				mDoubleTapHadOne = false;
			}
			else if (mDoubleTapHadOne) {					//released in time to matter and is second
				
				evtKey.modifiers |= doubleTapKeyMask;
				mDoubleTapHadOne = false;
				mKeyDoubleTapLastTime = 0;
			}
			else {											//released in time to matter and is first
				
				mKeyDoubleTapLastTime = now;				//time space till second
			}
		}
	}
	
	//it could be an auto-repeat
	if ((oldState == newState) && (newState & changedBit)) {
		if (changedBit & KEYS_NO_AUTO_REPEAT_MASK) {
			logst("autorepeat squashed\n");
			return;
		}
		evtKey.modifiers |= autoRepeatKeyMask;
	}
	
	if (newState & changedBit) {	//down or autorepeat are the times to send events
		
		//peculiarity of PalmOs's key code - these vchars need commandKeyMask set
		if (changedBit & (HARD_KEY_POWER | HARD_KEY_APP_1 | HARD_KEY_APP_2 | HARD_KEY_APP_3 | HARD_KEY_APP_3 | HARD_KEY_HOTSYNC))
			evtKey.modifiers |= commandKeyMask;
		
		if (changedBit & HARD_KEY_POWER)
			evtKey.ascii = vchrHardPower;
		else if (changedBit & HARD_KEY_PG_UP)
			evtKey.ascii = vchrPageUp;
		else if (changedBit & HARD_KEY_PG_DN)
			evtKey.ascii = vchrPageDown;
		else if (changedBit & HARD_KEY_APP_1)
			evtKey.ascii = vchrHard1;
		else if (changedBit & HARD_KEY_APP_2)
			evtKey.ascii = vchrHard2;
		else if (changedBit & HARD_KEY_APP_3)
			evtKey.ascii = vchrHard3;
		else if (changedBit & HARD_KEY_APP_4)
			evtKey.ascii = vchrHard4;
		else if (changedBit & HARD_KEY_HOTSYNC)
			evtKey.ascii = vchrHardCradle;
		else if (changedBit & HARD_KEY_JOG_UP)
			evtKey.ascii = vchrThumbWheelUp;
		else if (changedBit & HARD_KEY_JOG_DN)
			evtKey.ascii = vchrThumbWheelDown;
		else if (changedBit & HARD_KEY_JOG_SEL)
			evtKey.ascii = vchrThumbWheelPush;
		else if (changedBit & HARD_KEY_JOG_BACK)
			evtKey.ascii = vchrThumbWheelBack;
		else if (changedBit & HARD_KEY_ROCKER_UP)
			evtKey.ascii = vchrRockerUp;
		else if (changedBit & HARD_KEY_ROCKER_DN)
			evtKey.ascii = vchrRockerDown;
		else if (changedBit & HARD_KEY_ROCKER_LT)
			evtKey.ascii = vchrRockerLeft;
		else if (changedBit & HARD_KEY_ROCKER_RT)
			evtKey.ascii = vchrRockerRight;
		else if (changedBit & HARD_KEY_ROCKER_SEL)
			evtKey.ascii = vchrRockerCenter;
		
		if (evtKey.ascii)
			HALEventPost(HAL_EVENT_TYPE_KEY, &evtKey);
	}
}

static void halHardKeysPrvAutorepeatTimerCbk(void* timerData)
{
	uint32_t mask;
	
	mask = mKeyState & mKeyMask;
	if (!mask)
		return;
	
	while (mask) {
		
		uint32_t newMask = mask & (mask - 1);
		uint32_t bit = mask &~ newMask;
		
		halHardKeyDoEvents(mKeyState, mKeyState, bit);
		mask = newMask;
	}
	
	halHardKeysPrvAutorepeatTimerSet();
}

static void halHardKeysPrvAutorepeatTimerSet(void)
{
	if (mAutorepeatTimer)
		KALTimerDelete(mAutorepeatTimer);
	
	if (errNone == KALTimerCreate(&mAutorepeatTimer, CREATE_4CC('k','R','P','T'), halHardKeysPrvAutorepeatTimerCbk, NULL))
		(void)KALTimerSet(mAutorepeatTimer, 100);
}

//double tap is any press of length < 1 sec coming less than 1 sec after another like it while no other keys are pressed
void halHardKeyChanged(uint32_t keyBit, bool pressed)
{
	uint32_t curState, newState;
	
	curState = newState = mKeyState;
	
	if (!keyBit)
		fatal("WTF: no hard key changed but handling was requested\n");
	
	if (pressed)
		newState |= keyBit;
	else
		newState &=~ keyBit;
	
	logt("HARD KEYS 0x%08lx -> 0x%08lx (change is %c0x%08lx)\n", curState, newState, pressed ? '+' : '-', keyBit);
	
	mKeyState = newState;
	if (keyBit & mKeyMask) {
		
		halHardKeyDoEvents(curState, newState, keyBit /* so we do not have to recalculate it */);
		halHardKeysPrvAutorepeatTimerSet();
	}	
}



