#include "halDisplay.h"
#include <string.h>
#include "printf.h"
#include "timers.h"
#include "entry.h"
#include "boot.h"
#include "dal.h"


const static uint8_t mMonospaceChars[];

#define CHAR_W		6
#define CHAR_H		8

static char mErrorBufferInitial[128];
static char *mErrorBuffer = mErrorBufferInitial;
static bool mErrReportedAlready = false;

static void drawChar(uint8_t *dispFb, uint32_t w, uint32_t h, uint32_t d, bool indexedFmtIsLE, uint32_t fg, uint32_t bg, const uint8_t *img, uint32_t x, uint32_t y, bool doubled)
{
	uint32_t cr, cc, dupv, dupc, rawRowBytes = w * d / 8;
	
	dispFb += y * rawRowBytes;	//go to dst row;
	
	for (cr = 0; cr < CHAR_H; cr++) {
		
		for (dupv = 0; dupv < (doubled ? 2 : 1); dupv++, dispFb += rawRowBytes) {
			
			for (cc = 0; cc < CHAR_W; cc++) {
			
				uint32_t color = cr * CHAR_W + cc;
				
				color = ((img[color / 8] >> (color % 8)) & 1) ? fg : bg;
			
				for (dupc = 0; dupc < (doubled ? 2 : 1); dupc++) {
					
					uint32_t po, pm, px = x + cc * (doubled ? 2 : 1) + dupc;
					uint8_t *pb;
					
					pb = dispFb + (px * d / 8);
					po = px * d % 8;
					pm = (1 << d) - 1;
					
					if (!indexedFmtIsLE)
						po = 8 - d - po;
					
					if (d == 16) {
						
						pb[0] = color;
						pb[1] = color >> 8;
					}
					else if (d == 8) {
						
						pb[0] = color;
					}
					else {
						
						pb[0] = (pb[0] &~ (pm << po)) | ((color & pm) << po);
					}
				}
			}
		}
	}
}

static void drawChars(uint8_t *dispFb, uint32_t w, uint32_t h, uint32_t d, bool indexedFmtIsLE, uint32_t fg, uint32_t bg, const char *str, uint32_t cx, uint32_t cy, bool doubled)
{
	uint32_t cw = w / CHAR_W, ch = h / CHAR_H;
	char c;
	
	while ((c = *str++) != 0) {
		
		if (c == '\r') {
			cx = 0;
			continue;
		}
		
		if (c == '\n')
			cx = cw;
		else {
		
			if (cx >= cw) {
				
				cx = 0;
				cy += doubled ? 2 : 1;
				
				if (cy >= ch)
					return;
			}
			
			if (c < 0x20 || c > 0x7f)
				c = 0x7f;
			
			c -= 0x20;
			
			drawChar(dispFb, w, h, d, indexedFmtIsLE, fg, bg, &mMonospaceChars[6 * c], cx * CHAR_W, cy * CHAR_H, doubled);
		}
		
		cx += doubled ? 2 : 1;
	}
}

char* halErrorGetBuffer(void)
{
	return mErrorBuffer;
}

void halErrorHeapAvail(void)
{
	uint32_t bufferSz = 1024;
	
	//this buffer is used from exception levels of various types, so in ROMRAM, it might not be in the dynamic heap
	#ifdef USE_XRAM_FOR_DYN
		mErrorBuffer = kheapAllocEx(bufferSz, MEM_USABLE_AS_STACK | MEM_FAST | MEM_NO_OS_HEAP);
	#else
		mErrorBuffer = MemChunkNew(0, bufferSz, 0x1200);
	#endif
}

bool DALEXPORT impl_HALErrDisplay(const char *msg, bool forced, uint32_t *usersChoiceP, bool m68kDebugAllowed)
{
	uint32_t w, h, d, bg, fg, cw, ch;
	uint64_t time, nowTime;
	bool indexedFmtIsLE;
	uint8_t *dispFb;
	
	if (!mErrReportedAlready) {	//hard fault causes us to re-fault. do not draw over the original error report
		
		halScreenBreakLocks((void**)&dispFb, &w, &h, &d, &indexedFmtIsLE);
		bg = d > 8 ? 0xffff : 0x00;
		fg = d > 8 ? 0 : ((1 << d) - 1);
	
		memset(dispFb, bg, w * h * d / 8);
		
		//we expect two strings back to back, with a zero in between!
		drawChars(dispFb, w, h, d, indexedFmtIsLE, fg, bg, msg, 0, 0, true);
		drawChars(dispFb, w, h, d, indexedFmtIsLE, fg, bg, msg + strlen(msg) + 1, 0, 2, false);
	
		mErrReportedAlready = true;
	}
	
	halDisplayRefreshManual();
	halDisplayRefreshManual();
	
	machInit(STAGE_CRASH_LOCKS_BREAK, NULL);
	
	//wait a second for platforms with no manual disp update ability
	//due to how timer is implemented, irqs off might cause it to loop in a small vicinity. Watch out for that
	time = timerGetTime();
	do {
		nowTime = timerGetTime();
		if (nowTime <= time) {
			loge("timer appears INOP, skipping the wait\n");
			break;
		}
	} while (nowTime - time < TIMER_TICKS_PER_MSEC * 1000);
	
	return true;
}







#define _	0
#define W	1
	


#define R(a0,a1,a2,a3,a4,a5,b0,b1,b2,b3,b4,b5,c0,c1,c2,c3,c4,c5,d0,d1,d2,d3,d4,d5)				\
																								\
												(a0 << 0) | (a1 << 1) | (a2 << 2) | (a3 << 3) |	\
												(a4 << 4) | (a5 << 5) | (b0 << 6) | (b1 << 7),	\
												(b2 << 0) | (b3 << 1) | (b4 << 2) | (b5 << 3) |	\
												(c0 << 4) | (c1 << 5) | (c2 << 6) | (c3 << 7),	\
												(c4 << 0) | (c5 << 1) | (d0 << 2) | (d1 << 3) |	\
												(d2 << 4) | (d3 << 5) | (d4 << 6) | (d5 << 7)



const static uint8_t mMonospaceChars[]=
{

	//' ',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'!',
	
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,_,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'"',
	
	R(_,W,_,W,_,_,
	  _,W,_,W,_,_,
	  _,W,_,W,_,_,
	  _,_,_,_,_,_),
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'#',
	
	R(_,W,_,W,_,_,
	  _,W,_,W,_,_,
	  W,W,W,W,W,_,
	  _,W,_,W,_,_),
	R(W,W,W,W,W,_,
	  _,W,_,W,_,_,
	  _,W,_,W,_,_,
	  _,_,_,_,_,_),
	
	
	
	//'$',
	
	R(_,_,W,_,_,_,
	  _,W,W,W,W,_,
	  W,_,W,_,_,_,
	  _,W,W,W,_,_),
	R(_,_,W,_,W,_,
	  W,W,W,W,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'%',
	
	R(W,W,_,_,_,_,
	  W,W,_,_,W,_,
	  _,_,_,W,_,_,
	  _,_,W,_,_,_),
	R(_,W,_,_,_,_,
	  W,_,_,W,W,_,
	  _,_,_,W,W,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'&',
	
	R(_,W,_,_,_,_,
	  W,_,W,_,_,_,
	  W,_,W,_,_,_,
	  _,W,_,_,_,_),
	R(W,_,W,_,W,_,
	  W,_,_,W,_,_,
	  _,W,W,_,W,_,
	  _,_,_,_,_,_),
	
	
	
	
	//''',
	
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'(',
	
	R(_,_,W,_,_,_,
	  _,W,_,_,_,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_),
	R(W,_,_,_,_,_,
	  _,W,_,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//')',
	
	R(_,_,W,_,_,_,
	  _,_,_,W,_,_,
	  _,_,_,_,W,_,
	  _,_,_,_,W,_),
	R(_,_,_,_,W,_,
	  _,_,_,W,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'*',
	
	R(_,_,_,_,_,_,
	  _,_,W,_,_,_,
	  W,_,W,_,W,_,
	  _,W,W,W,_,_),
	R(W,_,W,_,W,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'+',
	
	R(_,_,_,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  W,W,W,W,W,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//',',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	R(_,_,_,_,_,_,
	  _,W,W,_,_,_,
	  _,_,W,_,_,_,
	  _,W,_,_,_,_),
	
	  
	  
	//'-',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,W,W,W,W,_),
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'.',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	R(_,_,_,_,_,_,
	  _,W,W,_,_,_,
	  _,W,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'/',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,W,_,
	  _,_,_,W,_,_,
	  _,_,W,_,_,_),
	R(_,W,_,_,_,_,
	  W,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'0',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,W,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'1',
	
	R(_,_,W,_,_,_,
	  _,W,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'2',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  _,_,_,_,W,_,
	  _,_,_,W,_,_),
	R(_,_,W,_,_,_,
	  _,W,_,_,_,_,
	  W,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'3',
	
	R(W,W,W,W,W,_,
	  _,_,_,W,_,_,
	  _,_,W,_,_,_,
	  _,_,_,W,_,_),
	R(_,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	  
	//'4',
	
	R(_,_,_,W,_,_,
	  _,_,W,W,_,_,
	  _,W,_,W,_,_,
	  W,_,_,W,_,_),
	R(W,W,W,W,W,_,
	  _,_,_,W,_,_,
	  _,_,_,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'5',
	
	R(W,W,W,W,W,_,
	  W,_,_,_,_,_,
	  W,W,W,W,_,_,
	  _,_,_,_,W,_),
	R(_,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	  
	//'6',
	
	R(_,_,W,W,_,_,
	  _,W,_,_,_,_,
	  W,_,_,_,_,_,
	  W,W,W,W,_,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'7',
	
	R(W,W,W,W,W,_,
	  _,_,_,_,W,_,
	  _,_,_,W,_,_,
	  _,_,W,_,_,_),
	R(_,W,_,_,_,_,
	  _,W,_,_,_,_,
	  _,W,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'8',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'9',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,W,_),
	R(_,_,_,_,W,_,
	  _,_,_,W,_,_,
	  _,W,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//':',
	
	R(_,_,_,_,_,_,
	  _,W,W,_,_,_,
	  _,W,W,_,_,_,
	  _,_,_,_,_,_),
	R(_,W,W,_,_,_,
	  _,W,W,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//';',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,W,W,_,_,
	  _,_,W,W,_,_),
	R(_,_,_,_,_,_,
	  _,_,W,W,_,_,
	  _,_,_,W,_,_,
	  _,_,W,_,_,_),
	
	
	
	
	//'<',
	
	R(_,_,_,W,_,_,
	  _,_,W,_,_,_,
	  _,W,_,_,_,_,
	  W,_,_,_,_,_),
	R(_,W,_,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'=',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,W,W,W,W,_,
	  _,_,_,_,_,_),
	R(W,W,W,W,W,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'>',
	
	R(_,W,_,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,W,_,_,
	  _,_,_,_,W,_),
	R(_,_,_,W,_,_,
	  _,_,W,_,_,_,
	  _,W,_,_,_,_,
	  _,_,_,_,_,_),
	  
	  
	//'?',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  _,_,_,_,W,_,
	  _,_,_,W,_,_),
	R(_,_,W,_,_,_,
	  _,_,_,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'@',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,W,_,W,_,
	  W,_,W,W,W,_),
	R(W,_,W,_,_,_,
	  W,_,_,_,_,_,
	  _,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'A',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'B',
	
	R(W,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,_,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'C',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_),
	R(W,_,_,_,_,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'D',
	
	R(W,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'E',
	
	R(W,W,W,W,W,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,W,W,W,_,_),
	R(W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'F',
	
	R(W,W,W,W,W,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,W,W,W,_,_),
	R(W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'G',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,_,_,
	  W,_,W,W,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'H',
	
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'I',
	
	R(_,W,W,W,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'J',
	
	R(_,_,W,W,W,_,
	  _,_,_,W,_,_,
	  _,_,_,W,_,_,
	  _,_,_,W,_,_),
	R(_,_,_,W,_,_,
	  W,_,_,W,_,_,
	  _,W,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'K',
	
	R(W,_,_,_,W,_,
	  W,_,_,W,_,_,
	  W,_,W,_,_,_,
	  W,W,_,_,_,_),
	R(W,_,W,_,_,_,
	  W,_,_,W,_,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'L',
	
	R(W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_),
	R(W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'M',
	
	R(W,_,_,_,W,_,
	  W,W,_,W,W,_,
	  W,_,W,_,W,_,
	  W,_,W,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'N',
	
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,_,_,W,_,
	  W,_,W,_,W,_),
	R(W,_,_,W,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'O',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'P',
	
	R(W,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,_,_),
	R(W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	  
	//'Q',
	
	R(_,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,W,_,W,_,
	  W,_,_,W,_,_,
	  _,W,W,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'R',
	
	R(W,W,W,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,_,_),
	R(W,_,W,_,_,_,
	  W,_,_,W,_,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	  
	//'S',
	
	R(_,W,W,W,W,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  _,W,W,W,_,_),
	R(_,_,_,_,W,_,
	  _,_,_,_,W,_,
	  W,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	  
	//'T',
	
	R(W,W,W,W,W,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	  
	//'U',
	
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'V',
	
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  _,W,_,W,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'W',
	
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,W,_,W,_,
	  W,_,W,_,W,_,
	  _,W,_,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'X',
	
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,_,W,_,_,
	  _,_,W,_,_,_),
	R(_,W,_,W,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'Y',
	
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,_,W,_,_,
	  _,_,W,_,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'Z',
	
	R(W,W,W,W,W,_,
	  _,_,_,_,W,_,
	  _,_,_,W,_,_,
	  _,_,W,_,_,_),
	R(_,W,_,_,_,_,
	  W,_,_,_,_,_,
	  W,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'[',
	
	R(_,_,W,W,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'\',
	
	R(_,_,_,_,_,_,
	  W,_,_,_,_,_,
	  _,W,_,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,_,W,_,_,
	  _,_,_,_,W,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//']',
	
	R(_,_,W,W,_,_,
	  _,_,_,W,_,_,
	  _,_,_,W,_,_,
	  _,_,_,W,_,_),
	R(_,_,_,W,_,_,
	  _,_,_,W,_,_,
	  _,_,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'^',
	
	R(_,_,W,_,_,_,
	  _,W,_,W,_,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'_',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'`',
	
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,W,_,_,
	  _,_,_,_,_,_),
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'a',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,W,W,W,_,_,
	  _,_,_,_,W,_),
	R(_,W,W,W,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'b',
	
	R(W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,_,W,W,_,_,
	  W,W,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'c',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,W,W,W,_,_,
	  W,_,_,_,_,_),
	R(W,_,_,_,_,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'d',
	
	R(_,_,_,_,W,_,
	  _,_,_,_,W,_,
	  _,W,W,_,W,_,
	  W,_,_,W,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'e',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,W,W,W,_,_,
	  W,_,_,_,W,_),
	R(W,W,W,W,W,_,
	  W,_,_,_,_,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'f',
	
	R(_,_,W,W,_,_,
	  _,W,_,_,W,_,
	  _,W,_,_,_,_,
	  W,W,W,_,_,_),
	R(_,W,_,_,_,_,
	  _,W,_,_,_,_,
	  _,W,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'g',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,W,W,W,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  _,W,W,W,W,_,
	  _,_,_,_,W,_,
	  _,W,W,W,_,_),
	
	  
	  
	//'h',
	
	R(W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,_,W,W,_,_,
	  W,W,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'i',
	
	R(_,_,W,_,_,_,
	  _,_,_,_,_,_,
	  _,W,W,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'j',
	
	R(_,_,_,W,_,_,
	  _,_,_,_,_,_,
	  _,_,W,W,_,_,
	  _,_,_,W,_,_),
	R(_,_,_,W,_,_,
	  _,_,_,W,_,_,
	  W,_,_,W,_,_,
	  _,W,W,_,_,_),
	
	  
	  
	//'k',
	
	R(_,W,_,_,_,_,
	  _,W,_,_,_,_,
	  _,W,_,_,W,_,
	  _,W,_,W,_,_),
	R(_,W,W,_,_,_,
	  _,W,_,W,_,_,
	  _,W,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'l',
	
	R(_,W,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'m',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,W,_,W,_,_,
	  W,_,W,_,W,_),
	R(W,_,W,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'n',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,_,W,W,_,_,
	  W,W,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'o',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,W,W,W,_,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  _,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'p',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,W,W,W,_,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,W,W,W,_,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_),
	
	  
	  
	//'q',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,W,W,_,W,_,
	  W,_,_,W,W,_),
	R(W,_,_,_,W,_,
	  _,W,W,W,W,_,
	  _,_,_,_,W,_,
	  _,_,_,_,W,_),
	
	  
	  
	//'r',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,_,W,W,_,_,
	  W,W,_,_,W,_),
	R(W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  W,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'s',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,W,W,W,_,_,
	  W,_,_,_,_,_),
	R(_,W,W,W,_,_,
	  _,_,_,_,W,_,
	  W,W,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'t',
	
	R(_,W,_,_,_,_,
	  _,W,_,_,_,_,
	  W,W,W,_,_,_,
	  _,W,_,_,_,_),
	R(_,W,_,_,_,_,
	  _,W,_,_,W,_,
	  _,_,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'u',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,W,W,_,
	  _,W,W,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'v',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  _,W,_,W,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'w',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,W,_,W,_,
	  W,_,W,_,W,_,
	  _,W,_,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'x',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,_,_,_,W,_,
	  _,W,_,W,_,_),
	R(_,_,W,_,_,_,
	  _,W,_,W,_,_,
	  W,_,_,_,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'y',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  _,W,W,W,W,_,
	  _,_,_,_,W,_,
	  _,W,W,W,_,_),
	
	  
	  
	//'z',
	
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  W,W,W,W,W,_,
	  _,_,_,W,_,_),
	R(_,_,W,_,_,_,
	  _,W,_,_,_,_,
	  W,W,W,W,W,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'{',
	
	R(_,_,W,W,_,_,
	  _,W,_,_,_,_,
	  _,W,_,_,_,_,
	  W,_,_,_,_,_),
	R(_,W,_,_,_,_,
	  _,W,_,_,_,_,
	  _,_,W,W,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'|',
	
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	//'}',
	
	R(W,W,_,_,_,_,
	  _,_,W,_,_,_,
	  _,_,W,_,_,_,
	  _,_,_,W,_,_),
	R(_,_,W,_,_,_,
	  _,_,W,_,_,_,
	  W,W,_,_,_,_,
	  _,_,_,_,_,_),
	
	  
	  
	//'~',
	
	R(_,_,_,_,_,_,
	  _,W,_,_,_,_,
	  W,_,W,_,W,_,
	  _,_,_,W,_,_),
	R(_,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_,
	  _,_,_,_,_,_),
	
	
	
	
	// '\0x7F',	//- no char
	
	R(W,W,W,W,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,_,_,_,W,_),
	R(W,_,_,_,W,_,
	  W,_,_,_,W,_,
	  W,W,W,W,W,_,
  _,_,_,_,_,_),


};

