#ifndef _HAL_DRAWING_H_
#define _HAL_DRAWING_H_

#include <Rect.h>
#include <stdbool.h>
#include <stdint.h>

#define PALM_SCREEN_LOCK_METHOD_COPY		(0)
#define PALM_SCREEN_LOCK_METHOD_ERASE		(1)
#define PALM_SCREEN_LOCK_METHOD_DNK			(2)	//do not care


#define PALM_DENSITY_STANDARD				(72)
#define PALM_DENSITY_ONE_AND_A_HALF			(108)
#define PALM_DENSITY_DOUBLE					(144)
#define PALM_DENSITY_TRIPLE					(216)
#define PALM_DENSITY_QUADRUPLE				(288)

#define PALM_BMP_PIXEL_FORMAT_INDEXED		(0)
#define PALM_BMP_PIXEL_FORMAT_RGB565_BE		(1)
#define PALM_BMP_PIXEL_FORMAT_RGB565_LE		(2)
#define PALM_BMP_PIXEL_FORMAT_INDEXED_LE	(3)

#define PALM_BMP_COMPRES_TYPE_SCAN_LINE		(0)
#define PALM_BMP_COMPRES_TYPE_RLE			(1)
#define PALM_BMP_COMPRES_TYPE_PACK_BITS 	(2)
#define PALM_BMP_COMPRES_TYPE_NONE			(0xFF)


struct PalmClutEntry {		//aka rgb color type
	uint8_t idx, r, g, b;
};

struct PalmClut {			//must be 4-byte aligned since it is often also used as an array of 32-bit integers
	uint16_t nEntries;
	uint16_t rfu;
	struct PalmClutEntry entries[];
}  __attribute__((aligned(4)));

struct PalmDrawStateFlags {
	uint16_t	unscaledBitmaps:1;			// if set, do not scale bitmaps
	uint16_t	unscaledText:1;				// if set, do not scale text
	uint16_t	unpaddedText:1;				// if set, do not insert pixels between glyphs
	uint16_t	useFloor:1;					// used internally by OS for underline behaviour
	uint16_t 	reserved:12;
};

enum PalmWinDrawOperation {
	palmWinDrawOpWinPaint,
	palmWinDrawOpWinErase,
	palmWinDrawOpWinMask,
	palmWinDrawOpWinInvert,
	palmWinDrawOpWinOverlay,
	palmWinDrawOpWinPaintInverse,
	palmWinDrawOpWinSwap,
};

enum PalmPatternType {
	palmPatternBlackPattern,
	palmPatternWhitePattern,
	palmPatternGrayPattern,
	palmPatternCustomPattern,
	palmPatternLightGrayPattern,
	palmPatternDarkGrayPattern,
	
	palmPatternNum,
};

enum PalmUnderlineMode {
	palmUnderlineModeNoUnderline,
	palmUnderlineModeGrayUnderline,
	palmUnderlineModeSolidUnderline,
	palmUnderlineModeColorUnderline,
};

union PalmCustomPattType {
	uint8_t data[8];	//8x8 1 bpp 
	uint32_t data32[2];
};

struct PalmBitmap {							//generic structure. all others are castable
	uint16_t	width;
	uint16_t	height;
	uint16_t	stride;						//in bytes
	uint16_t 	compressed			:1; 
	uint16_t 	hasColorTable		:1;		// if true, color table exists before data
	uint16_t 	hasTransparency		:1;	
	uint16_t 	indirect			:1;		// true if bits do not follow CLUT, but instead a pointer to them is
	uint16_t 	forScreen			:1;	
	uint16_t	directColor			:1;		// direct color bitmap
	uint16_t	indirectColorTable	:1;		// if true, CLUT does not follow header but instead a pointer to it does
	uint16_t	noDither			:1;
	uint16_t 	reserved			:8;
	uint8_t		pixelSz;					//bpp
	uint8_t		version;					//top bit set for LE images (the only ones we support)
};

struct PalmBitmapV0 {
	uint16_t	width;
	uint16_t	height;
	uint16_t	stride;						//in bytes
	uint16_t 	compressed			:1;
	uint16_t 	reserved			:15;
	uint16_t	reserved2[4];
};

struct PalmBitmapV1 {
	uint16_t	width;
	uint16_t	height;
	uint16_t	stride;						//in bytes
	uint16_t 	compressed			:1; 
	uint16_t 	hasColorTable		:1;		// if true, color table exists before data
	uint16_t 	reserved			:14;	
	uint8_t		pixelSz;					//bpp
	uint8_t		version;					//top bit set for LE images (the only ones we support)
	
	uint16_t	nextDepthOffset;			// offset in units of sizeof(uint32_t)
	uint16_t	reserved2[2];
};

struct PalmBitmapV2 {
	uint16_t	width;
	uint16_t	height;
	uint16_t	stride;						//in bytes
	uint16_t 	compressed			:1; 
	uint16_t 	hasColorTable		:1;		// if true, color table exists before data
	uint16_t 	hasTransparency		:1;	
	uint16_t 	indirect			:1;		// true if bits do not follow CLUT, but instead a pointer to them is
	uint16_t 	forScreen			:1;	
	uint16_t	directColor			:1;		// direct color bitmap
	uint16_t 	reserved			:10;
	uint8_t		pixelSz;					//bpp
	uint8_t		version;					//top bit set for LE images (the only ones we support)
	
	uint16_t	nextDepthOffset;			// offset in units of sizeof(uint32_t)
	uint8_t		transparentIndex;	
	uint8_t		compressionType;
	uint16_t	reserved2;
	
	//data is preceded by a uint16 "compressed data size in bytes" if image is compressed
};

struct PalmBitmapV3 {
	uint16_t	width;
	uint16_t	height;
	uint16_t	stride;						//in bytes
	uint16_t 	compressed			:1; 
	uint16_t 	hasColorTable		:1;		// if true, color table exists before data
	uint16_t 	hasTransparency		:1;	
	uint16_t 	indirect			:1;		// true if bits do not follow CLUT, but instead a pointer to them is
	uint16_t 	forScreen			:1;	
	uint16_t	directColor			:1;		// direct color bitmap
	uint16_t	indirectColorTable	:1;		// if true, CLUT does not follow header but instead a pointer to it does
	uint16_t	noDither			:1;
	uint16_t 	reserved			:8;	
	uint8_t		pixelSz;					//bpp
	uint8_t		version;					//top bit set for LE images (the only ones we support)
	
	uint8_t		structSz;					// size of this structure in bytes (0x18)
	uint8_t		pixelFormat;				// format of the pixel data, see pixelFormatType
	uint8_t		unused;						// officially unused, may be same as "neverDensityScale" infat v3 bmp
	uint8_t		compressionType;			// see BitmapCompressionType
	uint16_t	density;					// used by the blitter to scale bitmaps
	uint32_t	transparentValue;			// the index or RGB value of the transparent color (rgb value is in 16LE format)
	uint32_t	nextBitmapOffset;			// byte offset to next bitmap in bitmap family
	
	//data is preceded by a uint32 "compressed data size in bytes" if image is compressed
};

struct PalmBitmapV3fat {
	uint16_t			width;
	uint16_t			height;
	uint16_t			stride;						//in bytes
	uint16_t 			compressed			:1; 
	uint16_t 			hasColorTable		:1;		// if true, color table exists before data
	uint16_t 			hasTransparency		:1;	
	uint16_t 			indirect			:1;		// true if bits do not follow CLUT, but instead a pointer to them is
	uint16_t 			forScreen			:1;	
	uint16_t			directColor			:1;		// direct color bitmap
	uint16_t			indirectColorTable	:1;		// if true, CLUT does not follow header but instead a pointer to it does
	uint16_t			noDither			:1;
	uint16_t 			reserved			:8;	
	uint8_t				pixelSz;					//bpp
	uint8_t				version;					//top bit set for LE images (the only ones we support)
	
	uint8_t				structSz;					// size of this structure in bytes (0x24 or 0x22)
	uint8_t				pixelFormat;				// format of the pixel data, see pixelFormatType
	uint8_t				neverDensityScale	:1;		// officially undocumented. not sure what bitmap version this arrived in
	uint8_t     		rfu					:7;
	uint8_t				compressionType;			// see BitmapCompressionType
	uint16_t			density;					// used by the blitter to scale bitmaps
	uint32_t			transparentValue;			// the index or RGB value of the transparent color
	uint32_t			nextBitmapOffset;			// byte offset to next bitmap in bitmap family
	
	struct PalmClut*	clut;
	void*				data;
	uint32_t			compressedDataSz;			// sometimes this is a uint16_t if src bitmap is a v2
};

#define FONT_TYPE_MASK_V2			0x0200
#define FONT_MAP_MASK				0xc000			//both bits set means font map
#define FONT_MAP_APP_FONT_MAP		0xc800			//all 3 bits need to be set

struct PalmFontMetrics {
	uint16_t fontType;	// 0x0200 set if this is a new format font (v2 - has densities), if not set, it is v1
	uint16_t firstChar;
	uint16_t lastChar;
	int16_t maxWidth;
	int16_t kernMax;
	int16_t nDescent;
	int16_t fRectWidth;
	int16_t fRectHeight;
	int16_t owTLoc;
	int16_t ascent;
	int16_t descent;
	int16_t leading;
	uint16_t rowWords;	//in units of uint16_t
};

struct PalmFontV1 {			//V1 - old format - low density only
	struct PalmFontMetrics metrics;
	//image data here
	//char location data here
};

struct PalmFontDensityType {
	uint16_t density;
	uint16_t padding;
	uint32_t glyphBitsOffset;
};

struct PalmFont {			//V2 - multiple densities supported - from docs - reality may differ!
	struct PalmFontMetrics metrics;
	int16_t version;		//should be 1 for a v2 font as this is

	uint16_t densityCount;
	uint16_t padding;
	struct PalmFontDensityType densities[];

	//char location data here
	//image data here
};

struct PalmFontMapEntry {			// 0x1A
	uint8_t unk_0x00;
	uint8_t unk_0x01;
	uint16_t fontIdx;
};

struct PalmFontMap {
	struct PalmFontMetrics metrics;		//metrics to use for general line info (greatest common denominator dimensions of al the fonts in this map?)
	struct PalmFontMapEntry entries[];	// 0x1A
};

struct PalmDrawState {
	enum PalmWinDrawOperation	drawMode;
	enum PalmPatternType		pattern;
	enum PalmUnderlineMode		underlineMode;
	uint8_t						fontId;
	struct PalmFont*			fontPtr;
	union PalmCustomPattType	patternData;
	
	// These are only valid for indexed color bitmaps
	uint8_t						foreColorIdx;
	uint8_t						backColorIdx;
	uint8_t						textColorIdx;
	uint8_t						rfu;	

	// These are only valid for direct color bitmaps
	struct PalmClutEntry		foreColorRGB;
	struct PalmClutEntry		backColorRGB;
	struct PalmClutEntry		textColorRGB;
	
	uint16_t					coordinateSystem;		// active coordinate system
	struct PalmDrawStateFlags 	flags;
	union {						//keep in mind that this union has didefent sized members, on pre-garnet, PalmDrawState IS smaller
		struct {
			uint32_t		activeToNative;
			uint32_t		nativeToActive;
			uint32_t		standardToActive;
			uint32_t		activeToStandard;
		} garnet;		//16.16 format fixed point
		struct {
			uint16_t		activeToNative;
			uint16_t		nativeToActive;
			uint16_t		standardToActive;
			uint16_t		activeToStandard;
		} preGarnet;	//6.10 format
	} scaling;
};

struct PalmCanvas {
	struct RectangleType	clippingRect;
	struct PalmDrawState	*drawState;
	struct PalmBitmapV3fat	*bmp;
};

typedef bool (*PalmDrawCharsVerifyFunc)(uint16_t val);


Err HALDraw_FindIndexes(uint32_t numEntries, struct PalmClutEntry *colorsToMatch, const struct PalmClut* referenceClut);
void HALDraw_Rectangle(struct PalmCanvas* canvas, const struct RectangleType *rect, int16_t radius, int16_t penWidth);
void HALDraw_Line(struct PalmCanvas* canvas, int16_t x1, int16_t y1, int16_t x2, int16_t y2, int16_t penWidth);
Err HALDraw_Bitmap(struct PalmCanvas* canvas, const struct PalmBitmapV3fat *srcBitmapP, const struct RectangleType *dstClippedP, int16_t offsetX, int16_t offsetY);
void HALDraw_Chars(struct PalmCanvas* canvas, int16_t x, int16_t y, const char *str, int16_t len, const struct PalmFont *font, const struct PalmFontMap *fontMap, PalmDrawCharsVerifyFunc verifyCbk);
Err HALDraw_GetSupportedDensity(uint16_t *densityP);		//enumerates all supportd densities (call with 0 to start, call repeatedly tyo get next)
uint32_t HALDraw_GetPixel(const struct PalmBitmapV3fat *bitmapP, int16_t x, int16_t y, bool asIndex);
void HALDraw_Pixels(struct PalmCanvas* canvas, uint32_t nPoints, const struct PointType *pts, int16_t penWidth);

Err HALRedrawInputArea(const struct RectangleType *rectP, bool selected);
Err HALDisplayDrawBootScreen(int16_t x, int16_t y, const struct PalmBitmap *bmp);


void halDrawingSetWeirdFlag(uint8_t val);	//XXX: TODO: this should go away, sets [globals + 0x453] aka someFlagToIndicateIfCurrentClutIsSameAsDefault_not_sure_yet

void halDrawingSetClutsDirtyFlag(void);		//called when higher layers change clut, cleared by us via PrvCompareColorTableToDefault()


//internal use only


const struct PalmClut* halDrawingGetDefaultColorTableForDepth(uint32_t depth);



#endif

