#include <MemoryMgr.h>
#include <string.h>
#include <Window.h>
#include <string.h>
#include <alloca.h>
#include "zodiac/zodiac.h"
#include "halDisplay.h"
#include "halDrawing.h"
#include "kernel.h"
#include "printf.h"
#include "heap.h"
#include "disp.h"
#include "kal.h"
#include "dal.h"



#define HARD_DEPTH_LIMIT		16			//for testing, must be a sane value (a supported bpp value <= 16)

struct ScreenBitmap {
	struct PalmBitmapV3 bmp;
	struct PalmClut *clut;
	void* data;
};

static uint32_t mScreenLockCtMutex;			//protects the following fields
static struct ScreenBitmap mScreenBitmap;
static uint8_t mLockCount, mDispDepth = 0;
static uint16_t *mFbLocked;
static void *mSilkAreaSelectedBmp = NULL, *mSilkAreaUnselectedBmp = NULL;
static bool mLocksBroken = false, mIndexedFormatIsLe = false;


static uint16_t mDispW, mDispH, mSupportedDepths, mDispDensity, mDispRealDpi;
static uint8_t mDispBri, mDispContrast;
static uint16_t *mFb;


static struct PalmClut XRAM1 mClutCurrent = {.entries[255] = {0,}};	//current pallette

static uint8_t XRAM1 mColorXlateTab1[2];
static uint8_t XRAM1 mColorXlateTab2[4];
static uint8_t XRAM1 mColorXlateTab4[16];
static uint8_t XRAM1 mColorXlateTab8[256];



//used only early in boot for boot screen and by the refresh thread
void halDisplayRefreshManual(void)
{
	Err e;
	
	if (!mLocksBroken) {
		
		e = KALMutexReserve(mScreenLockCtMutex, -1);
		if (e)
			fatal("Cannot create reserve refresh mutex\n");
	}
	
	dispManualUpdate();
	
	if (!mLocksBroken) {
		
		e = KALMutexRelease(mScreenLockCtMutex);
		if (e)
			fatal("Cannot create release refresh mutex\n");
	}
}

static uint32_t halDisplayGetBootDepth(void)	//highest not limited depth
{
	uint32_t avail = mSupportedDepths & ((1 << HARD_DEPTH_LIMIT) - 1);
	
	return 32 - __builtin_clz(avail);
}

void halDisplayInit(void)
{
	const struct PalmClut *dfltClut;
	uint32_t depth;
	Err e;
	
	e = KALMutexCreate(&mScreenLockCtMutex, CREATE_4CC('_','s','r','f'));
	if (e)
		fatal("Cannot create screen refresh mutex\n");

	if (!dispDrvInit(&mDispW, &mDispH, &mDispDensity, &mDispRealDpi, &mSupportedDepths, (void**)&mFb, &mIndexedFormatIsLe))
		fatal("Display HW init failed\n");
	dispSetBri(mDispBri = 255);
	dispSetContrast(mDispContrast = 128);

	mDispDepth = depth = halDisplayGetBootDepth();
	dispSetDepth(depth);		//set depth in hw
	memset(mFb, depth == 16 ? 0xFF : 0x00, (uint32_t)mDispW * (uint32_t)mDispH * depth / 8);
	
	//set a default clut so that indexed-color boot images can be drawn
	dfltClut = halDrawingGetDefaultColorTableForDepth(depth > 8 ? 8 : depth);
	dispSetClut(0, dfltClut->nEntries, dfltClut->entries);
	
	halDisplayRefreshManual();
}

Err DALEXPORT impl_HALDisplaySetAttributes(uint16_t attr, uint32_t val)
{
	logt("HALDisplaySetAttributes(0x%04x, 0x%08x) from 0x%08lx\n", attr, val, __builtin_return_address(0));
	struct PalmClut *s;
	uint32_t i;
	
	if (attr <= hwrDispLastCommonAttrWithGarnet || attr >= hwrDispFistCommonAttrWithGarnetHiRange) switch (attr) {
		case hwrDispDepth:
			if (val != 1 && val != 2 && val != 4 && val != 8 && val != 16)
				return 0xffff;
			mDispDepth = val;
			dispSetDepth(val);
			return errNone;
		case hwrDispScreenBacklight:
			if (dispSetBacklight(!!val))
				return errNone;
			break;
		case hwrDispBrightness:
			logt("set bri %u\n", val);
			if (val > 0xff)
				val = 0xff;
			mDispBri = val;
			dispSetBri(val);
			return errNone;
		case hwrDispContrast:
			logt("set bri %u\n", val);
			if (val > 0xff)
				val = 0xff;
			mDispContrast = val;
			dispSetContrast(val);
			return errNone;
		case hwrDispOrient:
			logt("set orientation %u\n", val);
			return errNone;
	}
	else if (isGarnet()) switch (attr) {
		case hwrDispInputAreaRect:
			fatal("input area rect NOT settable\n");
			return 0x2201;
		case hwrDispInputAreaBitmapUnselected:
			logt("setting silk bitmap unsel to 0x%08x\n", val);
			mSilkAreaUnselectedBmp = (void*)val;
			return errNone;
		case hwrDispInputAreaBitmapSelected:
			logt("setting silk bitmap sel to 0x%08x\n", val);
			mSilkAreaSelectedBmp = (void*)val;
			return errNone;
	}
	else switch (attr) {

		case hwrDispUnknown_1a:
			logt("unknown attr 0x1a set to 0x%08x\n", val);
			return errNone;
	}
	
	if (isSony_safe(false)) switch(attr) {
		case hwrDispSonyUserAreaSz:
			if (val < 3) {
				static const uint8_t vals[] = {32, 45, 48};
				logt("SONY: new screen user area sz: %u0px\n", vals[val]);
				return errNone;
			}
			fatal("Not understodo value given to hwrDispSonyUserAreaSz\n");
			return 0xffff;
		case hwrDispSony8003:
		case hwrDispSony8004:
			return 0xffff;
	}

	fatal("%s: Unknown display attr being set: 0x%02x <- 0x%08lx from 0x%08lx\n", __func__, attr, val, __builtin_return_address(0));
	return 0x2201;					//this is the proper error
}

Err DALEXPORT impl_HALDisplayGetAttributes(uint16_t attr, void *valueP)
{
	logst("HALDisplayGetAttributes(0x%04x, ...) from 0x%08lx\n", attr, __builtin_return_address(0));
	
	if (attr <= hwrDispLastCommonAttrWithGarnet || attr >= hwrDispFistCommonAttrWithGarnetHiRange) switch (attr) {
		case hwrDispDrvName:
			*(const char**)valueP = "rePalm vDisplay";
			return errNone;
		case hwrDispHorizontal:
			*(uint32_t*)valueP = mDispW;
			return errNone;
		case hwrDispVertical:
			*(uint32_t*)valueP = mDispH;
			return errNone;
		case hwrDispStride:
			*(uint32_t*)valueP = mDispW * mDispDepth / 8;
			return errNone;
		case hwrDispScreenPpiX:
		case hwrDispScreenPpiY:
			*(uint32_t*)valueP = mDispDensity;
			return errNone;
		case hwrDispBootDepth:
			*(uint32_t*)valueP = halDisplayGetBootDepth();
			return errNone;
		case hwrDispDepth:
			*(uint32_t*)valueP = mDispDepth;
			return errNone;
		case hwrDispMaxDepth:
			*(uint32_t*)valueP = HARD_DEPTH_LIMIT;
			return errNone;
		case hwrDispAllDepths:
			*(uint32_t*)valueP = mSupportedDepths & ((1 << HARD_DEPTH_LIMIT) - 1);		//remove all above our hard limit
			return errNone;
		case hwrDispColorSupported:
			*(uint32_t*)valueP = (mSupportedDepths >> 4) ? 1 : 0;						//8bpp or more is considered color
			return errNone;
		case hwrDispScreenBacklight:
			*(uint32_t*)valueP = dispGetBacklight();
			return errNone;
		case hwrDispBrightness:
			*(uint32_t*)valueP = mDispBri;
			return errNone;
		case hwrDispContrast:
			*(uint32_t*)valueP = mDispContrast;
			return errNone;
		case hwrDispT3SliderState:
			*(uint32_t*)valueP = 0;				//sure, it's open
			return errNone;
		case hwrDispRealDpi:
			*(uint32_t*)valueP = mDispRealDpi;
			return errNone;
	}
	else if (isGarnet()) switch (attr) {
		case hwrDispDensity54:
			*(uint32_t*)valueP = mDispDensity;
			return errNone;
		case hwrDispDispScale:
			*(uint32_t*)valueP = 0x10000;
			return errNone;
		case hwrDispDgtScale:
			*(uint32_t*)valueP = 0x10000 * 72 / mDispDensity;
			return errNone;
		case hwrDispInputAreaRect:
			((struct RectangleType*)valueP)->topLeft.x = 0 * mDispDensity / 144;
			((struct RectangleType*)valueP)->topLeft.y = 320 * mDispDensity / 144;
			((struct RectangleType*)valueP)->extent.x = 320 * mDispDensity / 144;
			((struct RectangleType*)valueP)->extent.y = 160 * mDispDensity / 144;
			return errNone;
		case hwrDispInputAreaBitmapUnselected:
			logt("getting silk bitmap unsel\n");
			*(void**)valueP = mSilkAreaUnselectedBmp;
			return errNone;
		case hwrDispInputAreaBitmapSelected:
			logt("getting silk bitmap sel\n");
			*(void**)valueP = mSilkAreaSelectedBmp;
			return errNone;
		case hwrDispVirtSlkLoc_v2:
			return 0x2201;	//error - do not draw silk over boot screen
	}
	else switch (attr) {
	
		case hwrDispDensity:
			*(uint16_t*)valueP = mDispDensity;
			return errNone;
		case hwrDispVirtSlkLoc:
			return 0x2201;							//disabled for now
		case hwrDispDigitizerScaleToScreenScale:	//in fractions of 64. this means each digitizer "pixel" resolves to one screen pixel
			*(uint16_t*)valueP = 64;
			return errNone;
		case hwrDispDigitizerScaleToStandardScale:	//in fractions of 64.  this means each digitizer "pixel" resolves to one standard rez pixel if screeen is standard rez and 1/2 such pixel if screen is double density
			*(uint16_t*)valueP = 64 * 72 / mDispDensity;
			return errNone;
	}
	
	if (isSony_safe(false)) switch(attr) {
		case hwrDispSony8003:
		case hwrDispSony8004:
			return 0xffff;
	}
	
	fatal("%s: Unknown display attr being gotten: 0x%02x from 0x%08lx\n", __func__, attr, __builtin_return_address(0));
	return 0x2201;					//this is the proper error
}


//set hardware palette for screen (shouldnt need to touch any globals - only set hw pal)
Err DALEXPORT impl_HALDisplaySetPalette(int32_t startIdx, uint32_t numEntries, const struct PalmClutEntry *entries)	//assumes entries are in order (they better be)
{
	logt("%s\n", __func__);
	
	dispSetClut(startIdx, numEntries, entries);
	
	return errNone;
}

uint8_t* halScreenGetStandardPaletteXlationTableForDepth(uint32_t depth)
{
	switch (depth) {
		case 1:
			return mColorXlateTab1;
		case 2:
			return mColorXlateTab2;
		case 4:
			return mColorXlateTab4;
		case 8:
			return mColorXlateTab8;
		default:
			return 0;
	}
}

//set screen hardware palette to palettes[4], then lookup each of 0..3 in it and set them as logical palettes for each depth
Err DALEXPORT impl_HALScreenPalette(int32_t startIndex, uint32_t numEntries, uint32_t depth, struct PalmClut *tableP, const struct PalmClut **palettes)
{
	struct PalmClutEntry *tmpColors;
	uint32_t i, j;
	
	logt("%s(%u,%u, %u, 0x%08x(%d, ...), 0x%08x(0x%08x 0x%08x 0x%08x 0x%08x 0x%08x)\n",
		__func__, startIndex, numEntries, depth, tableP, tableP ? tableP->nEntries : -1,
		palettes, palettes ? palettes[0] : (void*)-1, palettes ? palettes[1] : (void*)-1, palettes ? palettes[2] : (void*)-1, palettes ? palettes[3] : (void*)-1, palettes ? palettes[4] : (void*)-1);
	
	halDrawingSetClutsDirtyFlag();
	
	(void)HALDisplaySetPalette(startIndex, numEntries, palettes[4]->entries);
	
	tmpColors = kheapAlloc(sizeof(*tmpColors) * 256);
	if (!tmpColors)
		fatal("Cannot allocate space for colorspace lookup precalculation\n");
	
	for (i = 0; i < 4; i++) {
		
		uint8_t *dst = halScreenGetStandardPaletteXlationTableForDepth(1 << i);
		const struct PalmClutEntry *src = palettes[i]->entries;
		uint32_t num = palettes[i]->nEntries;
		
		if (depth > 8) {	//direct map to colors
			
			memset(dst, 0x00, num);	//unused
		}
		else {				//look them up
			
			memcpy(tmpColors, src, num * sizeof(*tmpColors));
			(void)HALDraw_FindIndexes(num, tmpColors, tableP);
			
			for (j = 0; j < num; j++)
				*dst++ = tmpColors[j].idx;
		}
	}
	
	return errNone;
}

Err DALEXPORT impl_HALScreenUpdateBitmap(UInt8 depth)
{
	logt("%s(%d)\n", __func__, depth);
	uint8_t filler;
	void* bits;
	
	if (depth == mScreenBitmap.bmp.pixelSz)	//only do this if there was a change
		return errNone;
	
	bits = mScreenBitmap.data;
	
	mScreenBitmap.bmp.hasColorTable = 1;
	mScreenBitmap.bmp.indirectColorTable = 1;
	
	if (depth == 16) {
		mScreenBitmap.bmp.pixelFormat = PALM_BMP_PIXEL_FORMAT_RGB565_LE;
		mScreenBitmap.bmp.directColor = 1;
		filler = 0xFF;
		mClutCurrent.nEntries = 256;
	}
	else {
		
		mScreenBitmap.bmp.pixelFormat = (mIndexedFormatIsLe && depth < 8) ? PALM_BMP_PIXEL_FORMAT_INDEXED_LE : PALM_BMP_PIXEL_FORMAT_INDEXED;
		mScreenBitmap.bmp.directColor = 0;
		filler = 0x00;
		mClutCurrent.nEntries = 1 << depth;
	}
	
	mScreenBitmap.bmp.pixelSz = depth;
	mScreenBitmap.data = bits;
	mScreenBitmap.bmp.stride = mDispW * mScreenBitmap.bmp.pixelSz / 8;
	memset(mFb, filler, mDispW * mDispH * depth / 8);
	
	return errNone;
}

Err DALEXPORT impl_HALScreenInit(struct PalmBitmap **screenBmpP, const struct PalmClut *palletteP)
{
	Err e;
	
	logt("%s\n", __func__);
	
	mScreenBitmap.bmp.width = mDispW;
	mScreenBitmap.bmp.height = mDispH;
	mScreenBitmap.bmp.indirect = 1;
	mScreenBitmap.bmp.forScreen = 1;
	mScreenBitmap.bmp.pixelSz = 0;		//so HALScreenUpdateBitmap for sure does what i want
	mScreenBitmap.bmp.version = 0x83;
	mScreenBitmap.bmp.structSz = sizeof(struct PalmBitmapV3);
	mScreenBitmap.bmp.density = mDispDensity;
	mScreenBitmap.clut = &mClutCurrent;
	
	*screenBmpP = (struct PalmBitmap*)&mScreenBitmap;
	mScreenBitmap.data = mFb;
	
	dispSetDepth(mDispDepth);
	
	if (palletteP) {
		e = HALDisplaySetPalette(0, palletteP->nEntries, palletteP->entries);
		if (e)
			return e;
	}
	memcpy(&mClutCurrent, palletteP, sizeof(*palletteP) + sizeof(*palletteP->entries) * palletteP->nEntries);
	
	return impl_HALScreenUpdateBitmap(mDispDepth);		//call direct to no patches get in the way, will set stride
}

void halDisplayEarlyBootGetMetrics(void** fbP, uint32_t *widthP, uint32_t *heightP, uint32_t *strideP, uint32_t *densityP, uint32_t *depthP, bool *indexedFormatIsLeP)
{
	if (fbP)
		*fbP = mFb;
	if (widthP)
		*widthP = mDispW;
	if (heightP)
		*heightP = mDispH;
	if (strideP)
		*strideP = mDispW * mDispDepth / 8;
	if (densityP)
		*densityP = mDispDensity;
	if (depthP)
		*depthP = mDispDepth;
	if (indexedFormatIsLeP)
		*indexedFormatIsLeP = mIndexedFormatIsLe;
}

const struct PalmClut* DALEXPORT impl_HALScreenGetColortable(void)
{
	logst("%s\n", __func__);
	
	return &mClutCurrent;
}

void* DALEXPORT impl_HALScreenLock(uint32_t initMode)
{
	void *ret = NULL;
	Err e;
	
	logt("%s\n", __func__);
	
	e = KALMutexReserve(mScreenLockCtMutex, -1);
	if (e)
		fatal("Cannot create reserve refresh mutex\n");
	
	logt("%s: cur lock ct %u\n", __func__, mLockCount);
	
	if (mLockCount == 255)
		fatal("HALScreenLock count overflow\n");
	
	if (!mLockCount) {		//first time
		
		uint32_t screenBytes = mDispW * mDispH * mDispDepth / 8;
		
		mFbLocked = (uint16_t*)kheapAlloc(screenBytes);
		if (mFbLocked) {
			
			switch (initMode) {
				case winLockCopy:
					memcpy(mFbLocked, mFb, screenBytes);
					break;
				case winLockErase:
					memset(mFbLocked, mScreenBitmap.bmp.pixelSz == 16 ? 0xFF : 0x00, screenBytes);
					break;
				case winLockDontCare:
					break;
				default:
					fatal("Unknown screen lock method: %u\n", initMode);
					break;
			}
			mScreenBitmap.data = mFbLocked;
			mLockCount = 1;
			zodScreenLockNotif(true);
		}
	}
	else
		mLockCount++;
	
	ret = mFbLocked;
	
	e = KALMutexRelease(mScreenLockCtMutex);
	if (e)
		fatal("Cannot create release refresh mutex\n");

	return ret;
}

Err DALEXPORT impl_HALScreenUnlock(void)
{
	Err e;
	
	logt("%s\n", __func__);
	
	e = KALMutexReserve(mScreenLockCtMutex, -1);
	if (e)
		fatal("Cannot create reserve refresh mutex\n");
	
	if (!mLockCount)
		fatal("HALScreenLock count underflow\n");
	
	mLockCount--;
	logt("%s: new lock ct %u\n", __func__, mLockCount);
	
	if (!mLockCount) {
		
		//copy drawn data back
		memcpy(mFb, mFbLocked, mDispW * mDispH * mDispDepth / 8);
		mScreenBitmap.data = mFb;
		kheapFree(mFbLocked);
		mFbLocked = NULL;
		zodScreenLockNotif(false);
	}
	
	e = KALMutexRelease(mScreenLockCtMutex);
	if (e)
		fatal("Cannot create release refresh mutex\n");
	
	return errNone;
}

void halScreenBreakLocks(void **screnBuf, uint32_t *wP, uint32_t *hP, uint32_t *bppP, bool *indexedFormatIsLeP)
{
	if (screnBuf)
		*screnBuf = mFb;
	if (wP)
		*wP = mDispW;
	if (hP)
		*hP = mDispW;
	if (bppP)
		*bppP = mDispDepth;
	if (indexedFormatIsLeP)
		*indexedFormatIsLeP = mIndexedFormatIsLe;
	
	mLocksBroken = true;
}

bool halScreenIsLive(void)
{
	return !mLockCount;	//no point taking locks, result is stale by the time it is returned anyways. the only hope is that only one thread does ui
}

static void halDisplayRequestRefresh(void)
{
	dispRequestUpdate();
}

static void halDisplayUpdatePartial(uint32_t x, uint32_t y, uint32_t width, uint32_t height)
{
	logst("halDisplayUpdatePartial(%u,%u,%u,%u)\n", x, y, width, height);
	halDisplayRequestRefresh();
}

static void halDisplayUpdateEntire(void)
{
	logst("halDisplayUpdateEntire()\n");
	halDisplayRequestRefresh();
}

void DALEXPORT impl_HALScreenDrawNotify(int16_t updLeft, int16_t updTop, int16_t updWidth, int16_t updHeight)
{
	logst("%s(%d,%d,%d,%d)\n", __func__, updLeft, updTop, updWidth, updHeight);
	
	if (updLeft < 0)
		updLeft = 0;
	if (updTop < 0)
		updTop = 0;
	if (updLeft + updWidth > mDispW)
		updWidth = mDispW - updLeft;
	if (updTop + updHeight > mDispW)
		updHeight = mDispH - updTop;
	
	halDisplayUpdatePartial(updLeft, updTop, updWidth, updHeight);
}

void DALEXPORT impl_HALScreenSendUpdateArea(bool force)
{
	logst("%s\n", __func__);
	
	halDisplayUpdateEntire();
}

struct PalmBitmapV3* halScreenGetCurBitmap(void)
{
	return &mScreenBitmap.bmp;
}

Err DALEXPORT impl_HALDisplayDoze(void)
{
	return errNone;	//nothing to do to low-power-but-keep-on display
}

Err DALEXPORT impl_HALDisplaySleep(bool untilReset, bool emergency)
{
	dispSleep();
	
	dalModifyWakeFlags(DAL_WAKE_FLAG_LCD_ASLEEP, 0);

	return errNone;
}

Err DALEXPORT impl_HALDisplayWake(void)
{
	dispWake();
	
	dalModifyWakeFlags(0, DAL_WAKE_FLAG_LCD_ASLEEP);

	return errNone;
}
