#include <stdbool.h>
#include <string.h>
#include <stdint.h>
#include "../../kernel/hw/RP2040/palmcardProto.h"
#include "pilotPinout.h"
#include "printf.h"
#include "m68k.h"


extern uint16_t pccReq(uint16_t* dst, uint16_t req, uint16_t nWords, uint16_t *src);	//	-> u16 wordsInReply


#define CPU_CLOCK								(16384000ul)
#define TIMER_CLOCK								(32768)

#define TICKS_BETWEEN_BATTERY_SAMPLES			(TIMER_CLOCK * 3 / 2)						//sample battery every 1.5 sec
#define TICKS_MAX_UART_DELAY					(TIMER_CLOCK / 20)							//50ms max uart delay
#define BYTES_UART_SEND_ALWAYS					(8)											//if we have this many bytes, send them right away

#define WAKE_REASON_BUTTON_DOWN					0x01
#define WAKE_REASON_PEN_DOWN					0x02
#define WAKE_REASON_NEED_HW_ID					0x04





struct CircBuf {								//structure used from assembly
	uint16_t data[192];
	uint8_t writePtr, readPtr, numItems;
};

volatile uint32_t mTimerHi = 0x00000000;
static struct CircBuf mRxUartBuf, mTxUartBuf;
static uint8_t mHwID;

uint8_t __attribute__((aligned(2))) mAudioBuffer[1024];
register uint8_t *mAudioBufferCurPtr asm("a4"), *mAudioBufferEndPtr asm("a5");


void prPutchar(char chr)
{
	uint16_t ch = (uint8_t)chr;
	
	(void)pccReq(NULL, 0xa015, 1, &ch);
}

static void timerInit(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	mTimerHi++;					//make sure time neevr goes backwards if we enable/disable timer
	cpu->T[0].TPRER = 0;
	cpu->T[0].TCTL = 0x0119;	//on @ 32768, irq on
	cpu->T[0].TCMP = 0xffff;
	
	cpu->IMR &=~ M68K_IxR_TMR1;
}

static void timerDeinit(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->IMR |= M68K_IxR_TMR1;
	cpu->T[0].TCTL = 0;
}

static uint64_t timerGet(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	union {
		struct {
			uint16_t zero;
			uint32_t hi;
			uint16_t lo;
		};
		uint64_t ret;
	} mVal = {};
	uint32_t hi;
	uint16_t lo;
	
	do {
		hi = mTimerHi;
		lo = cpu->T[0].TCN;
	} while (mTimerHi != hi);
	
	mVal.hi = hi;
	mVal.lo = lo;
	
	return mVal.ret;
}

static uint8_t circBufGetSizeInItems(const struct CircBuf *cb)
{
	return sizeof(cb->data) / sizeof(*cb->data);
}

static uint8_t circBufGetNumItemsContained(const struct CircBuf *cb)
{
	return cb->numItems;
}

static bool circBufIsFull(const struct CircBuf *cb)
{
	return circBufGetNumItemsContained(cb) == circBufGetSizeInItems(cb);
}

static bool circBufIsEmpty(const struct CircBuf *cb)
{
	return circBufGetNumItemsContained(cb) == 0;
}

static bool circBufAdd(struct CircBuf *cb, uint16_t val)
{
	if (circBufIsFull(cb))
		return false;
	
	cb->data[cb->writePtr] = val;
	if (++cb->writePtr == circBufGetSizeInItems(cb))
		cb->writePtr = 0;
	cb->numItems++;
	
	return true;
}

static int32_t circBufGet(struct CircBuf *cb)
{
	uint16_t ret;
	
	if (circBufIsEmpty(cb))
		return -1;
	
	ret = cb->data[cb->readPtr];
	if (++cb->readPtr == circBufGetSizeInItems(cb))
		cb->readPtr = 0;
	cb->numItems--;
	return (uint32_t)ret;
}

static bool uartPrvCheck(void)		//to be done often and in all busy loops
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	bool anythingDone = false;
	
	if (cpu->USTCNT & 0x8000) {	//only bother if uart is on
		
		//we might have flow contorl ON, and reading the fifo might re-enable RX
		//so do not do it unless we have space
		
		while (!circBufIsFull(&mRxUartBuf)) {
			
			uint16_t rx = cpu->URX;
			
			if (!(rx & 0x2000))	//no more items in RX fifo
				break;
				
			circBufAdd(&mRxUartBuf, rx);
			anythingDone = true;
		}
		
		//if we have data to TX and space, TX
		while (!circBufIsEmpty(&mTxUartBuf) && (cpu->UTX & 0x2000)) {
			
			cpu->UTXlo = circBufGet(&mTxUartBuf);
			anythingDone = true;
		}
	}
	return anythingDone;
}

static void timerDelayMsec(uint32_t msec)	//each ms id 32.768 ticks, limits on provided value
{
	uint32_t ticks = msec * (TIMER_CLOCK / 1000);	//close enough
	uint64_t time = timerGet();
	
	while ((uint32_t)(timerGet() - time) < ticks) {
		
		uartPrvCheck();
	}
}

static bool lcdPrvIsEnabled(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	return !!(cpu->LCKCON & 0x80);
}

static void lcdPrvDisable(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	volatile uint32_t zero = 0;
	uint_fast8_t i;
	
	
	pr("** disp off\n");
							
	cpu->P[PORT_F].DATA &=~ 0x10;
	cpu->P[PORT_F].DATA &=~ 0x40;
	cpu->LCKCON &=~ 0x80;
	
	//blank the display to not leave any lines driven
	cpu->LXMAX = 1;
	cpu->LYMAX = 1;
	cpu->LVPW = 1;
	cpu->LBAR = 1;
	cpu->LSSA = (uintptr_t)&zero;
	
	cpu->LCKCON |= 0x82;	//mem is 8 bit on out stack
	timerDelayMsec(1);
	cpu->LCKCON &=~ 0x80;
	cpu->P[PORT_F].DATA |= 0x20;
		
	//drain
	for (i = 0; i < 64; i++)
		(void)*(volatile uint32_t*)0x00810000;
}

static void lcdPrvEnable(uint32_t baseAddr, bool memisX16, uint8_t depth, bool superFast)
{
	const uint16_t width = 160, height = 160, fps = 60, lineWords = (width * depth + 15) / 16;
	struct M68K *cpu = (struct M68K*)0xfffff000;
		
	cpu->P[PORT_F].DATA &=~ 0x20;
	cpu->P[PORT_F].DATA &=~ 0x10;
	
	cpu->LCKCON = 0;
	cpu->LSSA = baseAddr;
	cpu->LVPW = lineWords;
	cpu->LXMAX = width - 1;
	cpu->LYMAX = height - 1;
	cpu->LBAR = lineWords - 1;
	cpu->LPXCD = superFast ? 0 : 1;
	cpu->LOTCR = cpu->LBAR + 8;	//(cpuClock * fps / 1000000 / height + 15) / 16;
	cpu->LPICF = 0x04 + (depth == 2 ? 1 : 0);
	cpu->LCKCON = 0x68 + (memisX16 ? 0x00 : 0x02);
	cpu->LGPMR = 0x2073;
	/*
	code			0	1	2	3	4	5	6	7
	density	/16		0	4	5	8	11	12	16	16
	*/
	
	
	timerDelayMsec(20);
	
	cpu->LCKCON |= 0x80;
	
	timerDelayMsec(25);
	
	cpu->P[PORT_F].DATA |= 0x40;
	
	timerDelayMsec(50);
	
	cpu->P[PORT_F].DATA |= 0x10;
	
}

static void prvIrqsRequest(uint16_t *irqsOutP, uint16_t *devStaOutP, uint16_t irqsClear)
{
	uint16_t dataOut[2], dataOutLen, irqsToClear;
			
	dataOutLen = pccReq(dataOut, PCC_WRAP_REQ(PCC_REQ_IRQ_STA), 1, &irqsClear);
	
	if (dataOutLen != sizeof(dataOut) / sizeof(*dataOut)) {
		
		*irqsOutP = 0;
		*devStaOutP = 0;
	}
	else {
		
		*irqsOutP = dataOut[0];
		*devStaOutP = dataOut[1];
	}
}

static void pllConfig(uint16_t freq)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	uint16_t dummy;
	
	asm volatile(
		"1:						\n\t"
		"	move.w	(%1), %0	\n\t"
		"	bpl.b	1b			\n\t"
		"	move.w	%2, (%1)	\n\t"
		"1:						\n\t"
		"	move.w	(%1), %0	\n\t"
		"	bmi.b	1b			\n\t"
		"1:						\n\t"
		"	move.w	(%1), %0	\n\t"
		"	bpl.b	1b			\n\t"
		:"=&d"(dummy)
		:"a"(&cpu->PLLFSR), "d"(freq)
		:"memory", "cc"
	);
}

static void pllShutdown(void)		//xxx see also: PrvShutDownCPU() and HwrShutDownPLL
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	uint16_t dummy;
	
	asm volatile(
		"1:						\n\t"
		"	move.w	(%1), %0	\n\t"
		"	bpl.b	1b			\n\t"
		"	bset	#3, (%2)	\n\t"
		"	stop	#0x2000		\n\t"
		"	nop					\n\t"
		"	nop					\n\t"
		"	nop					\n\t"
		"	nop					\n\t"
		:"=&d"(dummy)
		:"a"(&cpu->PLLFSR), "a"(&cpu->PLLCR)
		:"memory", "cc"
	);
}

static void prvInit(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->SCR = 0xf0;		//no more double map, report faults, clear fault state
	cpu->IVR = 0x18;		//to match our table
	cpu->IMR = 0x00ffffff;	//mask all interrupts
	cpu->IWR = 0x00ffffff;	//all irqs do wake us up
	asm volatile("andi.w #0xf0ff, %%sr":::"cc","memory");	//interrupts on
	
	pr("Hello from 68K at 0x%08lx \n", &prvInit);

	//set clock rate
	pllConfig(0x0922);	//32768 * 500 = 16384000
	
	//enable irq3
	cpu->P[PORT_M].DIR &=~ (1 << 3);
	cpu->P[PORT_M].SEL &=~ (1 << 3);
	cpu->ICR &=~ 0x2200;					//irq3 is level triggered
	cpu->IMR &=~ M68K_IxR_IRQ3;	//enabled
	
	timerInit();
}

static uint8_t keyRead(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	return PORTD_BITS_ALL_KEYS & cpu->P[PORT_D].DATA;
}

static void keyInit(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->P[PORT_D].DIR &=~ PORTD_BITS_ALL_KEYS;
	cpu->P[PORT_D].POL |= PORTD_BITS_ALL_KEYS;
	cpu->P[PORT_D].PUEN |= PORTD_BITS_ALL_KEYS;
	cpu->P[PORT_D].IRQEN |= PORTD_BITS_ALL_KEYS;
	
	cpu->IMR &=~ M68K_IxR_KB;	//without this, we'll never wake up from idle
}

static uint8_t deviceIdentify(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->P[PORT_E].PUEN |= PORTE_BIT_HAVE_NO_BACKLITE;
	cpu->P[PORT_E].SEL |= PORTE_BIT_HAVE_NO_BACKLITE;
	cpu->P[PORT_E].DIR &=~ PORTE_BIT_HAVE_NO_BACKLITE;
	
	if (cpu->P[PORT_E].DATA & PORTE_BIT_HAVE_NO_BACKLITE) {	//either Pilot 1000/5000 or PalmIII+ - need to do more work
		
		uint8_t id, idBits = PORTD_BIT_KEY_APP1 | PORTD_BIT_KEY_APP2 | PORTD_BIT_KEY_APP3 | PORTD_BIT_KEY_APP4;
		
		//pullup off
		cpu->P[PORT_E].PUEN &=~ PORTE_BIT_HAVE_NO_BACKLITE;
		
		//keys serve as device ID as well. this is a race with the user. we may be slow, but the user is slower...
		cpu->P[PORT_D].DIR &=~ idBits;
		cpu->P[PORT_D].PUEN |= idBits;
		cpu->P[PORT_D].POL &=~ idBits;
		cpu->P[PORT_D].IRQEN &=~ idBits;
		
		while ((cpu->P[PORT_D].DATA & idBits) != idBits);	//wait for keys to NOT be held  (this raced with the check in the func that calls us)
		
		cpu->P[PORT_E].DIR |= PORTE_BIT_HAVE_NO_BACKLITE;
		cpu->P[PORT_E].DATA &=~ PORTE_BIT_HAVE_NO_BACKLITE;
		
		//you'd think we'd need to delay here, but we are so slow, we do not
		id = cpu->P[PORT_D].DATA & idBits;
		
		//now shut off that pin to let user use the buttons
		cpu->P[PORT_E].DIR &=~ PORTE_BIT_HAVE_NO_BACKLITE;
		
		//restore button config
		cpu->P[PORT_D].DIR &=~ idBits;
		cpu->P[PORT_D].POL |= idBits;
		cpu->P[PORT_D].PUEN |= idBits;
		cpu->P[PORT_D].IRQEN |= idBits;
		
		//and now parse the ID. we only support a few devices, so this is easy
		if (id == idBits) 		//Pilot 1000/5000
			return PCC_HW_IDENTIFIED;
		
		if (id == (idBits &~ PORTD_BIT_KEY_APP1))	//III
			return PCC_HW_IDENTIFIED | PCC_HW_HAS_IR | PCC_HW_HAS_BACKLIGHT | PCC_HW_HAS_BBADS;
		
		//some other device
		return 0;
	}
	else {													//Palm Personal / Professional
		
		//pullup off
		cpu->P[PORT_E].PUEN &=~ PORTE_BIT_HAVE_NO_BACKLITE;
		return PCC_HW_IDENTIFIED | PCC_HW_HAS_BACKLIGHT;
	}
}



static uint16_t bbadsDoRead(uint8_t cmd)
{
	uint16_t ret;
	
/*	--the c code
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->P[PORT_F].DATA &=~ PORTF_BIT_ADC_NCS;
	
	cpu->SPIMCONT = 0x2246;			//7 bits at Fclk / 8 = 2MHz
	cpu->SPIMDATA = cmd >> 1;
	cpu->SPIMCONT &=~ 0x0080;
	cpu->SPIMCONT |= 0x0100;
	while (!(cpu->SPIMCONT & 0x0080));
	
	cpu->SPIMCONT = 0xc240;			//1 bit at slow clock
	cpu->SPIMDATA = cmd;
	cpu->SPIMCONT &=~ 0x0080;
	cpu->SPIMCONT |= 0x0100;
	while (!(cpu->SPIMCONT & 0x0080));
	
	cpu->SPIMCONT = 0x224f;			//16 bits of result at Fclk / 8 = 2MHz
	cpu->SPIMDATA = 0;
	cpu->SPIMCONT &=~ 0x0080;
	cpu->SPIMCONT |= 0x0100;
	while (!(cpu->SPIMCONT & 0x0080));
	
	cpu->P[PORT_F].DATA |= PORTF_BIT_ADC_NCS;
	
	return (cpu->SPIMDATA >> 3) & 0xfff;
*/
	
	
	asm(
		".macro spistart				\n\t"
		"	andi.b	#0x7f,	0x803(%1)	\n\t"			//clear "done" bit
		"	ori.b	#0x01,	0x802(%1)	\n\t"			//start
		".endm							\n\t"
	
		".macro spiwait					\n\t"
		"	1:							\n\t"
		"		tst.b	0x803(%1)		\n\t"
		"		bpl.b	1b				\n\t"
		".endm							\n\t"
		
		
		"	andi.b	#0x7f,	0x429(%1)	\n\t"			//cpu->P[PORT_F].DATA &=~ PORTF_BIT_ADC_NCS;
	
		"	move.w	#0x2246,0x802(%1)	\n\t"			//7 bits at Fclk / 8 = 2MHz
		"	move.b	%2,		%0			\n\t"
		"	lsr.b	#1,		%0			\n\t"
		"	move.b	%0,		0x801(%1)	\n\t"			//send all but the last bit
		"	spistart					\n\t"
		"	spiwait						\n\t"
	
		"	move.w	#0x8240,0x802(%1)	\n\t"			//1 bits at Fclk / 256 
		"	move.b	%2,		0x801(%1)	\n\t"			//send all but the last bit
		"	spistart					\n\t"
		"	spiwait						\n\t"
		
		"	move.w	#0x224f,0x802(%1)	\n\t"			//16 bits of result at Fclk / 8 = 2MHz
		"	clr.w	0x800(%1)			\n\t"			//send a zero
		"	spistart					\n\t"
		"	spiwait						\n\t"
		"	move.w	0x800(%1),	%0		\n\t"
		"	lsr.w	#3, %0				\n\t"
		"	andi.w	#0x0fff, %0			\n\t"
	
		"	ori.b	#0x80,	0x429(%1)	\n\t"			//cpu->P[PORT_F].DATA |=PORTF_BIT_ADC_NCS;
		
		:"=&d"(ret)
		:"a"(0xfffff000), "d"(cmd)
		:"memory", "cc"
	);
	
	return ret;
}

static void penPrvSetModeOld(uint8_t mode)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->P[PORT_F].DATA = (cpu->P[PORT_F].DATA &~ PORTF_DIGI_MODE_MASK) | mode;
}


static void penInit(bool withBbads)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	//digitizer control pins are OUT GPIOs in the "Off" mode, nCS pin is out GPIO high
	cpu->P[PORT_F].SEL |= PORTF_BIT_DIGI_X_NEG_TO_GND_ON | PORTF_BIT_DIGI_X_POS_TO_VCC_DIS | PORTF_BIT_DIGI_Y_NEG_TO_GND_ON | PORTF_BIT_DIGI_Y_POS_TO_VCC_DIS | PORTF_BIT_ADC_NCS;
	cpu->P[PORT_F].DIR |= PORTF_BIT_DIGI_X_NEG_TO_GND_ON | PORTF_BIT_DIGI_X_POS_TO_VCC_DIS | PORTF_BIT_DIGI_Y_NEG_TO_GND_ON | PORTF_BIT_DIGI_Y_POS_TO_VCC_DIS | PORTF_BIT_ADC_NCS;
	cpu->P[PORT_F].DATA = (cpu->P[PORT_F].DATA &~ PORTF_DIGI_MODE_MASK) | PORTF_DIGI_MODE_OFF | PORTF_BIT_ADC_NCS;
	
	//penirq pin setup
	cpu->P[PORT_M].DIR |= PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].DATA &=~ PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].PUEN |= PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].SEL &=~ PORTM_BIT_PEN_IRQ;
	
	//battery measurement pin setup
	cpu->P[PORT_G].SEL |= PORTG_BATT_MEAS_DIS;
	cpu->P[PORT_G].DIR |= PORTG_BATT_MEAS_DIS;
	cpu->P[PORT_G].DATA |= PORTG_BATT_MEAS_DIS;
	
	//setup SPI master pins
	cpu->P[PORT_K].SEL &=~ 0x07;	//give GPIOs to SPIM
		
	if (withBbads)
		bbadsDoRead(0x90);	//irq on
	else
		penPrvSetModeOld(PORTF_DIGI_MODE_IRQ);
	cpu->IMR &=~ M68K_IxR_PENIRQ;	//irq on
}

static void penReadOld(uint16_t *xP, uint16_t *yP, bool *downP)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	//stop pen irq port from messing with us by making it a low output
	cpu->P[PORT_M].PUEN &=~ PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].SEL |= PORTM_BIT_PEN_IRQ;
	
	//measure Y
	penPrvSetModeOld(PORTF_DIGI_MODE_MEAS_Y);
	cpu->SPIMCONT = 0x427f;						//15 bits in mode 3 at CLK/16, irqen (see datasheet as to why)
	cpu->SPIMDATA = 0xc000;
	timerDelayMsec(1);
	cpu->P[PORT_F].DATA &=~ PORTF_BIT_ADC_NCS;
	cpu->SPIMCONT |= 0x0100;
	while (!(cpu->SPIMCONT & 0x0080)) {
		
		uartPrvCheck();
	}
	*yP = (cpu->SPIMDATA & 0x1fe) << 3;
	cpu->P[PORT_F].DATA |= PORTF_BIT_ADC_NCS;
	
	//measure X
	penPrvSetModeOld(PORTF_DIGI_MODE_MEAS_X);
	cpu->SPIMDATA = 0xffff;
	cpu->SPIMCONT &=~ 0x0080;
	timerDelayMsec(1);
	cpu->P[PORT_F].DATA &=~ PORTF_BIT_ADC_NCS;
	cpu->SPIMCONT |= 0x0100;
	while (!(cpu->SPIMCONT & 0x0080)) {
		
		uartPrvCheck();
	}
	*xP = (cpu->SPIMDATA & 0x1fe) << 3;
	cpu->P[PORT_F].DATA |= PORTF_BIT_ADC_NCS;
	
	//SPIM off
	cpu->SPIMCONT = 0;
	
	penPrvSetModeOld(PORTF_DIGI_MODE_IRQ);
	//restore penirq
	cpu->P[PORT_M].PUEN |= PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].DIR &=~ PORTM_BIT_PEN_IRQ;
	timerDelayMsec(2);
	
	*downP = !(cpu->P[PORT_M].DATA & PORTM_BIT_PEN_IRQ);
	cpu->P[PORT_M].SEL &=~ PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].DIR |= PORTM_BIT_PEN_IRQ;
}

static void penReadBbads(uint16_t *xP, uint16_t *yP, bool *downP)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	//stop pen irq port from messing with us by making it a low output
	cpu->P[PORT_M].PUEN &=~ PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].SEL |= PORTM_BIT_PEN_IRQ;
	
	*yP = bbadsDoRead(0x90);
	*xP = bbadsDoRead(0xd0);
	
	//restore penirq
	cpu->P[PORT_M].PUEN |= PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].DIR &=~ PORTM_BIT_PEN_IRQ;
	timerDelayMsec(2);
	
	*downP = !(cpu->P[PORT_M].DATA &PORTM_BIT_PEN_IRQ);
	cpu->P[PORT_M].SEL &=~ PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].DIR |= PORTM_BIT_PEN_IRQ;
}

static uint16_t batteryReadOld(void)	//assumes penInit() has been called, assumes touch chip is in PENIRQ mode
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	uint8_t ret;
	
	cpu->P[PORT_M].DIR |= PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].DATA &=~ PORTM_BIT_PEN_IRQ;
	cpu->P[PORT_M].SEL |= PORTM_BIT_PEN_IRQ;
	penPrvSetModeOld(PORTF_DIGI_MODE_OFF);
	
	cpu->P[PORT_G].DATA &=~ PORTG_BATT_MEAS_DIS;
	
	cpu->SPIMCONT = 0x427f;						//15 bits in mode 3 at CLK/16, irqen (see datasheet as to why)
	cpu->SPIMDATA = 0xffff;
	timerDelayMsec(1);
	cpu->P[PORT_F].DATA &=~ PORTF_BIT_ADC_NCS;
	cpu->SPIMCONT |= 0x0100;
	while (!(cpu->SPIMCONT & 0x0080)) {
		
		uartPrvCheck();
	}
	ret = cpu->SPIMDATA >> 1;
	cpu->P[PORT_F].DATA |= PORTF_BIT_ADC_NCS;
	
	cpu->P[PORT_G].DATA |= PORTG_BATT_MEAS_DIS;
	
	penPrvSetModeOld(PORTF_DIGI_MODE_IRQ);
	cpu->P[PORT_M].SEL &=~ PORTM_BIT_PEN_IRQ;
	
	return ((uint16_t)ret) << 8;
}

static uint16_t batteryReadBbads(void)	//assumes penInit() has been called, assumes touch chip is in PENIRQ mode
{
	return bbadsDoRead(0xa4) << 4;
}

static void uartPrvConfigure(uint16_t cfg, uint32_t *baudrateP)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	uint32_t presc32, baudrate = *baudrateP;
	uint16_t presc, shift = (uint16_t)-1;
	
	if (!baudrate)
		baudrate = 1;
	
	do {
		shift++;
		presc = presc32 = ((CPU_CLOCK >> shift) + baudrate / 2) / baudrate;
	} while (presc32 > 64);
	
	if (presc < 2)
		presc = 2;
	
	if (shift > 7)
		shift = 7;
	
	*baudrateP = (CPU_CLOCK >> shift) / presc;
	
	cpu->USTCNT = 0;
	cpu->UBAUD = (shift << 8) + (65 - presc);
	cpu->UMISC = 0;
	cpu->UTXhi = ((cfg & UART_CFG_TX_BREAK) ? 0x10 : 0x00) + ((cfg & UART_CFG_TX_FLOW_CONTROL) ? 0x00 : 0x08);
	cpu->USTCNT = ((cfg & (UART_CFG_MASK_RX_EN | UART_CFG_MASK_TX_EN)) ? 0x8000 : 0x0000) +		//on
					(cfg & (UART_CFG_MASK_RX_EN | UART_CFG_MASK_TX_EN | UART_CFG_MASK_PAR_ENA |
						UART_CFG_MASK_PAR_ODD | UART_CFG_MASK_STOP_BITS_2 | UART_CFG_MASK_BYTE_LEN_8));
}

static void miscInit(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->P[PORT_G].DIR |= PORTG_UART_CHIP_ENA | PORTG_BACKLITE_ENA;
	cpu->P[PORT_G].SEL |= PORTG_UART_CHIP_ENA | PORTG_BACKLITE_ENA;
	cpu->P[PORT_G].DATA &=~( PORTG_UART_CHIP_ENA | PORTG_BACKLITE_ENA);
	
	cpu->P[PORT_J].DIR |= PORTJ_BIT_IRDA_TX_ENA | PORTJ_BIT_IRDA_ENA;
	cpu->P[PORT_J].SEL |= PORTJ_BIT_IRDA_TX_ENA | PORTJ_BIT_IRDA_ENA;
	cpu->P[PORT_J].DATA &=~ (PORTJ_BIT_IRDA_TX_ENA | PORTJ_BIT_IRDA_ENA);
}

static void simpleSoundCreate(uint16_t freq, uint8_t volume)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	if (!freq || !volume) {
		
		cpu->PWMC = 0;
		cpu->P[PORT_G].SEL |= (1 << 2);
		cpu->P[PORT_G].DIR &=~ (1 << 2);
	}
	else {
		
		uint16_t period, duty, divShift = 0;
		uint32_t base = CPU_CLOCK / 2, t;
		
		cpu->P[PORT_G].SEL &=~ (1 << 2);
		cpu->P[PORT_G].DIR |= (1 << 2);
		cpu->PWMC = 0;
		
		while ((base >> 16) >= freq) {
			
			base >>= 1;
			divShift++;
		}
		
		//we know the division will not overflow, so force gcc to do it
		asm("divu %1, %0":"=d"(period):"d"(freq),"0"(base):"cc");
		
		//gcc just sucks at this, so we have to help it
		asm("mulu %1, %0" :"=d"(t):"d"(volume * volume),"0"(period):"cc");
		duty = t / 512 / 256;
		
		cpu->PWMCNT = 0;
		cpu->PWMP = period - 1;
		cpu->PWMW = duty;
		cpu->PWMC = 0x0110 + divShift;
	}
}

static uint16_t pccUartDataXcg(void)		//return num words given to host
{
	uint16_t buf[PCC_REQ_UART_DATA + 1], nItems = 0, i, ret;
	
	buf[0] = circBufGetSizeInItems(&mTxUartBuf) - circBufGetNumItemsContained(&mTxUartBuf);
	while (nItems < PCC_REQ_UART_DATA && !circBufIsEmpty(&mRxUartBuf))
		buf[++nItems] = circBufGet(&mRxUartBuf);
	ret = nItems;

	nItems = pccReq(buf, PCC_WRAP_REQ(PCC_REQ_UART_DATA), nItems + 1, buf);
	
	for (i = 0; i < nItems; i++)	//ignore failures - they asked for it by sendign too much data
		(void)circBufAdd(&mTxUartBuf, buf[i]);
	
	return ret;
}

static bool __attribute__((noinline)) hwIdAndInitAsNeeded(void)	//this func will self-patch-out the call to itself when it has done its job, thus noinline
{
	
	if (keyRead()) {
		
		//keys pressed - no ID yet
		return false;
	}
	else if (mHwID & PCC_HW_IDENTIFIED) {
		
		//already identified
		return true;
	}
	else {
	
		uint16_t id = mHwID = deviceIdentify();
		
		pr("ID success: %02x\n", mHwID);
		pccReq(NULL, PCC_WRAP_REQ(PCC_REQ_DEVICE_HWR_FLAGS), 1, &id);
		
		if (!(mHwID & PCC_HW_IDENTIFIED)) {
			
			pr("device id fail\n");
			return false;
		}
		else {
			
			uint16_t *retAddr16;
			uint8_t *retAddr8;
			void *retAddr;
						
			penInit(!!(mHwID & PCC_HW_HAS_BBADS));
			
			//patch self out
			retAddr = __builtin_return_address(0);
			retAddr16 = retAddr;
			retAddr8 = retAddr;
			
			if (retAddr8[-2] == 0b01100001)	//maybe a short BSR?
				retAddr16[-1] = 0x7000;
			else if (retAddr16[-2] == 0b0110000100000000)
				retAddr16[-1] = retAddr16[-2] = 0x7000;
			else
				pr("patch fail\n");
			
			return true;
		}
	}
}

static void sampledAudioOn(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	//with audio on, we cannot use interrupts to handle keyboard so disable that irq, we'll busy-poll
	cpu->IMR |= M68K_IxR_KB;
	
	cpu->P[PORT_G].SEL &=~ (1 << 2);
	cpu->P[PORT_G].DIR |= (1 << 2);
	memset(mAudioBuffer, 0, sizeof(mAudioBuffer));
	mAudioBufferCurPtr = mAudioBuffer;
	cpu->PWMP = 255;
	cpu->PWMW = 0;
	cpu->PWMC = 0x4110;		//p. 131
	cpu->IMR &=~ M68K_IxR_PWM;
}

static void sampledAudioOff(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->IMR |= M68K_IxR_PWM;
	cpu->PWMC = 0;

	cpu->P[PORT_G].SEL |= (1 << 2);
	cpu->P[PORT_G].DIR &=~ (1 << 2);
	
	mAudioBufferCurPtr = mAudioBuffer;
	
	cpu->IMR &=~ M68K_IxR_KB;	//re-enable keyboard interrupts, they will copy shit to PWM register, but who cares?
}

void micromain(void)
{
	uint8_t prevKeys = 0, wakeReasons = WAKE_REASON_NEED_HW_ID, audioPrevHalf = 0;
	uint64_t timeLastBattery = 0, timeHadUartRxData = 0;	///aaa
	struct M68K *cpu = (struct M68K*)0xfffff000;
	uint16_t prevDevState = 0;
	
	prvInit();	
	keyInit();
	
	
	miscInit();
	
	mAudioBufferEndPtr = mAudioBuffer + sizeof(mAudioBuffer);	//one-time init
	mAudioBufferCurPtr = mAudioBuffer;
		
	while (1) {
		
		uint8_t keysNow, numUartRxBytes;
		uint64_t currentTime;
		uint8_t audioNowHalf;
		
		if (!wakeReasons) {
			
			asm volatile("stop #0x2000");
		}
		
		if (hwIdAndInitAsNeeded()) {	//this call will be patched out, replaced with a return false
			
			timeLastBattery = 0;	//send a report asap
			wakeReasons &=~ WAKE_REASON_NEED_HW_ID;
		}
		
		keysNow = keyRead();
		if (keysNow) {
			wakeReasons |= WAKE_REASON_BUTTON_DOWN;
		}
		else {
			wakeReasons &=~ WAKE_REASON_BUTTON_DOWN;
			if (cpu->IMR & M68K_IxR_PWM)	//no audio? re-enable keys irqs
				cpu->IMR &=~ M68K_IxR_KB;
		}
		
		//if we have no audio on, re-enable uart irq
		if (cpu->IMR & M68K_IxR_PWM)	//no audio? re-enable keys irqs
			cpu->IMR &=~ M68K_IxR_UART;
		
		do {
			
			//do this often, and uart too
			audioNowHalf = mAudioBufferCurPtr >= &mAudioBuffer[sizeof(mAudioBuffer) / sizeof(*mAudioBuffer) / 2];
			if (audioNowHalf != audioPrevHalf) {
				
				audioPrevHalf = audioNowHalf;
				if (audioNowHalf) {
					
					pccReq((uint16_t*)(mAudioBuffer + 0), PCC_WRAP_REQ(PCC_REQ_AUDIO_DATA), 0, NULL);
				}
				else {
					
					pccReq((uint16_t*)(mAudioBuffer + 512), PCC_WRAP_REQ(PCC_REQ_AUDIO_DATA), 0, NULL);
				}
					
					
				(void)pccReq((uint16_t*)(audioNowHalf ? mAudioBuffer : &mAudioBuffer[sizeof(mAudioBuffer) / sizeof(*mAudioBuffer) / 2]), PCC_WRAP_REQ(PCC_REQ_AUDIO_DATA), 0, NULL);
			}
			
		} while (uartPrvCheck());
		
		currentTime = timerGet();
		
		numUartRxBytes = circBufGetNumItemsContained(&mRxUartBuf);
		if (numUartRxBytes && !timeHadUartRxData)
			timeHadUartRxData = currentTime;
		
		if (numUartRxBytes >= BYTES_UART_SEND_ALWAYS || (numUartRxBytes && (currentTime - timeHadUartRxData >= TICKS_MAX_UART_DELAY))) {
			
			uint16_t wordsToHost = pccUartDataXcg();
			
			if (circBufIsEmpty(&mRxUartBuf))
				timeHadUartRxData = 0;
			else
				timeHadUartRxData = currentTime;
		}
		
		if (currentTime - timeLastBattery >= TICKS_BETWEEN_BATTERY_SAMPLES) {
			
			if (mHwID & PCC_HW_IDENTIFIED) {
			
				uint16_t packet[] = {(mHwID & PCC_HW_HAS_BBADS) ? batteryReadBbads() : batteryReadOld(), 3300};
				(void)pccReq(NULL, PCC_WRAP_REQ(PCC_REQ_BATT_REPORT), sizeof(packet) / sizeof(*packet), packet);
			}
			timeLastBattery = currentTime;
		}
		
		if (cpu->IPR & M68K_IxR_IRQ3) {
			
			uint16_t irqs = 0, requestedDevSta, changedSta;
			
			do {		//handle irqs while they exist, clear as we go along
				
				uint16_t irqsToClear = irqs;
				
				prvIrqsRequest(&irqs, &requestedDevSta, irqsToClear);
				//pr("irqs gotten: %04x %04x\n", irqs, requestedDevSta);
			
				//if we just cleared the "go to sleep" irq, go to sleep now
				if (irqsToClear & PCC_IRQ_BIT_REQUEST_POWER_OFF) {
					
					pr("***shutting down 68k side...\n");
					sampledAudioOff();
					simpleSoundCreate(0, 0);
					timerDeinit();
					pr("***going to sleep\n");
					//erratum in some revisions of 68328
					cpu->PLLCR = (cpu->PLLCR &~ 0x0700) | 0x0100;	//goto "div 4"
					pllShutdown();
					cpu->PLLCR = (cpu->PLLCR &~ 0x0700) | 0x0400;	//go back to "div 1"
					pr("***awake\n");
					timerInit();
				}
			
				//handle irqs here, set bits in irqsToClear
				if (irqs & PCC_IRQ_BIT_HAVE_NEW_DEVICE_CONFIG) {
					
					uint16_t stateChanges = requestedDevSta ^ prevDevState;
					
					if (stateChanges & PCC_DEVSTATE_MASK_SCREEN) {
						
						switch ((requestedDevSta & PCC_DEVSTATE_MASK_SCREEN) >> PCC_DEVSTATE_SHIFT_SCREEN){
							case PCC_DEVSTATE_VAL_SCREEN_OFF >> PCC_DEVSTATE_SHIFT_SCREEN:
								if (lcdPrvIsEnabled())
									lcdPrvDisable();
								break;
							
							case PCC_DEVSTATE_VAL_SCREEN_1_BPP:
								if (lcdPrvIsEnabled())
									break;
								lcdPrvEnable(0x00810000, true, 1, false);
								break;
							
							case PCC_DEVSTATE_VAL_SCREEN_2_BPP:
								if (lcdPrvIsEnabled())
									break;
								lcdPrvEnable(0x00810000, true, 2, false);
								break;
							
							case PCC_DEVSTATE_VAL_SCREEN_4_BPP:
								if (lcdPrvIsEnabled())
									break;
								lcdPrvEnable(0x00810000, true, 2, true);
								break;
						}
					}
					
					if (stateChanges & PCC_DEVSTATE_MASK_SERIAL) {
						
						switch ((requestedDevSta & PCC_DEVSTATE_MASK_SERIAL) >> PCC_DEVSTATE_SHIFT_SERIAL) {
							case PCC_DEVSTATE_VAL_SERIAL_XCVR_OFF:
								
								cpu->P[PORT_G].DATA &=~ PORTG_UART_CHIP_ENA;
								cpu->P[PORT_J].DATA &=~ (PORTJ_BIT_IRDA_TX_ENA | PORTJ_BIT_IRDA_ENA);
								break;
							
							case PCC_DEVSTATE_VAL_SERIAL_XCVR_CRADLE:
								
								cpu->P[PORT_G].DATA |= PORTG_UART_CHIP_ENA;
								cpu->P[PORT_J].DATA &=~ (PORTJ_BIT_IRDA_TX_ENA | PORTJ_BIT_IRDA_ENA);
								break;
							
							case PCC_DEVSTATE_VAL_SERIAL_XCVR_IRDA:
							
								cpu->P[PORT_G].DATA &=~ PORTG_UART_CHIP_ENA;
								cpu->P[PORT_J].DATA |= (PORTJ_BIT_IRDA_TX_ENA | PORTJ_BIT_IRDA_ENA);
								break;
						}
					}
					
					if (stateChanges & PCC_DEVSTATE_BIT_BACKLIGHT_ON) {
						
						//pr("backlight: %s\n", (requestedDevSta & PCC_DEVSTATE_BIT_BACKLIGHT_ON) ? "ON" : "OFF");
						if (requestedDevSta & PCC_DEVSTATE_BIT_BACKLIGHT_ON)
							cpu->P[PORT_G].DATA |= PORTG_BACKLITE_ENA;
						else
							cpu->P[PORT_G].DATA &=~ PORTG_BACKLITE_ENA;
					}
					
					if (stateChanges & PCC_DEVSTATE_BIT_AUDIO_ON) {
						
						if (requestedDevSta & PCC_DEVSTATE_BIT_AUDIO_ON)
							sampledAudioOn();
						else
							sampledAudioOff();
						audioPrevHalf = 0;	//needed in either case
					}
				
					prevDevState = requestedDevSta;
				}
				
				if (irqs & PCC_IRQ_BIT_HAVE_NEW_UART_CONFIG) {
					
					struct PalmcardCommsUartConfig cfg;
					uint16_t dataOutLen = pccReq((uint16_t*)&cfg, PCC_WRAP_REQ(PCC_REQ_WANT_UART_CFG), 0, NULL);
					
					if (dataOutLen * sizeof(uint16_t) == sizeof(cfg))
						uartPrvConfigure(cfg.cfg, &cfg.baudrateBE);
					
					//report baudrate back
					(void)pccReq(NULL, PCC_WRAP_REQ(PCC_REQ_UART_BAUDRATE), sizeof(cfg.baudrateBE) / sizeof(uint16_t), (uint16_t*)&cfg.baudrateBE);
				}
				
				if (irqs & PCC_IRQ_BIT_HAVE_NEW_UART_DATA) {
					
					pccUartDataXcg();
				}
				
				if (irqs & PCC_IRQ_BIT_HAVE_NEW_SIMPLE_SOUND_REQ) {
					
					uint16_t req[2], dataOutLen = pccReq(req, PCC_WRAP_REQ(PCC_REQ_SIMPLE_AUDIO_REQ), 0, NULL);
					
					if (dataOutLen == sizeof(req) / sizeof(*req))
						simpleSoundCreate(req[0], req[1]);
				}
				
				if (irqs & PCC_IRQ_BIT_REQUEST_POWER_OFF) {		//sleep was requested
					
					pr("request to sleep RXed\n");
					//all work done when we clear the irq
				}
			
			} while (irqs);
			
			cpu->IMR &=~ M68K_IxR_IRQ3;	//re-enable irq
		}
		
		if ((mHwID & PCC_HW_IDENTIFIED) && (cpu->IMR & M68K_IxR_PENIRQ)) {	//penirq being masked means the pen is down
			
			uint16_t coords[2];
			bool penDown;
			
			if (mHwID & PCC_HW_HAS_BBADS)
				penReadBbads(&coords[0], &coords[1], &penDown);
			else
				penReadOld(&coords[0], &coords[1], &penDown);

			if (!penDown) {
				
				coords[0] = 0xffff;
				coords[1] = 0xffff;
			}

			(void)pccReq(NULL, PCC_WRAP_REQ(PCC_REQ_TOUCH_REPORT), sizeof(coords) / sizeof(uint16_t), coords);
						
			if (!penDown) {
				
				wakeReasons &=~ WAKE_REASON_PEN_DOWN;
				cpu->IMR &=~ M68K_IxR_PENIRQ;
			}
			else {
				
				wakeReasons |= WAKE_REASON_PEN_DOWN;
			}
		}
		
		if (prevKeys != keysNow) {
			
			uint16_t keysVal = keysNow;
			
			(void)pccReq(NULL, PCC_WRAP_REQ(PCC_REQ_BTN_REPORT), sizeof(keysVal) / sizeof(uint16_t), &keysVal);
			prevKeys = keysNow;
		}
	}
	
	while(1);
}


void __attribute__((interrupt,used)) __irq5(void)
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	
	cpu->IMR |= M68K_IxR_PENIRQ;
}

void __attribute__((interrupt,used)) __irq3(void)		//here we'll just mask the irq and let main code do the rest
{
	struct M68K *cpu = (struct M68K*)0xfffff000;
	uint16_t dataOut[2], dataOutLen;
	
//	pr("3 irq. ISR=%08xh IMR=%08x IPR=%08x\n", cpu->ISR, cpu->IMR, cpu->IPR);
	cpu->IMR |= M68K_IxR_IRQ3;
}


void __attribute__((interrupt,used)) __irq1(void)
{
	pr("1 irq\n");	
}

void __attribute__((interrupt,used)) __irq2(void)
{
	pr("2 irq\n");	
}

void __attribute__((interrupt,used)) __irq7(void)
{
	pr("7 irq\n");	
}


#define MISCHANDLER(name)	void __ ##name(void) { pr(#name "\n"); while(1); }

MISCHANDLER(fault)
MISCHANDLER(unusedVec)
MISCHANDLER(uninitIrq)
MISCHANDLER(spuriousIrq)
MISCHANDLER(trap)
