#include <stdbool.h>
#include <stdint.h>
#include <PalmOS.h>

//assumes no databases change while we are running
//no device we support has over 32M of ram, so our virtucal card is FAT16, 512b clusters
//partition starts at sec 1, all files are in /PALM/Launcher
//cluster 2 is "PALM"
//cluster 3 if "LAUNCHER"
//all other clusters are PRCs
//short names are just HEX versions of DB index followed by a dot and ".prc"
//as each long name in palmos is 32 chars at max, we prefix "ROM-" or "RAM-" to them, thus we need at most 3 LFN entries + 1 SFN entry per name
//16 entries per dir sector, we always use 4 entries per sector
//to avoid complex math, first sec of launcher dir only has the mandatory entries
///we basically hope there are no dupes...

#define SECTOR_SIZE					512
#define DIR_ENTRIES_PER_SEC			(SECTOR_SIZE / sizeof(struct DirEntry))
#define CHARS_PER_LFN_ENTRY			13

#define ATTR_READONLY				0x01
#define ATTR_HIDDEN					0x02
#define ATTR_SYSTEM					0x04
#define ATTR_VOL_LBL				0x08
#define ATTR_DIRECTORY				0x10
#define ATTR_ARCHIVE				0x20
#define ATTR_LFN					(ATTR_READONLY | ATTR_HIDDEN | ATTR_SYSTEM | ATTR_VOL_LBL)

#define LFN_ENTRIES_PER_DB			3
#define DIR_ENTRIES_PER_DB			(LFN_ENTRIES_PER_DB + 4)
#define DBS_PER_DIR_SECTOR			(DIR_ENTRIES_PER_SEC / DIR_ENTRIES_PER_DB)

#define SEC_PER_CLUS				1
#define NUM_CLUSTERS				0xf000	//includes the 2 that do not exist
#define ROOT_DIR_ENTRIES			16
#define FAT_SECS					((NUM_CLUSTERS * 2 + SECTOR_SIZE - 1) / SECTOR_SIZE)
#define ROOT_DIR_SECS				((ROOT_DIR_ENTRIES * 32 + SECTOR_SIZE - 1) / SECTOR_SIZE)
#define MBR_SEC						0
#define PBR_SEC						(MBR_SEC + 1)
#define FIRST_FAT_FIRST_SEC			(PBR_SEC + 1)
#define SECOND_FAT_FIRST_SEC		(FIRST_FAT_FIRST_SEC + FAT_SECS)
#define ROOT_DIR_FIRST_SEC			(SECOND_FAT_FIRST_SEC + FAT_SECS)
#define DATA_FIRST_SEC				(ROOT_DIR_FIRST_SEC + ROOT_DIR_SECS)
#define NUM_CARD_SECS				(DATA_FIRST_SEC + SEC_PER_CLUS * (NUM_CLUSTERS - 2))
#define NUM_PART_SECS				(NUM_CARD_SECS - PBR_SEC)
#define PALM_DIR_CLUS				2
#define LAUNCHER_DIR_1ST_CLUS		3
#define NUM_LAUNCHER_DIR_ENTRIES	(DmNumDatabases(0) * DIR_ENTRIES_PER_DB)
#define NUM_LAUNCHER_DIR_SECTORS	(1 + (NUM_LAUNCHER_DIR_ENTRIES + DIR_ENTRIES_PER_SEC - 1) / DIR_ENTRIES_PER_SEC)		//1 is for initial sector that has just the mandatory dot entries
#define NUM_LAUNCHER_DIR_CLUSTERS	((NUM_LAUNCHER_DIR_SECTORS + SEC_PER_CLUS - 1) / SEC_PER_CLUS)


struct PerDbInfo {
	uint32_t size;					//0 if not yet cached
	uint16_t cluster;
	char name[dmDBNameLength + 1];
	bool isRom, isResDb;
};

#define CHILD_IDX_APPINFO			(-1)
#define CHILD_IDX_SORTINFO			(-2)
#define CHILD_IDX_HEADER			(-3)


struct HostFs {
	
	//cache for speed
	uint16_t curCurDbIdx, numChildren;
	DmOpenRef db;			//NULL if none
	int32_t curChildIdx;	//appinfo is -1, sortinfo is -2, header is -3
	uint32_t curFileOffset, curChildOffset, appInfoSz, sortInfoSz;
	
	
	//info
	
	uint16_t firstFreeClus;
	struct PerDbInfo info[];
};

struct Part {
	uint8_t active;
	uint8_t chsStart[3];
	uint8_t partType;
	uint8_t chsEnd[3];
	uint32_t lbaStart;
	uint32_t lbaLen;
} __attribute__((packed));

struct Mbr {
	uint8_t code[0x1be];
	struct Part parts[4];
	uint8_t sig[2];
} __attribute__((packed));

struct Pbr {
	uint8_t jump[3];
	char oemName[8];
	uint16_t bytesPerSec;
	uint8_t secPerClus;
	uint16_t numReservedSecs;
	uint8_t numFats;
	uint16_t numRootDirEntries;
	uint16_t totalSecs16;
	uint8_t diskType;
	uint16_t secsPerFat;
	uint16_t secsPerTrack, headsPerDisk;	//fake CHS geometry
	uint32_t partFirstSec;
	uint32_t totalSecs32;
	uint8_t physicalDriveNo;
	uint8_t rfu0;
	uint8_t extendedBootSig;
	uint32_t volumeSerialNo;
	char label[11];
	char fsType[8];
} __attribute__((packed));

struct DirEntry {
	union {
		struct {
			union {
				struct {
					char name[8];
					char ext[3];
				} __attribute__((packed));
				char nameAndExt[11];
			} __attribute__((packed));
			uint8_t attrs;
			uint8_t rfu[14];
			uint16_t clusNo;
			uint32_t size;
		} __attribute__((packed));
		struct {
			uint8_t seq;
			uint16_t name1[5];
			uint8_t attrs;
			uint8_t rfu0;
			uint8_t csum;
			uint16_t name2[6];
			uint8_t rfu1[2];
			uint16_t name3[2];
		} __attribute__((packed)) lfn;
	} __attribute__((packed));
} __attribute__((packed));

struct DirSector {
	struct DirEntry entries[16];
};

struct PRCFILE_RsrcEntry {						//as it is on host filesystem. this is BIG ENDIAN and packed

	uint32_t					type; 					// 0x00
	uint16_t					id; 					// 0x04
	uint32_t					ofstInFile; 			// 0x06

} __attribute__((packed));

struct PRCFILE_RecEntry {						//as it is on host filesystem. this is BIG ENDIAN and packed

	uint32_t					ofstInFile; 			// 0x00
	uint8_t						attrs; 					// 0x04
	uint8_t						uniqId[3]; 				// 0x05		ACTUALLY stored LE

} __attribute__((packed));

struct PRCFILE_RecordList {						//as it is on host filesystem. this is BIG ENDIAN and packed

	uint32_t					nextRecordListID_ofst;	// 0x00
	uint16_t					numRecords;				// 0x04
	
	//struct PRCFILE_RsrcEntry	res[];					// 0x06
	// OR
	//struct PRCFILE_RecEntry	rec[];					// 0x06
} __attribute__((packed));

struct PRCFILE_DatabaseHdrType {				//as it is on host filesystem. this is BIG ENDIAN and packed

	char						name[32];				// 0x00 
	uint16_t					attributes;				// 0x20
	uint16_t					version;				// 0x22
	uint32_t					creationDate;			// 0x24
	uint32_t					modificationDate;		// 0x28
	uint32_t					lastBackupDate;			// 0x2C
	uint32_t					modificationNumber;		// 0x30
	uint32_t					appInfoID_offset;		// 0x34
	uint32_t					sortInfoID_offset;		// 0x38
	uint32_t					type;					// 0x3C
	uint32_t					creator;				// 0x40
	uint32_t					uniqueIDSeed;			// 0x44
	struct PRCFILE_RecordList	recordList;				// 0x48

} __attribute__((packed));

static uint16_t hostfsPrvPalmLatinToUCS(uint8_t chr)	//also removes invalid chars
{
	static const uint16_t thirdQuarterChars[] = {
		0x0402, 0x0403, 0x201A, 0x0453, 0x201E, 0x2026, 0x2020, 0x2021, 0x20AC, 0x2030, 0x0409, 0x2039, 0x040A, 0x040C, 0x040B, 0x040F,
		0x0452, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, 0xFFFD, 0x2122, 0x0459, 0x203A, 0x045A, 0x045C, 0x045B, 0x045F,
		0x00A0, 0x040E, 0x045E, 0x0408, 0x00A4, 0x0490, 0x00A6, 0x00A7, 0x0401, 0x00A9, 0x0404, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x0407,
		0x00B0, 0x00B1, 0x0406, 0x0456, 0x0491, 0x00B5, 0x00B6, 0x00B7, 0x0451, 0x2116, 0x0454, 0x00BB, 0x0458, 0x0405, 0x0455, 0x0457,
	};
	
	if (chr < 32 || chr == '"' || chr == '*' || chr == '/' || chr == ':' || chr == '<' || chr == '>' || chr == '?' || chr == '\\' || chr == '|' || chr == 0x7f)
		return '_';
	else if (chr >= 0xc0)
		return 0x0410 + chr;
	else if (chr >= 0x80)
		return thirdQuarterChars[chr];
	else
		return chr;
}

static void hostfsPrvGenerateLfnEntry(struct DirEntry *dst, uint8_t charIdx, uint16_t charVal)
{
	while (charIdx >= CHARS_PER_LFN_ENTRY) {
		charIdx -= CHARS_PER_LFN_ENTRY;
		dst--;
	}
	
	if (charIdx < sizeof(dst->lfn.name1) / sizeof(*dst->lfn.name1)) {
		
		dst->lfn.name1[charIdx] = charVal;
		return;
	}
	charIdx -= sizeof(dst->lfn.name1) / sizeof(*dst->lfn.name1);
	
	if (charIdx < sizeof(dst->lfn.name2) / sizeof(*dst->lfn.name2)) {
		
		dst->lfn.name2[charIdx] = charVal;
		return;
	}
	charIdx -= sizeof(dst->lfn.name2) / sizeof(*dst->lfn.name2);
	
	dst->lfn.name3[charIdx] = charVal;
}

static void hostfsPrvGenerateLauncherDirSec(struct HostFs *hfs, void *dst, uint16_t secNo)
{
	struct DirSector *sec = (struct DirSector*)dst;
	uint8_t i, j, k;

	if (secNo-- == 0) {
		
		static const struct DirEntry firstLauncherDirSecEntries[] = {
			[0] = {
				.name = ".       ",
				.ext = "   ",
				.attrs = ATTR_DIRECTORY,
				.clusNo = __builtin_bswap16(LAUNCHER_DIR_1ST_CLUS),
			},
			[1] = {
				.name = "..      ",
				.ext = "   ",
				.attrs = ATTR_DIRECTORY,
				.clusNo = __builtin_bswap16(PALM_DIR_CLUS),
			},
		};
		
		MemSet(sec, SECTOR_SIZE, 0);
		for (i = 0; i < sizeof(sec->entries) / sizeof(*sec->entries); i++)
			sec->entries[i].name[0] = 0xE5;
		MemMove(sec, &firstLauncherDirSecEntries, sizeof(firstLauncherDirSecEntries));
	}
	else {	//secNo is now 0-based
		
		uint16_t dbIdx = secNo * DBS_PER_DIR_SECTOR;
		struct DirEntry *dst = sec->entries;
		
		for (i = 0; i < DIR_ENTRIES_PER_SEC; i += DIR_ENTRIES_PER_DB, dbIdx++) {
			
			//clear all the entries
			for (j = 0; j < DIR_ENTRIES_PER_DB; j++) {
				
				MemSet(&dst[j], sizeof(dst[j]), 0);
				dst[j].name[0] = 0xe5;
			}
			
			if (dbIdx < DmNumDatabases(0)) {		//have DB? record it
				
				const struct PerDbInfo *nfo = &hfs->info[dbIdx];
				UInt8 csum = 0;
				
				//make SFN entry
				StrIToH(dst[LFN_ENTRIES_PER_DB].name, dbIdx);	//precisely 8 chars
				dst[LFN_ENTRIES_PER_DB].ext[0] = 'P';
				if (nfo->isResDb) {
					dst[LFN_ENTRIES_PER_DB].ext[1] = 'R';
					dst[LFN_ENTRIES_PER_DB].ext[2] = 'C';
				}
				else  {
					dst[LFN_ENTRIES_PER_DB].ext[1] = 'D';
					dst[LFN_ENTRIES_PER_DB].ext[2] = 'B';
				}
				dst[LFN_ENTRIES_PER_DB].clusNo = __builtin_bswap16(nfo->cluster);
				dst[LFN_ENTRIES_PER_DB].size = __builtin_bswap32(nfo->size);
				
				//calc csum
				for (j = 0; j < sizeof(dst[LFN_ENTRIES_PER_DB].nameAndExt); j++)
					csum = (csum >> 1) + ((csum & 1) ? 0x80 : 0x00) + (uint8_t)dst[LFN_ENTRIES_PER_DB].nameAndExt[j];
				
				//make LFN entries
				j = 0;
				hostfsPrvGenerateLfnEntry(&dst[LFN_ENTRIES_PER_DB - 1], j++, 'R');
				hostfsPrvGenerateLfnEntry(&dst[LFN_ENTRIES_PER_DB - 1], j++, nfo->isRom ? 'O' : 'A');
				hostfsPrvGenerateLfnEntry(&dst[LFN_ENTRIES_PER_DB - 1], j++, 'M');
				hostfsPrvGenerateLfnEntry(&dst[LFN_ENTRIES_PER_DB - 1], j++, '-');
				for (k = 0; nfo->name[k]; k++)
					hostfsPrvGenerateLfnEntry(&dst[LFN_ENTRIES_PER_DB - 1], j++, hostfsPrvPalmLatinToUCS(nfo->name[k]));
				if (j % CHARS_PER_LFN_ENTRY)
					hostfsPrvGenerateLfnEntry(&dst[LFN_ENTRIES_PER_DB - 1], j++, 0);
				while (j % CHARS_PER_LFN_ENTRY)
					hostfsPrvGenerateLfnEntry(&dst[LFN_ENTRIES_PER_DB - 1], j++, 0xffff);
				
				//finish filling them
				k = (j - 1) / CHARS_PER_LFN_ENTRY;
				for (j = 0; j < k; j++) {
					dst[j].lfn.csum = csum;
					dst[j].lfn.attrs = ATTR_LFN;
					dst[j].lfn.seq = (i ? 0 : 0x40) + k - j;
				}
			}
		}
	}
}

static uint32_t hostfsPrvOutputGeneratedData(uint8_t *dst, const void *srcDataP, uint32_t srcDataSz, uint32_t filePosDataStarts, uint32_t filePosWantedToOutput, uint32_t maxBytesOutput)
{
	uint32_t bytesIntoData = filePosWantedToOutput - filePosDataStarts;
	
	const uint8_t *srcData = srcDataP;
	
	//data is not allowed in the future
	if (filePosDataStarts > filePosWantedToOutput)
		SysFatalAlert("non-fast-forward");
	
	//if the entire data is before the desired start, we can output nothing
	if (bytesIntoData >= srcDataSz)
		return 0;
	
	//skip the part of data that is in the past
	srcData += bytesIntoData;
	srcDataSz -= bytesIntoData;
	
	if (srcDataSz > maxBytesOutput)
		srcDataSz = maxBytesOutput;
	
	MemMove(dst, srcData, srcDataSz);
	return srcDataSz;
}

static void hostfsPrvGenerateDataSecEx(struct HostFs *hfs, const struct PerDbInfo *dbi, uint16_t dbIdx, uint8_t *dst)
{
	uint16_t outSpaceAvail = SECTOR_SIZE, bytesUsed;
	
	while (outSpaceAvail) {
		
		if (hfs->curChildIdx == CHILD_IDX_HEADER) {
			
			uint32_t childListSz = (dbi->isResDb ? sizeof(struct PRCFILE_RsrcEntry) : sizeof(struct PRCFILE_RecEntry)) * hfs->numChildren, hdrSz = sizeof(struct PRCFILE_DatabaseHdrType) + childListSz;
			
			if (hfs->curFileOffset < sizeof(struct PRCFILE_DatabaseHdrType)) {
				
				uint32_t crDate, modDate, bkpDate, modNum, type, crid;
				struct PRCFILE_DatabaseHdrType hdr;
				uint16_t attrs, version;
				
				
				DmDatabaseInfo(0, DmGetDatabase(0, dbIdx), hdr.name, &attrs, &version, &crDate, &modDate, &bkpDate, &modNum, NULL, NULL, &type, &crid);
				hdr.attributes = __builtin_bswap16(attrs);
				hdr.version = __builtin_bswap16(version);
				hdr.creationDate = __builtin_bswap32(crDate);
				hdr.modificationDate = __builtin_bswap32(modDate);
				hdr.lastBackupDate = __builtin_bswap32(bkpDate);
				hdr.modificationNumber = __builtin_bswap32(modNum);
				hdr.appInfoID_offset = __builtin_bswap32(hfs->appInfoSz ? hdrSz : 0);
				hdr.sortInfoID_offset = __builtin_bswap32(hfs->sortInfoSz ? hdrSz + hfs->appInfoSz : 0);
				hdr.type = __builtin_bswap32(type);
				hdr.creator = __builtin_bswap32(crid);
				hdr.uniqueIDSeed = 0;
				
				bytesUsed = hostfsPrvOutputGeneratedData(dst, &hdr, sizeof(hdr), 0, hfs->curFileOffset, outSpaceAvail);
				dst += bytesUsed;
				hfs->curFileOffset += bytesUsed;
				hfs->curChildOffset += bytesUsed;
				outSpaceAvail -= bytesUsed;
				continue;
			}
			if (hfs->curFileOffset < hdrSz) {
				
				uint32_t fileDataLoc = hdrSz + hfs->appInfoSz + hfs->sortInfoSz, fileOutputPos = sizeof(struct PRCFILE_DatabaseHdrType);
				uint16_t childIdx = 0;
				
				for (childIdx = 0; childIdx < hfs->numChildren && outSpaceAvail; childIdx++) {
					
					LocalID chunk;
					MemHandle mh;
					union {
						struct PRCFILE_RecEntry rec;
						struct PRCFILE_RsrcEntry res;
					} data;
					UInt16 entrySz;
					
					if (dbi->isResDb) {
						
						UInt32 resType;
						UInt16 resID;
						
						DmResourceInfo(hfs->db, childIdx, &resType, &resID, &chunk);
						data.res.type = __builtin_bswap32(resType);
						data.res.id = __builtin_bswap16(resID);
						data.res.ofstInFile = __builtin_bswap32(chunk ? fileDataLoc : 0);
						entrySz = sizeof(struct PRCFILE_RsrcEntry);
					}
					else {
						
						UInt16 attr;
						UInt32 uniqID;
						
						DmRecordInfo(hfs->db, childIdx, &attr, &uniqID, &chunk);
						data.rec.ofstInFile = __builtin_bswap32(chunk ? fileDataLoc : 0);
						data.rec.attrs = attr;
						data.rec.uniqId[0] = uniqID >> 16;
						data.rec.uniqId[1] = uniqID >> 8;
						data.rec.uniqId[2] = uniqID;
						entrySz = sizeof(struct PRCFILE_RecEntry);
					}
					
					if (chunk)
						fileDataLoc += MemPtrSize(MemLocalIDToGlobal(0, chunk));
					
					bytesUsed = hostfsPrvOutputGeneratedData(dst, &data, entrySz, fileOutputPos, hfs->curFileOffset, outSpaceAvail);
					dst += bytesUsed;
					hfs->curFileOffset += bytesUsed;
					hfs->curChildOffset += bytesUsed;
					outSpaceAvail -= bytesUsed;
					
					fileOutputPos += entrySz;
				}
				continue;
			}
			//if we got here, we are done with the header
			if (hfs->appInfoSz)
				hfs->curChildIdx = CHILD_IDX_APPINFO;
			else if (hfs->sortInfoSz)
				hfs->curChildIdx = CHILD_IDX_SORTINFO;
			else
				hfs->curChildIdx = 0;
			hfs->curChildOffset = 0;
			continue;
		}
		
		if (hfs->curChildIdx == CHILD_IDX_APPINFO || hfs->curChildIdx == CHILD_IDX_SORTINFO) {
			
			uint32_t chunkStartFilePos = hfs->curFileOffset - hfs->curChildOffset, chunkSz;
			LocalID chunk;
			void *ptr;
			
			DmDatabaseInfo(0, DmGetDatabase(0, dbIdx), NULL, NULL, NULL, NULL, NULL, NULL, NULL, hfs->curChildIdx == CHILD_IDX_APPINFO ? &chunk : NULL, hfs->curChildIdx == CHILD_IDX_APPINFO ? NULL : &chunk, NULL, NULL);
			ptr = MemLocalIDToGlobal(0, chunk);
			chunkSz = MemPtrSize(ptr);
			
			bytesUsed = hostfsPrvOutputGeneratedData(dst, ptr, chunkSz, chunkStartFilePos, hfs->curFileOffset, outSpaceAvail);
			dst += bytesUsed;
			hfs->curFileOffset += bytesUsed;
			hfs->curChildOffset += bytesUsed;
			outSpaceAvail -= bytesUsed;
			
			if (hfs->curChildOffset == chunkSz) {
				
				if (hfs->curChildIdx == CHILD_IDX_APPINFO && hfs->sortInfoSz)
					hfs->curChildIdx = CHILD_IDX_SORTINFO;
				else
					hfs->curChildIdx = 0;
				hfs->curChildOffset = 0;
			}
			continue;
		}
		
		//data chunk
		if (hfs->curChildIdx < hfs->numChildren) {
			
			uint32_t chunkStartFilePos = hfs->curFileOffset - hfs->curChildOffset, chunkSz;
			MemHandle mh;
			
			
			if (dbi->isResDb)
				mh = DmGetResourceIndex(hfs->db, hfs->curChildIdx);
			else
				mh = DmQueryRecord(hfs->db, hfs->curChildIdx);
			
			if (!mh)
				chunkSz = 0;
			else {
	
				chunkSz = MemHandleSize(mh);
			
				bytesUsed = hostfsPrvOutputGeneratedData(dst, MemHandleLock(mh), chunkSz, chunkStartFilePos, hfs->curFileOffset, outSpaceAvail);
				MemHandleUnlock(mh);
				if (dbi->isResDb)
					DmReleaseResource(mh);
				
				dst += bytesUsed;
				hfs->curFileOffset += bytesUsed;
				hfs->curChildOffset += bytesUsed;
				outSpaceAvail -= bytesUsed;
			}
			
			if (hfs->curChildOffset == chunkSz) {
				
				hfs->curChildIdx++;
				hfs->curChildOffset = 0;
			}
			continue;
		}
		
		//end of file
		MemSet(dst, outSpaceAvail, 0);
		outSpaceAvail = 0;
		continue;
	}
}

static void hostfsPrvGenerateDataSecFromDatabase(struct HostFs *hfs, void *dstP, uint16_t dbIdx, uint16_t sec)
{
	const struct PerDbInfo *dbi = &hfs->info[dbIdx];
	LocalID LID = DmGetDatabase(0, dbIdx);
	
	if (hfs->db && hfs->curCurDbIdx != dbIdx) {
		DmCloseDatabase(hfs->db);
		hfs->db = NULL;
	}
	
	if (!hfs->db) {
		
		LocalID appInfoLID, sortInfoLID;
		
		hfs->db = DmOpenDatabase(0, LID, dmModeReadOnly | (dbi->isResDb ? 0 : dmModeShowSecret));
		hfs->curCurDbIdx = dbIdx;
		hfs->curFileOffset = 0;
		hfs->curChildIdx = CHILD_IDX_HEADER;
		hfs->curChildOffset = 0;
		hfs->numChildren = dbi->isResDb ? DmNumResources(hfs->db) : DmNumRecords(hfs->db);
		
		DmDatabaseInfo(0, LID, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &appInfoLID, &sortInfoLID, NULL, NULL);
		hfs->appInfoSz = appInfoLID ? MemPtrSize(MemLocalIDToGlobal(0, appInfoLID)) : 0;
		hfs->sortInfoSz = sortInfoLID ? MemPtrSize(MemLocalIDToGlobal(0, sortInfoLID)) : 0;
	}
	
	if (hfs->curFileOffset > SECTOR_SIZE * sec) {			//no rewind, so we go to the start
		
		hfs->curFileOffset = 0;
		hfs->curChildIdx = CHILD_IDX_HEADER;
		hfs->curChildOffset = 0;
	}
	
	hostfsPrvGenerateDataSecEx(hfs, dbi, dbIdx, dstP);
}

static void hostfsPrvGenerateDataSec(struct HostFs *hfs, void *dst, uint16_t clusNo, uint8_t secInClus)
{
	if (clusNo >= LAUNCHER_DIR_1ST_CLUS && clusNo - LAUNCHER_DIR_1ST_CLUS < NUM_LAUNCHER_DIR_CLUSTERS)
		hostfsPrvGenerateLauncherDirSec(hfs, dst, (clusNo - LAUNCHER_DIR_1ST_CLUS) * SEC_PER_CLUS + secInClus);
	else if (clusNo >= hfs->firstFreeClus || clusNo < hfs->info[0].cluster)
		MemSet(dst, SECTOR_SIZE, 0);
	else {
		UInt16 i;
		
		for (i = 1; i < DmNumDatabases(0) && clusNo < hfs->info[i].cluster; i++);
		i--;
		
		hostfsPrvGenerateDataSecFromDatabase(hfs, dst, i, (clusNo - hfs->info[i].cluster) * SEC_PER_CLUS + secInClus);
	}
}

static void hostfsPrvGenerateFatSec(struct HostFs *hfs, void *dst, uint16_t fatSecNo)
{
	if (fatSecNo >= FAT_SECS)
		MemSet(dst, SECTOR_SIZE, 0);
	else {
		
		uint16_t i, fatEntriesPerSec = SECTOR_SIZE / sizeof(uint16_t), curClus = fatSecNo / fatEntriesPerSec, *dstP = dst;
		
		for (i = 0; i < fatEntriesPerSec; i++, curClus++) {
			
			uint16_t firstClusPastLauncherDir = DmNumDatabases(0) ? hfs->info[0].cluster : hfs->firstFreeClus;
			uint16_t val = 0xffff;		//EOC by default
			
			if (curClus == 0)								//clus 0 is spwcial
				val = 0xfff8;
			
			//all our chains are simple - they go to the next cluster or end.
			else if (curClus == 1 || curClus == PALM_DIR_CLUS)	//clus 1 and palm directory always end immediately
				val = 0xffff;
			
			//"launcher" dir may have a chain we can tell by comparing current cluster with before-first-data-cluster-of-first-file
			else if (curClus >= LAUNCHER_DIR_1ST_CLUS && curClus < firstClusPastLauncherDir - 1)
				val = curClus + 1;
			
			//launcher dir last clus needs eoc
			else if (curClus == firstClusPastLauncherDir - 1)
				val = 0xffff;
			
			//unused clusters are free
			else if (curClus < firstClusPastLauncherDir || curClus >= hfs->firstFreeClus)
				val = 0x0000;
			
			//data chains. every cluster goes to next except if next starts a new file
			else {
				uint16_t j, num = DmNumDatabases(0);
				
				for (j = 0; j < num && curClus != hfs->info[j].cluster - 1; j++);
				
				if (j == num && j != hfs->firstFreeClus - 1)
					val = curClus + 1;
				else
					val = 0xffff;
			}
			
			*dstP = __builtin_bswap16(val);
		}
	}
}

void hostfsReadSec(struct HostFs *hfs, void *dstP, uint32_t secNo)
{
	if (secNo == 0) {
		
		static const struct Part part = {
			.active = 0x80,
			.partType = (NUM_PART_SECS < 65536) ? 0x04 : 0x006,
			.lbaStart = __builtin_bswap32(PBR_SEC),
			.lbaLen = __builtin_bswap32(NUM_PART_SECS),
		};
		struct Mbr *dst = (struct Mbr*)dstP;
		
		MemSet(dstP, SECTOR_SIZE, 0);
		
		dst->code[0] = 0xe9;
		dst->code[1] = 0x10;
		dst->code[2] = 0x90;
		dst->parts[0] = part;
		dst->sig[0] = 0x55;
		dst->sig[1] = 0xaa;
	}
	else if (secNo == 1) {
		
		static const struct Pbr pbr = {
			.jump = {0xe9, 0x10, 0x90},
			.oemName = "MSWIN4.1",
			.bytesPerSec = __builtin_bswap16(SECTOR_SIZE),
			.secPerClus = SEC_PER_CLUS,
			.numReservedSecs = __builtin_bswap16(FIRST_FAT_FIRST_SEC - PBR_SEC),
			.numFats = 2,
			.numRootDirEntries = __builtin_bswap16(ROOT_DIR_ENTRIES),
			.totalSecs16 = __builtin_bswap16((NUM_PART_SECS >= 65536) ? 0 : NUM_PART_SECS),
			.diskType = 0xf8,
			.secsPerFat = __builtin_bswap16(FAT_SECS),
			.secsPerTrack = __builtin_bswap16(1),
			.headsPerDisk = __builtin_bswap16(1),
			.partFirstSec = __builtin_bswap32(PBR_SEC),
			.totalSecs32 = __builtin_bswap32((NUM_PART_SECS >= 65536) ? NUM_PART_SECS : 0),
			.physicalDriveNo = 0,
			.rfu0 = 0,
			.extendedBootSig = 0x29,
			.volumeSerialNo = __builtin_bswap32(1234567890),
			.label = "HOSTFS     ",
			.fsType = "FAT16   ",
		};
		
		uint8_t *dst = dstP;
		MemSet(dstP, SECTOR_SIZE, 0);
		
		*(struct Pbr*)dstP = pbr;
		dst[SECTOR_SIZE - 2] = 0x55;
		dst[SECTOR_SIZE - 1] = 0xaa;
	}
	else if (secNo >= FIRST_FAT_FIRST_SEC && secNo - FIRST_FAT_FIRST_SEC < FAT_SECS) {
		
		hostfsPrvGenerateFatSec(hfs, dstP, secNo - FIRST_FAT_FIRST_SEC);
	}
	else if (secNo >= SECOND_FAT_FIRST_SEC && secNo - SECOND_FAT_FIRST_SEC < FAT_SECS) {
		
		hostfsPrvGenerateFatSec(hfs, dstP, secNo - SECOND_FAT_FIRST_SEC);
	}
	else if (secNo == ROOT_DIR_FIRST_SEC) {
		
		static const struct DirEntry rootDirEntries[] = {
			{
				.name = "HOSTFS  ",
				.ext = "   ",
				.attrs = ATTR_VOL_LBL,
			},
			{
				.name = "PALM    ",
				.ext = "   ",
				.attrs = ATTR_DIRECTORY,
				.clusNo = __builtin_bswap16(PALM_DIR_CLUS),
			},
		};
		struct DirSector *dst = (struct DirSector*)dstP;
		
		MemSet(dstP, SECTOR_SIZE, 0);
		
		dst->entries[0] = rootDirEntries[0];
		dst->entries[1] = rootDirEntries[1];
	}
	else if (secNo >= DATA_FIRST_SEC && secNo - DATA_FIRST_SEC < NUM_CARD_SECS - DATA_FIRST_SEC) {
		
		hostfsPrvGenerateDataSec(hfs, dstP, (secNo - DATA_FIRST_SEC) / SEC_PER_CLUS, (secNo - DATA_FIRST_SEC) % SEC_PER_CLUS);
	}
	else {
		
		MemSet(dstP, SECTOR_SIZE, 0);
	}
}

struct HostFs* hostfsInit(void)
{
	uint16_t i, j, num, nextClus = LAUNCHER_DIR_1ST_CLUS + NUM_LAUNCHER_DIR_CLUSTERS;
	struct HostFs *hfs;
	uint32_t size;
	
	size = sizeof(struct HostFs) + sizeof(*hfs->info) * DmNumDatabases(0);
	hfs = MemPtrNew(size);
	
	if (!hfs)
		return NULL;
	
	MemSet(hfs, size, 0);
	
	for (i = 0, num = DmNumDatabases(0); i < num; i++) {
	
		LocalID LID = DmGetDatabase(0, i), appInfoLID, sortInfoLID;
		uint16_t attrs, num = 0, j;
		MemHandle mh;
		DmOpenRef db;
		Err e;
		
		size = sizeof(struct PRCFILE_DatabaseHdrType);
		hfs->info[i].name[sizeof(hfs->info[i].name) - 1] = 0;
		if (errNone == DmDatabaseInfo(0, LID, hfs->info[i].name, &attrs, NULL, NULL, NULL, NULL, NULL, &appInfoLID, &sortInfoLID, NULL, NULL) && (db = DmOpenDatabase(0, LID, dmModeReadOnly | ((attrs & dmHdrAttrResDB) ? 0 : dmModeShowSecret))) != NULL) {
			
			if (attrs & dmHdrAttrResDB) {
				
				hfs->info[i].isResDb = true;
				for (j = 0, num = DmNumResources(db); j < num; i++) {
					
					size += sizeof(struct PRCFILE_RsrcEntry);
					mh = DmGetResourceIndex(db, j);
					if (mh) {
						size += MemHandleSize(mh);
						DmReleaseResource(mh);
					}
				}
			}
			else {
			
				hfs->info[i].isResDb = false;
				for (j = 0, num = DmNumRecords(db); j < num; j++) {
					
					size += sizeof(struct PRCFILE_RecEntry);
					mh = DmQueryRecord(db, j);
					if (mh)
						size += MemHandleSize(mh);
				}
			}
			
			if (appInfoLID)
				size += MemPtrSize(MemLocalIDToGlobal(0, appInfoLID));
			if (sortInfoLID)
				size += MemPtrSize(MemLocalIDToGlobal(0, sortInfoLID));
			
			DmCloseDatabase(db);
			
			e = DmDatabaseProtect(0, LID, true);
			if (e == dmErrROMBased)
				hfs->info[i].isRom = true;
			else if (e == errNone) {
				hfs->info[i].isRom = false;
				(void)DmDatabaseProtect(0, LID, false);
			}
		}
		hfs->info[i].cluster = nextClus;
		nextClus += (size + SECTOR_SIZE * SEC_PER_CLUS - 1) / (SECTOR_SIZE * SEC_PER_CLUS);
	}
	hfs->firstFreeClus = nextClus;
	
	return hfs;
}

void hostfsDeinit(struct HostFs *hfs)
{
	MemPtrFree(hfs);
}