#include <ExpansionMgr.h>
#include <SlotDrvrLib.h>
#include <PalmOS.h>
#include "nandPublic.h"

typedef unsigned long long UInt64;
typedef signed long long Int64;

static Boolean findSlotAndLib(UInt16 *libRefP, UInt16 *slotRefP)
{
	UInt32 slotIter = expIteratorStart;
	UInt16 slotRef, libRef;
	
	while (slotIter != expIteratorStop && errNone == ExpSlotEnumerate(&slotRef, &slotIter)) {
		
		ExpCardInfoType ci;
		
		if (errNone != ExpCardInfo(slotRef, &ci))
			continue;
		
		if (StrCompare(ci.deviceClassStr, NAND_SLOT_DEVICE_CLASS))
			continue;
		
		if (errNone != ExpSlotLibFind(slotRef, &libRef))
			continue;
		
		if (libRefP)
			*libRefP = libRef;
		
		if (slotRefP)
			*slotRefP = slotRef;
		
		return true;
	}
	return false;
}

static UInt64 getNandInfo(UInt16 slotLibRef, UInt16 slotRef, enum FtlDataType which)
{
	struct NandGetInfoRequest req;
	UInt16 len = sizeof(req);
	Err e;
	
	req.slotRefNum = slotRef;
	req.infoType = which;
	
	e = SlotCustomControl(slotLibRef, NAND_SLOT_CUSTOM_CONTROL_API_CRID, NAND_SLOT_CUSTOM_CONTROL_SEL_GET_INFO, &req, &len);
	if (e != errNone)
		return 0;
	
	if (len != sizeof(req))
		return 0;
	
	return req.value;
}

static UInt8 div10(UInt64 *vP)
{
	UInt64 nv = *vP, dv = 0xa000000000000000ULL, rv = 0, mv = 0x1000000000000000ULL;
	Int8 i, j;
	
	for (i = 0; i <= 60; i++) {
		
		if (nv >= dv) {
			nv -= dv;
			rv |= mv;
		}
		mv >>= 1;
		dv >>= 1;
	}
	
	i = nv;
	*vP = rv;
	
	return i;
}


static void StrULLToA(char *dst, UInt64 v)
{
	UInt8 i, pos = 0;
	char val[24];
	
	val[pos++] = 0;
	
	do {
		val[pos++] = '0' + div10(&v);
	} while (v);		//avoid long long
	
	for (i = 0; i < pos; i++)
		dst[pos - i - 1] = val[i];
}

static void setValBasic(FormPtr fp, UInt16 lblId, UInt16 slotLibRef, UInt16 slotRef, enum FtlDataType which)
{
	char x[32];
	
	StrULLToA(x, getNandInfo(slotLibRef, slotRef, which));
	
	FrmCopyLabel(fp, lblId, x);
}

static void setValPermille(FormPtr fp, UInt16 lblId, UInt16 slotLibRef, UInt16 slotRef, enum FtlDataType which)
{
	UInt32 val = getNandInfo(slotLibRef, slotRef, which);
	UInt16 integer, frac;
	char x[32];
	
	//contort ourselves into weird shapes to not need long division
	integer = (UInt16)(val / 16) / (UInt16)625;
	frac = val - (UInt32)integer * (UInt32)10000;
	
	StrPrintF(x, "%u.%02u%%", integer, frac / (UInt16)100);
	
	FrmCopyLabel(fp, lblId, x);
}

static void setValGeometry(FormPtr fp, UInt16 lblId, UInt16 slotLibRef, UInt16 slotRef, enum FtlDataType which)
{
	UInt64 val = getNandInfo(slotLibRef, slotRef, which);
	UInt16 pageSz = val, pagesPerBlock = val >> 16;
	UInt32 numBlocks = val >> 32;
	char x[32];
	
	if (pagesPerBlock)
		StrPrintF(x, "%uB x %u x %lu", pageSz, pagesPerBlock, numBlocks);
	else
		StrPrintF(x, "%uB x %lu", pageSz, numBlocks);
	
	FrmCopyLabel(fp, lblId, x);
}

static void ui(void)
{
	UInt16 slotLibRef, slotRef, i;
	EventType e;
	FormPtr fp;
	
	if (!findSlotAndLib(&slotLibRef, &slotRef)) {
//		ErrAlertCustom(0, "Slot not found", NULL, NULL);
		return;
	}
	
	if (!getNandInfo(slotLibRef, slotRef, FtlVersion)) {
//		ErrAlertCustom(0, "Cannot communicate with the NAND driver", NULL, NULL);
		return;
	}
	
	fp = FrmInitForm(1000);
	setValBasic(fp, 8000, slotLibRef, slotRef, FtlVersion);
	setValGeometry(fp, 8001, slotLibRef, slotRef, FtlNandGeometry);
	setValGeometry(fp, 8002, slotLibRef, slotRef, FtlGeometry);
	
	setValBasic(fp, 8010, slotLibRef, slotRef, FtlLifetimeSectorsRead);
	setValBasic(fp, 8011, slotLibRef, slotRef, FtlLifetimeSectorsWritten);
	setValBasic(fp, 8012, slotLibRef, slotRef, FtlLifetimeSectorsTrimmed);
	setValBasic(fp, 8013, slotLibRef, slotRef, FtlLifetimeSectorsRelocated);
	
	setValBasic(fp, 8020, slotLibRef, slotRef, FtlLifetimePagesWritten);
	setValBasic(fp, 8021, slotLibRef, slotRef, FtlLifetimeBlocksErased);
	
	setValPermille(fp, 8030, slotLibRef, slotRef, FtlAverageBlockCleanliness);
	setValPermille(fp, 8031, slotLibRef, slotRef, FtlAverageBlockWear);
	setValBasic(fp, 8032, slotLibRef, slotRef, FtlNumBadBlocks);
	setValPermille(fp, 8033, slotLibRef, slotRef, FtlGetLifetimeLeft);
	FrmDoDialog(fp);
	FrmDeleteForm(fp);
}

UInt32
#ifdef __GNUC__
__attribute__((section(".vectors")))
#endif
__Startup__(void)
{
	MemPtr			prevGlobalsP;
	MemPtr			globalsP;
	SysAppInfoPtr	appInfoP;
	

	if (errNone != SysAppStartup(&appInfoP, &prevGlobalsP, &globalsP))
		return 0;

	if (appInfoP->cmd == sysAppLaunchCmdNormalLaunch || appInfoP->cmd == sysAppLaunchCmdPanelCalledFromApp)
		ui();

	SysAppExit(appInfoP, prevGlobalsP, globalsP);
	
	return 0;
}