/******************************************************************************
 * Copyright (c) 1995-2005 palmOne, Inc. or its subsidiaries.
 * All rights reserved.
 *****************************************************************************/
/**
 * @ingroup PIM
 */

/**
 * @file ToDoDB.h
 *
 * @brief Contains database record type and constants for Tasks application.
 *
 * Tasks application uses a different record format than the old ToDo
 * application due to some feature enhancement and new data fields. This file
 * contains the structure of the record in Tasks DB and can be used to access
 * the database directly. One way to utilize this header file is to combine it
 * with the old To Do source code so that the record packing and unpacking
 * routines are adjusted for the new structure.
 *
 * Please note that accessing the database directly is generally not recommended
 * and this is offered as a temporary solution for 3rd party developers. The
 * structure might change again in the future.
 *
 * <hr>
 */

#ifndef __TDTODOMGR_H__
#define __TDTODOMGR_H__

#include <DataMgr.h>
#include <DateTime.h>

/** PalmSource ToDo app database version at time of change. */
#define TODO_DB_VERSION_CLASSIC 0

/** palmOne new Tasks app database version. */
#define TODO_DB_VERSION_MULLET  1

/** Current database version. */
#define TODO_DB_VERSION         TODO_DB_VERSION_MULLET

/** The app info string resource ID for localized strings to replace the
 *  strings in the application info block depending on the current
 *  system locale state (e.g.: Spanish). */
#define LocalizedAppInfoStr     1000

/**
 * @name Miscellaneous
 */
/*@{*/
#define todoLabelLength         12
#define todoNumFields           16
/*@}*/

/** Maximum priority value; range is 1 - toDoMaxPriority. */
#define toDoMaxPriority         5

/** Dirty flags for to do application info. */
#define toDoSortByPriorityDirty 0x0001

/**
 * @name Database Sort Orders
 */
/*@{*/
#define soDueDatePriority       0
#define soPriorityDueDate       1
#define soCategoryPriority      2
#define soCategoryDueDate       3
/*@}*/

/** @brief Application Info Block
 *
 * This structure is used to store info applicable to all records
 * in the database, specific to the application, inter-session (like
 * preferences), etc.
 */
typedef struct {

    /** Bitfield of categories with a different name */
    UInt16          renamedCategories;

    char            categoryLabels[dmRecNumCategories][dmCategoryLength];

    UInt8           categoryUniqIDs[dmRecNumCategories];

    /**
     * Uniq IDs generated by the device are between 0 - 127.  Those
     * from the PC are 128 - 255.
     */
    UInt8           lastUniqID;

    /** From the compiler word aligning things */
    UInt8           reserved1;

    /**
     * Whether category colors were edited since last sync.
     * Least significant bit first.
     */
    UInt16          categoryColorsEdited;

    /**
     * Added as part of the Mullet version of this application so that we can
     * later support color categories without breaking the conduits.
     */
    UInt8           categoryColors[dmRecNumCategories];

    UInt16          reserved2;

    UInt16          dirtyAppInfo;

    /** The database sort order. */
    UInt8           sortOrder;

    UInt8           reserved3;

} ToDoAppInfoType;

typedef ToDoAppInfoType * ToDoAppInfoPtr;

/***********************************************************************
 *  Application Record Format Related Data - CLASSIC
 ***********************************************************************/
#if TODO_DB_VERSION == TODO_DB_VERSION_CLASSIC

/***********************************************************************
 *  Application Packed Record Format Related Data
 *
 *  Note: Records are stored in the database in packed format to
 *        conserve space.  When creating a new record, ToDoNewRecord(),
 *        or saving a record, ToDoChangeRecord(), the packing is handled
 *        for you.
 ***********************************************************************/

/**
 * @brief   Record format
 *
 * This is the packed database record format; record format as
 * stored. All strings are null terminated and appear within
 * the record.
 */
typedef struct {
    DateType dueDate;
    UInt8 priority;     /**< high bit is complete flag */
    char description;   /**< start of the task description string */
} ToDoDBRecord;

typedef ToDoDBRecord            ToDoDBRecordType;
typedef ToDoDBRecord*           ToDoDBRecordPtr;

/**
 * @brief   Get the minimum size of the packed record.
 *
 * Minimum size assumes empty 'description' and 'note' fields.
 * An extra byte is calculated into the minimum size for the
 * NULL character of a note string.
 *
 * @return  Minimum size of the packed record.
 */
#define sizeDBRecord (sizeof (ToDoDBRecord) + 1)

/***********************************************************************
 *  Application Unpacked Record Format Related Data
 ***********************************************************************/

/**
 * @brief   Record format
 *
 * This is the unpacked record format as used by the app.  All
 * pointers are either NULL or point to data within the PackedDB
 * record.  All strings are NULL character terminated.
 */
typedef struct {
    DateType            dueDate;
    UInt8               priority;       /**< high bit is complete flag */
    UInt8               reserved;
    Char *              description;
    Char *              note;
} ToDoItemType;

typedef ToDoItemType * ToDoItemPtr;

/** Indicates a task record has been completed. */
#define completeFlag    0x80

/** Bit mask for checking the priority bits of the priority member. */
#define priorityOnly    ~completeFlag

/***********************************************************************
 *  Application Record Format Related Data - NEW PIM
 ***********************************************************************/
#elif TODO_DB_VERSION == TODO_DB_VERSION_MULLET

/**
 * @brief   Record data flags
 *
 * These flags indicate if a record contains specific data
 * features. e.g.: if the dueDate flag is set, then the record
 * has due date data.
 */
typedef struct {
    unsigned    dueDate         : 1;
    unsigned    completionDate  : 1;
    unsigned    alarm           : 1;
    unsigned    repeat          : 1;
    unsigned    description     : 1;
    unsigned    note            : 1;
    unsigned    reserved        : 10;
} ToDoDBDataFlags;

/***********************************************************************
 *  Application Unpacked Record Format Related Data
 ***********************************************************************/

/**
 * @brief   Record alarm data
 *
 * Records with due dates can have an alarm set for the due
 * date, and for each day preceding the due date as specified
 * by the number of days in advance of the due date that the
 * alarm should first occur.
 */
typedef struct {
    TimeType    alarmTime;
    UInt16      alarmAdvanceDays;
} ToDoAlarmInfoType;

typedef ToDoAlarmInfoType   *ToDoAlarmInfoPtr;

#ifndef _REPEAT_TYPE_
#define _REPEAT_TYPE_

/**
 * @brief   Repeating record frequency
 *
 * This enum is used to specify the frequency of
 * repeating appointments.
 */

enum repeatTypes {
    repeatNone,
    repeatDaily,
    repeatWeekly,
    repeatMonthlyByDay,
    repeatMonthlyByDate,
    repeatYearly
};

typedef enum repeatTypes RepeatType;

/**
 * @brief   Repeating task data.
 *
 * This structure contains information about repeat tasks.
 * The repeatOn member is only used by weekly and monthly-by-day
 * repeating tasks.
 *
 * For weekly the byte is a bit field that contains the days
 * of the week the for which the task is scheduled
 * (bit: 0-sun, 1-mon, 2-tue, etc.).
 * For monthly-by-day the byte contains the day for which
 * the task is scheduled, (ex: the 3rd friday), and the
 * byte is of type DayOfMonthType.
 */

typedef struct {
    RepeatType  repeatType;         /**< daily, weekly, monthlyByDay, etc. */
    UInt8       reserved1;
    DateType    repeatEndDate;      /**< minus one if forever */
    UInt8       repeatFrequency;    /**< i.e. every 2 days if repeatType daily
    */
    UInt8       repeatOn;           /**< monthlyByDay and repeatWeekly only */
    UInt8       repeatStartOfWeek;  /**< repeatWeekly only */
    UInt8       reserved2;
} RepeatInfoType;

typedef RepeatInfoType * RepeatInfoPtr;

#endif // _REPEAT_TYPE_

/**
 * @brief   Does the task repeat on only one day per week.
 *
 * The form (x & (x - 1)) masks out the lowest order bit
 * in x.  (K&R, p. 51)
 * If only one bit is set, which occurs iff the task is only
 * once per week, then (x & (x - 1)) == 0.
 *
 * @param R A repeat info type struct.
 *
 * @retval  True if the task repeats only once per week.
 */
#define OnlyRepeatsOncePerWeek(R)  (((R).repeatOn & ((R).repeatOn - 1)) == 0)

/**
 * @brief   Does the task repeat on the given day of the week.
 *
 * @param R Ptr to a repeat info type struct.
 * @param D The day of the week to check.
 *              For weekly repeating tasks, this
 *              is a bit field (see RepeatInfoType).
 *              For monthly-by-day repeating tasks,
 *              this is a DayOfMonthType.
 *
 * @retval  True if repeat info R repeats on day of week D.
 */
#define RepeatOnDOW(R, D)   ((1 << (D)) & ((R)->repeatOn))

/**
 * @brief   Task repeating data.
 */
typedef struct {
    DateType        repeatStartDate;
    RepeatInfoType  repeatInfo;
}
ToDoRepeatInfoType;

typedef ToDoRepeatInfoType  *ToDoRepeatInfoPtr;

/***********************************************************************
 *  Application Packed Record Format Related Data
 *
 *  Note: Records are stored in the database in packed format to
 *        conserve space.  When creating a new record, ToDoNewRecord(),
 *        or saving a record, ToDoChangeRecord(), the packing is handled
 *        for you.
 ***********************************************************************/

/**
 * @brief   Record format
 *
 * This is the packed database record format; record format
 * as stored.  All strings are null terminated and appear within
 * the record.
 *
 * optionalData is actually a variable-length compacted set
 * of optional data:
 *
 * if flags.dueDate, a DateType for the due date
 * then if flags.completionDate, a DateType for the completion date
 * then if flags.alarm, a TimeType (alarmTime) and a UInt16 (advance days)
 * then if flags.repeat, a DateType for the start date and a RepeatInfoType
 * then a null terminated description string
 * then a null terminated note string
 */
typedef struct {
    ToDoDBDataFlags dataFlags;
    UInt16          recordFlags;
    UInt16          priority;
    char            optionalData[];
} ToDoDBRecord;

typedef ToDoDBRecord        ToDoDBRecordType;
typedef ToDoDBRecord*       ToDoDBRecordPtr;

/**
 * @brief   Get the minimum size of the packed record.
 *
 * Minimum size assumes all empty fields, which always includes
 * a NULL byte for a zero length description, and a NULL byte
 * for a zero length note.
 * This allows GetToDoDescriptionPtr and GetToDoNotePtr to always
 * return sensible values, and also allows FldSetText, etc. to
 * edit them directly.
 *
 * @return  Minimum size of the packed record.
 */
#define sizeDBRecord    (sizeof( ToDoDBRecord ) + 2)

/** @brief Record format
 *
 * This is the unpacked record format as used by the app.  All pointers
 * are either NULL or point to data allocated in the heap.  All
 * strings are NULL character terminated.
 */
typedef struct {
    ToDoDBDataFlags     dataFlags;
    UInt16              recordFlags;
    UInt16              priority;

    DateType            dueDate;

    DateType            completionDate;

    ToDoAlarmInfoType   alarmInfo;

    ToDoRepeatInfoType  repeatInfo;

    Char                *descriptionP;
    Char                *noteP;
} ToDoItemType;

typedef ToDoItemType * ToDoItemPtr;

/** Indicates a task record has been completed. */
#define TODO_RECORD_FLAG_COMPLETE           0x0001

/** Indicates that the next repeat date for the task should be based on the
 *  completion date of the task. i.e.: the next repeat date of the task
 *  will be the completion date plus the time of the repeat interval.
 *  e.g.: if the task was due yesterday, but not completed until today,
 *  and the repeat interval is 1 week, then the next due date of the task
 *  is 1 week from today instead of 1 week from yesterday.
 */
#define TODO_RECORD_FLAG_REPEAT_ON_COMPLETE 0x0002

#else
#error "Invalid TODO_DB_VERSION"
#endif

/**
 * @name Record Feature Negatives
 */
/*@{*/
#define toDoNoDueDate           0xffff
#define toDoNoCompletionDate    0xffff
#define toDoNoAlarmDate         0xffff
#define toDoNoAlarmTime         (noTime)
/*@}*/

/** Used for ToDoChangeRecord. */
typedef enum {
    toDoRecordFieldCategory,
    toDoRecordFieldDataFlags,
    toDoRecordFieldRecordFlags,
    toDoRecordFieldPriority,
    toDoRecordFieldComplete,
    toDoRecordFieldDueDate,
    toDoRecordFieldCompletionDate,
    toDoRecordFieldAlarm,
    toDoRecordFieldRepeat,
    toDoRecordFieldDescription,
    toDoRecordFieldNote,
    toDoRecordFieldEndOfRecord
} ToDoRecordFieldType;

#define TODO_FILTER_ALL					(0x0000)
#define TODO_FILTER_CATEGORY			(0x0001)
#define TODO_FILTER_DATE				(0x0002)

#define TODO_SUBFILTER_DATE_TODAY		(0x0000)
#define TODO_SUBFILTER_DATE_LAST_SEVEN	(0x0001)
#define TODO_SUBFILTER_DATE_NEXT_SEVEN	(0x0002)
#define TODO_SUBFILTER_DATE_PAST_DUE	(0x0003)

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************
 *
 *  Function Prototypes
 *
 *  Note: Filters are used by several functions in searching and
 *        retrieval of task records.  The following is an example
 *        of how filters are used to effect what is displayed:
 *
 *  Imagine you have a task that repeats every Thursday, and you
 *  haven't checked it off in a couple of weeks. Given that today
 *  is Thursday, 9/25:
 *
 *  Using the due today subfilter, you would see 9/25! as the due date.
 *  Using the last seven days sub filter, you would see 9/25! as the due date.
 *  Using the next seven days sub filter, you would see 10/2 as the due date.
 *  Using the past due sub filter, you would see 9/11! as the due date.
 *
 *  If none of the incomplete instances of a repeat pass a given
 *  subfilter, that task is not displayed for that subfilter.  So, if
 *  your repeat were every 2 weeks, you wouldn't see the 10/2 instance
 *  using the next seven days sub filter in the above example.
 *
 *************************************************************/

/**
 * @brief   Create a new record in sorted position.
 *
 * @param dbP               Open database ptr.
 * @param item              Ptr to the record data to copy into the new
 *                          record.
 * @param category          Category of the new record.
 * @param filter            Used to refine how to sort. Currently ignored.
 * @param subFilter         Used to refine how to sort. Currently ignored.
 * @param index             Index of the new record.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoNewRecord( DmOpenRef dbP, ToDoItemPtr item, UInt16 category,
                          UInt16 filter, UInt16 subFilter, UInt16 *index);

/**
 * @brief   Create a new record and insert it after the specified position.
 *
 * The new record is assigned the same priority and due date as
 * the record it is inserted after.
 *
 * @param dbP               Open database ptr.
 * @param filter            Used to init the due date data.
 * @param subFilter         Used to init the due date data.
 * @param index             On entry, index of the record to be inserted
 *                          after. On exit, index of the new record.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoInsertNewRecord( DmOpenRef dbP, UInt16 filter, UInt16 subFilter,
                                UInt16 *index );

/**
 * @brief   Change a record in the database.
 *
 * Records are not stored with extra padding - they are
 * always resized to their exact storage space (i.e.: packed).
 * This avoids a database compression issue.
 *
 * @param dbP               Open database ptr.
 * @param index             On entry, index of the record to change.
 *                              On exit, new index of the record after
 *                              sorting.
 * @param filter            Used to refine how to sort. Currently ignored.
 * @param subFilter         Used to refine how to sort. Currently ignored.
 * @param changedField      The record field to change.
 * @param data              Ptr to the new field data.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoChangeRecord(DmOpenRef dbP, UInt16 * index, UInt16 filter,
                            UInt16 subFilter, ToDoRecordFieldType changedField,
                            const void * data);

/**
 * @brief   Change the database sort order.
 *
 * Records are not stored with extra padding - they are
 * always resized to their exact storage space (i.e.: packed).
 * This avoids a database compression issue.
 *
 * @param dbP               Open database ptr.
 * @param sortOrder         How the records should be sorted. e.g.: by
 *                              priority and then due date, soPriorityDueDate.
 * @param filter            Used to refine how to sort. Currently ignored.
 * @param subFilter         Used to refine how to sort. Currently ignored.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoChangeSortOrder( DmOpenRef dbP, UInt8 sortOrder, UInt16 filter,
                                UInt16 subFilter );

/**
 * @brief   Create an app info chunk if missing and initialize with defaults.
 *
 * @param dbP               Open database ptr.
 * @return                  0 if successful, error code if not.
 */
extern Err ToDoAppInfoInit(DmOpenRef dbP);

/**
 * @brief   Get handle to the application database info block.
 *
 * @param dbP               Open database ptr.
 * @return                  Handle to the AddrAppInfo or NULL.
 */
extern MemHandle ToDoGetAppInfo  (DmOpenRef dbP);

/**
 * @brief   Get the database's current sort order.
 *
 * @param dbP               Open database ptr.
 * @return                  The sort order of the database.
 *                              e.g.: soDueDatePriority
 */
extern UInt8 ToDoGetSortOrder (DmOpenRef dbP);

/**
 * @brief   Get the current filter preferences.
 *
 * @param filter            filter preference.
 * @param subFilter         sub-filter preference.
 * @return                  Nothing.
 */
extern void ToDoGetFilters( UInt16 *filterP, UInt16 *subFilterP );

/**
 * @brief   Sort the database.
 *
 * @param dbP               Open database ptr.
 * @return                  Nothing.
 */
extern void ToDoSort (DmOpenRef dbP);

/**
 * @brief   Get the application's database, opening or creating it as
 *          neccessary.
 *
 * @param dbPP              Ptr to a database ref var (DmOpenRef) to set.
 * @param mode              How to open the database (e.g.: dmModeReadWrite).
 * @return                  0 if successful, error code if not.
 */
extern Err ToDoGetDatabase (DmOpenRef *dbPP, UInt16 mode);

/**
 * @brief   Set the specified data flags on the specified record.
 *
 * Record data flags indicate whether the record has a due date,
 * alarm, description, note, etc.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param flagsP            The record data flags to set.
 *
 * @return                  Nothing.
 */
extern void ToDoDBRecordSetDataFlags( DmOpenRef dbP, UInt16 index,
                                      ToDoDBDataFlags *flagsP );

/**
 * @brief   Clear the specified data flags on the specified record.
 *
 * Record data flags indicate whether the record has a due date,
 * alarm, description, note, etc.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param flagsP            The record data flags to clear.
 *
 * @return                  Nothing.
 */
extern void ToDoDBRecordClearDataFlags( DmOpenRef dbP, UInt16 index,
                                        ToDoDBDataFlags *flagsP );

/**
 * @brief   Get the flags on the specified record.
 *
 * An example record flag is TODO_RECORD_FLAG_COMPLETE.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param flagsP            Ptr to a var to initialize with the record
 *                              flags.
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordGetRecordFlags( DmOpenRef dbP, UInt16 index,
                                       UInt16 * flagsP );

/**
 * @brief   Set the specified flags on the specified record.
 *
 * An example record flag is TODO_RECORD_FLAG_COMPLETE.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param flags             The record flags to set.
 *
 * @return                  Nothing.
 */
extern Err ToDoDBRecordSetRecordFlags( DmOpenRef dbP, UInt16 index,
                                       UInt16 flags );

/**
 * @brief   Clear the specified flags on the specified record.
 *
 * An example record flag is TODO_RECORD_FLAG_COMPLETE.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param flags             The record flags to clear.
 *
 * @return                  Nothing.
 */
extern Err ToDoDBRecordClearRecordFlags( DmOpenRef dbP, UInt16 index,
                                         UInt16 flags );

/**
 * @brief   Get the due date (if any) on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param hasDueDateP       Ptr to var to set to true if the record has a due
 *                              date. Pass NULL if you don't care.
 * @param dueDateP          Ptr to var to set with the record due date, if there
 *                              is a due date. Pass NULL if you don't care.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordGetDueDate( DmOpenRef dbP, UInt16 index,
                                   Boolean * hasDueDateP,
                                   DateType *dueDateP );

/**
 * @brief   Calculate the next due date (if any) on the specified record
 *          based on it being completed as of the specified date.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param completionDate    The date of the completed instance from which to
 *                              calculate the next due date.
 * @param gotNextP          Ptr to var to set to true if the record has a next
 *                              due date. Pass NULL if you don't care.
 * @param nextDueDateP      Ptr to var to set with the record next due date, if
 *                              any. Pass NULL if you don't care.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordGetNextDueDate( DmOpenRef dbP, UInt16 index,
                                       DateType completionDate,
                                       Boolean *gotNextP,
                                       DateType *nextDueDateP );

/**
 * @brief   Set or add a due date on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param dueDateP          Ptr to due date with which to set the record.
 *
 * @return                  0 if successful; error code otherwise.
 */
extern Err ToDoDBRecordSetDueDate( DmOpenRef dbP, UInt16 index,
                                   DateType * dueDateP );

/**
 * @brief   Clear the due date on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordClearDueDate( DmOpenRef dbP, UInt16 index );

/**
 * @brief   Get the completion date (if any) on the specified record.
 *
 * @param dbP                Open database ptr.
 * @param index              The record index.
 * @param hasCompletionDateP Ptr to var to set to true if the record has a
 *                               completion date. Pass NULL if you don't care.
 * @param completionDateP    Ptr to var to set with the completion date, if any.
 *                           Pass NULL if you don't care.
 *
 * @retval                   0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordGetCompletionDate( DmOpenRef dbP, UInt16 index,
                                          Boolean *hasCompletionDateP,
                                          DateType *completionDateP );

/**
 * @brief   Set the completion date on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param completionDateP   Ptr to the completion date with which to set the
 *                              record.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordSetCompletionDate( DmOpenRef dbP, UInt16 index,
                                          DateType *completionDateP );

/**
 * @brief   Clear the completion date on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordClearCompletionDate( DmOpenRef dbP, UInt16 index );

/**
 * @brief   Get the alarm data (if any) on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param hasAlarmP         Ptr to var to set to true if the record has an
 *                              alarm. Pass NULL if you don't care.
 * @param dueDateP          Ptr to var to set with the due date, if any.
 *                              Pass NULL if you don't care.
 * @param alarmInfoP        Ptr to var to set with the alarm data, if any.
 *                              Pass NULL if you don't care.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordGetAlarmInfo( DmOpenRef dbP, UInt16 index,
                                     Boolean * hasAlarmP,
                                     DateType *dueDateP,
                                     ToDoAlarmInfoType *alarmInfoP );

/**
 * @brief   Set the alarm data on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param alarmInfoP        Ptr to the alarm data with which to set the record.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordSetAlarmInfo( DmOpenRef dbP, UInt16 index,
                                     ToDoAlarmInfoType *alarmInfoP );

/**
 * @brief   Clear the alarm data and flag, if any, on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordClearAlarmInfo( DmOpenRef dbP, UInt16 index );

/**
 * @brief   Get the repeat info (if any) on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param hasRepeatP        Ptr to var to set to true if the record is
 *                              repeating. Pass NULL if you don't care.
 * @param repeatOnCompleteP Ptr to var to set to true if the record is set to
 *                              calculate the next occurance based on the
 *                              completion date. Pass NULL if you don't care.
 * @param repeatInfoP       Ptr to var to set with the repeat data, if any.
 *                              Pass NULL if you don't care.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordGetRepeatInfo( DmOpenRef dbP, UInt16 index,
                                      Boolean * hasRepeatP,
                                      Boolean *repeatOnCompleteP,
                                      ToDoRepeatInfoType *repeatInfoP );

/**
 * @brief   Set the repeat data on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 * @param repeatOnComplete  True if the record should be set to calculate the
 *                              next occurance in the repeat based on the
 *                              completion date.
 * @param alarmInfoP        Ptr to the repeat data with which to set the record.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordSetRepeatInfo( DmOpenRef dbP, UInt16 index,
                                      Boolean repeatOnComplete,
                                      ToDoRepeatInfoType *repeatInfoP );

/**
 * @brief   Clear the repeat data and flag, if any, on the specified record.
 *
 * @param dbP               Open database ptr.
 * @param index             The record index.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBRecordClearRepeatInfo( DmOpenRef dbP, UInt16 index );

/**
 * @brief    Get a pointer to the description field.
 *
 * @param recordP           Ptr to a record.
 * @return                  Ptr to the record description field.
 */
extern Char *GetToDoDescriptionPtr( ToDoDBRecordPtr recordP );

/**
 * @brief    Get a pointer to the note field.
 *
 * @param recordP           Ptr to a record.
 * @return                  Ptr to the record note field.
 */
extern Char *GetToDoNotePtr( ToDoDBRecordPtr recordP );

/**
 * @brief    Get the date of the next repeat occurance.
 *
 * @param repeatP           Ptr to the repeat data.
 * @param dateP             Ptr to the date to set.
 * @param searchForward     Search for the next occurance forward or backward
 *                              from the specified date.
 * @return                  .
 */
extern Boolean ToDoDBFindNextRepeat( ToDoRepeatInfoPtr repeatP, DatePtr dateP,
                                     Boolean searchForward );

/**
 * @brief    Get the due date of the specified record as it best matches
 *           the given filters.
 *
 * If a record has a due date, and the date filter is passed,
 * TODO_FILTER_DATE, then the sub-filter is used to determine if
 * the due date stored in the record should be returned, or if
 * a due date should be calculated based on the sub-filter.
 * e.g.: if the sub-filter is TODO_SUBFILTER_DATE_LAST_SEVEN,
 *       then the due date returned will be either the
 *       record's stored due date or it will be a date
 *       within the last 7 days if the due date is not
 *       today or later and if the next repeat occurance
 *       of the due date falls within the last 7 days.
 *
 * @param recordP           Ptr to a record.
 * @param filter            Used to refine the due date returned. See remarks.
 * @param subFilter         Used to refine the due date returned for repeating
 *                              records.
 * @param today             Today's date.
 * @param displayedDateP    Ptr to the var to set with the due date.
 *
 * @return                  Always returns errNone.
 */
extern Err ToDoDBRecordGetBestDueDateForFilter( ToDoDBRecordPtr recordP,
                                                UInt16 filter, UInt16 subFilter,
                                                DateType today,
                                                DateType *displayedDateP );

/**
 * @brief    Does the record at the given record index have the qualities
 *            expressed by the given filters.
 *
 * e.g.: If a record has a due date, and the date filter is passed,
 *       TODO_FILTER_DATE, then if the record matches the qualities
 *       expressed by the sub-filter, e.g.: the due date is past due,
 *       for sub-filter TODO_SUBFILTER_DATE_PAST_DUE, true is returned.
 *
 * @param dbP               Open database ptr.
 * @param index             Index of the record to check.
 * @param filter            What qualities for which to check this record.
 * @param subFilter         What qualities for which to check this record.
 * @param category          Used to refine the category quality when the
 *                              filter is TODO_FILTER_CATEGORY. i.e: does
 *                              this record belong to the given category or
 *                              indirectly by belonging to the All category.
 * @param today             Today's date.
 *
 * @return                  True if the given record has the given
 *                              qualifications.
 */
extern Boolean ToDoDBRecordPassesFilter( DmOpenRef dbP, UInt16 index,
                                         UInt16 filter, UInt16 subFilter,
                                         UInt16 category, DateType today );

/**
 * @brief    Find a record in the database that has the qualities
 *           expressed by the given filters.
 *
 * @param dbP               Open database ptr.
 * @param indexP            Ptr to the record index from which to begin the
 *                              search, and to set to the index of the found
 *                              record. Note that the first record examined is
 *                              the record at the given index.
 * @param offset            The number of records having the specified
 *                              qualities to skip before returning a matching
 *                              record. e.g.: if the offset is 0, then the first
 *                              record found will be returned.
 * @param direction         The direction in which to search, dmSeekForward or
 *                              dmSeekBackward.
 * @param filter            What qualities for which to seek in a record.
 * @param subFilter         What qualities for which to seek in a record.
 * @param category          Used to refine the category quality when the
 *                              filter is TODO_FILTER_CATEGORY. i.e: match
 *                              the record to the given category.
 * @param today             Today's date.
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBSeekRecordInFilter( DmOpenRef dbP, UInt16 *indexP,
                                     UInt16 offset, Int16 direction,
                                     UInt16 filter, UInt16 subFilter,
                                     UInt16 category, DateType today );
/**
 * @brief    Calculate the number of records in the database with due dates
 *           of today, within the last 7 days, within the next 7 days, and
 *           that are overdue.
 *
 * @param dbP               Open database ptr.
 * @param today             Today's date.
 * @param countCompleted    True if completed records should be included in
 *                              the count.
 * @param numTodayP         Ptr to the var to set with the number of records
 *                              with due dates of today.  Pass NULL if you don't
 *                              care.
 * @param numLastSevenP     Ptr to the var to set with the number of records
 *                              with due dates in the last 7 days.  Pass NULL
 *                              if you don't care.
 * @param numNextSevenP     Ptr to the var to set with the number of records
 *                              with due dates in the next 7 days.  Pass NULL
 *                              if you don't care
 * @param numPastDueP       Ptr to the var to set with the number of records
 *                              with due dates overdue.  Pass NULL if you don't
 *                              care
 *
 * @return                  0 if successfull; error code otherwise.
 */
extern Err ToDoDBNumRecordsInDateSubFilters( DmOpenRef dbP, DateType today,
                                             Boolean countCompleted,
                                             UInt16 *numTodayP,
                                             UInt16 *numLastSevenP,
                                             UInt16 *numNextSevenP,
                                             UInt16 *numPastDueP );

#if ERROR_CHECK_LEVEL == ERROR_CHECK_FULL
extern void ECToDoDBValidate( DmOpenRef dbP );
#endif

#ifdef __cplusplus
}
#endif

#endif // __TDTODOMGR_H__
