#include <avr/wdt.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdint.h>
#include "LEDs.h"
#include "nwk.h"
#include "nwk_hw.h"
#include "protocol.h"


static uint8_t gBri = 31;	//updates with up/down buttons   	0..31
static uint8_t gParam = 128;	//updates with left/right buttons	0...256


static void init()
{
	
	//wdt
	{		
		cli();
		wdt_reset();
		wdt_disable();
	}
	
	//ports
	{
		MCUCR |= 0x40;
		ACSR = 0x80;	//disable comparator
		DIDR1 = 0;
		DIDR0 = 0;
	}
}


static uint8_t rnd(void)
{
	static uint32_t seed1 = 4294906183ULL;
	
	seed1 *= 214013UL;
	seed1 += 2531011UL;
	
	return seed1 >> 24;
}

static uint8_t rnd_nz(uint8_t mask){
	uint8_t i;
	
	do{
		i = rnd();
	}while(!(i & mask));
	
	return i;
}

static uint16_t rnd_16(void){
	
	uint16_t i = rnd();
	
	i <<= 8;
	
	return i + rnd();
}

static void cooler_rainbow(char weirder)
{
	static const uint8_t PROGMEM sintabR[] = {
		255, 255, 254, 253, 252, 250, 247, 245, 242, 238, 235, 230, 226, 221, 216, 211, 205, 200, 194, 188, 181, 175, 168, 162, 155, 148, 142, 135, 128, 122, 115, 109, 102, 96, 90, 84, 78, 73, 67, 62, 57, 52, 48, 43, 39, 36, 32, 29, 26, 23, 20, 18, 15, 13, 12, 10, 9, 7, 6, 5, 4, 4, 3, 3, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 3, 3, 4, 4, 5, 6, 7, 9, 10, 12, 13, 15, 18, 20, 23, 26, 29, 32, 36, 39, 43, 48, 52, 57, 62, 67, 73, 78, 84, 90, 96, 102, 109, 115, 122, 128, 135, 142, 148, 155, 162, 168, 175, 181, 188, 194, 200, 205, 211, 216, 221, 226, 230, 235, 238, 242, 245, 247, 250, 252, 253, 254, 255
	};
	static const uint8_t PROGMEM sintabG[] = {
		255, 255, 254, 252, 249, 246, 242, 237, 232, 226, 220, 213, 205, 198, 190, 181, 173, 164, 155, 146, 137, 128, 119, 111, 102, 94, 86, 78, 71, 64, 57, 51, 45, 39, 34, 30, 26, 22, 18, 15, 13, 11, 9, 7, 6, 4, 3, 3, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 3, 4, 5, 6, 8, 10, 12, 14, 17, 20, 24, 28, 32, 37, 42, 48, 54, 60, 67, 74, 82, 90, 98, 106, 115, 124, 133, 142, 151, 160, 168, 177, 186, 194, 202, 209, 216, 223, 229, 235, 240, 244, 247, 250, 253, 254
	};
	static const uint8_t PROGMEM sintabB[] = {
		255, 254, 253, 250, 246, 241, 235, 228, 220, 211, 202, 192, 181, 171, 160, 149, 137, 126, 115, 104, 94, 84, 75, 66, 57, 49, 42, 36, 30, 25, 20, 16, 13, 10, 8, 6, 4, 3, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 3, 4, 5, 7, 9, 11, 15, 18, 22, 27, 33, 39, 46, 53, 61, 70, 79, 89, 99, 110, 121, 132, 143, 154, 165, 176, 187, 197, 206, 215, 224, 231, 238, 243, 248, 251, 254, 255
	};
	uint16_t posR = 0;
	uint16_t posG = 0;
	uint16_t posB = 0;
	uint8_t step = 0;
	
	while(!nwkPoll(&gBri, &gParam)){
		uint16_t i;
		uint8_t div = (35 - gBri);
		
		for(i = 0; i < NUM_RGBS; i++) rgbSet(i, pgm_read_byte(&sintabR[(i + posR) % sizeof(sintabR)])/div + 1, pgm_read_byte(&sintabG[(i + posG) % sizeof(sintabG)])/div + 1, pgm_read_byte(&sintabB[(i + posB) % sizeof(sintabB)])/div + 1);
		
		if(rnd()){
			if (posG == 0 && weirder) posG = sizeof(sintabG) - 1;
			else if (weirder) posG--;
			else if (++posG == sizeof(sintabG)) posG = 0;
		}
		if(rnd()) {
			if (++posR == sizeof(sintabR)) posR = 0;
		}
		if(rnd()) {
			if(((++step) & 1) && (++posB == sizeof(sintabB))) posB = 0;
		}
		
		rgbSync_nowait();
	}
}

static void color_wheel_val(uint8_t phase, uint8_t ctr, uint8_t *pR, uint8_t *pG, uint8_t *pB)	//phase is 0..255
{
	uint8_t tR = 0, tG = 0, tB = 0;
	
	switch(phase){
		case 0:
			tR = ctr;
			tG = 255;
			tB = 0;
			break;

		case 1:
			tR = 255;
			tG = 255 - ctr;
			tB = 0;
			break;

		case 2:
			tR = 255;
			tG = 0;
			tB = ctr;
			break;

		case 3:
			tR = 255 - ctr;
			tG = 0;
			tB = 255;
			break;

		case 4:
			tR = 0;
			tG = ctr;
			tB = 255;
			break;

		case 5:
			tR = 0;
			tG = 255;
			tB = 255 - ctr;
			break;
	}
	
	*pR = tR;
	*pG = tG;
	*pB = tB;
}

static void cool_color_wheel()
{
	uint16_t i;
	uint8_t phase = 0;
	uint16_t ctr = 0;

	while(!nwkPoll(&gBri, &gParam)){
		uint8_t tR, tG, tB;
		uint8_t div = (35 - gBri);
		
		
		for(i = 0; i < NUM_RGBS - 1; i++){
			rgbGet(i + 1, &tR, &tG, &tB);
			rgbSet(i, tR, tG, tB);
		}
		if((ctr += 213) >= 0x8000){ // 213/128 approx == 256*6 / NUM_RGBS
			ctr -= 0x8000;
			if(++phase == 6) phase = 0;
		}
		
		color_wheel_val(phase, ctr / 128, &tR, &tG, &tB);
		rgbSet(NUM_RGBS - 1, tR / div, tG / div, tB / div);
		rgbSync_nowait();
	}
}



#define SCALE(x)	x = (((uint32_t)x * scale_factor) >> 24)
			
static void raindrops(uint32_t scale_factor, char use_self, uint8_t num_per_flash)
{
	uint8_t p_r, p_g, p_b;
	uint8_t p0_r, p0_g, p0_b;
	uint16_t t_r, t_g, t_b;
	uint16_t i;
	
	
	while(!nwkPoll(&gBri, &gParam)){
		
		uint8_t tr, tg, tb, j;
			
		for(j = 0; j < num_per_flash; j++){
			
			do{
				i = rnd_16();
				
			} while(i > ((0x10000UL / NUM_RGBS) * NUM_RGBS));
			
			i %= NUM_RGBS;
			
			color_wheel_val(rnd() % 6, rnd(), &tr, &tg, &tb);
			
			tr /= (32 - gBri);
			tg /= (32 - gBri);
			tb /= (32 - gBri);
			
			rgbSet(i % NUM_RGBS, tr, tg, tb);
		}
		
		rgbGet(0, &p0_r, &p0_g, &p0_b);
		rgbGet(NUM_RGBS - 1, &p_r, &p_g, &p_b);
		
		for(i = 0; i < NUM_RGBS; i++){
			
			t_r = p_r;
			t_g = p_g;
			t_b = p_b;
			
			if(use_self){
				
				rgbGet(i, &tr, &tg, &tb);
				
				t_r += tr;
				t_g += tg;
				t_b += tb;
			}
			
			if(i == NUM_RGBS - 1){
				t_r += p0_r;
				t_g += p0_g;
				t_b += p0_b;
			} else {
				rgbGet(i + 1, &tr, &tg, &tb);
				
				t_r += tr;
				t_g += tg;
				t_b += tb;
			}
			
			SCALE(t_r);
			SCALE(t_g);
			SCALE(t_b);
			
			rgbGet(i, &p_r, &p_g, &p_b);
			rgbSet(i, t_r, t_g, t_b);
		}
		
		rgbSync_nowait();
	}
}

static void ants(uint32_t scale_factor)
{
	#define NUM_ANTS 32
	uint16_t i;
	uint8_t tr, tg, tb, j;
	uint8_t antR[NUM_ANTS];		//ant color
	uint8_t antG[NUM_ANTS];
	uint8_t antB[NUM_ANTS];
	uint8_t antL[NUM_ANTS];		//ant life
	uint16_t antP[NUM_ANTS];	//ant pos
	uint8_t antS[NUM_ANTS];		//ant speed (top bit is direction)
	uint8_t antC[NUM_ANTS];		//ant countsr
	
	for(j = 0; j < NUM_ANTS; j++){
		antL[j] = rnd_nz(0xFF);
		antS[j] = 1;
		antC[j] = 0;
	}
	
	while(!nwkPoll(&gBri, &gParam)){
		
		#define ADD_C(dst, src1, src2) dst = (((uint16_t)(src1)) + ((uint16_t)(src2)) >= 256) ? 255 : (src1) + (src2)
		
		for(j = 0; j < NUM_ANTS; j++){
			rgbGet(antP[j], &tr, &tg, &tb);
			ADD_C(tr, tr, antR[j]);
			ADD_C(tg, tg, antG[j]);
			ADD_C(tb, tb, antB[j]);
			rgbSet(antP[j], tr, tg, tb);
			
			if (++antC[j] == (antS[j] & 0x7F)){
				antC[j] = 0;
				if (antL[j]--){
					if (antS[j] & 0x80) {
						antP[j]++;
						if(antP[j] == NUM_RGBS) antP[j] = 0;
					} else {
						antP[j] = antP[j] ? antP[j] - 1 : NUM_RGBS - 1;
					}
					SCALE(antR[j]);
					SCALE(antG[j]);
					SCALE(antB[j]);
					if(!antR[j] && !antG[j] && !antB[j]) antL[j] = 0; //if faded, die
				} else {
					color_wheel_val(rnd() % 6, rnd(), antR + j, antG + j, antB + j);
					
					antR[j] /= (33 - gBri);
					antG[j] /= (33 - gBri);
					antB[j] /= (33 - gBri);
					
					antL[j] = rnd_nz(0xFF) | 0x80;
					antP[j] = rnd_16() & NUM_RGBS;
					antS[j] = (rnd_nz(0x03) & 0x03) | (rnd() & 0x80);
				}
			}
		}
		
		for(i = 0; i < NUM_RGBS; i++){
			
			rgbGet(i, &tr, &tg, &tb);
			SCALE(tr);
			SCALE(tg);
			SCALE(tb);
			SCALE(tr);
			SCALE(tg);
			SCALE(tb);
			rgbSet(i, tr, tg, tb);
		}
		
		rgbSync_nowait();
	}
}

static void solidColor(uint8_t color_on){
	
	uint16_t i;
	uint8_t tr, tg, tb;
	
	
	while(!nwkPoll(&gBri, &gParam)){
		
		uint8_t div = (33 - gBri) * 3;
		
		for(i = 0; i < NUM_RGBS; i++) {
			
			tr = gParam >= 252 ? 251 : gParam;
			
			color_wheel_val(tr / 42, (tr % 42) * 6, &tr, &tg, &tb);
			tr  = ((uint16_t)tr + (uint16_t)div/2) / div;
			tg  = ((uint16_t)tg + (uint16_t)div/2) / div;
			tb  = ((uint16_t)tb + (uint16_t)div/2) / div;
			if(!color_on) tr = tg = tb = 0;
			rgbSet(i, tr, tg, tb);
		}
		rgbSync_nowait();
	}
}

static void pulseColor(void)
{
	static const uint8_t PROGMEM pulseTab[] = {
		255, 255, 254, 253, 252, 250, 247, 245, 242, 238, 235, 230, 226, 221, 216, 211, 205, 200, 194, 188, 181, 175, 168, 162, 155, 148, 142, 135, 128, 122, 115, 109, 102, 96, 90, 84, 78, 73, 67, 62, 57, 52, 48, 43, 39, 36, 32, 29, 26, 23, 20, 18, 15, 13, 12, 10, 9, 7, 6, 5, 4, 4, 3, 3, 2, 2, 2, 2, 2, 2, 3, 3, 4, 4, 5, 6, 7, 9, 10, 12, 13, 15, 18, 20, 23, 26, 29, 32, 36, 39, 43, 48, 52, 57, 62, 67, 73, 78, 84, 90, 96, 102, 109, 115, 122, 128, 135, 142, 148, 155, 162, 168, 175, 181, 188, 194, 200, 205, 211, 216, 221, 226, 230, 235, 238, 242, 245, 247, 250, 252, 253, 254, 255
	};
		
	uint16_t i, j = 0;
	uint8_t tr, tg, tb;
	
	
	while(!nwkPoll(&gBri, &gParam)){
		
		uint8_t div = (33 - gBri) * 3;
		
		tr = gParam >= 252 ? 251 : gParam;
		
		color_wheel_val(tr / 42, (tr % 42) * 6, &tr, &tg, &tb);
		tr = ((uint16_t)tr * (uint16_t)pgm_read_byte(&pulseTab[j])) >> 8;
		tg = ((uint16_t)tg * (uint16_t)pgm_read_byte(&pulseTab[j])) >> 8;
		tb = ((uint16_t)tb * (uint16_t)pgm_read_byte(&pulseTab[j])) >> 8;
		
		tr  = ((uint16_t)tr + (uint16_t)div - 1) / div;
		tg  = ((uint16_t)tg + (uint16_t)div - 1) / div;
		tb  = ((uint16_t)tb + (uint16_t)div - 1) / div;
		
		for(i = 0; i < NUM_RGBS; i++) {
			
			rgbSet(i, tr, tg, tb);
		}
		if(++j == sizeof(pulseTab)) j = 0;
		rgbSync_nowait();
	}
}

int __attribute__((noreturn)) main()
{

	init();
	rgbInit();
	nwkHwInit();
	nwkInit(0x3A50B211);
	while(nwkPoll(&gBri, &gParam));	//before we start, let network layer clear the packet backlog
	
	rgbSync();
	rgbSync();

	while(1){
		solidColor(0);		//off
		solidColor(1);		//color
		pulseColor();
		cooler_rainbow(1);
		raindrops(5592384UL, 1, 1);	//- very chill
		raindrops(6400000UL, 0, 2);	//- very spazzy but cool for dance music
	//	ants(16000000UL); - somehow crashes...
		cool_color_wheel();
	}
	
	while(1);	//stop gcc warning...
}



