#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <curses.h>
#include "cpu.h"
#include "soc.h"
#include "vDISP.h"

#define VERBOSE				0

#define DISP_REAL_WIDTH		40		//controls DDRAM wraparound and display sizing
#define DISP_ADDR_WIDTH		64		//controls DDRAM addresses
#define DISP_HEIGHT			2

#define CGRAM_SIZE			64

//display shift not emulated as it is unused
//blinking and cursor not emulated (i am lazy)

enum DispState {
	DispStateDeselected,
	DispStateStartByteWait,
	DispStateInstrWriteWait,
	DispStateInstrReadWait,
	DispStateDataWriteWait,
	DispStateDataReadWait,

	DispStatePostCommandIdle,
};

struct VDISP {
	struct VSPI *vspi;

	enum DispState state;
	uint8_t AC;
	uint8_t brightness;
	bool ACincr;		//else decr
	bool dispOn, acPointsToDDRAM;
	uint8_t CGRAM[CGRAM_SIZE];
	uint8_t DDRAM[DISP_HEIGHT][DISP_ADDR_WIDTH];
};

static void vdispPrvRedraw(struct VDISP *disp)
{
	unsigned row, col;

	for (row = 0; row < DISP_HEIGHT + 2; row++) {
		for (col = 0; col < DISP_REAL_WIDTH + 2; col++) {
			mvaddch(row, col, 'X');
		}
	}

	for (row = 0; row < DISP_HEIGHT; row++) {
		for (col = 0; col < DISP_REAL_WIDTH; col++) {
			
			mvaddch(row + 1, col + 1, disp->DDRAM[row][col]);
		}
	}
	refresh();
}

static void vdispPrvPostAccessAdjustAC(struct VDISP *disp)
{
	if (disp->acPointsToDDRAM) {
		if (disp->ACincr) {
			disp->AC++;
			if (disp->AC % DISP_ADDR_WIDTH == DISP_REAL_WIDTH) {

				if (VERBOSE)
					fprintf(stderr, "COL OVERFLOW\n");

				disp->AC /= DISP_ADDR_WIDTH;
				if (disp->AC == DISP_HEIGHT)
					disp->AC = 0;
				disp->AC *= DISP_ADDR_WIDTH;
			}
		}
		else {
			if (disp->AC % DISP_ADDR_WIDTH) {

				disp->AC /= DISP_ADDR_WIDTH;
				if (disp->AC)
					disp->AC--;
				else
					disp->AC = DISP_HEIGHT - 1;
				disp->AC *= DISP_ADDR_WIDTH;
				disp->AC += DISP_REAL_WIDTH - 1;
			}
			else  {

				disp->AC--;
			}
		}
	}
	else {
		if (disp->ACincr) {
			if (++disp->AC == CGRAM_SIZE)
				disp->AC = 0;
		}
		else {
			if (disp->AC)
				disp->AC--;
			else
				disp->AC = CGRAM_SIZE - 1;
		}
	}
}

static uint8_t vdispPrvSpiProvideByteF(void *userData)
{
	struct VDISP *disp = (struct VDISP *)userData;
	uint8_t ret = 0;

	switch (disp->state) {

		case DispStateInstrReadWait:
			ret = disp->AC;
			break;

		case DispStateDataReadWait:
			if (disp->acPointsToDDRAM) {

				ret = disp->DDRAM[disp->AC / DISP_ADDR_WIDTH][disp->AC % DISP_ADDR_WIDTH];
		
				if (VERBOSE)
					fprintf(stderr, "VDISP: read char '%c' (0x%02x) from row %u col %u\n", ret, ret, disp->AC / DISP_ADDR_WIDTH, disp->AC % DISP_ADDR_WIDTH);
			}
			else
				ret = disp->CGRAM[disp->AC];
			vdispPrvPostAccessAdjustAC(disp);
			break;

		default:
			//we have nothing to say
			break;
	}

	if (VERBOSE)
		fprintf(stderr, "VDISP: sent 0x%02x\n", ret);
	return ret;
}

static void vdispPrvErase(struct VDISP *disp)
{
	unsigned i, j;

	for (i = 0; i < DISP_HEIGHT; i++) {
		for (j = 0; j < DISP_ADDR_WIDTH; j++) {
			disp->DDRAM[i][j] = ' ';
		}
	}
}

static void vdispPrvSpiAcceptByteF(void *userData, uint8_t byte)
{
	struct VDISP *disp = (struct VDISP *)userData;
	bool needRedraw = false;

	if (VERBOSE)
		fprintf(stderr, "VDISP: got  0x%02x\n", byte);

	switch (disp->state) {

		case DispStateInstrReadWait:	//called after vdispPrvSpiProvideByteF() so this is the palce to do this
			disp->state = DispStatePostCommandIdle;
			break;

		case DispStateDataReadWait:	//called after vdispPrvSpiProvideByteF() so this is the palce to do this
			disp->state = DispStatePostCommandIdle;
			break;

		case DispStateStartByteWait:
			if ((byte & 0xf9) != 0xf8) {
				fprintf(stderr, "VDISP: invalid start byte: 0x%02x\n", byte);
				abort();
			}
			switch ((byte >> 1) & 3) {
				case 0:
					disp->state = DispStateInstrWriteWait;
					break;
				case 1:
					disp->state = DispStateDataWriteWait;
					break;
				case 2:
					disp->state = DispStateInstrReadWait;
					break;
				case 3:
					disp->state = DispStateDataReadWait;
					break;
			}
			break;

		case DispStateDataWriteWait:
			if (disp->acPointsToDDRAM)
				disp->DDRAM[disp->AC / DISP_ADDR_WIDTH][disp->AC % DISP_ADDR_WIDTH] = byte;
			else
				disp->CGRAM[disp->AC] = byte;
			if (VERBOSE)
				fprintf(stderr, "VDISP: wrote char '%c' (0x%02x) to row %u col %u\n", byte, byte, disp->AC / DISP_ADDR_WIDTH, disp->AC % DISP_ADDR_WIDTH);
			vdispPrvPostAccessAdjustAC(disp);
			needRedraw = true;
			disp->state = DispStatePostCommandIdle;
			break;

		case DispStateInstrWriteWait:
			switch (byte) {
				case 0x01:	//clear display
					vdispPrvErase(disp);
					disp->ACincr = true;
					needRedraw = true;
					break;
				
				case 0x02:
				case 0x03:
					disp->AC = 0;
					break;
				
				case 4:
				case 5:
					disp->ACincr = false;
					break;

				case 6:
				case 7:
					disp->ACincr = true;
					break;
				
				case 8 ... 11:
					disp->dispOn = false;
					needRedraw = true;
					break;

				case 12 ... 15:
					disp->dispOn = true;
					needRedraw = true;
					break;

				case 0x10 ... 0x1f:
					//cursor display/shift command not emulated - i am not clear as to why it exists
					break;

				case 0x20 ... 0x3f:
					if (!(byte & 8)) {
						fprintf(stderr, "display set to one-line mode!\n");
						abort();
					}
					disp->brightness = byte & 3;
					needRedraw = true;
					break;

				case 0x40 ... 0x7f:
					disp->acPointsToDDRAM = false;
					disp->AC = byte & 0x3f;
					break;

				case 0x80 ... 0xff:
					disp->acPointsToDDRAM = true;
					disp->AC = byte & 0x7f;
					break;
			}
			disp->state = DispStatePostCommandIdle;
			break;

		case DispStatePostCommandIdle:
			fprintf(stderr, "VDISP: got data after the requisite two bytes: 0x%02x\n", byte);
			abort();
			break;

		case DispStateDeselected:
			//we do not care
			break;
	}

	if (needRedraw)
		vdispPrvRedraw(disp);
}

static void vdispPrvSpiSelectionChanged(void *userData, bool selected)
{
	struct VDISP *disp = (struct VDISP *)userData;

	if (selected) {

		if (VERBOSE)
			fprintf(stderr, "===== VDISP :selected =====\n");

		switch (disp->state) {
			case DispStateDeselected:
				disp->state = DispStateStartByteWait;
				break;

			default:
				fprintf(stderr, "VDISP: select in state %u\n", disp->state);
				abort();
		}
	}
	else {

		if (VERBOSE)
			fprintf(stderr, "==== VDISP: deselected ====\n");

		switch (disp->state) {
			case DispStatePostCommandIdle:
				disp->state = DispStateDeselected;
				break;

			default:
				fprintf(stderr, "VDISP: deselect in state %u\n", disp->state);
				abort();
		}
	}
}

struct VDISP* vdispInit(void)
{
	struct VDISP *disp = calloc(1, sizeof(struct VDISP));

	if (disp) {
		disp->vspi = vspiInit("VDISP", SpiMode3);
		
		if (disp->vspi) {
			
			disp->acPointsToDDRAM = true;
			disp->ACincr = true;
			disp->brightness = 3;
			vspiDeviceRegister(disp->vspi, vdispPrvSpiProvideByteF, vdispPrvSpiAcceptByteF, vdispPrvSpiSelectionChanged, disp);

			vdispPrvErase(disp);
			vdispPrvRedraw(disp);
			return disp;
		}
		free(disp);
		disp = NULL;
	}


	return disp;
}

struct VSPI* vdispGetVSPI(struct VDISP *disp)
{
	return disp->vspi;
}

void vdispDeinit(struct VDISP *disp)
{
	vspiDestroy(disp->vspi);
	free(disp);
}


