;;-------------------------------------------------------------------
;; File: bridge.inc
;;
;; Original Copyright (C) 1989 Northwestern University, Vance Morrison
;; Changes 05may93 Alessandro Fanelli and Luigi Rizzo (luigi@iet.unipi.it)
;;
;; ORIGINAL COPYRIGHT NOTICE:
;; Permission to view, compile, and modify for LOCAL (intra-organization)
;; USE ONLY is hereby granted, provided that this copyright and permission
;; notice appear on all copies.  Any other use by permission only.
;;
;; Northwestern University makes no representations about the suitability
;; of this software for any purpose.  It is provided "as is" without expressed
;; or implied warranty.  See the copywrite notice file for complete details.
;;
;; BDG_IF_R_ACCESS waits for the next packet to come from the board
;; associated with 'name'. Source and Destination address are searched
;; in the table and compared with the bridge's port address, causing
;; the following behaviour:
;;
;;	Dest. port	Resent to:
;;	==============================================
;;	bridge		Upper layer
;;	this port	Dropped
;;	other port	Destination port
;;	not found	All ports
;;	multi/broadcast	All ports and upper layer
;;
;; After processing, the packet is dropped and space freed.
;;
;; Registers: DS:always data segment
;;	IN:	--
;;	OUT:	--
;;	CHANGE: AX, BX, CX, DX, BP, DI, SI, ES

BDG_IF_R_ACCESS MACRO name
    LOCAL got_packet
    LOCAL forward_all
    LOCAL forward, not_me, drop
    LOCAL bridge_end ;; can become an external label
    LOCAL broadcast, collision, no_collision
    LOCAL end_filters, next_filter, outside ;; for statistics

	;; Reg. Used: -- Free: ALL
    cmp if_found[name],0	;; skip non existing interfaces
    jz bridge_end

    IF_R_ACCESS name, bridge_end ;; ES:BX=buffer, CX= pkt_len
	;; Free: AX, DX, SI, DI, BP

;;------- check if packet is for the bridge ---------------
    mov current_rx_segment,ES
    mov AX, ES:[BX+4]
    cmp AX, if_address[8*name+4]
    jnz not_me
    mov AX, ES:[BX+2]
    cmp AX, if_address[8*name+2]
    jnz not_me
    mov AX, ES:[BX]
    cmp AX, if_address[8*name]
    jz got_packet		;; it is for me

not_me: ;;------- an ordinary packet - must be bridged ------------

    IFDEF use_filters ;; ------ filtering code ------------
	;; Remember, CX and ES:BX should be preserved
	mov ax, es:[BX+12] ;; packet type
	xchg ah,al ;; back to Intel (little endian) format
	mov bp,ax
	lea si, filters
    next_filter:
	lodsw
	and ax, ax
	jz end_filters;
	mov dx, ax ;; save type. DL= outside, DH=inside 
	lodsw ;; DI=low boundary
	mov di, ax
	lodsw
	cmp bp, di
	jb outside
	cmp bp,ax
	jae outside
	xchg dh,dl ;; here is inside
    outside:
	cmp dl,1 ; KEEP
	je end_filters ;; keep packet
	cmp dl,2 ; DROP
	jne next_filter
	COUNT_CYCLE %(name + LOCAL_DROP_BASE_CTR)
	jmp drop

    end_filters:
    ENDIF ;;----------- filtering code --------------------

    ;; Used: ES:BX = buffer address, CX= pkt_len
    ;; Free: AX, DX, SI, DI, BP

;; WARNING: DS is saved in BP, ES (was pkt-seg) goes to DS, BX saved in SI
;;---------------- find table entry --------------------
    mov BP, DS			;; save DS
    mov SI, BX			;; save BX (buf pointer)
 
    mov AX, ES
    mov ES, bdg_table_ptr	;; ES = table
    mov DS, AX			;; now DS:SI = pkt_start
 
    ;; Used: DS:SI(=BX) = rx pkt address, CX= pkt_len, BP=data seg.
    ;;       ES=bridge table
    ;; Free: AX, DX, DI

    add SI, 6                ;; SI= Source_Address
    mov DI, [SI+4]           ;; compute hash address 
    xor DI, [SI+2]
    shl DI, 3                ;; now ES:DI is the hash address
    
;;----------- check table collisions on source addresses -------------
    push si                  ;; DS:SI=pkt(src addr), ES:DI=table
    push di

    cmp byte ptr es:[di+6],0ffh;; interface #
    jz no_collision          ;; empty -> no collision, no loop
    cmpsw                    ;; ES:DI=table, DS:SI=pkt
    jnz collision	     ;; different -> collision, no loop
    cmpsw
    jnz collision	     ;; different -> collision, no loop
    cmpsw
    IFDEF no_loops ;;-------------- loop avoidance -------------------
	jnz collision	     ;; different -> collision, no loop
	cmp byte ptr es:[di],name ;; found address, check source port
	je no_collision ;; same port, no loops
	    ;; found a loop. Temporarily disable this interface, flush entry.
	mov byte ptr es:[di], 0ffh ;; flush entry
	mov ds, bp               ;; restore DS=data seg  
	or byte ptr if_address[8*name+7],IF_FLAG_DISABLE
	pop di
	pop si
	jmp drop
    ELSE
	jz no_collision
    ENDIF ;;------------- loop avoidance -----------------------------

collision: ;; collision in hash table, update counters
    mov ax, ds			;; save DS
    mov ds, bp			;; restore DS=data seg  
    COUNT_CYCLE %COLLSN_CTR	;; counter for table collisions
    mov ds,ax			;; restore DS:SI Source_Address
no_collision:
    pop di
    pop si
 
;;------------- drop packet if IF is disabled. ----------------------
;; Here DS:SI=pkt+6, ES:DI=table entry, BP=data segment, BX=pkt offset,CX=len
    mov ax, ds	;; save pkt segment in AX
    mov ds,bp	;; restore DS=data segment
    test byte ptr if_address[8*name+7],IF_FLAG_DISABLE ;; status
    jnz drop ;; drop if disabled. Should count...
    mov ds,ax	;; restore  packet segment

;;------------- now write the source address into the table. --------
    movsw 
    movsw
    movsw
    mov byte ptr es:[di], name	;; save source IF
    sub SI, 12			;; DS:SI back to pkt beginning

    ;; Used: DS:SI(=BX) = rx pkt address, CX= pkt_len, BP=data seg.
    ;;       ES=bridge table
    ;; Free: AX, DX, DI

;;-------------- check destination -----------------------------------
    test byte ptr [SI], 1	;; dst is a multicast address, forward all
    jnz broadcast		;; really the same as forward_all
    mov DI, [SI+4]		;; hash destination
    xor DI, [SI+2]
    shl DI, 3
    cmp byte ptr es:[di+6],0ffh	;; empty entry ?
    jz forward_all		;; yes, must forward all
    cmpsw			;; search into table
    jnz forward_all		;; not in table
    cmpsw
    jnz forward_all		;; not in table
    cmpsw
    jnz forward_all		;; not in table
 	;; found destination
    mov DS, BP			;; restore DS to data segment
    mov AL, ES:[DI]		;; AL = dst card
    cmp AL, name		;; check for local destination
    jnz forward			;; forward if not
 
    COUNT_CYCLE %(name + LOCAL_DROP_BASE_CTR) ;; drop because local destination
    jmp drop
 
forward: ;;----- forward to a specific interface (in AL) ----------------
    mov BP, CX ;; BP,CX = pkt size, AL= destination IF
    ;; Reg. Used: DS= dataseg, BX = rx pkt address
    ;; Free: DX, SI, DI, ES
    ;;--------- old code - can be shortened ------------------
    ;; The problem here is that macros expect a constant (idx) as argument,
    ;; while we'd like to use AL instead.
    ;; "name" is the source port, "idx" is the destination port
    FOR idx, <1,2,3,4,5>
	LOCAL next,drop1
	IF (idx le num_dls) AND (idx ne name)
	    cmp AL, idx      ;; skip if not for this IF
	    jnz next
		;; now have found the if, no need to check if_found
	    test byte ptr if_address[8*idx+7], IF_FLAG_DISABLE
	    jnz drop1 ;; disabled ports don't work
		;; BX=source pkt offset, BP=source pkt len
		;; AX, DX, SI available
	    IF_W_ACCESS idx, drop1 ;; get output buffer in ES:DI
	    mov SI, BX ;; probably can do it outside
	    COUNT_CYCLE %(idx + TXPKT_BASE_CTR) ;; update tx. ctr
	    IF_COPY name ;; seg:SI -> ES:DI for CX bytes
		;; CX is lost, as are SI and DI
	    mov CX, BP ;; restore CX
		;; AX, DX, SI, DI available
	    IF_W_WRITE idx ;; forward data. Output buffer for card
		;; is at fixed position, so pointers are not needed
	    jmp drop
	drop1: ;; packet dropped -- IF not ready or disabled
	    COUNT_CYCLE %(idx + DROP_BASE_CTR)
	    jmp drop
	next:
	ENDIF ;; "if (idx le num_dls)..."
    ENDM
    ;; should only reach here if something goes wrong!
    ;; should really mark it, in counter %name
    COUNT_CYCLE %(name + BAD_RX_BASE) ;; test discard packet 
    jmp drop
;;------------- end of forwarding code for specific IF -------------
 
;;------------- forwarding code for all interfaces -----------------
broadcast:
forward_all:
    mov DS, BP ;; restore DS (on entry here, it is in BP)
    mov BP, CX ;; save CX for later use
    COUNT_CYCLE %(name + RXALL_BASE_CTR)

    FOR idx, <1,2,3,4,5>
	LOCAL drop2, next
	IF (idx le num_dls) AND (idx ne name) ;; skip local address
	    cmp if_found[idx], 0
	    jz next
	    test byte ptr if_address[8*idx+7], IF_FLAG_DISABLE
	    jnz drop2 ;; disabled ports don't work
	    IF_W_ACCESS idx, drop2; ES:DI=packet ptr
	    mov SI, BX
	    COUNT_CYCLE %(idx + TXPKT_BASE_CTR)
	    IF_COPY name ;; CX is lost, as are SI and DI
	    mov CX, BP
		;; AX, DX, SI, DI available
	    IF_W_WRITE idx
	    jmp next
	drop2: ;; skip this if, goto next
	    COUNT_CYCLE %(idx + DROP_BASE_CTR)
	next:
	ENDIF ;; "if (idx le num_dls)..."
    ENDM
;;--------- end forwarding code for all interfaces -----------------

;;---------- deal with possible broadcasts for the bridge ---------
    mov ES, current_rx_segment ;; Set ES to the buffer segment, which
    test byte ptr es:[bx],1 ;; BX points to the input PKT
    jz drop
    mov CX, BP ;; restore length.
got_packet: ;; send the pkt up to the higher level protocols
    COUNT_CYCLE %(name + RXPKT_BRIDGE)  ;; count # packets to bridge
    ;; CX=length. ES:BX=packet. BP=name. Either for me, or multicast.
    mov BP, name
    call process_pkt
drop:
    IF_R_FREE name   ;; free memory, no par.
bridge_end:
ENDM
;;******************************************************************
;;------------- end of file -------------------
