/***
 *** ns8390.c
 ***
 *** ed driver for pcbridge 3.xx
 ***
 *** Luigi Rizzo 17 oct 96
 ***

 This code is based heavily on David Greenman's if_ed.c driver
 and Martin Renters' FreeBSD netboot code. The original copyright
 follows:

 Copyright (C) 1993-1994, David Greenman, Martin Renters.
  This software may be used, modified, copied, distributed, and sold, in
  both source and binary form provided that the above copyright and these
  terms are retained. Under no circumstances are the authors responsible for
  the proper functioning of this software, nor do the authors assume any
  responsibility for damages incurred with its use.

3c503 support added by Bill Paul (wpaul@ctr.columbia.edu) on 11/15/94
SMC8416 support added by Bill Paul (wpaul@ctr.columbia.edu) on 12/25/94

***/

/* #define SINK	/*** to test SINK interfaces ***/

#include "bridge.h"
#include "ns8390.h"

int eth_pio_read(struct if_vars *ifp, u_short src, char *dst, u_short cnt);
int eth_pio_write(struct if_vars *ifp, u_char *src, u_short dst, u_short cnt);

int eth_transmit(struct if_vars *ifp, u_short s, char *p);

char eth_driver[] = "ed0";

/***
 *** ETH_PROBE - Look for an adapter
 ***/
eth_probe(struct if_vars *ifp, int base_addr, int mem_seg)
{
    struct wd_board *brd;
    char *name;
    u_short chksum;
    u_char c;
    int i, base_low, base_high;

    ifp->name = (ifp - &c_if[0]);
    printf("\nProbing board %d IO 0x%x MEM 0x%X\n",
	ifp->name, base_addr, mem_seg);
    ifp->eth_vendor = VENDOR_NONE;
    if (!base_addr)
	goto probe_ne;
    base_low=base_high=base_addr;

#ifdef INCLUDE_WD
    /*** Search for WD/SMC cards ***/
    for (ifp->eth_asic_base=base_low; ifp->eth_asic_base <= base_high;
	ifp->eth_asic_base += 0x20) {
	    chksum = 0;
	    for (i=8; i<16; i++)
		chksum += inb(i+ifp->eth_asic_base);
	    if ((chksum & 0x00FF) == 0x00FF)
		break;
    }
    if (ifp->eth_asic_base <= base_high) { /* We've found a board */
	ifp->eth_vendor = VENDOR_WD;
	ifp->eth_nic_base = ifp->eth_asic_base + WD_NIC_ADDR;
	c = inb(ifp->eth_asic_base+WD_BID);	/* Get board id */
	for (brd = wd_boards; brd->name; brd++)
	    if (brd->id == c) break;
	if (!brd->name) {
	    printf("\nUnknown Ethernet type %x\n", c);
	    return(0);	/* Unknown type */
	}
	ifp->eth_flags = brd->flags;
	ifp->eth_memsize = brd->memsize;
	ifp->eth_tx_start = 0;
	if ((c == TYPE_WD8013EP) &&
		(inb(ifp->eth_asic_base + WD_ICR) & WD_ICR_16BIT)) {
	    ifp->eth_flags = FLAG_16BIT;
	    ifp->eth_memsize = MEM_16384;
	}
	if ((c & WD_SOFTCONFIG) && (!(ifp->eth_flags & FLAG_790))) {
	    ifp->eth_bmem = (char *)(0x80000 |
		 ((inb(ifp->eth_asic_base + WD_MSR) & 0x3F) << 13));
	} else
	    ifp->eth_bmem = (char *)mem_seg;
	if (brd->id == TYPE_SMC8216T || brd->id == TYPE_SMC8216C) {
	    (u_int) *(ifp->eth_bmem + 8192) = (u_int)0;
	    if ((u_int) *(ifp->eth_bmem + 8192)) {
		brd += 2;
		ifp->eth_memsize = brd->memsize;
	    }
	}
	outb(ifp->eth_asic_base + WD_MSR, 0x80);	/* Reset */
	printf("\n%s base 0x%x, memory 0x%X, addr ",
		brd->name, ifp->eth_asic_base, ifp->eth_bmem);
	{
	    u_char *p= (u_char *)&(ifp->eth_node_addr);
	    for (i=0; i<6; i++) {
		p[i]=inb(i+ifp->eth_asic_base+WD_LAR);
		printf("%b",(int)(p[i]));
		if (i < 5) printf (":");
	    }
	}
	if (ifp->eth_flags & FLAG_790) {
	    outb(ifp->eth_asic_base+WD_MSR, WD_MSR_MENB);
	    outb(ifp->eth_asic_base+0x04,
		    (inb(ifp->eth_asic_base+0x04) | 0x80));
	    outb(ifp->eth_asic_base+0x0B,
		    (((u_int)ifp->eth_bmem >> 13) & 0x0F) |
		    (((u_int)ifp->eth_bmem >> 11) & 0x40) |
		    (inb(ifp->eth_asic_base+0x0B) & 0xB0));
	    outb(ifp->eth_asic_base+0x04,
		    (inb(ifp->eth_asic_base+0x04) & ~0x80));
	} else {
	    outb(ifp->eth_asic_base+WD_MSR,
		    (((u_int)ifp->eth_bmem >> 13) & 0x3F) | 0x40);
	}
	if (ifp->eth_flags & FLAG_16BIT) {
	    if (ifp->eth_flags & FLAG_790) {
		ifp->eth_laar = inb(ifp->eth_asic_base + WD_LAAR);
		outb(ifp->eth_asic_base + WD_LAAR, WD_LAAR_M16EN);
		inb(0x84);
	    } else {
		outb(ifp->eth_asic_base + WD_LAAR, (ifp->eth_laar =
			WD_LAAR_M16EN | WD_LAAR_L16EN | 1));
	    }
	}
	goto found_board;
    }
#endif
#ifdef INCLUDE_3COM
    /*** Search for 3Com 3c503 if no WD/SMC cards ***/
    if (ifp->eth_vendor == VENDOR_NONE) {
	ifp->eth_asic_base = _3COM_BASE + _3COM_ASIC_OFFSET;
	ifp->eth_nic_base = _3COM_BASE;
	/*** ifp->eth_vendor = VENDOR_3COM; ***/
        /*
         * Note that we use the same settings for both 8 and 16 bit cards:
         * both have an 8K bank of memory at page 1 while only the 16 bit
         * cards have a bank at page 0.
         */
	ifp->eth_memsize = MEM_16384;
	ifp->eth_tx_start = 32;

        /* Check our base address */

	switch(inb(ifp->eth_asic_base + _3COM_BCFR)) {
	    case _3COM_BCFR_300:
		if ((int)ifp->eth_nic_base != 0x300) return(0);
		break;
	    case _3COM_BCFR_310:
		if ((int)ifp->eth_nic_base != 0x310) return(0);
		break;
	    case _3COM_BCFR_330:
		if ((int)ifp->eth_nic_base != 0x330) return(0);
		break;
	    case _3COM_BCFR_350:
		if ((int)ifp->eth_nic_base != 0x350) return(0);
		break;
	    case _3COM_BCFR_250:
		if ((int)ifp->eth_nic_base != 0x250) return(0);
		break;
	    case _3COM_BCFR_280:
		if ((int)ifp->eth_nic_base != 0x280) return(0);
		break;
	    case _3COM_BCFR_2A0:
		if ((int)ifp->eth_nic_base != 0x2a0) return(0);
		break;
	    case _3COM_BCFR_2E0:
		if ((int)ifp->eth_nic_base != 0x2e0) return(0);
		break;
	    default:
		return (0);
	}

        /* Now get the shared memory address */

	switch (inb(ifp->eth_asic_base + _3COM_PCFR)) {
	    case _3COM_PCFR_DC000:
		ifp->eth_bmem = (char *)0xdc000;
		break;
	    case _3COM_PCFR_D8000:
		ifp->eth_bmem = (char *)0xd8000;
		break;
	    case _3COM_PCFR_CC000:
		ifp->eth_bmem = (char *)0xcc000;
		break;
	    case _3COM_PCFR_C8000:
		ifp->eth_bmem = (char *)0xc8000;
		break;
	    default:
		return (0);
	}

        /* Need this to make ifp->eth_poll() happy. */

	ifp->eth_rmem = ifp->eth_bmem - 0x2000;

        /* Reset NIC and ASIC */

	outb (ifp->eth_asic_base + _3COM_CR , _3COM_CR_RST | _3COM_CR_XSEL);
	outb (ifp->eth_asic_base + _3COM_CR , _3COM_CR_XSEL);

        /* Get our ethernet address */

	outb(ifp->eth_asic_base + _3COM_CR, _3COM_CR_EALO | _3COM_CR_XSEL);
	printf("\n3Com 3c503 base 0x%x, memory 0x%X addr ",
		     ifp->eth_nic_base, ifp->eth_bmem);
	{
	    u_char *p= (u_char *)&(ifp->eth_node_addr);
	    for (i=0; i<6; i++) {
		p[i]=inb(i+ifp->eth_nic_base);
		printf("%b",(int)(p[i]));
		if (i < 5) printf (":");
	    }
	}
	outb(ifp->eth_asic_base + _3COM_CR, _3COM_CR_XSEL);
        /*
         * Initialize GA configuration register. Set bank and enable shared
         * mem. We always use bank 1.
         */
	outb(ifp->eth_asic_base + _3COM_GACFR, _3COM_GACFR_RSEL |
		_3COM_GACFR_MBS0);

	outb(ifp->eth_asic_base + _3COM_VPTR2, 0xff);
	outb(ifp->eth_asic_base + _3COM_VPTR1, 0xff);
	outb(ifp->eth_asic_base + _3COM_VPTR0, 0x00);
        /*
         * Clear memory and verify that it worked (we use only 8K)
         */
	bzero(ifp->eth_bmem, 0x2000);
	for(i = 0; i < 0x2000; ++i)
	    if (*((ifp->eth_bmem)+i)) {
		printf ("Failed to clear 3c503 shared mem.\n");
		goto _3com_not_found;
	    }
        /*
         * Initialize GA page/start/stop registers.
         */
	outb(ifp->eth_asic_base + _3COM_PSTR, ifp->eth_tx_start);
	outb(ifp->eth_asic_base + _3COM_PSPR, ifp->eth_memsize);
	ifp->eth_vendor = VENDOR_3COM;

	goto found_board;
    }
_3com_not_found: ;
#endif
probe_ne: ;
#ifdef INCLUDE_NE
    /*** Search for NE1000/2000 if no WD/SMC or 3com cards ***/
    if (ifp->eth_vendor == VENDOR_NONE) {
	char romdata[16], testbuf[32];
	char test[] = "NE1000/2000 memory";

	ifp->eth_bmem = NULL;	/* No shared memory */
	ifp->eth_asic_base = base_addr + NE_ASIC_OFFSET;
	ifp->eth_nic_base = base_addr;
	/* ifp->eth_vendor = VENDOR_NOVELL; */
	ifp->eth_flags = FLAG_PIO;
	ifp->eth_memsize = MEM_16384;
	ifp->eth_tx_start = 32;
	c = inb(ifp->eth_asic_base + NE_RESET);
	outb(ifp->eth_asic_base + NE_RESET, c);
	inb(0x84);
	outb(ifp->eth_nic_base + ED_P0_CR, ED_CR_STP | ED_CR_RD2);
	outb(ifp->eth_nic_base + D8390_P0_RCR, D8390_RCR_MON);
	outb(ifp->eth_nic_base+ D8390_P0_DCR, D8390_DCR_FT1 | D8390_DCR_LS);
	outb(ifp->eth_nic_base + D8390_P0_PSTART, MEM_8192);
	outb(ifp->eth_nic_base + D8390_P0_PSTOP, MEM_16384);
	eth_pio_write(ifp, test, 8192, sizeof(test));
	eth_pio_read(ifp, 8192, testbuf, sizeof(test));
	if (!bcompare(test, testbuf, sizeof(test))) {
	    ifp->eth_flags |= FLAG_16BIT;
	    ifp->eth_memsize = MEM_32768;
	    ifp->eth_tx_start = 64;
	    outb(ifp->eth_nic_base + D8390_P0_DCR, D8390_DCR_WTS |
		    D8390_DCR_FT1 | D8390_DCR_LS);
	    outb(ifp->eth_nic_base + D8390_P0_PSTART, MEM_16384);
	    outb(ifp->eth_nic_base + D8390_P0_PSTOP, MEM_32768);
	    eth_pio_write(ifp, test, 16384, sizeof(test));
	    eth_pio_read(ifp, 16384, testbuf, sizeof(test));
	    if (!bcompare(testbuf, test, sizeof(test)))
		goto ne_notfound;
	}
	eth_pio_read(ifp, 0, romdata, 16);
	printf("\nNE1000/NE2000 base 0x%x, addr ", ifp->eth_nic_base);
	{
	    u_char *p= (u_char *)&(ifp->eth_node_addr);
	    for (i=0; i<6; i++) {
		p[i]= romdata[i + ((ifp->eth_flags & FLAG_16BIT) ? i : 0)];
		printf("%b",(int)(p[i]));
		if (i < 5) printf (":");
	    }
	}
	printf(" mem 0x%x", ifp->eth_memsize*256);
	ifp->eth_vendor = VENDOR_NOVELL;
	goto found_board;
    }
ne_notfound:
#endif
found_board:
    printf("\n");
    if (ifp->eth_vendor == VENDOR_NONE) {
#ifdef SINK
	printf("SINK: addr 0x%x mem0x%X\r",base_addr, mem_seg);
	ifp->eth_vendor=VENDOR_SINK;
	ifp->eth_node_addr[0]=base_addr;
	ifp->eth_node_addr[1]= (mem_seg >>16);
	ifp->eth_node_addr[2]=mem_seg & 0xffff;
#else
	return 0;
#endif
    }

    if (ifp->eth_vendor != VENDOR_3COM) ifp->eth_rmem = ifp->eth_bmem;

    if (ifp->eth_flags & FLAG_790)
        ifp->proto=0;
    else    
        ifp->proto= ED_CR_RD2;

    eth_reset(ifp);
    ifp->flags=0;	/* clear operating flags */
    ifp->queued=0;	/* clear operating flags */
    ifp->rst_ctr=0;	/* number of reset */
    ifp->tx_head=ifp->tx_tail=NULL;
    ifp->tx_delay=0;
    return(ifp->eth_vendor);
}

/***
 *** eth_reset - Reset adapter
 ***/
eth_reset(struct if_vars *ifp)
{
    int n, i;
    
    ifp->rst_ctr++;
    print_addr(23, 18+i*14, &(ifp->rst_ctr), -2);
#ifdef	SINK
    if (ifp->eth_vendor ==VENDOR_SINK) return;
#endif

    outb(ifp->eth_nic_base+ED_P0_CR,
		ED_CR_PAGE_0 | ifp->proto | ED_CR_STP);

	/*** wait (for a limited time...) the board to reset ***/
    n=5000;
    while (((inb(ifp->eth_nic_base+D8390_P0_ISR) & D8390_ISR_RST) == 0)
	    && --n);

    ifp->flags &= ~IF_SENDING;
/*** this was in ed_init in the original driver ***/

    /*
     * Set FIFO threshold to 8, No auto-init Remote DMA, byte
     * order=80x86, word/byte-wide DMA xfers
     */
    outb(ifp->eth_nic_base+D8390_P0_DCR,
	D8390_DCR_FT1 | D8390_DCR_LS |
	(ifp->eth_flags & FLAG_16BIT ? D8390_DCR_WTS : 0) );

    /*
     * Clear Remote Byte Count Registers
     */
    outb(ifp->eth_nic_base+D8390_P0_RBCR0, 0);
    outb(ifp->eth_nic_base+D8390_P0_RBCR1, 0);

    outb(ifp->eth_nic_base+D8390_P0_RCR,
	D8390_RCR_AB |	/* allow broadcast frames */
	D8390_RCR_AM |	/* allow multicast frames */
	D8390_RCR_PRO );/* allow promiscuous mode */

    outb(ifp->eth_nic_base+D8390_P0_TCR, 2);
    outb(ifp->eth_nic_base+D8390_P0_TPSR, ifp->eth_tx_start);
    outb(ifp->eth_nic_base+D8390_P0_PSTART, ifp->eth_tx_start+D8390_TXBUF_SIZE);
    if (ifp->eth_flags & FLAG_790)
	outb(ifp->eth_nic_base + 0x09, 0);
    outb(ifp->eth_nic_base+D8390_P0_PSTOP, ifp->eth_memsize);
    outb(ifp->eth_nic_base+ED_P0_BNRY, ifp->eth_tx_start+D8390_TXBUF_SIZE);
    outb(ifp->eth_nic_base+D8390_P0_ISR, 0xFF);	/* clear int */
    outb(ifp->eth_nic_base+D8390_P0_IMR, 0);	/* disable int */
    outb(ifp->eth_nic_base+ED_P0_CR, ED_CR_PAGE_1 |
		ifp->proto | ED_CR_STP);
    {
	char *p= (char *)(ifp->eth_node_addr);
	for (i=0; i<6; i++)
	    outb(ifp->eth_nic_base+D8390_P1_PAR0+i, p[i]);
    }
    for (i=0; i<8; i++)	/* XXX this is wrong in ns8390.c */
	outb(ifp->eth_nic_base+D8390_P1_MAR0+i, 0xFF);
    outb(ifp->eth_nic_base+D8390_P1_CURR, ifp->eth_tx_start+D8390_TXBUF_SIZE+1);
    outb(ifp->eth_nic_base+ED_P0_CR, ED_CR_PAGE_0 |
		    ifp->proto | ED_CR_STA);
    outb(ifp->eth_nic_base+D8390_P0_ISR, 0xFF);
    outb(ifp->eth_nic_base+D8390_P0_TCR, 0);
#ifdef INCLUDE_3COM
    if (ifp->eth_vendor == VENDOR_3COM) {
	/*
	 * No way to tell whether or not we're supposed to use
         * the 3Com's transceiver unless the user tells us.
         * 'aui' should have some compile time default value
         * which can be changed from the command menu.
         */
	outb(ifp->eth_asic_base + _3COM_CR, _3COM_AUI);
    }
#endif
    return(1);
}

eth_transmit(struct if_vars *ifp, u_short s, char *p)
{
    u_char c;

#ifdef	SINK
    if (ifp->eth_vendor==VENDOR_SINK) return 1;
#endif
	/*** the following is redundant... */
    if (inb(ifp->eth_nic_base+ED_P0_CR) & ED_CR_TXP)
	return 1;
    ifp->flags |= IF_SENDING ;
    COUNT(ifp->name + TXPKT_CTR);
#ifdef INCLUDE_3COM
    if (ifp->eth_vendor == VENDOR_3COM) {
	bcopy(p, ifp->eth_bmem, s);
	while (s < ETH_MIN_PACKET) *(ifp->eth_bmem+(s++)) = 0;
	goto tx_end;
    }
#endif
#ifdef INCLUDE_WD
    if (ifp->eth_vendor == VENDOR_WD) {		/* Memory interface */
	if (ifp->eth_flags & FLAG_16BIT) {
	    outb(ifp->eth_asic_base + WD_LAAR, ifp->eth_laar | WD_LAAR_M16EN);
	    inb(0x84);
	}
	if (ifp->eth_flags & FLAG_790) {
	    outb(ifp->eth_asic_base + WD_MSR, WD_MSR_MENB);
	    inb(0x84);
	}
	inb(0x84);
	if (ifp->eth_flags & FLAG_16BIT)
	    wcopy(p, ifp->eth_bmem, (s+1)/2);
	else
	    bcopy(p, ifp->eth_bmem, s);

	while (s < ETH_MIN_PACKET) *(ifp->eth_bmem+(s++)) = 0;
	if (ifp->eth_flags & FLAG_790) {
	    outb(ifp->eth_asic_base + WD_MSR, 0);
	    inb(0x84);
	}
	if (ifp->eth_flags & FLAG_16BIT) {
	    outb(ifp->eth_asic_base + WD_LAAR, ifp->eth_laar & ~WD_LAAR_M16EN);
	    inb(0x84);
	}
	goto tx_end;
    }
#endif
#ifdef INCLUDE_NE
    if (ifp->eth_vendor == VENDOR_NOVELL) {	/* Programmed I/O */
	eth_pio_write(ifp, p, (ifp->eth_tx_start<<8), s);
	if (s < ETH_MIN_PACKET) s = ETH_MIN_PACKET;
    }
#endif
tx_end:
    outb(ifp->eth_nic_base+ED_P0_CR, ED_CR_PAGE_0 |
		ifp->proto | ED_CR_STA);
    outb(ifp->eth_nic_base+D8390_P0_TPSR, ifp->eth_tx_start);
    outb(ifp->eth_nic_base+D8390_P0_TBCR0, s);
    outb(ifp->eth_nic_base+D8390_P0_TBCR1, s>>8);
    outb(ifp->eth_nic_base+ED_P0_CR, ED_CR_PAGE_0 |
		ED_CR_TXP | ifp->proto | ED_CR_STA);
    return(0);
}

/***
 *** eth_poll - Wait for a frame
 ***/
eth_poll(struct if_vars *ifp)
{
    u_char bound,curr,rstat;
    u_short len, pktoff;
    u_char *p, *tmp;
    struct ringbuffer pkthdr;

    struct pkt_buf *newp;

#ifdef	SINK
    if (ifp->eth_vendor==VENDOR_SINK) return 1;
#endif

    /*** XXX added ED_CR_STA, lr 961102 ***/
    outb(ifp->eth_nic_base+ED_P0_CR, ED_CR_PAGE_0 | ifp->proto | ED_CR_STA);

    rstat = inb(ifp->eth_nic_base+D8390_P0_RSR);

    if (rstat & (D8390_RSTAT_OVER|D8390_RSTAT_MPA|D8390_RSTAT_DIS) ) {
	eth_reset(ifp);
	return(0);
    }
    if (!(rstat & D8390_RSTAT_PRX)) return(0);
	/*
	 * here we should have new data
	 */

    bound = inb(ifp->eth_nic_base+ED_P0_BNRY)+1;
    if (bound == ifp->eth_memsize)
	bound = ifp->eth_tx_start + D8390_TXBUF_SIZE;
    outb(ifp->eth_nic_base+ED_P0_CR, ED_CR_PAGE_1 | ifp->proto | ED_CR_STA);
    curr = inb(ifp->eth_nic_base+D8390_P1_CURR);
    outb(ifp->eth_nic_base+ED_P0_CR, ED_CR_PAGE_0);
    if (curr == ifp->eth_memsize)
	curr=ifp->eth_tx_start + D8390_TXBUF_SIZE;
    if (curr == bound)
	return(0);
    if (ifp->eth_vendor == VENDOR_WD) {
	if (ifp->eth_flags & FLAG_16BIT) {
	    outb(ifp->eth_asic_base + WD_LAAR, ifp->eth_laar | WD_LAAR_M16EN);
	    inb(0x84);
	}
	if (ifp->eth_flags & FLAG_790) {
	    outb(ifp->eth_asic_base + WD_MSR, WD_MSR_MENB);
	    inb(0x84);
	}
    }
    pktoff = (bound << 8);
    if (ifp->eth_flags & FLAG_PIO)
	eth_pio_read(ifp, pktoff, (char *)&pkthdr, 4);
    else
	bcopy(ifp->eth_rmem + pktoff, &pkthdr, 4);
    len = pkthdr.len - 4; /* sub CRC */
	/*
	 * should deal with buggy NICs which duplicate the low byte
	 * in the high byte.
	 */
    pktoff += 4;
	/*
	 * here read the remaining of the packet, if must be retransmitted.
	 */
    if (len > 1514) len = 1514; /* isn't it ETH_MAX_PACKET ? */
    bound = pkthdr.bound;		/* New bound ptr */
    if (ifp->eth_flags & FLAG_PIO)
	eth_pio_read(ifp, pktoff, (char *)&packet, ETHER_HDR_SIZE);
    else
	bcopy(ifp->eth_rmem + pktoff, (char *)&packet, ETHER_HDR_SIZE);

    /***
     *** first bridging decision: should we drop the packet ?
     ***/
    if (bdg_drop(ifp, &packet, len))
	goto eth_drop;

    if ( (pkthdr.status & D8390_RSTAT_PRX) &&
	    (len > ETHER_HDR_SIZE) && (len <= ETH_MAX_PACKET)) {
	packetlen = len;
	/***
	 *** get a buffer from the free list to store the packet
	 ***/
	if ( (newp= buf_alloc())==NULL ) {
	    COUNT(ifp->name + DROP_CTR);
	    goto eth_drop;
	}
	newp->len = packetlen;
	p= (char *)&(newp->pkt);
	/***
	 *** compute remaining room in buffer ring.
	 ***/
	len = (ifp->eth_memsize << 8) - pktoff;
	if (len < packetlen) {		/* We have a wrap-around */
	    u_short newoff = (ifp->eth_tx_start + D8390_TXBUF_SIZE) << 8;
	    if (ifp->eth_flags & FLAG_PIO) {
		eth_pio_read(ifp, pktoff, p, len);
		eth_pio_read(ifp, newoff, p+len, packetlen - len);
	    } else {
		if (ifp->eth_flags & FLAG_16BIT) {
		    wcopy(ifp->eth_rmem + pktoff, p, len/2);
		    wcopy(ifp->eth_rmem + newoff, p+len, (packetlen+1 - len)/2);
		} else {
		    bcopy(ifp->eth_rmem + pktoff, p, len);
		    bcopy(ifp->eth_rmem + newoff, p+len, packetlen - len);
		}
	    }
	} else {
	    if (ifp->eth_flags & FLAG_PIO)
		eth_pio_read(ifp, pktoff, p, packetlen);
	    else {
		if (ifp->eth_flags & FLAG_16BIT)
		    wcopy(ifp->eth_rmem + pktoff, p, (packetlen+1)/2);
		else
		    bcopy(ifp->eth_rmem + pktoff, p, packetlen);
	    }
	}

	/***
	 *** Bridging #2: link the outgoing packet in the
	 *** relevant queues.
	 ***/
	bdg_forward(ifp, newp);
	if (newp->count==0) buf_free(newp);
    } else {
	/*** XXX invalid ? ***/
        if ( pkthdr.status & ~(D8390_RSTAT_PRX | D8390_RSTAT_PHY) ) {
	    eth_reset(ifp);
	    return(0);
	}
    }
    /***
     *** enter here if the packet must be dropped
     ***/
eth_drop:
    if (ifp->eth_vendor == VENDOR_WD) {
	if (ifp->eth_flags & FLAG_790) {
	    outb(ifp->eth_asic_base + WD_MSR, 0);
	    inb(0x84);
	}
	if (ifp->eth_flags & FLAG_16BIT) {
	    outb(ifp->eth_asic_base + WD_LAAR, ifp->eth_laar &
		    ~WD_LAAR_M16EN);
	    inb(0x84);
	}
    }
    if (bound == (ifp->eth_tx_start + D8390_TXBUF_SIZE))
	bound = ifp->eth_memsize;

    /***
     *** The 'ed' driver does the following:
     ***
     ***	outb(ifp->eth_nic_base+ED_P0_CR,
     ***		sc->cr_proto | ED_CR_STA);
     ***	outb(...boundary...) as below
     ***	outb(ifp->eth_nic_base+ED_P0_CR,
     ***		sc->cr_proto | ED_CR_PAGE_1 | ED_CR_STA);
     ***/
    outb(ifp->eth_nic_base+ED_P0_BNRY, bound-1);
    return;
}

#ifdef INCLUDE_NE
/***
 *** eth_pio_read - Read a frame via Programmed I/O
 ***/
eth_pio_read(struct if_vars *ifp, u_short src, char *dst, u_short cnt)
{
    if (cnt & 1) cnt++;

    /*** abort any previous DMA ***/
    outb(ifp->eth_nic_base + ED_P0_CR, ED_CR_RD2 | ED_CR_STA);

    outb(ifp->eth_nic_base + D8390_P0_RBCR0, cnt);
    outb(ifp->eth_nic_base + D8390_P0_RBCR1, cnt>>8);
    outb(ifp->eth_nic_base + D8390_P0_RSAR0, src);
    outb(ifp->eth_nic_base + D8390_P0_RSAR1, src>>8);

    /*** remote read ***/
    outb(ifp->eth_nic_base + ED_P0_CR, ED_CR_RD0 | ED_CR_STA);

    if (ifp->eth_flags & FLAG_16BIT)
	insw(ifp->eth_asic_base + NE_DATA, dst, cnt/2);
    else
	insb(ifp->eth_asic_base + NE_DATA, dst, cnt);
}

/***
 *** eth_pio_write - Write a frame via Programmed I/O
 ***/
eth_pio_write(struct if_vars *ifp, u_char *src, u_short dst, u_short cnt)
{
    int maxwait = 200; /* about 240us -- are we sure ? */

    /*** abort any previous DMA ***/
    outb(ifp->eth_nic_base + ED_P0_CR, ED_CR_RD2 | ED_CR_STA);
    outb(ifp->eth_nic_base + D8390_P0_ISR, D8390_ISR_RDC);
    outb(ifp->eth_nic_base + D8390_P0_RBCR0, cnt);
    outb(ifp->eth_nic_base + D8390_P0_RBCR1, cnt>>8);
    outb(ifp->eth_nic_base + D8390_P0_RSAR0, dst);
    outb(ifp->eth_nic_base + D8390_P0_RSAR1, dst>>8);

    /*** remote write ***/
    outb(ifp->eth_nic_base + ED_P0_CR, ED_CR_RD1 | ED_CR_STA);
    if (ifp->eth_flags & FLAG_16BIT) {
	if (cnt & 1) cnt++;		/* Round up */
	outsw(ifp->eth_asic_base + NE_DATA, src, cnt/2);
    }
    else {
	outsb(ifp->eth_asic_base + NE_DATA, src, cnt);
    }
    /*
     * wait for the remote DMA to complete
     */
    while( ((inb(ifp->eth_nic_base + D8390_P0_ISR) & D8390_ISR_RDC)
	    != D8390_ISR_RDC) && --maxwait) ;
    if (maxwait == 0) {
	eth_reset(ifp);
    }
}
#else
/***
 *** eth_pio_read - Dummy routine when NE2000 not compiled in
 ***/
eth_pio_read(struct if_vars *ifp) {}
#endif
