/*
 * Addons for extra webcam support in the pwcbsd driver
 *
 *  Copyright (C) 2007 Luigi Rizzo
 *
 * Based on the Linux pwc driver.
 *  
 *  Copyright (C) 1999-2003 Nemosoft Unv.
 *  Copyright (C) 2004-2006 Luc Saillard
 *  ...
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301  USA
 */

#include "pwc.h"

#include "spca5xx.h"

#include "hdcs.h"
#include "pb0100.h"
#include "vv6410.h"
#include "quickcam.h"

/*
 * Support routines extracted from the qce-ga driver
 */

/*
 * In order to identify the sensor, we read at the address specified
 * by id_reg, for a size of l bytes, and look for a match of the last
 * byte with the id specified.
 */
static struct sensor_data sensors [] = {
  /* name           reg23 i2c_addr     id_reg          id    l  load           */
  /* ------------   ----- -----------  --------------  ----  -  -------------- */
  { SENSOR_HDCS1000,   0, HDCS_ADDR,   HDCS_IDENT + 1, 0x08, 1, load_hdcs_mod   },
  { SENSOR_BP100,      1, PB_ADDR,     PB_IDENT,       0x64, 2, load_pb0100_mod },
  { SENSOR_VV6410,     5, VV6410_ADDR, VV6410_IDENT,   0x19, 1, load_vv6410_mod },
  { SENSOR_HDCS1020,   0, HDCS_ADDR,   HDCS_IDENT + 1, 0x10, 1, load_hdcs20_mod },
  { -1 }
};

static int usb_quickcam_i2c_in(struct usb_device *dev, int reg,
         unsigned char sensor_addr)
{ 
	char buff[35]; /* why 35 = 23 hex? */
  
	printf("usb_quickcam_i2c_in reg 0x%x addr 0x%x a\n", reg, sensor_addr);
        buff[0]=reg;
 
        buff[0x20]=sensor_addr;
        buff[0x21]=0; // 1 value
        buff[0x22]=3; // Read cmd.
 
        return usb_control_msg(dev, usb_sndctrlpipe(dev, 0),
                0x04,
                UT_WRITE_VENDOR_DEVICE,   /* UT_WRITE | UT_VENDOR | UT_DEVICE */
                0x1400, 0,
                buff, sizeof(buff), HZ);
}
  
/* read one byte identification register for HDCS.
 * write command to sensor.
 * read the STV0600.
 */
int usb_quickcam_get_i2c(struct usb_device *dev, unsigned char sensor_addr, int reg, void *buf, int len)
{
        if (usb_quickcam_i2c_in(dev,reg,sensor_addr)<0) {
                printf("usb_quickcam_i2c_in failed\n");
                return(-1);
	}
        return usb_control_msg(dev, usb_rcvctrlpipe(dev, 0),
                0x04,
                UT_READ_VENDOR_DEVICE,   /* UT_READ | UT_VENDOR | UT_DEVICE */
                0x1410, 0, buf, len, HZ);
}
 
/*
 * Set register one byte 
 */
int usb_quickcam_set1(struct usb_device *dev, short reg, char val)
{
	char buff[1];
        buff[0] = val;
        return usb_control_msg(dev, usb_sndctrlpipe(dev, 0),
                0x04,  UT_WRITE_VENDOR_DEVICE, reg, 0,  buff, 1, HZ);
}

/*
 * Set register two byte   
 */
int usb_quickcam_set2(struct usb_device *dev, short reg, short val)
{
	char buff[2];
        buff[0] = val&0xFF;
        buff[1] = (val>>8)&255;

        return usb_control_msg(dev, usb_sndctrlpipe(dev, 0),
                0x04, UT_WRITE_VENDOR_DEVICE, reg, 0, buff, 2, HZ);
}
/* Send a command to the sensor */

static int quickcam_usb_control_msg(struct usb_device *dev, char *buff)
{
        return usb_control_msg(dev, usb_sndctrlpipe(dev, 0),
                0x04, UT_WRITE_VENDOR_DEVICE, 0x400, 0, buff, 0x23 , HZ);
}

/**
 * Clean buffer for I2C messages.
 */
void usb_quickcam_i2c_new(struct quickcam_i2c *i2cbuff)
{
        i2cbuff->length=0;
        memset(i2cbuff->buff,'\0',0x23);
}

/**
 * Add register and byte value.
 */
void usb_quickcam_i2c_add(struct quickcam_i2c *i2cbuff,
unsigned char reg, unsigned char value)
{
        i2cbuff->buff[i2cbuff->length] = reg;
        i2cbuff->buff[i2cbuff->length+0x10] = value;
        i2cbuff->length++;
}

/**
 * Add register and 2 bytes value.
 */
void usb_quickcam_i2c_add2(struct quickcam_i2c *i2cbuff,
unsigned char reg, unsigned short value)
{
        i2cbuff->buff[i2cbuff->length] = reg;
        i2cbuff->buff[(i2cbuff->length*2)+0x10] = value&255;
        i2cbuff->buff[(i2cbuff->length*2)+0x11] = (value>>8)&255;
        i2cbuff->length++;
}

/**
 * Send the I2C message.
 */
int usb_quickcam_i2c_send(struct usb_device *dev, struct quickcam_i2c *i2cbuff,
unsigned char sensor_add)
{
	int ret;

        i2cbuff->buff[0x20]=sensor_add;
        i2cbuff->buff[0x21]=i2cbuff->length-1;
        i2cbuff->buff[0x22]=1; // Write cmd, 03 would be read.
        ret = quickcam_usb_control_msg(dev,i2cbuff->buff);
        usb_quickcam_i2c_new(i2cbuff);
        return(ret);
}

int qc_probe_sensor(struct pwc_softc *sc)
{
	struct usb_device *dev = sc->udev;
	struct sensor_data *sensor;
	unsigned char id[2];

	if (dev == NULL) {	/* this can happen in the probe routine */
		printf("%s dev is null\n", __FUNCTION__);
		return 0;
	}
	if (usb_quickcam_set1(dev, STV_ISO_ENABLE, 0) < 0) {
		printf("usb_quickcam_set1 STV_ISO_ENABLE(0) failed\n");
		return -EBUSY;
	}
        /* Probe for the sensor type. */
        for (sensor = sensors; sensor->name >= 0; sensor++) {
                if (usb_quickcam_set1(dev, STV_REG23, sensor->reg23) < 0) {
                        printf("usb_quickcam_set1 STV_REG23 failed\n");
                        return -EBUSY;
                }
                if (usb_quickcam_get_i2c(dev, sensor->i2c_addr, sensor->id_reg, id, sensor->length_id) < 0) {
                        printf("usb_quickcam_get_i2c error\n");
                        return -EBUSY;
                }
                printf("quickcam: probe of sensor %d = %02x %02x id: %02x\n", sensor->name, id[0], id[1],sensor->id);
                if (id[sensor->length_id-1] == sensor->id) {
			printf("found sensor %d i2c base %d\n", sensor->name, sensor->i2c_addr);
                        break;
		}
        }
	sc->pwc_info.sensor = sensor->name;
	sc->spca50x.i2c_base = sensor->i2c_addr;
	sensor->load(&(sc->sensor_ctrl) );
	return 0;	/* success */
}

int config_quickcam(struct usb_spca50x *spca50x)
{
	struct usb_device *dev = spca50x->dev;

        /* Probe for the sensor type already done */

        // Disable data stream.
        if (usb_quickcam_set1(dev, STV_ISO_ENABLE, 0) < 0) {
                printf("usb_quickcam_set1 STV_ISO_ENABLE(0) failed\n");
                return -EBUSY;
        }
        if (usb_quickcam_set1(dev, STV_REG23, 1) < 0) {
                /* I see no reason that the others failed if this one is OK */
                printf("usb_quickcam_set1 STV_REG23(1) failed\n");
                return -EBUSY;
        }
    memset(spca50x->mode_cam, 0x00, TOTMODE * sizeof(struct mwebcam));
    spca50x->mode_cam[SIF].width = 352;
    spca50x->mode_cam[SIF].height = 288;
    spca50x->mode_cam[SIF].t_palette =
        P_RAW | P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[SIF].pipe = 1023;
    spca50x->mode_cam[SIF].method = 0;

    spca50x->mode_cam[CIF].width = 320;
    spca50x->mode_cam[CIF].height = 240;
    spca50x->mode_cam[CIF].t_palette =
        P_RAW | P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[CIF].pipe = 1023;
    spca50x->mode_cam[CIF].method = 0;

#if 0
        /* set default values for the camera */
        if(bright<=0 || bright>=65535) bright=EXPO_VAL;
        quickcam->brightness = bright;
        quickcam->shutter_val= SHUTTER_VAL;
        quickcam->scanstate  = STATE_OK;
        quickcam->x          = quickcam->yy=0;
        quickcam->readframe  = -1;

        // Setup internal video_picture
        quickcam->vpic.hue        = (rgain -bgain +0xFF) << 7;
        quickcam->vpic.colour     = ggain;
        quickcam->vpic.contrast   = 32768;
        quickcam->vpic.brightness = bright;
        quickcam->vpic.whiteness  = 0;
        quickcam->vpic.palette    = VIDEO_PALETTE_RGB24;
        quickcam->vpic.depth      = quickcam_get_depth(quickcam);

        // Setup internal video_window
        quickcam->vwin.x = 0;
        quickcam->vwin.y = 0;
        quickcam->vwin.chromakey = 0;
        quickcam->vwin.flags = 30;              /* 30 fps */
#endif

        printf("config_quickcam\n");
        return 0;
}

int qc_sensor_init(struct usb_spca50x *spca50x)
{
	static int rgain = 0, bgain = 0, ggain = 0;
	static int mode = 0; /* 0=normal, 1=subsample (faster) */
	struct sensorctrl *sensor_ctrl = &(PWC_SC(spca50x)->sensor_ctrl) ;

	if (spca50x->streaming == 1) {
		/* stop streaming on the ISOC endpoint */
		if (usb_quickcam_set1(spca50x->dev, STV_ISO_ENABLE, 0) < 0) {
			printf("usb_quickcam_set1 STV_ISO_ENABLE(0) failed\n");
			return -EBUSY;
		}
		sensor_ctrl->stop(spca50x->dev, sensor_ctrl);
	}
        if (sensor_ctrl->init(spca50x->dev,mode, &rgain, &bgain, &ggain, sensor_ctrl)<0) {
                printf("qc_sensor_init failed\n");
                return(-1);
        }
#if 0
/* gain is the average of blue, green and red gain */
        if(!keepexposure) {
                quickcam->gain =10;
        }
        quickcam->blue =qcmin(255,qcmax(2,bgain));
        quickcam->red  =qcmin(255,qcmax(2,rgain));
        quickcam->green=qcmin(255,qcmax(2,ggain));
        if (usb_quickcam_set_gains(quickcam)<0) {
                printk("set_gains sensor failed\n");
                return(-1);
        }
#endif
        if (sensor_ctrl->set_shutter(spca50x->dev, 0x80 /* val */,0x100, sensor_ctrl)<0) {
                printf("set_shutter sensor failed\n");
                return(-1);
        }
        /* Set the size otherwise the read() will fail */
        /*
         * JFC have to arrange this .... and use the value from the quickcam structure!
         * NO: It is called in open() ... For the moment...
         */

        if (sensor_ctrl->set_size(spca50x->dev, mode)<0) {
		printf(" set size failed\n");
                return(-1);
	}
	if (sensor_ctrl->set_window(spca50x->dev, 0, 0, 352, 288, sensor_ctrl)<0) {
                printf("set_window sensor failed\n");
                return(-1);
        }

	/* XXX start grabbing ? */
	if (sensor_ctrl->start(spca50x->dev, sensor_ctrl) < 0 ) {
		printf("failed to start capture\n");
		return -1;
	}
	/* actually enable streaming on the ISOC endpoint */
	if (usb_quickcam_set1(spca50x->dev, STV_ISO_ENABLE, 1) < 0) {
		printf("usb_quickcam_set1 STV_ISO_ENABLE(1) failed\n");
		return -EBUSY;
	}
        return 0;
}

int spca561_consume(struct pwc_softc *sc, unsigned char *iso_buf, int flen)
{
	int awake = 0;
	int seq;
	int skip_bytes;
	struct pwc_frame_buf *fbuf = sc->fill_frame;

	if (flen == 0)
		return 0;
	seq = iso_buf[SPCA50X_OFFSET_SEQUENCE];
	if (seq == 0xff)
		return 0;	/* drop packet */
	if(sc->vsync != VSYNC_SYNCHED) {
		if (seq != 0) /* sync hunting */
			return 0;
		sc->vsync = VSYNC_SYNCHED;
		printf("found sync\n");
	}
	skip_bytes = (seq == 0) ? 10 : 1;
	if (seq == 0 && fbuf->filled > 0) { /* new frame */
		awake = 1;
		pwc_next_fill_frame(sc);
		fbuf = sc->fill_frame;
		fbuf->filled = 0;
	}
	if (fbuf->filled + flen - skip_bytes > sc->frame_total_size) {
		/* error */
		sc->vsync = VSYNC_SYNCHUNT;
		fbuf->filled = 0;
	} else {
		memcpy(fbuf->data + fbuf->filled, iso_buf + skip_bytes, flen - skip_bytes);
		fbuf->filled += flen - skip_bytes;
	}
	return awake;
}

int zc3xx_consume(struct pwc_softc *sc, unsigned char *iso_buf, int flen)
{
	int awake = 0;
	int skip_bytes = 0;
	struct pwc_frame_buf *fbuf = sc->fill_frame;

	if (flen == 0)
		return 0;
	if (iso_buf[0] == 0xff && iso_buf[1] == 0xd8) {
		sc->vsync = VSYNC_SYNCHED;
		skip_bytes = 2;
	} else if (sc->vsync != VSYNC_SYNCHED) {
		return 0;
	}
	if (skip_bytes > 0 && fbuf->filled > 0) { /* new frame */
                awake = 1;
		printf("%s: full frame size %d\n", __FUNCTION__, fbuf->filled);
                pwc_next_fill_frame(sc);
                fbuf = sc->fill_frame;
                fbuf->filled = 0;
        }
        if (fbuf->filled + flen - skip_bytes > sc->frame_total_size) {
                /* error */
                sc->vsync = VSYNC_SYNCHUNT;
                fbuf->filled = 0;
        } else {
                memcpy(fbuf->data + fbuf->filled, iso_buf + skip_bytes, flen - skip_bytes);
                fbuf->filled += flen - skip_bytes;
        }
        return awake;
}

int qc_consume(struct pwc_softc *sc, unsigned char *iso_buf, int flen)
{
	int awake = 0;	/* need to awake the caller at the end */
	struct pwc_frame_buf *fbuf = sc->fill_frame;
	unsigned char *fillptr = fbuf->data + fbuf->filled;

	static char buf[8192]; /* XXX fixme */
	int more_data = 0;

	/* XXX there is no individual framestatus in FreeBSD usbstack
	 * so just assume all frames are good
	 */
	int framesize;

again:
	if (flen == 0)
		return awake;

	if(sc->vsync != VSYNC_SYNCHED) { /* sync hunting */
		int x;
		switch (sc->pwc_info.bridge) {
		default:
			printf("unsupported bridge %d, cannot find sync\n",
				sc->pwc_info.bridge);
			break;

		case BRIDGE_SPCA561:
			/* 10-byte header with various info, see spca5xx.h */
			printf("got %d bytes seqnum %d\n", flen, iso_buf[0]);
			if (iso_buf[SPCA50X_OFFSET_SEQUENCE] == 0)
				sc->vsync = VSYNC_SYNCHED;
			break;

		case BRIDGE_STV0602:
			/* see quickcam_parse_store */
			if (flen < 4)
				return 0;
			for (x = 0; x < flen - 4; x++) {
				if ( (iso_buf[x] & ~0x40) != 0x80 ||
				    (iso_buf[x+1] & ~0x4) != 0x01 ||
				    iso_buf[x+2] != 0 || iso_buf[x+3] != 0)
					continue;
				sc->vsync = VSYNC_SYNCHED;
				x += 4;
				flen -= x;
				iso_buf += x;
				fbuf->totlength = 0;
			}
			break;
		}
		if (sc->vsync != VSYNC_SYNCHED)
			return awake;
	}
	/* ...copy data to frame buffer, if possible */
	if (flen + fbuf->filled > sc->frame_total_size) {
		Trace(TRACE_ISOC, "Frame buffer overflow (flen = %d,frame_total_size = %d).\n",flen, sc->frame_total_size);

		goto done; // XXX try see what we get
	}
	else {
		memcpy(fillptr, iso_buf, flen);
		fillptr += flen;
	}

	fbuf->filled += flen;

	/* now find end of packet, if any */

	switch (sc->pwc_info.bridge) {
	default:
		printf("sorry, unknown bridge %d\n", sc->pwc_info.bridge);
		break;

	case BRIDGE_STV0602:
		framesize = sc->image.x*sc->image.y;	/* the full image */
		while (fbuf->totlength < fbuf->filled - 4) {
			unsigned char *xx = fbuf->data + fbuf->totlength;
			int l = xx[2]*256 + xx[3];
			if ((xx[0] & ~0x40) != 2 || xx[1] != 0 || fbuf->totlength + l >= fbuf->filled) {
				printf("missed header %x %x %x %x tot %d\n",
					xx[0], xx[1], xx[2], xx[3], fbuf->totlength);
				/* if we find a end-of-frame marker put a fake one */
				if ((xx[0] & ~0x40) == 0x80 && (xx[1] & ~0x4) == 2)
					fbuf->totlength = fbuf->filled = framesize;
				break;
			}
			/* skip the 4 header bytes, copy the block */
			memcpy(xx, xx + 4, l);
			fbuf->totlength += l+4;
		}
		if (fbuf->totlength >= framesize) {
			more_data = fbuf->filled - framesize;
			memcpy(buf, fbuf->data + framesize, more_data);
			fbuf->filled = framesize;
			printf("full frame left %d\n", more_data);
			goto done;
		}
		break;
	}
	goto eof_done;
		
	/* Shorter packet... We probably have the end of an image-frame; 
	   wake up read() process and let select()/poll() do something.
	   Decompression is done in user time over there.
	 */
	
	if (sc->vsync != VSYNC_SYNCHED)
		goto start_new_frame;
	
	/* In case we were instructed to drop the frame, do so silently.
	   The buffer pointers are not updated either (but the counters are reset below).
	 */
	if(sc->drop_frames > 0) {
		sc->drop_frames--;
	}
	/* Check for underflow first */
	else if(sc->pwc_info.type != 0 && fbuf->filled < sc->frame_total_size) {
		Trace(TRACE_ISOC,"Frame buffer underflow (have %d bytes need %d); discarded.\n", fbuf->filled, sc->frame_total_size);
		sc->vframes_error++;
	}
	else {
done: /* recognised frame */
		/* Send only once per EOF */
		awake = 1; /* delay wake_ups */
		/* Find our next frame to fill. This will always succeed, since we
		 * nick a frame from either empty or full list, but if we had to
		 * take it from the full list, it means a frame got dropped.
		 */
		pwc_next_fill_frame(sc);
		fbuf = sc->fill_frame;
	}
	sc->vframe_count++;
start_new_frame:
	fbuf->filled = 0;
	fillptr = fbuf->data;
	sc->vsync = VSYNC_SYNCHUNT;
	if (more_data) {
		flen = more_data;
		iso_buf = buf;
		more_data = 0;
		goto again;
	}
eof_done:
	sc->vlast_packet_size = flen;
	return awake;
}

