/* --------------------------------------------------------------------
   Project: PAL: Palmtop Application Library
   Module:  IHDATE.C
   Author:  John McDonald
	    (Based on Gilles Kohl original IHCOMBO.C)
   Started: 1 Jan. 1996
   Subject: Functions that implement the 'date box' dialog item class
   -------------------------------------------------------------------- */

/* --------------------------------------------------------------------
		       standard includes
   -------------------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define DATE_X 4
#define DATE_Y 29
#define DATE_X_OFF 24
#define DATE_Y_OFF 12
#define TITLE_OFF 25
#define DATE_WIDE 17
#define DATE_HIGH 10
#define X 0
#define Y 1
#define HIGH_X_OFF 24
#define HIGH_Y_OFF 12
#define HIGH_START_X 4
#define HIGH_START_Y 3
#define DAY 0
#define MON 1
#define YR 2
#define WIN_WIDTH 170
#define WIN_HEIGHT 102
#define ENTER 0x1C0D

/* --------------------------------------------------------------------
			 local includes
   -------------------------------------------------------------------- */
#include "pal.h"
#include "palpriv.h"

/* --------------------------------------------------------------------
			 global data
   -------------------------------------------------------------------- */
unsigned char DtArrowImg[] = {
   0x01, 0x00, 0x01, 0x00, 0x08, 0x00, 0x08, 0x00, 0xFF, 0xE7,
   0xE7, 0xE7, 0x81, 0xC3, 0xE7, 0xFF,
};

	int     DayDisp;
	int     CurDayX;
	int     CurDayY;
	char    MonthArray[7][6];
	char    FirstDay[2];
	char    LastDay[2];
	char    Today[2];
	int     CurHighDate[3];
	char    days_per_month[12] = 
		{ 31,28,31,30,31,30,31,31,30,31,30,31 };
	PALWIN  *pwin;

/* --------------------------------------------------------------------
			   functions
   -------------------------------------------------------------------- */


/**********************************************************
*
*       This routine will take a year as input and return
*       a 0 if it isn't a leap year and a 1 if it is
*
***********************************************************/

int ihDtLeapYr( int year )
/*
*   If the year is divisible by 4, it' a leap year
*   If it's divisible by 400, it's a leap year
*/
{
	int     retval;

	retval = 0;
	if ( !( year % 4 ) ) retval = 1;
	if ( !( year % 100 ) )
		{
		retval = 0;
		if ( !( year % 400 ) ) retval = 1;
		}
	return( retval );
}
/*************************************************************
*
*       This routine will create the month array and draw
*       the month
*
**************************************************************/
void ihDtCalDraw( int highlight_date[3], int first_dow, int WIN_X, int WIN_Y )
{
	int x,y;
	char buf[80];
	char months[12][10] = {
	"January","February","March","April","May","June",
	"July","August","September",
	" October"," November"," December" };
	char    done;
/*
*   Put up the header
*/
	SelectFont( SMALL_FONT );
	SetColor( WHITE_COLOR );
	Rectangle( WIN_X+1, WIN_Y+1, 
		   WIN_X+WIN_WIDTH-1, WIN_Y+14, SOLID_FILL );
	SetColor( BLACK_COLOR );
	sprintf( buf, "%s %4d", months[highlight_date[MON]], 
		 highlight_date[YR] );
	x = TextExt( SMALL_FONT, buf );
	WinText( pwin, (WIN_WIDTH - x )/2, 2, buf );
	SetColor( BLACK_COLOR );
	if ( ihDtLeapYr( highlight_date[YR] ))
		{
		days_per_month[1] = 29;
		}
	else
		{
		days_per_month[1] = 28;
		}

	CurDayX = DATE_X;
	CurDayY = DATE_Y;
	DayDisp = -1;
	memset( MonthArray, 0, 35 );
	LastDay[X] = LastDay[Y] = -1;
	done = FALSE;
	ClrBlock( WIN_X+1, WIN_Y+TITLE_OFF+1, 
		  WIN_X+WIN_WIDTH-1, WIN_Y+WIN_HEIGHT-1, 
		  WHITE_COLOR );
	for ( y=0; y<6; y++ )
		{
		for ( x=0; x<7; x++ )
			{
			if ( y == 0 && x == first_dow ) 
				{
				DayDisp = 1;
				FirstDay[X] = x;
				FirstDay[Y] = y;
				}
			else if ( y == 0 && x < first_dow )
				{
				DayDisp = -1;
				}
			else if ( DayDisp >= 
			     days_per_month[highlight_date[MON]] )
				{
				DayDisp = -1;
				if ( LastDay[X] == -1 )
					{
					LastDay[X] = x-1;
					LastDay[Y] = y;
					done = TRUE;
					break;
					}
				}
			else  
				{
				DayDisp++;
				}
			if ( DayDisp == highlight_date[DAY] )
				{
				Today[X] = x;
				Today[Y] = y;
				}
			MonthArray[x][y] = DayDisp;
			if ( DayDisp == -1 )
				{
				WinText( pwin, DATE_X + ( x * DATE_X_OFF ), 
					 DATE_Y + ( y * DATE_Y_OFF), "  " );
				}
			else
				{
				sprintf( buf, "%2d", DayDisp ); 
				WinText( pwin, DATE_X + (x * DATE_X_OFF ), 
					DATE_Y + ( y * DATE_Y_OFF),
					 buf );
				}
			}
			if ( done ) break;
		}
		if ( LastDay[X] == -1 )
			{
			LastDay[X] = 6;
			LastDay[Y] = y-1;
			}
}
/**********************************************************************
*
*       This routine will return the day of the week for a given
*       date using Zeller's Congruance
*
***********************************************************************/
int GetDOW( int day, int month, int year )
{
	int century;

	if(month > 2) month -= 2;
	else
		{
		month = month += 10;
		year -= 1;
		}
	century = year / 100;
	year = (year % 100);
	return((day - 1 + ((13 * month - 1) / 5) + (5 * year / 4) +
	  century / 4 - 2 * century + 1) % 7);
}
/********************************************************************
*
*   This routine will process the calendar if the user presses the
*   down arrow
*
*********************************************************************/

void IhDtProcCal( int WIN_X, int WIN_Y, int highlight_date[3], IHDATE *pDat )
{
	BYTE    fmask[] = {0x55,0xaa,0x55,0xaa,0x55,0xaa,0x55,0xaa};
	char    *keys[10] = { "Help", NULL, NULL, NULL, NULL, NULL,
			      NULL, NULL, NULL, "Exit" };
	char    buf[80];
	int     cur_high[4];
	int     keystroke;
	int     done;
	int     cur_high_row;
	int     cur_high_col;
	int     x,y;
	int     first_dow;
	time_t  CurDateTime;
	struct tm *pLocalTime;
	int     new_month[3];
	int     new_first_dow;
/*
*   Create the calendar window
*/
	pwin = OpenWin( WS_BORDER|WS_SHADOW|WS_SAVEBG|WS_SMLFONT, 
			WIN_X, WIN_Y, WIN_X+WIN_WIDTH, WIN_Y+WIN_HEIGHT,
			NULL );
	SetMask( fmask );
	SetColor( BLACK_COLOR );
	Line( WIN_X, WIN_Y+25, WIN_X+WIN_WIDTH, WIN_Y+25 );
	SelectFont( SMALL_FONT );
	WinText( pwin, 4, 14, "Su Mo Tu We Th Fr Sa" );
	y = 1;
/*
*   Find out what day of the week the 1st of the requested
*      month is
*/
	first_dow = GetDOW( 1, highlight_date[MON]+1, highlight_date[YR] );
/*
*   Draw the calendar
*/
	ihDtCalDraw( highlight_date, first_dow, WIN_X, WIN_Y );
	cur_high[0] = WIN_X + HIGH_START_X + ( Today[X] * HIGH_X_OFF ); 
	cur_high[1] = WIN_Y+ HIGH_START_Y + TITLE_OFF + 
		      ( Today[Y] * HIGH_Y_OFF );
	cur_high[2] = cur_high[0] + DATE_WIDE;
	cur_high[3] = cur_high[1] + DATE_HIGH;
	RevBlock( cur_high[0], cur_high[1], cur_high[2], cur_high[3] );
	cur_high_row = Today[Y];
	cur_high_col = Today[X];
	CurHighDate[DAY] = highlight_date[DAY];
	CurHighDate[MON] = highlight_date[MON];
	CurHighDate[YR] = highlight_date[YR];
/*
*   Process the keystrokes
*/
	keystroke = GetKey();
	done = FALSE;
	if ( keystroke == F10_KEY ||
	     keystroke == ESC_KEY ) done = TRUE;
	while ( !done )
		{
		switch ( keystroke )
		{
/*
* RIGHT KEY
*/
		case RIGHT_KEY:
			RevBlock( cur_high[0], cur_high[1], 
				  cur_high[2], cur_high[3] );
/*
*   Are we on the last day?
*   If so, go to the next month
*/
			if ( cur_high_row == LastDay[Y] &&
			     cur_high_col == LastDay[X] )
				{
				new_month[DAY] = 1;
				new_month[MON] = highlight_date[MON] + 1;
				new_month[YR] = highlight_date[YR];
				if ( new_month[MON] > 11 ) 
					{
					new_month[MON] = 0;
					new_month[YR]++;
					}
				first_dow = LastDay[X] + 1;
				if ( first_dow == 7 ) first_dow = 0;
				ihDtCalDraw( new_month, first_dow, 
					     WIN_X, WIN_Y );
				memcpy( highlight_date, new_month,
					3 * sizeof(int) );
				cur_high[0] = WIN_X + HIGH_START_X + 
					( FirstDay[X] * HIGH_X_OFF ); 
				cur_high[1] = WIN_Y+TITLE_OFF + 
					      HIGH_START_Y;
				cur_high[2] = cur_high[0] + DATE_WIDE;
				cur_high[3] = cur_high[1] + DATE_HIGH;
				RevBlock( cur_high[0], cur_high[1], 
					  cur_high[2], cur_high[3] );
				cur_high_row = FirstDay[Y];
				cur_high_col = FirstDay[X];
				CurHighDate[DAY] = new_month[DAY];
				CurHighDate[MON] = new_month[MON]+1;
				CurHighDate[YR] = new_month[YR];
				}
/*
*   If there's days to the right
*      Move right
*/
			else if ( cur_high_col < 6 )
				{
				cur_high[0] += HIGH_X_OFF;
				cur_high[2] = cur_high[0] + DATE_WIDE;
				RevBlock( cur_high[0], cur_high[1], 
					  cur_high[2], cur_high[3] );
				cur_high_col++;
				CurHighDate[DAY]++;
				}
/*
*   If we're in the right column
*   Go to the first in the next row
*/
			else
				{
				cur_high[0] = WIN_X + HIGH_START_X; 
				cur_high[1] += HIGH_Y_OFF;
				cur_high[2] = cur_high[0] + DATE_WIDE;
				cur_high[3] = cur_high[1] + DATE_HIGH;
				RevBlock( cur_high[0], cur_high[1], 
					  cur_high[2], cur_high[3] );
				cur_high_col = 0;
				cur_high_row++;
				CurHighDate[DAY]++;
				}
			break;
/*
*   LEFT KEY
*/
		case LEFT_KEY:
		RevBlock( cur_high[0], cur_high[1], 
			  cur_high[2], cur_high[3] );
/*
*   Are we at the first day?
*   If so, move to the previous month
*/
			if ( cur_high_row == FirstDay[Y] &&
			     cur_high_col == FirstDay[X] )
				{
				new_month[MON] = highlight_date[MON] - 1;
				new_month[YR] = highlight_date[YR];
				if ( new_month[MON] < 0 ) 
					{
					new_month[MON] = 11;
					new_month[YR]--;
					}
				new_month[DAY] = 
					days_per_month[new_month[MON]];
				first_dow = 6 - ((((6 - FirstDay[X]) + 
				   days_per_month[new_month[MON]] ) % 7));
				ihDtCalDraw( new_month, first_dow, 
					     WIN_X, WIN_Y );
				memcpy( highlight_date, new_month,
					3 * sizeof(int) );
				cur_high[0] = WIN_X + HIGH_START_X +
					( LastDay[X] * HIGH_X_OFF ); 
				cur_high[1] = WIN_Y +
					( LastDay[Y] * HIGH_Y_OFF) + 
					HIGH_START_Y + TITLE_OFF;
				cur_high[2] = cur_high[0] + DATE_WIDE;
				cur_high[3] = cur_high[1] + DATE_HIGH;
				RevBlock( cur_high[0], cur_high[1], 
					  cur_high[2], cur_high[3] );
				cur_high_row = LastDay[Y];
				cur_high_col = LastDay[X];
				CurHighDate[DAY] = new_month[DAY];
				CurHighDate[MON] = new_month[MON]+1;
				CurHighDate[YR] = new_month[YR];
				}
/*
*   Are there days to our left?
*   If so, move left
*/
			else if ( cur_high_col > 0 )
				{
				cur_high[0] -= HIGH_X_OFF;
				cur_high[2] = cur_high[0] + DATE_WIDE;
				RevBlock( cur_high[0], cur_high[1], 
					  cur_high[2], cur_high[3] );
				CurHighDate[DAY]--;
				cur_high_col--;
				}
/*
*   We must be in the left column
*   Move to the end of the previous row
*/
			else
				{
				cur_high[0] = WIN_X + HIGH_START_X + 
					      (6 * HIGH_X_OFF); 
				cur_high[1] -= HIGH_Y_OFF;
				cur_high[2] = cur_high[0] + DATE_WIDE;
				cur_high[3] = cur_high[1] + DATE_HIGH;
				RevBlock( cur_high[0], cur_high[1], 
					  cur_high[2], cur_high[3] );
				cur_high_col = 6;
				cur_high_row--;
				CurHighDate[DAY]--;
				}
			break;
/*
*   DOWN key
*/
		case DOWN_KEY:
			RevBlock( cur_high[0], cur_high[1], 
				  cur_high[2], cur_high[3] );
/*
*   Are we on the last row?
*/
			if ( cur_high_row < LastDay[Y] )
				{
/*
*   Is there a day below us?
*      No, then move to the last day on the next line
*/
				if ( cur_high_row == LastDay[Y]-1 &&
				     cur_high_col > LastDay[X] )
					{
					cur_high[1] += HIGH_Y_OFF;
					cur_high[3] = cur_high[1] + DATE_HIGH;
					cur_high[0] -= 
						(cur_high_col - LastDay[X]) *
						HIGH_X_OFF;
					cur_high[2] = cur_high[0] + DATE_WIDE;
					RevBlock( cur_high[0], cur_high[1], 
						  cur_high[2], cur_high[3] );
					cur_high_col = LastDay[X];
					cur_high_row++;
					CurHighDate[DAY] = 
					MonthArray[LastDay[X]][LastDay[Y]];
					}
/*
*   Yes, there's a day below us, so move the cursor down
*/
				else
					{
					cur_high[1] += HIGH_Y_OFF;
					cur_high[3] = cur_high[1] + DATE_HIGH;
					RevBlock( cur_high[0], cur_high[1], 
						  cur_high[2], cur_high[3] );
					cur_high_row++;
					CurHighDate[DAY] += 7;
					}
				}
/*
*   We're on the last row, so move to the next month
*/
			else
				{
				new_month[DAY] = 1;
				new_month[MON] = highlight_date[MON] + 1;
				new_month[YR] = highlight_date[YR];
				if ( new_month[MON] > 11 ) 
					{
					new_month[MON] = 0;
					new_month[YR]++;
					}
				first_dow = LastDay[X] + 1;
				if ( first_dow == 7 ) first_dow = 0;
				ihDtCalDraw( new_month, first_dow, 
					     WIN_X, WIN_Y );
				memcpy( highlight_date, new_month,
					3 * sizeof(int) );
				cur_high[0] = WIN_X + HIGH_START_X +
					( FirstDay[X] * HIGH_X_OFF ); 
				cur_high[1] = WIN_Y+TITLE_OFF + HIGH_START_Y;
				cur_high[2] = cur_high[0] + DATE_WIDE;
				cur_high[3] = cur_high[1] + DATE_HIGH;
				RevBlock( cur_high[0], cur_high[1], 
					  cur_high[2], cur_high[3] );
				cur_high_row = FirstDay[Y];
				cur_high_col = FirstDay[X];
				CurHighDate[DAY] = new_month[DAY];
				CurHighDate[MON] = new_month[MON]+1;
				CurHighDate[YR] = new_month[YR];
				}
			break;
/*
*   UP KEY
*/
		case UP_KEY:
			RevBlock( cur_high[0], cur_high[1], 
				  cur_high[2], cur_high[3] );
/*
*   Are we on the first row?
*/
			if ( cur_high_row != 0 )
				{
/*
*   Is there a day above us?
*      No, then move to the first day on the previous line
*/
				if ( cur_high_row == 1 &&
				     cur_high_col < FirstDay[X] )
					{
					cur_high[1] -= HIGH_Y_OFF;
					cur_high[3] = cur_high[1] + DATE_HIGH;
					cur_high[0] += 
						(FirstDay[X]-cur_high_col) *
						HIGH_X_OFF;
					cur_high[2] = cur_high[0] + DATE_WIDE;
					RevBlock( cur_high[0], cur_high[1], 
						  cur_high[2], cur_high[3] );
					cur_high_col = FirstDay[X];
					CurHighDate[DAY] = 1;
					cur_high_row--;
					}
/*
*   Yes, there's a day above us, so move the cursor up
*/
				else
					{
					cur_high[1] -= HIGH_Y_OFF;
					cur_high[3] = cur_high[1] + DATE_HIGH;
					RevBlock( cur_high[0], cur_high[1], 
						  cur_high[2], cur_high[3] );
					cur_high_row--;
					CurHighDate[DAY] -= 7;
					}
				}
/*
*   We're on the first row, so move to the previous month
*/
			else
				{
				new_month[MON] = highlight_date[MON] - 1;
				new_month[YR] = highlight_date[YR];
				if ( new_month[MON] < 0 ) 
					{
					new_month[MON] = 11;
					new_month[YR]--;
					}
				new_month[DAY] = 
					days_per_month[new_month[MON]];
				first_dow = 6 - ((((6 - FirstDay[X]) + 
				   days_per_month[new_month[MON]] ) % 7));
				ihDtCalDraw( new_month, first_dow, 
					     WIN_X, WIN_Y );
				memcpy( highlight_date, new_month,
					3 * sizeof(int) );
				cur_high[0] = WIN_X + HIGH_START_X +
					( LastDay[X] * HIGH_X_OFF ); 
				cur_high[1] = WIN_Y + HIGH_START_Y +
					( LastDay[Y] * HIGH_Y_OFF)+ TITLE_OFF;
				cur_high[2] = cur_high[0] + DATE_WIDE;
				cur_high[3] = cur_high[1] + DATE_HIGH;
				RevBlock( cur_high[0], cur_high[1], 
					  cur_high[2], cur_high[3] );
				cur_high_row = LastDay[Y];
				cur_high_col = LastDay[X];
				CurHighDate[DAY] = new_month[DAY];
				CurHighDate[MON] = new_month[MON]+1;
				CurHighDate[YR] = new_month[YR];
				}
			break;
/*
*   PAGE UP KEY
*/
		case PGUP_KEY:
			RevBlock( cur_high[0], cur_high[1], 
				  cur_high[2], cur_high[3] );
/*
*   Move to the previous month
*/
			new_month[MON] = highlight_date[MON] - 1;
			new_month[YR] = highlight_date[YR];
			if ( new_month[MON] < 0 ) 
				{
				new_month[MON] = 11;
				new_month[YR]--;
				}
			new_month[DAY] = days_per_month[new_month[MON]];
			first_dow = 6 - ((((6 - FirstDay[X]) + 
				   days_per_month[new_month[MON]] ) % 7));
			ihDtCalDraw( new_month, first_dow, WIN_X, WIN_Y );
			memcpy( highlight_date, new_month, 3 * sizeof(int) );
			cur_high[0] = WIN_X + ( LastDay[X] * HIGH_X_OFF ) +
				      HIGH_START_X; 
			cur_high[1] = WIN_Y + HIGH_START_Y +
				      ( LastDay[Y] * HIGH_Y_OFF)+ TITLE_OFF;
			cur_high[2] = cur_high[0] + DATE_WIDE;
			cur_high[3] = cur_high[1] + DATE_HIGH;
			RevBlock( cur_high[0], cur_high[1], 
				  cur_high[2], cur_high[3] );
			cur_high_row = LastDay[Y];
			cur_high_col = LastDay[X];
			CurHighDate[DAY] = new_month[DAY];
			CurHighDate[MON] = new_month[MON]+1;
			CurHighDate[YR] = new_month[YR];
			break;
/*
*   PAGE DOWN key
*/
		case PGDN_KEY:
			RevBlock( cur_high[0], cur_high[1], 
				  cur_high[2], cur_high[3] );
/*
*   Move to the next month
*/
			new_month[DAY] = 1;
			new_month[MON] = highlight_date[MON] + 1;
			new_month[YR] = highlight_date[YR];
			if ( new_month[MON] > 11 ) 
				{
				new_month[MON] = 0;
				new_month[YR]++;
				}
			first_dow = LastDay[X] + 1;
			if ( first_dow == 7 ) first_dow = 0;
			ihDtCalDraw( new_month, first_dow, WIN_X, WIN_Y );
			memcpy( highlight_date, new_month, 3 * sizeof(int) );
			cur_high[0] = WIN_X + ( FirstDay[X] * HIGH_X_OFF ) +
				      HIGH_START_X; 
			cur_high[1] = WIN_Y+TITLE_OFF + HIGH_START_Y;
			cur_high[2] = cur_high[0] + DATE_WIDE;
			cur_high[3] = cur_high[1] + DATE_HIGH;
			RevBlock( cur_high[0], cur_high[1], 
				  cur_high[2], cur_high[3] );
			cur_high_row = FirstDay[Y];
			cur_high_col = FirstDay[X];
			CurHighDate[DAY] = new_month[DAY];
			CurHighDate[MON] = new_month[MON]+1;
			CurHighDate[YR] = new_month[YR];
			break;
/*
*   ENTER key
*/
		case ENTER:
			done = TRUE;
			break;
		}
		if ( !done )
			{
			keystroke = GetKey();
			if ( keystroke == F10_KEY || 
				keystroke == ESC_KEY) done = TRUE;
			}
	}
	CloseWin( pwin );
	CurHighDate[MON]++;
	pDat->ReturnDate[0] = CurHighDate[0];
	pDat->ReturnDate[1] = CurHighDate[1];
	pDat->ReturnDate[2] = CurHighDate[2];
}
int IhDtInit(DIALOG *pDlg, DLGITEM *pItm, int GotFocus)
{
   IHDATE   *pDat;              /* points to private data of item */
   IHDTINIT *pIni = pItm->Init; /* points to eventual init data   */
   int n, i;
   time_t  CurDateTime;
   struct tm *pLocalTime;

   /* an init structure is mandatory - fail if it is missing */
   if(!pIni) return FALSE;

   /* Set the text field's initial value to a date string */

      if ( pIni->InitialDate[0] == 0 )  /* If date is null, set to today */
		{
		time( &CurDateTime );
		pLocalTime = localtime( &CurDateTime );
		pIni->InitialDate[DAY] = pLocalTime->tm_mday;
		pIni->InitialDate[MON] = pLocalTime->tm_mon;
		pIni->InitialDate[YR]  = pLocalTime->tm_year + 1900;
		}
      else                              /* Else, check the date */
		{
		if ( pIni->InitialDate[MON] < 1 ||
		     pIni->InitialDate[MON] > 12 ) return( FALSE );
		if ( pIni->InitialDate[DAY] < 1 ||
		     pIni->InitialDate[DAY] > 
			days_per_month[pIni->InitialDate[DAY]] ) 
			return ( FALSE );
		if ( pIni->InitialDate[YR] < 1970 ||
		     pIni->InitialDate[MON] > 2071 ) return( FALSE );
		pIni->InitialDate[MON]--;   /* Start months at 0 */
		}
      if ( pIni->Format == MMDDYY )
		{
		sprintf( pIni->Ei.Contents, "%d/%d/%d", 
			 pIni->InitialDate[MON]+1, 
			 pIni->InitialDate[DAY], 
			 pIni->InitialDate[YR] - 
				(( pIni->InitialDate[YR] / 100 ) * 100) );
		}
      else
		{
		sprintf( pIni->Ei.Contents, "%d/%d/%d", 
			 pIni->InitialDate[DAY], 
			 pIni->InitialDate[MON]+1, 
			 pIni->InitialDate[YR] - 
				(( pIni->InitialDate[YR] / 100 ) * 100) );
		}

   /* the date item is derived from the edit item - initialize it first,
      fail if this doesn't work for some reason */
   if(!IhEdit(IM_INIT, GotFocus, NULL, pDlg, pItm)) return FALSE;

   /* point to what the edit item set up for us */
   pDat = pItm->Data;

   /* store pointer to private data in item structure */
   pItm->Data = pDat;

   /* that's all folks */
   return TRUE;
}

int DtGetItem( int p[3], DLGITEM *pItm, DIALOG *pDlg )
{
   IHDATE   *pDat;              /* points to private data of item */
   char     *CurStr;
   IHDTINIT *pIni = pItm->Init; /* points to eventual init data   */
/*
*   Get the text field from the edit item
*/
   IhEdit( IM_GETDATA, EDGI_PTR, &CurStr, pDlg, pItm );
/*
*   Pull the info out of it
*/
	if ( pIni->Format == MMDDYY )
		{
		sscanf( CurStr, "%d/%d/%d", &p[MON], &p[DAY], &p[YR] );
		}
	else
		{
		sscanf( CurStr, "%d/%d/%d", &p[DAY], &p[MON], &p[YR] );
		}
/*
*   If the year is < 1900
*      Assume the user didn't enter the full four-digit year
*      If the year is > 70, assume it's the 1900's
*         Else, assume the 2000's
*/
	if ( p[YR] < 1900 )
		{
		if ( p[YR] > 70 )
			{
			p[YR] += 1900;
			}
		else
			{
			p[YR] += 2000;
			}
		}
	return TRUE;
}

int IhDtShow(DIALOG *pDlg, DLGITEM *pItm)
{
   int  x, y;
   int  h;
   int  WinPos;
   int  WinLen;
   IHEDIT *pDat = pItm->Data; /* a small lie, but makes it easier :-) */
   IMGHDR *pArrow = (IMGHDR *)DtArrowImg;

   /* display edit control, give up if that fails */
   if(!IhEdit(IM_SHOW, 0, NULL, pDlg, pItm)) return FALSE;

   /* set some local variables for speed */
   WinPos = pDat->WinPos;
   WinLen = pDat->WinLen;

   /* compute absolute position of bitmap */
   x = pDlg->pWin->PosX + pItm->PosX + WinPos+WinLen;
   y = pDlg->pWin->PosY + pItm->PosY - 2;

   /* compute height of bitmap */
   h = FontHeight(pDat->Font) + 4;

   /* display black rectangle */
   ClrBlock(x, y, x+h-1, y+h-1, BLACK_COLOR);

   /* center arrow bitmap in black rect */
   PutImg(x+(h-pArrow->Width)/2, y+(h-pArrow->Depth)/2,
	  FORCE_RULE, pArrow);
   return TRUE;
}


int IhDtKey(DIALOG *pDlg, DLGITEM *pItm, int Key)
{
   int x;
   int y;
   int n;
   IHDATE *pDat = pItm->Data;
   IHDTINIT *pIni = pItm->Init; /* points to eventual init data   */
   char buf[80];
   char *CurStr;
   int  Date[3];

/* 
*   All keys except cursor down and F10 (done) are handled by 
*   the edit item 
*/
	if ( Key != DOWN_KEY && Key != F10_KEY) 
		return IhEdit(IM_KEY, Key, NULL, pDlg, pItm);
/*
*   If this is the F10 (done) key, validate the date
*/
	if ( Key == F10_KEY )
		{
		IhEdit( IM_GETDATA, EDGI_PTR, &CurStr, pDlg, pItm );
		if ( pIni->Format == MMDDYY )
			{
			sscanf( CurStr, "%d/%d/%d", &Date[MON], &Date[DAY], 
						    &Date[YR] );
			}
		else
			{
			sscanf( CurStr, "%d/%d/%d", &Date[DAY], &Date[MON], 
						    &Date[YR] );
			}
/*
*   Make sure the user entered a valid date
*/
		if ( Date[MON] < 1 || Date[MON] > 12 )
			{
			MsgBox( "Error", "Invalid Month entered", NULL, "OK" );
			return FAILED_VALIDATION;
			}
		if ( Date[DAY] < 1 || Date[DAY] > days_per_month[Date[MON]-1] )
			{
			MsgBox( "Error", 
				"Invalid Day entered | for selected month", 
			NULL, "OK" );
			return FAILED_VALIDATION;
			}
/*
*   The date's OK, let the edit item finish it up 
*/
		return IhEdit(IM_KEY, Key, NULL, pDlg, pItm);
		}

   /* compute absolute position of menu */
   x = pDlg->pWin->PosX + pItm->PosX + pDat->Ed.WinPos;
   y = pDlg->pWin->PosY + pItm->PosY + FontHeight(pDat->Ed.Font);

      /* Display and process the actual calendar */
      IhDtProcCal( x, y, pIni->InitialDate, pDat );

      /* if user selected an entry, update edit item */
      if ( pDat->ReturnDate[0] != 0 )
	{
	if ( pIni->Format == MMDDYY )
		{
		sprintf( buf, "%d/%d/%d", pDat->ReturnDate[MON], 
			 pDat->ReturnDate[DAY], 
			 pDat->ReturnDate[YR] - 
				(( pDat->ReturnDate[YR] / 100 ) * 100) );
		}
	else
		{
		sprintf( buf, "%d/%d/%d", pDat->ReturnDate[DAY], 
			 pDat->ReturnDate[MON], 
			 pDat->ReturnDate[YR] - 
				(( pDat->ReturnDate[YR] / 100 ) * 100) );
		}

		pDat->Ed.SelFlag = TRUE;
		IhEdit(IM_SETDATA, EDSI_TXT, buf, pDlg, pItm);
	}

   return TRUE;
}
int IhDtSetData(int Msg, int n, void *p, DIALOG *pDlg, DLGITEM *pItm)
{
   int x;
   int y;
   IHDATE *pDat = pItm->Data;
   IHDTINIT *pIni = pItm->Init; /* points to eventual init data   */
   int  Date[3];
/*
*   Validate the new date string
*/
	if ( pIni->Format == MMDDYY )
		{
		sscanf( p, "%d/%d/%d", &Date[MON], &Date[DAY], 
			&Date[YR] );
		}
	else
		{
		sscanf( p, "%d/%d/%d", &Date[DAY], &Date[MON], 
			&Date[YR] );
		}
	if ( Date[MON] < 1 || Date[MON] > 12 ) return FALSE;
	if ( Date[DAY] < 1 || Date[DAY] > days_per_month[Date[MON]-1] )
		return FALSE;
	if ( Date[YR] < 0 || Date[YR] > 99 ) return FALSE;
	pIni->InitialDate[DAY] = Date[DAY];
	pIni->InitialDate[MON] = Date[MON];
	pIni->InitialDate[MON] = Date[MON];
/*
*   The date's OK, let the edit item finish it up 
*/
	return IhEdit(IM_SETDATA, EDSI_TXT, p, pDlg, pItm);
}

int IhDtKill(DIALOG *pDlg, DLGITEM *pItm)
{
   IHDATE *pDat = pItm->Data;

   /* ... and let the edit item do the rest of the work */
   return IhEdit(IM_KILL, 0, NULL, pDlg, pItm);
}
int IhDtLoseFocus(int Msg, int n, void *p, DIALOG *pDlg, DLGITEM *pItm)
{
	IHDATE  *pDat = pItm->Data;
	IHDTINIT *pIni = pItm->Init; /* points to eventual init data   */
	char    *CurStr;
	int     Date[3];
/*
*   Get the text field from the edit item
*/
   IhEdit( IM_GETDATA, EDGI_PTR, &CurStr, pDlg, pItm );
/*
*   Pull the date info out of it
*/
	if ( pIni->Format == MMDDYY )
		{
		sscanf( CurStr, "%d/%d/%d", &Date[MON], &Date[DAY], 
					    &Date[YR] );
		}
	else
		{
		sscanf( CurStr, "%d/%d/%d", &Date[DAY], &Date[MON], 
					    &Date[YR] );
		}
/*
*   Make sure the user entered a valid date
*/
	if ( Date[MON] < 1 || Date[MON] > 12 )
		{
		MsgBox( "Error", "Invalid Month entered", NULL, "OK" );
		return FALSE;
		}
	if ( Date[DAY] < 1 || Date[DAY] > days_per_month[Date[MON]-1] )
		{
		MsgBox( "Error", "Invalid Day entered | for selected month", 
			NULL, "OK" );
		return FALSE;
		}
/* 
*   Let the edit item do the rest of the work 
*/
	return IhEdit(Msg, n, p, pDlg, pItm);
}

int IhDate(int Msg, int n, void *p, DIALOG *pDlg, DLGITEM *pItm)
{
   switch(Msg) {
      case IM_INIT:      return IhDtInit(pDlg, pItm, n);
      case IM_KEY:       return IhDtKey(pDlg, pItm, n);
      case IM_SHOW:      return IhDtShow(pDlg, pItm);
      case IM_KILL:      return IhDtKill(pDlg, pItm);
      case IM_LOSEFOCUS: return IhDtLoseFocus(Msg, n, p, pDlg, pItm); 
      case IM_SETDATA:   return IhDtSetData(Msg, n, p, pDlg, pItm);

      /* the following cases are all handled by the edit item */
      case IM_GETDATA:   return DtGetItem( p, pItm, pDlg );
      case IM_SETFOCUS:  return IhEdit(Msg, n, p, pDlg, pItm);

      default:           return FALSE;
   }
}
