/* LDATE.C -- Date/Time Routines

	Written April 1994 by Craig A. Finseth
	Copyright 1994 by Craig A. Finseth
*/

#include "loki.h"

	/* Cumulative month start day number.  Yes, I know this has 14
	months: index is Jan == 1 and continues through Jan of next year. */
static int mdays[] = {
	0,
	0,
	0 + 31,
	0 + 31 + 28,
	0 + 31 + 28 + 31,
	0 + 31 + 28 + 31 + 30,
	0 + 31 + 28 + 31 + 30 + 31,
	0 + 31 + 28 + 31 + 30 + 31 + 30,
	0 + 31 + 28 + 31 + 30 + 31 + 30 + 31,
	0 + 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31,
	0 + 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30,
	0 + 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31,
	0 + 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31 + 30,
	0 + 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31 + 30 + 31 };

void D_MakeConsistent(int *yearptr, int *monptr);
long D_ToDate(long day);
long D_ToDayN(long day);

/* ------------------------------------------------------------ */

/* Enter the date. */

void
DDate(void)
	{
	char buf[SMALLBUFFSIZE];
	struct tm t;

	DNow(&t);
	xsprintf(buf, "%4d%02d%02dy", t.tm_year, t.tm_mon + 1, t.tm_mday);
	UEnter(&x, buf, RES_UN_DATE / UNIT_SIZE);
	m.stack_lift = TRUE;
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Return the new data AMT days before or after DATE. */

long
DDateN(long date, long amt)
	{
	return(D_ToDate(D_ToDayN(date) + amt));
	}


/* ------------------------------------------------------------ */

/* Return the number of days between the dates. */

long
DDays(long date1, long date2)
	{
	return(D_ToDayN(date1) - D_ToDayN(date2));
	}


/* ------------------------------------------------------------ */

/* Return the current date and time in TPTR. */

void
DNow(struct tm *tptr)
	{
#if defined(UNIX)
	time_t now;

	now = time(NULL);
	*tptr = *localtime(&now);
	tptr->tm_year += 1900;
#endif
#if defined(MSDOS)
	char buf[4];

	setmem((char *)tptr, sizeof(*tptr), 0);

	PGetDate(buf);
	tptr->tm_year = buf[1] * 256 + buf[0];
	tptr->tm_mon = buf[3] - 1;
	tptr->tm_mday = buf[2];

	PGetTime(buf);
	tptr->tm_hour = buf[1];
	tptr->tm_min = buf[0];
	tptr->tm_sec = buf[3];
#endif
	}


/* ------------------------------------------------------------ */

/* Do the time command. */

void
DTime(void)
	{
	char buf[SMALLBUFFSIZE];
	struct tm t;

	DNow(&t);
	xsprintf(buf, "%d:%d:%d", t.tm_hour, t.tm_min, t.tm_sec);
	UEnter(&x, buf, RES_UN_TIME / UNIT_SIZE);
	m.stack_lift = TRUE;
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Bring the month into the range 1-12 and adjust the year
accordingly. */

void
D_MakeConsistent(int *yearptr, int *monptr)
	{
	while (*monptr > 12) {
		(*monptr) -= 12;
		(*yearptr)++;
		}
	while (*monptr < 1) {
		(*monptr) += 12;
		(*yearptr)--;
		}
	if (*yearptr > 9999) {
		*monptr = 12;
		*yearptr = 9999;
		}
	if (*yearptr < 1583) {
		*monptr = 1;
		*yearptr = 1583;
		}
	}


/* ------------------------------------------------------------ */

/* Convert the day number to a date.  Should do sanity checking,
but... */

long
D_ToDate(day)
	long day;
	{
	long dn;
	long ltmp;
	int y;
	int m;
	int d;

/* Divide the day number by 365.2422 to get real close to the correct
year.  We must do this with ints, so multiply by 10,000 and divide by
3652422.  But our day numbers range up to 9999 * 366 (or so) =
3,600,000.  Multiplying by 10,000 exceeds a 32-bit int.  We have to
reduce the 10,000 by a factor of 20 or so in order not to overflow.

If we call the orgininal number 365.2425 and so get 10,000 and
3652425, we can remove a factor of 25 and obain 400 and 146,097.  This
works but is incorrect to the tune of 3 parts in (roughly) 3,000,000
or 1 in 1,000,000.  As there are only about 2,500 leap year days that
can foul things up, we are still close enough.  Later steps will
correct any error. */

	ltmp = day * 400;
	ltmp /= 146097;

/* Now the corrections start.  First, make sure that we are before the
correct year. */

	y = ltmp - 2;

/* Now, count up until we get to the correct year. */

	m = 1;
	d = 1;
	for (dn = 0; dn < day; y++) {
		dn = D_ToDayN((long)y * 10000 + m * 100 + d);
			/* done! */
		if (dn == day) return((long)y * 10000 + m * 100 + d);
		}
	y -= 2;

/* We now have the correct year. On to the month and day. */

	day -= D_ToDayN((long)y * 10000 + m * 100 + d);

	if (day < mdays[2]) {
		d = day + 1;	/* Jan */
		return((long)y * 10000 + m * 100 + d);
		}
	else if (day < mdays[3]) {
		m = 2;
		d = day - mdays[2] + 1; /* Feb 28 */
		return((long)y * 10000 + m * 100 + d);
		}

/* It is either Feb 29 (if we have a leap y), or some day after that */

/* See if we are a leap y. */
	if (y / 4 == 0 &&
		(y / 100 != 0 || y / 400 == 0)) {	/* leap year */
		if (day == mdays[3]) {
			m = 2;
			d = 29;
			return((long)y * 10000 + m * 100 + d);
			}
		day--;		/* treat as regular day */
		}

	for (m = 3; day >= mdays[m + 1]; m++) ;
	d = day - mdays[m] + 1;
	return((long)y * 10000 + m * 100 + d);
	}


/* ------------------------------------------------------------ */

/* Convert the date to a day number and return the day number. */

long
D_ToDayN(long day)
	{
	long tmp;
	int y;
	int m;
	int d;

	y = day / 10000;
	day %= 10000; 
	m = day / 100;
	d = day % 100;

	D_MakeConsistent(&y, &m);
	if (d < 1 || d > 31) d = 1;

	tmp = (365 * (long)y) + mdays[m] + d - 1;

		/* Jan and Feb get previous year's leap year counts */
	if (m <= 2) y--;

	tmp += y / 4;		/* add leap years */
	tmp -= y / 100;		/* subtract non-leap cents. */
	tmp += y / 400;		/* add back 400 years */
	return(tmp);
	}


/* end of LDATE.C -- Date/Time Routines */
