/* --------------------------------------------------------------------
   Project: PlayList
   Module:  PlayList
   Author:  Bob Chernow
   Started: 09/02/95
   Subject: Build, Manipulate and Play groups of Sound Files.
   -------------------------------------------------------------------- */

/* --------------------------------------------------------------------
                       standard includes
   -------------------------------------------------------------------- */
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <stat.h>
#include <stdlib.h>
#include <string.h>

/* --------------------------------------------------------------------
                         local includes
   -------------------------------------------------------------------- */
#include "pal.h"
#include "palsmsvc.h"

/* --------------------------------------------------------------------
                         type definitions
   -------------------------------------------------------------------- */

/* Define a window with the largest possible border, for test
*/
#define WS_TEST (WS_SHADOW|WS_BMASK|WS_SAVEBG|WS_SMLFONT)

#define WINX1   140
#define WINY1    70
#define WINX2   480
#define WINY2   110

/* --------------------------------------------------------------------
                         type definitions
   -------------------------------------------------------------------- */

/* let the compiler take care of CMD numbering with an enum
*/
typedef enum{
    CMD_OPEN = 1 , CMD_NEW,      CMD_SAVE, CMD_SAVEAS, CMD_QUIT,
    CMD_ADD,       CMD_DELETE,   CMD_EDIT, CMD_PLAY,
    CMD_DIR,       CMD_PLAYLOOP, CMD_TITLE,
    CMD_HELPGEN,   CMD_ABOUT
} MENUCOMMANDS;

/* --------------------------------------------------------------------
                        local prototypes
   -------------------------------------------------------------------- */
int PlayListCallback(MENU *pMnu);
void LoadConfig( char * MyName );
void LoadConfig( char * Name );
void menu_open(void);
void menu_new(void);
void menu_saveas(void);
void menu_save(void);
void menu_quit(void);
void menu_add(void);
void menu_delete(void);
void menu_edit(void);
void menu_play(void);
void menu_dir(void);
void menu_playloop( void );
void menu_title( void );
void menu_helpgen(void);
void menu_about(void);
int PlayListCallback(MENU *pMenu);
void ShowRetval(int RetVal);
void interact(void);
int RunMenu( void);
int main(int argc, char *argv[]);

/* --------------------------------------------------------------------
                        global variables
   -------------------------------------------------------------------- */
char        szSoundDir[128] = "";
char        szPlayList[128] = "";
char        szTitle[128] = "";
PICKRESULTS *pPR, *pPlayList = NULL, *pPlayLast = NULL;
int         iLoopCount;

/* --------------------------------------------------------------------
                  The Menues and their Items
   -------------------------------------------------------------------- */

/* the 'List' pulldown: Items and Menu
*/
MENUITEM ListItems[] = {
    { " &Open... ",      MA_VAL,          MENUVAL(CMD_OPEN)   },
    { " &New... ",       MA_VAL,          MENUVAL(CMD_NEW)    },
    { " &Save... ",      MA_VAL,          MENUVAL(CMD_SAVE)   },
    { " Save &As ... ",  MA_VAL,          MENUVAL(CMD_SAVEAS) },
    { " &Quit",          MA_VAL,          MENUVAL(CMD_QUIT)   },
};
MENU ListMenu = { NULL, MS_PULLDN, 0, 0, 5, 5, ListItems };

/* the 'Item' pulldown: Items and Menu
*/
MENUITEM ItemItems[] = {
    { " &Add... ",      MA_VAL,          MENUVAL(CMD_ADD)    },
    { " &Delete... ",   MA_VAL,          MENUVAL(CMD_DELETE) },
    { " &Edit... ",     MA_VAL | MI_GRY, MENUVAL(CMD_EDIT)   },
    { " &Play... ",     MA_VAL,          MENUVAL(CMD_PLAY)   },
};
MENU ItemMenu = { NULL, MS_PULLDN, 0, 0, 4, 4, ItemItems };

/* The 'OptionsMenu' pulldown: Items and Menu
*/
MENUITEM OptionsItems[] = {
    { "&Directory ...", MA_VAL,          MENUVAL(CMD_DIR)      },
    { "&Playloop ...",  MA_VAL,          MENUVAL(CMD_PLAYLOOP) },
    { "&Title ...",     MA_VAL,          MENUVAL(CMD_TITLE)    },
};
MENU OptionsMenu = { NULL, MS_PULLDN, 0, 0, 3, 3, OptionsItems };

/* the 'Help' pulldown: Items and Menu
*/
MENUITEM HelpItems[] = {
    { "&General Help",    MA_VAL | MI_GRY, MENUVAL(CMD_HELPGEN) },
    { "Separ",   MT_SEPARATOR | MA_VAL,    MENUVAL(99)          },
    { "&About",           MA_VAL,          MENUVAL(CMD_ABOUT)   },
};
MENU HelpMenu = { NULL, MS_PULLDN, 0, 0, 3, 3, HelpItems };

/* Finally, the menu bar itself, branching off to the four submenues
*/
MENUITEM BarItems[] = {
    { "&File",                      MA_MNU, &ListMenu    },
    { "&Item",                      MA_MNU, &ItemMenu    },
    { "&Options",                   MA_MNU, &OptionsMenu },
    { "Separ",   MT_SEPARATOR |     MA_VAL, MENUVAL(99)  },
    { "&Help",                      MA_MNU, &HelpMenu    },
};
MENU BarMenu = { NULL, MS_HORIZ|MS_TOPLVL, 0, 0, 5, 5, BarItems };


/* --------------------------------------------------------------------
                           Functions
   -------------------------------------------------------------------- */

/* --------------------------------------------------------------------
                                 Help window
   -------------------------------------------------------------------- */

void Help(void)

{
    PALWIN *pWin;
    int x=40,y=8;

    pWin  = OpenWin(WS_HELP, 30, 10, 600, 190, "Help");
    SelectFont(MEDIUM_FONT);

    WinText(pWin, x,y,    "File                                           ");
    WinText(pWin, x,y+10, "  New    | Initialize an empty playlist.       ");
    WinText(pWin, x,y+20, "  Open   | Open an existing playlisy.          ");
    WinText(pWin, x,y+30, "  Save   | Save the current playlist.          ");
    WinText(pWin, x,y+40, "  SaveAs | Save current playlist w/ new name.  ");
    WinText(pWin, x,y+50, "  Quit   | Get out of here.                    ");
    WinText(pWin, x,y+60, "Item     |                                     ");
    WinText(pWin, x,y+70, "  Add    | Add more songs.                     ");
    WinText(pWin, x,y+80, "  Delete | Delete a song.                      ");
    WinText(pWin, x,y+90, "  Edit   | Not Available Yet.                  ");
    WinText(pWin, x,y+100,"  Play   | Play the current list.              ");
    WinText(pWin, x,y+110,"                                               ");
    WinText(pWin, x,y+120,"                                               ");
    WinText(pWin, x,y+133,"  page 1 - any key for more                    ");

    GetKey();
    CloseWin(pWin);

    pWin  = OpenWin(WS_HELP, 30, 10, 600, 190, "Help");
    SelectFont(MEDIUM_FONT);

    WinText(pWin, x,y,    "Options                                        ");
    WinText(pWin, x,y+10, "  Directory | Set soundfile directory.         ");
    WinText(pWin, x,y+20, "  LoopCount | Set times to play songs.         ");
    WinText(pWin, x,y+30, "  Title     | Set the playlists title.         ");
    WinText(pWin, x,y+40, "Help        |                                  ");
    WinText(pWin, x,y+50, "  General   | These screens.                   ");
    WinText(pWin, x,y+60, "  About     | Product Information              ");
    WinText(pWin, x,y+70, "                                               ");
    WinText(pWin, x,y+80, "                                               ");
    WinText(pWin, x,y+90, "                                               ");
    WinText(pWin, x,y+100,"                                               ");
    WinText(pWin, x,y+110,"                                               ");
    WinText(pWin, x,y+120,"                                               ");
    WinText(pWin, x,y+133,"  page 2 - any key for more                    ");

    GetKey();
    CloseWin(pWin);

    pWin  = OpenWin(WS_HELP, 30, 10, 600, 190, "Help");
    SelectFont(MEDIUM_FONT);

    WinText(pWin, x,y,    "                                               ");
    WinText(pWin, x,y+10, " Command Line Syntax:                          ");
    WinText(pWin, x,y+20, "                                               ");
    WinText(pWin, x,y+30, "   Playlist [-A] [-L list]                     ");
    WinText(pWin, x,y+40, "                                               ");
    WinText(pWin, x,y+50, "     -A : Autoplay will play the list and exit.");
    WinText(pWin, x,y+60, "                                               ");
    WinText(pWin, x,y+70, "     -L : Load a specific list file            ");
    WinText(pWin, x,y+80, "                                               ");
    WinText(pWin, x,y+90, "    list: List to load.                        ");
    WinText(pWin, x,y+100,"                                               ");
    WinText(pWin, x,y+110,"                                               ");
    WinText(pWin, x,y+120,"                                               ");
    WinText(pWin, x,y+133,"  page 3                                       ");

    GetKey();
    CloseWin(pWin);
}

/*-----------------------------------------------------------------------------
     Function: LoadConfig
       Syntax: LoadConfig( name );
   Parameters: name: ( char *) Name of config file to load
      Returns: Nothing
        Notes: Reads default config file and sets default values.
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/17/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void LoadConfig( char * Name )
{
    char * pszKey = "";
    char * pszSong = "";

    PICKRESULTS *pPlayTemp;

    /* If we have already loaded a config free it
    */
    FreePickResults(pPlayList);
    pPlayList = NULL;

    ReadConfig( Name );

    strcpy( szSoundDir, GetConfigString("CONFIG","snddir","A:" ) );
    strcpy( szTitle, GetConfigString("CONFIG","title","" ) );
    iLoopCount = GetConfigInt("CONFIG","loopcount",1 );

    pPlayLast = NULL;
    pszSong = GetFirstEntry("PLAYLIST", &pszKey );
    if ( pszSong ){
        do {
            /* title is in the songlist section
            */
            if ( strcmpi( pszKey,"title" ) == 0 ) {
                strcpy(szTitle,pszSong);

            /* add a new song
            */
            } else if (strcmpi( pszKey, "song" ) == 0 ) {

                pPlayTemp = (PICKRESULTS *)malloc(sizeof(PICKRESULTS));
                pPlayTemp->String = strdup(pszSong);
                pPlayTemp->Next = NULL;

                /* First time?
                */
                if (!pPlayList) {
                    pPlayList = pPlayTemp;
                } else {
                    pPlayLast->Next = pPlayTemp;
                }
                pPlayLast = pPlayTemp;
            }
            pszSong = GetNextEntry(&pszKey);
        } while ( pszSong );
    }
    DiscardConfig();

}
/*-----------------------------------------------------------------------------
     Function: menu_open
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_open(void)
{
    char buf[81];
    int  rc;
    char prompt[128] = "The file entered here will be used |" \
                       "as the current PlayList.||PlayList: |";

    strcat(prompt,szPlayList);

    rc = MsgBox("Sound File Directory", prompt, buf, "!OK    |CANCEL" );
    if (rc == DN_OK || rc == 1) {
        strcpy(szPlayList, buf);
        LoadConfig(szPlayList);
    }
}

/*-----------------------------------------------------------------------------
     Function: menu_new
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_new(void)
{
    /* set up defaults
    */
    strcpy(szSoundDir, "C:");
    strcpy(szPlayList, "");
    strcpy(szTitle, "");

    pPlayList = NULL;
    pPlayLast = NULL;
    iLoopCount = 1;
}

/*-----------------------------------------------------------------------------
     Function: menu_saveas
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_saveas(void)
{
    char buf[81];
    int  rc;
    char prompt[128] = "The file you enter will be used to savel|" \
                       "the current playlist information.||Filespec: |";

    strcat(prompt,szPlayList);

    rc = MsgBox("Play List FileSpec", prompt, buf, "!OK    |CANCEL" );
    if (rc == DN_OK || rc == 1) {
        strcpy(szPlayList, buf);
    }
    /* fall thru to menu_save
    */
    menu_save();
}

/*-----------------------------------------------------------------------------
     Function: menu_save
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_save(void)
{
    char buf[256];
    PICKRESULTS *pPRTemp;

    /* do we have a file name?
    */
    if ( strlen(szPlayList) != 0 ) {

        NewConfig(szPlayList);

        /* write the CONFIG section
        */

        /* write the snddir
        */
        AddConfigString("CONFIG", "snddir", szSoundDir);

        /* write the loopcount
        */
        AddConfigInt("CONFIG", "loopcount", iLoopCount);

        /* write the PLAYLIST section
        */

        /* write the title
        */
        AddConfigString("PLAYLIST", "title", szTitle);

        /* now add the songs
        */
        pPRTemp = pPlayList;

        while(pPRTemp) {
            AddConfigString("PLAYLIST", "song", pPRTemp->String);
            pPRTemp = pPRTemp->Next;
        }

        /* write the file
        */
        WriteConfig(szPlayList);
        DiscardConfig();
    /* get a file name
    */
    } else {
        menu_saveas();
    }
}


/*-----------------------------------------------------------------------------
     Function: menu_quit
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_quit(void)
{
    /* Place clean-up code here
    */
    TextMode();
}

/*-----------------------------------------------------------------------------
     Function: menu_add
        Notes: Called from MAIN by menu selection
               Presents a FilePickList for the Current SoundDir.
               Sets the global pPR to the PickList results.
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_add(void)
{
    char        szFileSpec[256];
    PICKRESULTS *pPRTemp;
    char        *pszLastPos;
    char        *pszTempString;
    int         iPathLen;

    /* use current sound dir
    */
    strcpy(szFileSpec, szSoundDir);

    /* fixup if last char is a \
    */
    pszLastPos = strrchr(szFileSpec, '\\');

    if (strcmp( pszLastPos, "\\") == 0) {
        memset(pszLastPos,0,1);
    }

    /* show list of all files
    */
    strcat(szFileSpec, "\\*.*");
    iPathLen = strlen(szFileSpec) + 2;
    pPRTemp = FilePickList( szFileSpec, 7, 100, 40 );

    /* if any selected, add them
    */
    if ( PickCount( pPRTemp ) > 0 ) {
        pPR = pPRTemp;
        /* add the path information
        */
        while (pPRTemp) {
            pszTempString = (char *)malloc((iPathLen + strlen(pPRTemp->String)) *
                                        sizeof(char) );

            strcpy(pszTempString, szSoundDir);
            strcat(pszTempString, "\\");
            strcat(pszTempString, pPRTemp->String);
            free(pPRTemp->String);
            pPRTemp->String = pszTempString;
            pPRTemp = pPRTemp->Next;
        }

        /* Add the selections to the Playlist
        */
        if (pPlayList) {
            pPlayLast->Next = pPR;
        } else {
            pPlayList = pPR;
        }
        pPlayLast = FindLastPick(pPlayList);
    }
}


/*-----------------------------------------------------------------------------
     Function: menu_delete
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_delete(void)
{
    PICKRESULTS *pPlay, *pDel, *pTemp, *pNext;
    char *Array[512];
    int iCount, p, d;

    pPlay = pPlayList;

    /* Build the array of songs and count for the Picklist call
    */
    for(iCount = 0 ; pPlay; iCount++, pPlay = pPlay->Next) {
        Array[iCount] = strdup(pPlay->String);
    }

    /* Let the user select songs to delete
    */
    pDel = PickList( Array, iCount, 10, 100, 20 );

    pPlay = pPlayList;

    /* Handle all deletions
    */
    while (pDel) {
        /* get the song to remove
        */
        d = pDel->Index;
        p = 0;
        /* Advance the PlayList to 1 before the Index unless the index is 0
        */
        if (pPlay == pPlayList && p == d ) {
            /* Relink the list
            */
            pTemp = pPlay;
            pPlay = pTemp->Next;
            pPlayList = pPlay;
            p++;
        } else {
            /* Move thru Playlist
            */
            while ( p++ < ( d-1 ) ) {
                pPlay = pPlay->Next;
            }
            /* Relink the list
            */
            pTemp = pPlay->Next;
            pNext = pTemp->Next;
            pPlay->Next = pNext;
        }
        /* Free up the deleted song
        */
        free(pTemp->String);
        free(pTemp);
        pDel = pDel->Next;
    }
}

/*-----------------------------------------------------------------------------
     Function: menu_edit
        Notes: Called from MAIN by menu selection (not available)
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_edit(void)
{
    MsgBox("Function Entered", "menu_edit", NULL, NULL );
}

/*-----------------------------------------------------------------------------
     Function: menu_play
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_play(void)
{

    char    Filename[128],
            data[1024];

    PALWIN  *pWin;

    int     Font = MEDIUM_FONT;
    int     fh,
            i,
            PosX,
            PosY,
            Len,
            Height;

    PICKRESULTS *pPRTemp;

    for ( i=0; i < iLoopCount; i++ ){

        pPRTemp = pPlayList;

        while ( pPRTemp ) {

            /* Get a file name
            */
            strcpy( Filename, pPRTemp->String );

            /* Open it
            */
            fh = open(Filename, O_RDONLY, S_IREAD);

            /* read in the data
            */
            memset(data,'\0',1024);
            read(fh, data, 1024);


            /* pass the string to sysmgr to play
            */
            if (CheckSysMgr()) {

                /* get length of string
                */
                Len    = TextExt(Font, Filename);
                Height = FontHeight(Font);

                /* center string horizontally
                */
                PosX = (WINX2 - WINX1 - Len)/2;

                /* center string vertically
                */
                PosY = (WINY2 - WINY1 - Height)/2;

                pWin = OpenWin(WS_HP200, WINX1, WINY1, WINX2, WINY2,
                                 " Now Playing ");

                WinText(pWin, PosX, PosY, Filename);

                while (KeyWaiting()) {
                    GetKey();
                }

                m_play(data);
                msDelay(700);

                CloseWin(pWin);

            } else {
                MsgBox("System Manager Not Available",
                "Cannot Play %s", NULL, NULL, Filename );
            }

            /* close the file
            */
            close(fh);

            /* Next entry
            */
            pPRTemp = pPRTemp->Next;
        }
    }
}

/*-----------------------------------------------------------------------------
     Function: menu_dir
        Notes: Called from MAIN by menu selection
               Prompts the user for a new SoundDir, offering the current
               one as a default.
               Sets sxSoundDir.
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_dir(void)
{
    char buf[81];
    int  rc;
    char prompt[128] = "The directory entered here will|" \
                       "be searched for sound files.||Directory: |";

    strcat(prompt,szSoundDir);

    rc = MsgBox("Sound File Directory", prompt, buf, "!OK    |CANCEL" );
    if (rc == DN_OK || rc == 1) {
        strcpy(szSoundDir, buf);
    }
}

/*-----------------------------------------------------------------------------
     Function: menu_playloop
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_playloop( void )
{
    char buf[81];
    char num[10];
    int  rc;
    char prompt[128] = "The number you enter here will|" \
                       "determine the number of times|" \
                       "each song will play|Directory: |";

    strcat(prompt,itoa(iLoopCount,num,10));

    rc = MsgBox("Sound File Directory", prompt, buf, "!OK    |CANCEL" );
    if (rc == DN_OK || rc == 1) {
        iLoopCount = atoi(buf);
    }
}

/*-----------------------------------------------------------------------------
     Function: menu_title
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_title( void )
{
    char buf[81];
    int  rc;
    char prompt[128] = "The string you enter here will be used|" \
                       "as the title of this playlist.||Title: |";

    strcat(prompt,szTitle);

    rc = MsgBox("PlayList Title", prompt, buf, "!OK    |CANCEL" );
    if (rc == DN_OK || rc == 1) {
        strcpy(szTitle, buf);
    }
}

/*-----------------------------------------------------------------------------
     Function: menu_helpgen
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_helpgen(void)
{
    help();
}
/*-----------------------------------------------------------------------------
     Function: menu_about
        Notes: Called from MAIN by menu selection
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     mm/dd/yy rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void menu_about(void)
{
   PALWIN *pWin;
   int Font = MEDIUM_FONT;
   int PosX = 12;
   int PosY = 12;

   pWin = OpenWin(WS_HP200, 180, 50, 440, 130,
                  " About PLAYLIST ");
   WinText(pWin, PosX, PosY, "     HP SND Player    "); PosY += FNTD(Font)+2;
   WinText(pWin, PosX, PosY, "    Beta version .1   "); PosY += FNTD(Font)+2;
   WinText(pWin, PosX, PosY, "                      "); PosY += FNTD(Font)+2;
   WinText(pWin, PosX, PosY, " -------------------- "); PosY += FNTD(Font)+2;
   WinText(pWin, PosX, PosY, " R. Chernow  Sept. 95 "); PosY += FNTD(Font)+2;
   WinText(pWin, PosX, PosY, " -------------------- "); PosY += FNTD(Font)+2;
   GetKey();
   CloseWin(pWin);
}

/*-----------------------------------------------------------------------------
     Function: interact
        Notes: Handle function key interaction
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
void interact() {

    char    buf[80];
    WORD    Key;
    int     Cmd;
    char *KeyLabels[10] = {
    "Help", "Open"  , "New",  "Save", "SaveAs",
    "Add" , "Delete", "Edit", "Play",  "Quit"
    };
    char *NoKeyLabels[10] = {
    "", ""  , "",  "", "", "" , "", "", "",  ""
    };


    do {
        sprintf(buf, "PlayList: %s", szTitle);
        ShowTopTime(buf, 1);
        ShowFKeys(KeyLabels);

        Key = GetKey();
        if(Key & 0xff) Key &= 0xff;

        if(Key == MENU_KEY || Key == AF10_KEY) {
            ShowFKeys(NoKeyLabels);
            Cmd = RunMenu();
            ShowFKeys(KeyLabels);
        }
        if (Cmd == CMD_QUIT) {
            return;
        }


        switch(Key) {
            case F1_KEY:
                menu_helpgen();
                break;
            case F2_KEY:
                menu_open();
                break;
            case F3_KEY:
                menu_new();
                break;
            case F4_KEY:
                menu_save();
                break;
            case F5_KEY:
                menu_saveas();
                break;
            case F6_KEY:
                menu_add();
                break;
            case F7_KEY:
                menu_delete();
                break;
            case F8_KEY:
                menu_edit();
                break;
            case F9_KEY:
                menu_play();
                break;
            case F10_KEY:
                menu_quit();
                break;
        }
   } while(Key != F10_KEY);
}

/*-----------------------------------------------------------------------------
     Function: RunMenu
       Syntax: retval =RunMenu()
      Returns: selection from menu
        Notes: Handles Menu interaction
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
int RunMenu( void) {
    char buf[80];

    int RetVal;

    do {

        sprintf(buf, "PlayList: %s", szTitle);
        ShowTopTime(buf, 1);

        RetVal =  HandleMenu(&BarMenu, 10, 10);
        switch(RetVal) {
            case CMD_OPEN:
                menu_open();
                break;
            case CMD_NEW:
                menu_new();
                break;
            case CMD_SAVE:
                menu_save();
                break;
            case CMD_SAVEAS:
                menu_saveas();
                break;
            case CMD_QUIT:
                menu_quit();
                break;
            case CMD_ADD:
                menu_add();
                break;
            case CMD_DELETE:
                menu_delete();
                break;
            case CMD_EDIT:
                menu_edit();
                break;
            case CMD_PLAY:
                menu_play();
                break;
            case CMD_DIR:
                menu_dir();
                break;
            case CMD_PLAYLOOP:
                menu_playloop();
                break;
            case CMD_TITLE:
                menu_title();
                break;
            case CMD_HELPGEN:
                menu_helpgen();
                break;
            case CMD_ABOUT:
                menu_about();
                break;
        }
    } while(RetVal != CMD_QUIT & RetVal != MSC_ESC );
    return(RetVal);

}

/*-----------------------------------------------------------------------------
     Function: HandleArgs
       Syntax: rc = HandleArgs( argc, argv );
   Parameters: Command line parameter from main
      Returns: 0=Go GUI, 1=Played in Automode so exit
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/14/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
int HandleArgs(int argc, char *argv[])
{
    int RetVal = 0,
        i,
        bListNext = 0,
        iAuto = 0;

    for (i = 1; i < argc; i++)
    {
        /* autoplay?
        */
        if ( strcmpi( argv[i], "-A" ) == 0 )
        {
            iAuto = 1;

        /* Load file?
        */
        } else if ( strcmpi( argv[i], "-L" ) == 0 )
        {
            bListNext = 1;

        /* Get Load Filename
        */
        } else if (bListNext == 1 )
        {
            strcpy( szPlayList, argv[i] );
        }
    }

    /* If Autoplay then play songlist and signal exit
    */
    if (iAuto) {
        LoadConfig( szPlayList );
        menu_play();
        RetVal = 1;

    }
    return(RetVal);
}
/*-----------------------------------------------------------------------------
     Function: main
       Syntax: none
   Parameters: argc, argv
      Returns: nothing
        Notes: driver
  -----------------------------------------------------------------------------
     History
     Date     Who       What                                            Tag
     -------- --------- ----------------------------------------------- -------
     09/02/95 rlc       Creation                                        None
 ----------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
    int RetVal;
    char *KeyLabels[10] = {
    "Help", "Open"  , "New",  "Save", "SaveAs",
    "Add" , "Delete", "Edit", "Play",  "Quit"
    };

    char DefPath[64];
    char *s;

    /* remove warning
    */
    argc = argc;

    if(!PalInit(1)) {
        FatalExit("Init failed - CGAGRAPH not loaded ?", 1);
    }

    /* Load the default config file
    */
    strcpy( DefPath, argv[0] );
    s = strrchr(DefPath, '.');
    if ( s ) {
        *s = '\0';
    }
    strcat(DefPath, ".LST");
    strcpy(szPlayList, DefPath);

    /* Handle command line arguments
    */
    RetVal = HandleArgs(argc, argv);

    /* If not autoplay then go gui
    */
    if (RetVal == 0) {
        LoadConfig( szPlayList );
        ShowFKeys(KeyLabels);
        Interact();
    }
    PalDeInit(1);
}

