/****************************************************************************/
/*                                                                          */
/* wizcsv.c                                                                 */
/*                                                                          */
/* Read Windows Schedule data in Comma Separated Wizard format and export   */
/* it into an HP100/200LX Appointment book Comma Separated file.            */
/*                                                                          */
/* Much of this code was shamelessly borrowed from the ADBIO programs       */
/* produced by A. Garzotto, April '94.                                      */
/*                                                                          */
/* The Sharp Wizard DLL formats appointments as follows:                    */
/*                                                                          */
/* 19960516,"10:00","10:30","10:00","Demo appt, no warning","N"             */
/* 19960516,"12:00","15:00","11:45","Staff Meeting	Club Rasuna...","Y" */
/*                                                                          */
/* The fields are, in order:                                                */
/* YYYYMMDD	Date of appointment                                         */
/* "hh:mm"	Start time of appointment				    */
/* "hh:mm"	End time of appointment					    */
/* "hh:mm"	Time for alarm						    */
/* "text..."	Variable length text field... may need some formatting	    */
/*		(the first characters are copied into the appointment desc  */
/*		and the entire text into the notes field)		    */
/* "f"		a flag (Y or N) indicating if an alarm should go off	    */
/*                                                                          */
/* Produced by Peter van Es (pvanes@ibm.net)				    */
/*                                                                          */
/* Version History							    */
/*                                                                          */
/* 25 May 1996	Version 1.0 first attempt to check Merge behaviour	    */
/*                                                                          */
/****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <io.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>

#define YES 1
#define NO  0
#define NIL 0

#define DESCLEN 51	/* max length of description field */
#define LOCLEN	19	/* max length of location field */

#ifndef O_BINARY
#define O_BINARY 0
#endif

#define S_ALARM     1
#define S_CHECKOFF  2
#define S_MONTHLY   2
#define S_CARRY     4
#define S_WEEKLY    4
#define S_TODO     16
#define S_EVENT    32
#define S_STUB     64
#define S_APPT    128

/****************************************************************************/

struct rec_desc
{
   char year;               /* start date; YYYYMMDD */
   char month;
   char day;
   int stime;               /* start time HHMM */
   int etime;               /* end time HHMM */
   int alarm;		    /* 1 = yes, 0 = no */
   int lead;                /* alarm lead time in HHMM */
   char *desc;              /* description, max 51 chars */
   char *note;              /* note, max 512 bytes, TAB becomes CRLF */
   char *location;          /* location */
			    /* the end date is absent cause it's always the same as start */
   int weekview;	    /* 1 = visible, 0 is not */
   int monthview;
   struct rec_desc *next;   /* pointer to next element in list */
};

typedef struct rec_desc *RECORD;

/****************************************************************************/

FILE *fin;               /* CDF file */
FILE *fout;              /* resulting .csv file */

RECORD records = NIL;    /* list of data records */
int numrecords = 0;      /* number of records in database */
int notenum = 0;         /* number of current note */

int debug = NO;          /* debug mode */
char dateformat[80] = "yyyymmdd"; /* date input format for Wizard */
int retain_crlf = NO;    /* expect CR/LF in text fields */
int tquotes = YES;       /* expect quotes around text fields */
int nquotes = NO;        /* expect quotes around number fields */

char *monthnames[] =
{
   "JAN",
   "FEB",
   "MAR",
   "APR",
   "MAY",
   "JUN",
   "JUL",
   "AUG",
   "SEP",
   "OCT",
   "NOV",
   "DEC"
};

/****************************************************************************/
/* allocate memory */

char *my_malloc(int size)
{
   char *p;

   p = (char *)malloc(size);
   if (!p)
   {
      fprintf(stderr, "Memory allocation problems.\nAborted!\n");
      exit(1);
   }
   return p;
}

/****************************************************************************/
/* create a new record struct */

RECORD new_rec()
{
   RECORD r;

   r = (RECORD)my_malloc(sizeof(struct rec_desc));
   r->year = r->month = r->day = 0;
   r->stime = r->etime = r->lead = 0;
   r->desc = r->location = r->note = NIL;
   r->alarm = r->weekview = r->monthview = 1;
   r->next = records;
   records = r;
   return r;
}

/****************************************************************************/
/* check if file format is ok */

void check_syntax(char *buf, char *p, char ch, char *msg)
{
   if (*p == ch) return;
   if (!*p || ((p > buf) && !p[-1]))
      fprintf(stderr, "Syntax error %s: unexpected end of line!\n", msg);
   else if (ch != '?')
      fprintf(stderr, "Syntax error %s: `%c' expected!\n", msg, ch);
   else
      fprintf(stderr, "Syntax error %s: unexpected character!\n", msg);
   ch = *p;
   *p = '\0';
   fprintf(stderr, "%s<***HERE***>%c%s", buf, ch, &p[1]);
   exit(1);
}

/****************************************************************************/
/* read number from ascii string */

int get_number(char **p)
{
   int n;
   
   if (nquotes) check_syntax(*p, (*p)++, '\"', "before number");
   sscanf(*p, "%d", &n);
   while (**p && (**p >= '0') && (**p <= '9')) (*p)++;
   if (nquotes) check_syntax(*p, (*p)++, '\"', "after number");
   return n;
}

/****************************************************************************/
/* compare two strings ignoring case */

int my_strncasecmp(char *s1, char *s2, int n)
{
   while (*s1 && *s2 && n && (toupper(*s1) == toupper(*s2)))
   {
      s1++; s2++; n--;
   }
   if (n) return YES;
   return NO;
}

/****************************************************************************/
/* copy s1 to s2 for a maximum of n characters */

int my_strncpy(char *s1, char *s2, int n)
{
   while (*s1 && n )
   {
      *s2++ = *s1++;
      n--;
   }
   return YES;
}

/****************************************************************************/
/* read a date */

void get_date(char **p, char *y, char *m, char *d)
{
   /* the date format is fixed YYYYMMDD so use a simple scheme */
   int year, month, day;

   sscanf(*p, "%4d%2d%2d", &year, &month, &day);
   while (**p && (**p >= '0') && (**p <= '9')) (*p)++;
   *d = (char)day-1;
   *m = (char)month-1;
   *y = (char)(year-1900);
}

/****************************************************************************/
/* read a time field */

int get_time(char **p)
{
   int h, m;
   char *buf = *p;

   if (tquotes) check_syntax(buf, (*p)++, '\"', "before 'time'");
   sscanf(*p, "%d:%d", &h, &m);
   while (**p && (**p != '\"') && (tquotes || (**p != ','))) (*p)++;
   if (tquotes) check_syntax(buf, (*p)++, '\"', "after 'time'");
   return (60 * h + m);
}

/****************************************************************************/
/* read a text or note field -- but skip spurious spaces and tabs */

void get_text(char **p, char **str)
{
   char *q, *qq, *qend;
   char *buf = *p;

   if (tquotes) check_syntax(buf, (*p)++, '\"', "before text field");

   q = *p;
   while (*q)
   {
      if (retain_crlf && (*q == '\n') && (q != *p))
         fgets(&q[1], 1024, fin);
      if (tquotes && (*q == '\"') && (q[-1] != '\\')) break;
      if (!tquotes && (*q == ',') && (q[-1] != '\\')) break;
      if (!tquotes && ((*q == '\n') || (*q == '\r'))) break;
      q++;
   }
   qend = q;

   *str = (char *)my_malloc(qend - *p + 2);
   q = *p;
   qq = *str;
   while (q < qend)
   {
      if (*q == '\\')
      {
         q++;
         switch (*q)
         {
         case 'r': *(qq++) = '\r'; break;
         case 'n': *(qq++) = '\n'; break;
         default: *(qq++) = *q;
         }
         q++;
      }
      else {
	 /* previous character was a space or tab, ignore further spaces */
	 if ((*q == ' ') && ((*(qq-1) == ' ') || (*(qq-1) == '\t')))
	    q++;
	 else
            *(qq++) = *(q++);
      }
   }
   *qq = '\0';
   *p = qend;
   if (tquotes) check_syntax(buf, (*p)++, '\"', "after text field");
}


/****************************************************************************/
/* add records from CDF file */

void load_new_records()
{
   char *buf;
   char *p, *q;
   char **flag;
   RECORD r;
   int num, len, i;

   buf = (char *)my_malloc(1024);
   while (fgets(buf, 1022, fin))
   {
      if (*buf > ' ')
      {
         r = new_rec();
         p = buf;
	 /* Note - Wizard files have a date without quotes */
         /* if (tquotes) check_syntax(buf, p++, '\"', "(is this a CDF file?)"); */
         get_date(&p, &(r->year), &(r->month), &(r->day));
         /* if (tquotes) check_syntax(buf, p++, '\"', "after 'date'"); */
         check_syntax(buf, p++, ',', "after 'date'");
         r->stime = get_time(&p);
         check_syntax(buf, p++, ',', "after 'start time'");
         r->etime = get_time(&p);
         check_syntax(buf, p++, ',', "after 'end time'");
	 r->lead = get_time(&p);
	 check_syntax(buf, p++, ',', "after 'alarm time'");

	 /* if the total duration is less than an hour, do not put in weekview */
	 if ((r->etime - r->stime) < 60) r->weekview = 0;
	 /* if the total duration is less than 2 hours, do not put in monthview */
	 if ((r->etime - r->stime) < 120) r->monthview = 0;

	 /* now get the note field and extract the description from it */
         get_text(&p, &(r->note));
         check_syntax(buf, p++, ',', "after 'description'");
	 /* get the description from the note */
	 len = (strlen(r->note) > DESCLEN ? DESCLEN : strlen(r->note));
	 /* now check if there are tabs that we want to stop at */
	 q = r->note;
	 for (i=0; (i < len && q[i] != '\t'); i++);
	 if (q[i] == '\t') len = i;
	 r->desc = (char *) my_malloc (len);
	 strncpy (r->desc, r->note, len);
	 r->desc[len] = '\x0';

	 /* wizards have no location but the word after an @ is moved there */
	 /* scan the note for an @ sign */
	 q = r->note;
	 while (*q && (*q != '@')) q++;
	 /* if *q is @ then we have a location */
	 r->location = (char *) my_malloc (LOCLEN);
	 if (*q == '@') {
		q++;
		i = 0;
		while (isalnum(*q) && i < LOCLEN)
			r->location[i++] = *q++;
		r->location[i] = '\0';
		}
	 else
		r->location[0] = '\0';

	 get_text(&p, flag);
	 if (**flag == 'Y') r->alarm = 1;
	 else r->alarm = 0;

      }
   }
   fclose(fin);
   free(buf);
}

/****************************************************************************/
/* output text field */

void put_text(str)
char *str;
{
   char tquotes[2] = "\"";  /* quotes around CDF text fields */
   char *p = str;
   
   fprintf(fout, "%s", tquotes);
   if (!p)
   {
      fprintf(fout, "%s", tquotes);
      return;
   }
   
   while (*p)
   {
      switch (*p)
      {
      case '\r': if (retain_crlf)
                    fprintf(fout, "\r");
                 else
                    fprintf(fout, "\\r");
                 break;
      case '\n': if (retain_crlf)
                    fprintf(fout, "\n");
                 else
                    fprintf(fout, "\\n");
                 break;
      case '\"': fprintf(fout, "\\\"");
                 break;
      case ',':  if (!*tquotes)
                    fprintf(fout, "\\,");
                 else
                    fprintf(fout, ",");
                 break;
      default: fprintf(fout, "%c", *p);
      }
      p++;
   }
   fprintf(fout, "%s", tquotes);
}

/****************************************************************************/
/* write a data record to the .CSV file */

void write_record(RECORD r)
{
   char tquotes[2] = "\"";  /* quotes around CDF text fields */

   /* start date as YYYYMMDD */
   fprintf(fout, "%s", tquotes);
   fprintf(fout, "%d", 1900 + (r->year & 255));
   fprintf(fout, "%.2d", r->month + 1);
   fprintf(fout, "%.2d", r->day + 1);
   fprintf(fout, "%s,", tquotes);

   /* start time HHMM */
   fprintf(fout, "%s%.2d%.2d%s,", tquotes, r->stime / 60,
              r->stime % 60, tquotes);
   /* end time HHMM */
   fprintf(fout, "%s%.2d%.2d%s,", tquotes, r->etime / 60,
              r->etime % 60, tquotes);
   /* alarm */
   fprintf(fout, "%s%d%s,", tquotes, r->alarm, tquotes);
   /* lead time HHMM */
   fprintf(fout, "%s%.2d%.2d%s,", tquotes, r->lead / 60,
              r->lead % 60, tquotes);

   /* give description */
   put_text(r->desc);
   fprintf(fout, ",");
   put_text(r->note);
   fprintf(fout, ",");
   put_text(r->location);
   fprintf(fout, ",");

   /* print the enddate, the same as start date */
   fprintf(fout, "%s", tquotes);
   fprintf(fout, "%d", 1900 + (r->year & 255));
   fprintf(fout, "%.2d", r->month + 1);
   fprintf(fout, "%.2d", r->day + 1);
   fprintf(fout, "%s,", tquotes);

   /* weekview and monthview */
   fprintf(fout, "%s%d%s,", tquotes, r->weekview, tquotes);
   fprintf(fout, "%s%d%s\n", tquotes, r->monthview, tquotes);
}

/****************************************************************************/
/* add new records to .ADB file */

void write_new_records()
{
   RECORD r = records;
   
   while (r)
   {
      write_record(r);
      r = r->next;
   }
   fclose(fout);
}

/****************************************************************************/
/* display usage information */

void help(char *prog)
{
   fprintf(stderr, "WIZCSV version 1.0 by Peter van Es\n");
   fprintf(stderr, "Based on code by Andreas Garzotto\n\n");
   fprintf(stderr, "Usage: %s <.ozw file> <.csv file> \n", prog);

   exit(1);
}

/****************************************************************************/
/* decode options */

void get_options(int argc, char **argv)
{
   int state = 0, i = 1;

   while (i < argc)
   {
      if (argv[i][0] == '-')
      {
         switch (argv[i][1])
         {
         default: help(argv[0]);
         }
      }
      else
      {
         if (state == 0)
         {
            fin = fopen(argv[i], "r");
            if (fin == NULL)
            {
               fprintf(stderr, "Cannot open input file '%s'\n", argv[i]);
               exit(1);
            }
            state = 1;
         }
         else if (state == 1)
         {
            fout = fopen(argv[i], "w+");
            if (fout == NULL)
            {
               fprintf(stderr, "Cannot open output file '%s'\n", argv[i]);
               exit(1);
            }
            state = 2;
         }
         else
            help(argv[0]);
      }
      i++;
   }
}

/****************************************************************************/

void main(int argc, char **argv)
{
   if (argc < 3) help(argv[0]);
   
   get_options(argc, argv);
   load_new_records();
   write_new_records();
   exit(0);
}

/****************************************************************************/

