#
# Object which communicates with a digital camera over the serial port.
#
#  http://www.berkhirt.com/HomerProductions/
#  bhirt@berkhirt.com
#
# Copyright (c) 1998 by Brian Hirt, all rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#


package DCHandle;

use IO::File;
use Carp;

# Control bytes
$PKT_CTRL_RECV    = 0x01;
$PKT_CTRL_SEND    = 0x00;
$PKT_CTRL_EOF     = 0x80;
$PKT_CTRL_CANCEL  = 0xFF;

# Kodak System Codes
$DC_COMMAND_COMPLETE  = 0x00;
$DC_COMMAND_ACK       = 0xD1;
$DC_CORRECT_PACKET    = 0xD2;
$DC_COMMAND_NAK       = 0xE1;
$DC_ILLEGAL_PACKET    = 0xE3;
$DC_BUSY              = 0xF0;

# Commands common to all implemented Kodak cameras
$DC_SET_SPEED         = 0x41;

$debug = 0;

sub new
{
  my $class = shift;
  my $tty = shift;
  my $this = {};

  my $fileHandle = IO::File->new($tty,2) || die "cannot open $tty";

  system("stty raw -parenb -parodd -onlcr -iexten -echonl -noflsh -tostop -echoprt -ixoff -olcuc -ocrnl -onocr -onlret -ofill -ofdel -echo -echoe -echok -echoctl -echoke min 255 time 5 cs8 9600 < $tty");

  bless $this,$class;

  $this->setTty($tty);

  $this->setFileHandle($fileHandle);

  # Probe for a known camera
#  $this->probeCamera;

  return $this;
}

sub probeCamera
{
#
# This function probes the camera.  It cannot use many of the DCHandle
# methods since DC50/DC120 speak slightly different protocol than the
# DC210.   Once the camera is detected, all DCHandle methods should work.
#
  my $this = shift;

  # send the status command
  $this->sendCommand($DC_STATUS);

  # DC120 & DC50 do not send packet control bytes, DC210 does.
  # Regardless, DC120 and DC50 must send 0x01 as the first byte of the camera status
  my $firstByte = $this->readByte();
  if ($firstByte != 0x01)
  {
    carp "DCHandle::probeCamera - unexpected response from camera ($firstByte)\n";
  }
  else
  {
    my $secondByte = $this->readByte();
    # Okay, if the second byte is 0x01, then the camera must send
    # control packet; reason? first byte in status info must be 0x01
    # second byte is camera type.  Camera type 0x01 is DC50 and we 
    # know that DC50 doesn't send packet control.
    if ($secondByte == 0x01)
    {
      $cameraType = $this->readByte();
      $this->setHostPacketControl(1);
    }
    else
    {
      $cameraType = $secondByte;
      $this->setHostPacketControl(0);
    }
   
    my $data = pack("C2",0x01,$cameraType);
    my $read = 0;
    while ($read < 254)
    {
      my $stream;
      my $numRead = sysread($this->fileHandle,$stream,254 - $read);

      $data .= $stream;
      $read += $numRead;
    }

    my $sentChecksum = $this->readByte();
    my $computedChecksum = $this->checksum($data,256);

    if ($sentChecksum != $computedChecksum)
    {
      print STDERR "bad checksum $computedChecksum != $sentChecksum";
      $this->writeByte($DC_ILLEGAL_PACKET);
    }
    else
    {
      print STDERR "checksums MATCH!  $computedChecksum == $sentChecksum";
      $this->writeByte($DC_CORRECT_PACKET);
    }

    if ($cameraType == $DCUtils::DC50_CAMERA_TYPE)
    {
      print STDERR "DCHandle::probeCamera - DC50 detected\n";
    }
    elsif ($cameraType == $DCUtils::DC120_CAMERA_TYPE)
    {
      print STDERR "DCHandle::probeCamera - DC120 detected\n";
    }
    elsif ($cameraType == $DCUtils::DC210_CAMERA_TYPE)
    {
      print STDERR "DCHandle::probeCamera - DC210 detected\n";
    }
    else
    {
      print STDERR "DCHandle::probeCamera - unknown camera\n";
    }

    $this->commandComplete;
  }
}

sub setPortSpeed
{
  my $this = shift;
  my $speed = shift;

  my @args;

  if ($speed == 9600)
  {
    @args = (0x96,0x00);
  }
  elsif ($speed == 19200)
  {
    @args = (0x19,0x20);
  }
  elsif ($speed == 38400)
  {
    @args = (0x38,0x40);
  }
  elsif ($speed == 57600)
  {
    @args = (0x57,0x60);
  }
  elsif ($speed == 115200)
  {
    @args = (0x11,0x52);
  }
  else
  {
    carp "speed not supported";
    return undef;
  }

  $this->sendCommand($DC_SET_SPEED,@args,0x00,0x00);
  system("stty raw min 255 time 5 cs8 $speed < " . $this->tty);
}

sub fileHandle
{
  my $this = shift;

  $this->{'fileHandle'};
}

sub setFileHandle
{
  my $this = shift;

  $this->{'fileHandle'} = shift;
}

sub setTty
{
  my $this = shift;

  $this->{'tty'} = shift;
}

sub tty
{
  my $this = shift;

  $this->{'tty'};
}

sub sendCommand
{
# sends 8 byte packet to camera and waits for an ACK.  Returns true if 
# successful, false otherwise.
  my $this = shift;
  my $command = shift;
  my $arg1 = shift;
  my $arg2 = shift;
  my $arg3 = shift;
  my $arg4 = shift;

  my $success = 1;

  # make sure a command was supplied
  if (defined($command))
  {
    $arg1 = 0x00 if !defined($arg1);
    $arg2 = 0x00 if !defined($arg2);
    $arg3 = 0x00 if !defined($arg3);
    $arg4 = 0x00 if !defined($arg4);

    my $data = pack("C8",$command,0x00,$arg1,$arg2,$arg3,$arg4,0x00,0x1A);
    my $wrote = syswrite($this->fileHandle,$data,8);

    if ($wrote != 8)
    {
      carp "DCHandle::sendCommand - only wrote partial command ($wrote)";
      $success = 0;
    }
    else
    {
      # Get ack from camera 
      my $ack = $this->readByte();
      if ($ack != $DC_COMMAND_ACK)
      {
	carp "DCHandle::sendCommand - bad ack from camera ($ack)";
	$success = 0;
      }
    }
  }
  else
  {
    carp "DCHandle::sendCommand - command is undef";
    $success = 0;
  }

  return $success;
}

sub sendCorrectPacket
{
  my $this = shift;

  my $success = 1;  # lets assume it works.

  $this->writeByte($DC_CORRECT_PACKET);

  my $complete = $this->readByte();
  if (($ack != 0) && ($ack != 1))
  {
    carp "DCHandle::sendAck - bad response ($ack)";
    $success = 0;
  } 

  return $success;
}

sub checksum
{
# Computes the *primitave* kodak checksum using XOR - yuck.
  my $this = shift;
  my $data = shift;
  my $length = shift;

  my $chksum = 0;

  foreach (unpack("C$length",$data))
  {
    $chksum ^= $_;
  }

  return $chksum;
}

sub sendPacket
{
  my $this = shift;
  my $controlByte = shift;
  my $data = shift;
  my $length = shift;

  if (!defined($controlByte))
  {
    carp "DC210::sendPacket - controlByte not supplied";
  }
  elsif (!defined($data))
  {
    carp "DC210::sendPacket - packet data not supplied";
  }
  elsif (!defined($length))
  {
    carp "DC210::sendPacket - length not supplied";
  }
  else
  {
    my $wrote = 0;

    $this->writeByte($controlByte);

    while ($wrote < $length)
    {
      $wrote += syswrite($this->fileHandle,$data,$length - $wrote,$wrote);
    }

    my $chkSum = $this->checksum($data,$length);

    # send the checksum to the camera
    $this->writeByte($chkSum);

    # Make sure the camera got the packet ok
    my $correctPacket = $this->readByte;
    if ($correctPacket != $DC_CORRECT_PACKET)
    {
      carp "DC210::sendPacket - camera didn't like the packet ($correctPacket)";
    }
  }

  return undef;
}

sub commandComplete
{
  my $this = shift;

  my $status;

  # Wait for the camera to be ready.
  do
  {
    $status = $this->readByte();
  } 
  while ($status == $DC_BUSY);

  if ($status != $DC_COMMAND_COMPLETE)
  {
    if ($status == $DC_ILLEGAL_PACKET)
    {
      carp "DCHandle::commandComplete - illegal packet received from host";
    }
    else
    {
      carp "DCHandle::commandComplete - command not completed ($status)";
    }
  }
}

sub cameraType
{
  my $this = shift;

  return $this->{'cameraType'};
}

sub setCameraType
{
  my $this = shift;

  $this->{'cameraType'} = shift;
}

sub hostPacketControl
{
  my $this = shift;

  return $this->{'hostPacketControl'};
}

sub setHostPacketControl
{
  my $this = shift;

  $this->{'hostPacketControl'} = shift;
}

sub readPacket
{
  my $this = shift;
  my $length = shift;

  # Newer cameras send packet control byte
  if ($this->hostPacketControl)
  {
    # Read the control byte from the camera - and packet coming from
    # the camera must ALWAYS be 0x01 according to Kodak Host Protocol.
    my $controlByte = $this->readByte();
    if ($controlByte != 0x01)
    {
      carp "DCHandle::readPacket - packet control byte invalid ($controlByte)\n";
    }
  }

  my $data = '';
  my $read = 0;
  my $stream;
  while ($read < $length)
  {
    my $numRead = sysread($this->fileHandle,$stream,$length - $read);

    $data .= $stream;
    $read += $numRead;
    print STDERR "$numRead/$read/$length\n" if $debug;
  }

  my $sentChecksum = $this->readByte();
  my $computedChecksum = $this->checksum($data,$length);

  if ($sentChecksum != $computedChecksum)
  {
    print STDERR "bad checksum $computedChecksum != $sentChecksum";
    $this->writeByte($DC_ILLEGAL_PACKET);
  }
  else
  {
    $this->writeByte($DC_CORRECT_PACKET);
  }


  return $data;
}

sub writeByte
{
  my $this = shift;
  my $byte = shift;

  my $wrote = 0;

  do
  {
    $wrote = syswrite($this->fileHandle,pack('C1',$byte),1);
  }
  while ($wrote == 0);

  if ($wrote != 1)
  {
    carp "DC210::writeByte - wanted to write one byte, wrote ($read) bytes";
  }

  return;
}

sub readByte
{
  my $this = shift;

  my $read = 0;
  my $byte;

  # read a byte via sysread
  do 
  {
    $read += sysread($this->fileHandle,$byte,1);
  }
  while ($read == 0);

  if ($read != 1)
  {
    carp "DC210::readByte - wanted to read one byte, read ($read) bytes";
  }

  return unpack('C1',$byte);
}

sub close
{
  my $this = shift;

  $this->fileHandle->close;
}

sub DESTROY
{
  my $this = shift;

  print STDERR "DCHandle.pm - garbage collected\n" if $debug;
}

 

1;
