#
# Object which manages a DCHandle object and implements the application
# level protocol of the camera.
#
#
#  http://www.berkhirt.com/HomerProductions/
#  bhirt@berkhirt.com
#
# Copyright (c) 1998 by Brian Hirt, all rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#

package DC120Status;

use DCHandle;
use DCPictureInfo;
use DCUtils;
use POSIX;

use Carp;

$DC120Epoc = 852094800;

$debug = 0;

sub new
{
  my $class = shift;
  my $data = shift;

  my $this = bless {}, $class;

  print STDERR "DC120Status::new\n" if $debug;

  (undef,
   $cameraTypeId,
   $firmwareMajor,
   $firmwareMinor,
   $blah1,
   $blah2,
   $blah3,
   $blah4,
   $batteryStatusId,
   $acStatusId,
   undef,               # reserved
   $time,
   $zoomMode,
   undef,		# reserved
   $flashCharged,
   $compressionModeId,
   $flashMode,
   $exposureCompensation,
   undef,
   undef,
   $manualExposureTime,		 
   undef,			# not implemented yet.
   $numPicturesInMemory,	# the number of pictures in internal memory
   undef,			# not implemented yet
   $numPicturesInCard,		# the number of pictures in Compact Flash
   undef,			# not implemented yet
   $compactFlashVolumeId,	# volume id of the ATA card
   undef,			# reserved
   $cameraIdent			# camera identification
	) = unpack('C10 a2 N1 C8 N1 a8 n1 a18 n1 a19 A11 a2 a32',$data);

#  print STDERR "firmware: $firmwareMajor, $firmwareMinor\n";
#  print STDERR "blah 1-4: $blah1, $blah2, $blah3, $blah4\n";
#  print STDERR "exporuse time: $manualExposureTime\n";

  $this->_setFlashModeId($flashMode);
  $this->_setBatteryStatusId($batteryStatusId);
  $this->_setAcStatusId($acStatusId);
  $this->_setPicturesInCompactFlash($numPicturesInCard);
  $this->_setPicturesInInternalMemory($numPicturesInMemory);
  $this->_setCameraIdentification($cameraIdent);
  $this->_setCameraTypeId($cameraTypeId);
  $this->_setCompressionModeId($compressionModeId);
  $this->_setFlashCharged($flashCharged);
  $this->_setCameraTime($time);
  $this->_setCompactFlashVolumeId($compactFlashVolumeId);

  return $this;
}

sub attributes
{
  ('flashMode',
   'batteryStatus',
   'acStatus',
   'compactFlashVolumeId',
   'picturesInCompactFlash',
   'picturesInInternalMemory',
   'numberOfPictures',
   'cameraIdentification',
   'cameraType',
   'compressionMode',
   'flashCharged',
   'cameraTime');
}

sub cameraTime
{
  my $this = shift;


  my $time = $DC120Epoc + int($this->{'cameraTime'} / 2);

  return POSIX::strftime("%a, %d %b %y %T",localtime($time));
}

sub flashCharged
{
# returns the flash charged bit
  my $this = shift;

  return $this->{'flashCharged'};
}

sub compressionModeId
{
# returns the compression mode id
  my $this = shift;

  return $this->{'compressionModeId'};
}

sub compressionMode
{
# returns the compression mode (low/med/high)
  my $this = shift;

  return DCUtils->compressionModeWithId($this->compressionModeId);
}

sub cameraTypeId
{
# returns the camera type id
  my $this = shift;

  return $this->{'cameraTypeId'};
}

sub cameraType
{
# returns the camera type description
  my $this = shift;

  return DCUtils->cameraTypeWithId($this->cameraTypeId);
}

sub flashModeId
{
# returns the flash mode id
  my $this = shift;

  return $this->{'flashModeId'};
}

sub flashMode
{
# returns the flash mode description
  my $this = shift;

  return DCUtils->flashModeWithId($this->flashModeId);
}

sub acStatusId
{
# returns the ac adaptor status id
  my $this = shift;

  return $this->{'acStatusId'};
}

sub acStatus
{
# returns the status of the ac adaptor
  my $this = shift;

  my $status = 'unknwon';
  
  if (defined($this->acStatusId))
  {
    if ($this->acStatusId == 1)
    {
      $status = 'connected';
    }
    elsif ($this->acStatusId == 0)
    {
      $status = 'not connected';
    }
  }
  else
  {
    carp "DC210::acStatus - the ac status is not defined, has the camera been initialized?";
  }

  return $status;
}

sub batteryStatusId
{
# returns the battery status id
  my $this = shift;

  return $this->{'batteryStatusId'};
}

sub batteryStatus
{
# returns the status of the battery
  my $this = shift;

  return DCUtils->batteryStatus($this->batteryStatusId);
}

sub cameraIdentification
{
  my $this = shift;

  $this->{'cameraIdentification'};
}

sub setCameraIdentification
{
  my $this = shift;
  my $cameraId = shift;

  print STDERR "DC210::setCameraIdentification\n" if $debug;

  # Create a 58 bytes null padded string
  my $packet = pack("a58",$cameraId);

  # Send set camera id packet
  $this->dcHandle->sendCommand($DCHandle::DC_SET_CAMERA_ID);
  $this->dcHandle->sendPacket($DCHandle::PKT_CTRL_EOF,$packet,58);
  $this->dcHandle->commandComplete;

  $this->loadStatus;
}

sub picturesInCompactFlash
{
# returns number of pictures in compact flash
  my $this = shift;

  return $this->{'pictureInCompactFlash'};
}

sub compactFlashVolumeId
{
# returns the volume id of the compact flash
  my $this = shift;

  return $this->{'compactFlashVolumeId'};
}


sub picturesInInternalMemory
{
# returns number of pictures in internal memory
  my $this = shift;

  return $this->{'pictureInInternalMemory'};
}

sub numberOfPictures
{
# returns the number of pictures in the camera
  my $this = shift;

  $this->picturesInCompactFlash + $this->picturesInInternalMemory;
}

## Private helper methods
sub _setCompactFlashVolumeId
{
# sets the volumn id of the compact flash, private method
  my $this = shift;

  $this->{'compactFlashVolumeId'} = shift;
}

sub _setCompressionModeId
{
# sets the compression mode id flag (low/med/high), private method
  my $this = shift;

  $this->{'compressionModeId'} = shift;
}

sub _setAcStatusId
{
# sets the ac adaptor id flag (plugged in/not plugged in), private method
  my $this = shift;

  $this->{'acStatusId'} = shift;
}

sub _setPicturesInInternalMemory
{
# sets the number of pictures in the camera's internal memory, private method
  my $this = shift;

  $this->{'pictureInInternalMemory'} = shift;
}

sub _setPicturesInCompactFlash
{
# sets the number of pictures in the camera's internal memory, private method
  my $this = shift;

  $this->{'pictureInCompactFlash'} = shift;
}

sub _setFlashModeId
{
# sets the flash mode id, private method
  my $this = shift;

  $this->{'flashModeId'} = shift;
}

sub _setBatteryStatusId
{
# sets the battery status id, private method
  my $this = shift;

  $this->{'batteryStatusId'} = shift;
}

sub _setCameraTypeId
{
# sets the camera type id, private method
  my $this = shift;

  $this->{'cameraTypeId'} = shift;
}

sub _setFlashCharged
{
# sets the flash charged bit
  my $this = shift;

  $this->{'flashCharged'} = shift;
}

sub _setTotalPicturesTaken
{
# sets the total number of pictures taken 
  my $this = shift;

  $this->{'totalPicturesTaken'} = shift;
}

sub _setTotalFlashesFired
{
# sets the number of total flashes fired
  my $this = shift;

  $this->{'totalFlashesFired'} = shift;
}

sub _setCameraIdentification
{
# sets the camera identification string
  my $this = shift;

  $this->{'cameraIdentification'} = shift;
}

sub _setCameraTime
{
# sets the camera time, private method
  my $this = shift;

  $this->{'cameraTime'} = shift;
}

sub DESTROY
{
  my $this = shift;

  print STDERR "DC120Status::DESTROY\n" if $debug;
}


1;
