
#  http://www.berkhirt.com/HomerProductions/
#  bhirt@berkhirt.com
#
# Copyright (c) 1998 by Brian Hirt, all rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#

package DC50;

# Generic packages
use DCHandle;
use DCUtils;
use DC;

# DC50 specific libraries
use DC50Status;
use DC50PictureInfo;

# Perl libraries
use POSIX;
use Carp;

$DC50_EPOC = 852094800;

# Kodak System Commands
$DC50_GET_PICTURE_IN_CAMERA  = 0x51;
$DC50_GET_PICTURE_IN_CARD    = 0x61;
$DC50_PICTURE_INFO_IN_CAMERA = 0x55;
$DC50_PICTURE_INFO_IN_CARD   = 0x65;
$DC50_ERASE_IMAGE_IN_CAMERA  = 0x7A;
$DC50_ERASE_IMAGE_IN_CARD    = 0x7B;
$DC50_TAKE_PICTURE_TO_CAMERA = 0x77;
$DC50_TAKE_PICTURE_TO_CARD   = 0x7C;
$DC50_INITIALIZE             = 0x7E;
$DC50_STATUS                 = 0x7F;

$DC50_PICTURE_HEADER_SIZE = 15680;

$debug = 0;

@ISA = qw(DC);

sub new
{
  my $class = shift;
  my $port = shift;
  my $speed = shift;

  my $this = {};
 
  print STDERR "DC50::new\n" if $debug;

  # default to 9600bps
  $speed = 9600 if !defined($speed);

  bless $this, $class;

  # craete a new handle
  $this->setDcHandle(DCHandle->new($port));

  # DC50 cameras DO NOT send host packet control
  $this->dcHandle->setHostPacketControl(0);

  # initialize the connection with the camera
  if ($this->initialize())
  {
    $this->dcHandle->setPortSpeed($speed);
    $this->initialize();
  }
  else
  {
    # clear the DC handle since object destruction tries to
    # reset the oprt speed.
    $this->setDcHandle(undef);

    # fail on constructor.
    $this = undef;
  }

  return $this;
}

sub initialize
{
  my $this = shift;

  print STDERR "DC50::initialize\n" if $debug;

  eval {
    # Kodak Host Interface Document say that camera will response to
    # an initialize event within 3 seconds, put a timer on the initialize
    # to trap turned off cameras / bad baud rates.
    local $SIG{ALRM} = sub { die "timeout" };
    alarm(5);
    $this->dcHandle->sendCommand($DC50_INITIALIZE);
    $this->dcHandle->commandComplete;
  };

  # reset alarm so it doesn't fire again
  alarm(0);

  # Check to see if we were able to initialize
  if (defined($@) && ($@ =~ /timeout/))
  {
    print STDERR "DC50::initialize camera initialization timed out\n";
    return 0;
  }
  else
  {
    return 1;
  }
}

sub loadStatusFromCamera
{
  my $this = shift;

  print STDERR "DC50::status\n" if $debug;

  $this->dcHandle->sendCommand($DC50_STATUS);
  my $data = $this->dcHandle->readPacket(256);
  $this->dcHandle->commandComplete;

  return DC50Status->new($data);
}

sub deleteAllPictures
{
# deletes all pictures in the camera.
  my $this = shift;

  print STDERR "DC50::deleteAllPictures\n" if $debug;

  # erase pictures with picture number 0 = erase all 
  $this->dcHandle->sendCommand(
		$DC50_ERASE_IMAGE_IN_CARD, 0x00, 0x00, 0x00, 0x00);
  $this->dcHandle->commandComplete;

  # erase pictures with picture number 0 = erase all 
  $this->dcHandle->sendCommand(
		$DC50_ERASE_IMAGE_IN_CAMERA, 0x00, 0x00, 0x00, 0x00);
  $this->dcHandle->commandComplete;

  # Pictures have been deleted, reload status
  $this->clearCachedStatus;
}

sub deletePicture
{
  my $this = shift;
  my $number = shift;

  print STDERR "DC50 doesn't support deletion of single pictures in internal memory.\n";
  print STDERR "kdcpi doesn't support deletion of single pictures in ATA card.\n";
}

sub takePicture
{
# Takes a picture, no zoom/flash control yet.
  my $this = shift;

  print STDERR "DC50::takePictures\n" if $debug;

  # internal memory card
  $this->dcHandle->sendCommand($DC50_TAKE_PICTURE_TO_CAMERA);
  $this->dcHandle->commandComplete;

# ATA card
#  $this->dcHandle->sendCommand($DC50_TAKE_PICTURE);
#  $this->dcHandle->commandComplete;

  # Picture has been taken, reload status
  $this->clearCachedStatus;
}

sub pictureInfo
{
# returns information about the requested picture.
  my $this = shift;
  my $number = shift;

  print STDERR "DC50::pictureInfo\n" if $debug;

  my $pictureInfo = undef;

  if (($number < 1) || ($number > $this->status->numberOfPictures))
  {
    carp "DC50::getPictureInfo - invalid picture number";
  }
  else
  {
    my $dcHandle = $this->dcHandle;
    if ($number <= $this->status->picturesInInternalMemory)
    {
      if ($dcHandle->sendCommand($DC50_PICTURE_INFO_IN_CAMERA,unpack('C2',pack('n1',$number)),0x00,0x00))
      {
	my $data = $dcHandle->readPacket(256);
	$dcHandle->commandComplete;

	my @attrs = unpack('C8 N2 C12 N1 C5 A8',$data);

	$pictureInfo = DC50PictureInfo->new();

	if ($attrs[1] == 1)
	{
	  $pictureInfo->_setPictureFormat('KDC');
	  $pictureInfo->_setFileName($attrs[28] . ".KDC");
	}
	else
	{
	  $pictureInfo->_setPictureFormat('???');
	  $pictureInfo->_setFileName($attrs[28] . ".???");
	}

	$pictureInfo->_setPictureNumber($number);
	$pictureInfo->_setFileSize($attrs[8]);
        $pictureInfo->_setCompressionModeId($attrs[4]);
        $pictureInfo->_setResolutionId(50);
      }
      else
      {
	carp "DC50::pictureInfo - sendCommand failed for internalMemory";
        $pictureInfo = undef;;
      }
    }
    else
    {
      my $arg = $number - $this->status->picturesInInternalMemory;

      # send picture info command, receive data, send ack
      if ($dcHandle->sendCommand($DC50_PICTURE_INFO_IN_CARD,unpack('C2',pack('n1',$arg)),0x00,0x00))
      {
	my $data = $dcHandle->readPacket(256);
	$data .= $dcHandle->readPacket(256);
	$data .= $dcHandle->readPacket(256);
	$data .= $dcHandle->readPacket(256);
	$data .= $dcHandle->readPacket(256);

	$dcHandle->commandComplete;

	my @data = unpack('a440A8a259C1a70N1',$data);

	$pictureInfo = DC50PictureInfo->new();

	# Check the file format
	if ($data[3] == 1)
	{
	  $pictureInfo->_setPictureFormat('KDC');
	  $pictureInfo->_setFileName($data[1] . ".KDC");
	}
	elsif ($data[3] == 7)
	{
	  $pictureInfo->_setPictureFormat('JPG');
	  $pictureInfo->_setFileName($data[1] . ".JPG");
	}
	else
	{
	  $pictureInfo->_setPictureFormat('???');
	  $pictureInfo->_setFileName($data[1] . ".???");
	}

	$pictureInfo->_setPictureNumber($number);
	$pictureInfo->_setFileSize($data[5]);
        $pictureInfo->_setResolutionId(50);
  #      $pictureInfo->_setCompressionModeId($compression);
      }
      else
      {
	carp "DC50::pictureInfo - sendCommand failed for compactFlash";
        $pictureInfo = undef;;
      }
    }
  }

  return $pictureInfo;
}

sub thumbnail
{
  my $this = shift;

  print STDERR "kdcpi does support thumbnail pictures for DC50.\n";

  return undef;
}

sub picture
{
# Returns the data for the requested picture
  my $this = shift;
  my $number = shift;

  my $picture = '';

  print STDERR "DC50::picture\n" if $debug;
  if (($number < 1) || ($number > $this->status->numberOfPictures))
  {
    carp "DC50::getPicture - invalid picture number";
  }
  else
  {
    # initialize local vars
    my $dcHandle = $this->dcHandle;
    my $pictureInfo = $this->pictureInfo($number);
    my $fileSize = $pictureInfo->fileSize;
    my $blockSize = 1024;
    my $read = 0;
    my $cancel = 0;

    my $cmd;
    my $arg1;
    my $arg2;

    if ($pictureInfo->pictureFormat eq 'KDC')
    {
      if ($number <= $this->status->picturesInInternalMemory)
      {
	# picture is in internal memory
	$cmd =  $DC50_GET_PICTURE_IN_CAMERA;
	($arg1,$arg2) = unpack('C2',pack('n1',$number));
      }
      else
      {
	# picture is in CompactFlash memory
	$cmd = $DC50_GET_PICTURE_IN_CARD;
	($arg1,$arg2) = unpack('C2',pack('n1',$number - $this->status->picturesInInternalMemory));
      }

      $dcHandle->sendCommand( $cmd, $arg1, $arg2, 0x00, 0x00);

      while ($read < $fileSize)
      {
	my $block = $dcHandle->readPacket($blockSize);

	if ($read + $blockSize > $fileSize)
	{
	  $blockSize = $fileSize - $read;
	  $block = substr($block,0,$blockSize);
	}

	$picture .= $block;
	$read += $blockSize;

	if (defined($this->{'downloadNotify'}))
	{
	  $cancel = &{$this->{'downloadNotify'}}($pictureInfo,$read);
	}
      }

      $this->dcHandle->commandComplete;
    }
    else
    {
      print STDERR "picture $number is in an unrecognized format\n";
    }
  }

  return $picture;
}

sub DESTROY
{
  my $this = shift;

  print STDERR "DC50::DESTROY\n" if $debug;

  # set the port speed back to 9600 since that's what the default
  # speed of the camera is.
  if (defined($this->dcHandle))
  {
    $this->dcHandle->setPortSpeed(9600);

    # close the dcHandle
    $this->dcHandle->close;

    # break reference to dcHandle for proper garbage collection
    $this->setDcHandle(undef);
  }
}


1;
