#
# Object which manages a DCHandle object and implements the application
# level protocol of the camera.
#
#
#  http://www.berkhirt.com/HomerProductions/
#  bhirt@berkhirt.com
#
# Copyright (c) 1998 by Brian Hirt, all rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#

package DC210;

use DCHandle;
use DCUtils;
use DC;

use DC210PictureInfo;
use DC210Status;

use POSIX;

use Carp;

$DC210_LOW_RES_THUMBNAIL  = 0;
$DC210_HIGH_RES_THUMBNAIL = 1;
$DC210_EPOC = 852094800;

# Kodak System Commands
$DC210_SET_RESOLUTION      = 0x36;
$DC210_PICTURE_DOWNLOAD    = 0x64;
$DC210_PICTURE_INFO        = 0x65;
$DC210_PICTURE_THUMBNAIL   = 0x66;
$DC210_SET_SOMETHING       = 0x75;
$DC210_TAKE_PICTURE        = 0x7C;
$DC210_ERASE               = 0x7A;
$DC210_ERASE_IMAGE_IN_CARD = 0x7B;
$DC210_INITIALIZE          = 0x7E;
$DC210_STATUS              = 0x7F;
$DC210_SET_CAMERA_ID       = 0x9E;


$debug = 0;

@ISA = qw(DC);

sub new
{
  my $class = shift;
  my $port = shift;
  my $speed = shift;

  my $this = $class->SUPER::new();
 
  print STDERR "DC210::new\n" if $debug;

  # default to 9600bps
  $speed = 9600 if !defined($speed);

  bless $this, $class;

  # craete a new handle
  $this->setDcHandle(DCHandle->new($port));

  # DC210 cameras send host packet control
  $this->dcHandle->setHostPacketControl(1);

  # initialize the connection with the camera
  if ($this->initialize())
  {
    $this->dcHandle->setPortSpeed($speed);
    $this->initialize();
  }
  else
  {
    # clear the DC handle since object destruction tries to
    # reset the oprt speed.
    $this->setDcHandle(undef);

    # fail on constructor.
    $this = undef;
  }

  return $this;
}

sub initialize
{
  my $this = shift;

  print STDERR "DC210::initialize\n" if $debug;

  eval {
    # Kodak Host Interface Document say that camera will response to
    # an initialize event within 3 seconds, put a timer on the initialize
    # to trap turned off cameras / bad baud rates.
    local $SIG{ALRM} = sub { die "timeout" };
    alarm(5);
    $this->dcHandle->sendCommand($DC210_INITIALIZE);
    $this->dcHandle->commandComplete;
  };

  # reset alarm so it doesn't fire again
  alarm(0);

  # Check to see if we were able to initialize
  if (defined($@) && ($@ =~ /timeout/))
  {
    print STDERR "DC210::initialize camera initialization timed out\n";
    return 0;
  }
  else
  {
    return 1;
  }
}

sub loadStatusFromCamera
{
  my $this = shift;

  print STDERR "DC210::status\n" if $debug;

  $this->dcHandle->sendCommand($DC210_STATUS);
  my $data = $this->dcHandle->readPacket(256);
  $this->dcHandle->commandComplete;

  return DC210Status->new($data);
}

sub deleteAllPictures
{
# deletes all pictures in the camera.
  my $this = shift;

  print STDERR "DC210::deleteAllPictures\n" if $debug;
  $this->dcHandle->sendCommand(
		$DC210_ERASE_IMAGE_IN_CARD,
		0xFF,
		0xFF,
		0x00,
		0x00);
  $this->dcHandle->commandComplete;

  # Pictures have been deleted, reload status
  $this->clearCachedStatus;
}

sub deletePicture
{
# Takes a picture, no zoom/flash control yet.
  my $this = shift;
  my $number = shift;

  print STDERR "DC210::deletePicture\n" if $debug;
  if (($number < 1) || ($number > $this->status->numberOfPictures))
  {
    carp "DC210::deletePicture - invalid picture number";
  }
  else
  {
    my $lsb;
    my $msb;

    ($msb,$lsb) = unpack('C2',pack('n1',$number-1));

    $this->dcHandle->sendCommand(
		$DC210_ERASE_IMAGE_IN_CARD,
		$msb,
		$lsb,
		0x00,
		0x00);
    $this->dcHandle->commandComplete;

    # Picture has been deleted, reload status
    $this->clearCachedStatus;
  }
}

sub takePicture
{
# Takes a picture, no zoom/flash control yet.
  my $this = shift;

  print STDERR "DC210::takePictures\n" if $debug;

  $this->dcHandle->sendCommand($DC210_TAKE_PICTURE);
  $this->dcHandle->commandComplete;

  # Picture has been taken, reload status
  $this->clearCachedStatus;
}

sub pictureInfo
{
# returns information about the requested picture.
  my $this = shift;
  my $number = shift;

  print STDERR "DC210::pictureInfo\n" if $debug;

  my $pictureInfo = undef;

  if (($number < 1) || ($number > $this->status->numberOfPictures))
  {
    carp "DC210::getPictureInfo - invalid picture number";
  }
  else
  {
    # initialize local vars
    my $fileSize = undef;
    my $fileName = undef;
    my $pictureNumber = undef;

    my $lsb;
    my $msb;

    ($msb,$lsb) = unpack('C2',pack('n1',$number - 1));

    # send picture info command, receive data, send ack
    if ($this->dcHandle->sendCommand(
		$DC210_PICTURE_INFO,
		$msb,
		$lsb,
		0x0,
		0x0))
    {
      my $data = $this->dcHandle->readPacket(256);

      $this->dcHandle->commandComplete;

      ( undef,
	$resolution,
	$compression,
	undef,
	$pictureNumber,
	$fileSize,
	$elapsedTime,
	undef,
	$fileName) = unpack('a3C3n1N2a16a12',$data);

      $pictureInfo = DC210PictureInfo->new();
      if ($fileName =~ /JPG/)
      {
        $pictureInfo->_setPictureFormat('JPG');
      }
      elsif ($fileName =~ /FPX/)
      {
        $pictureInfo->_setPictureFormat('FPX');
      }
      else
      {
        $pictureInfo->_setPictureFormat('???');
      }

      $pictureInfo->_setPictureNumber($pictureNumber+1);
      $pictureInfo->_setFileSize($fileSize);
      $pictureInfo->_setFileName($fileName);
      $pictureInfo->_setResolutionId($resolution);
      $pictureInfo->_setCompressionModeId($compression);
    }
    else
    {
      carp "DC210::pictureInfo - sendCommand failed";
exit;
    }
  }

  return $pictureInfo;
}

sub thumbnail
{
# Returns the data for the requested thumbnail, data is in
# a 4 bit Bayer CFA encoding, no functions provided for decoding yet.
  my $this = shift;
  my $number = shift;
  my $resolution = shift;

  my $thumbnail = '';

  print STDERR "DC210::thumbnail\n" if $debug;

  if (($number < 1) || ($number > $this->status->numberOfPictures))
  {
    carp "DC210::thumbnail - invalid picture number";
  }
  else
  {
    # initialize local vars
    my $dcHandle = $this->dcHandle;

    my $read = 0;
    my $fileSize = ($resolution == $DC210_LOW_RES_THUMBNAIL) ? 2345 : 20736;
    my $blockSize = 1024;

    my $lsb;
    my $msb;

    ($msb,$lsb) = unpack('C2',pack('n1',$number-1));

    $dcHandle->sendCommand(
		$DC210_PICTURE_THUMBNAIL,
		$msb,
		$lsb,
		$resolution,
		0x00);

    while ($read < $fileSize)
    {
      my $block = $dcHandle->readPacket($blockSize);

      if ($read + $blockSize > $fileSize)
      {
        $blockSize = $fileSize - $read;
        $block = substr($block,0,$blockSize);
      }

      $thumbnail .= $block;
      $read += $blockSize;
    }

    $this->dcHandle->commandComplete;
  }

  my @bmp = (
	0x42, 0x4d, 0x36, 0x24,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x28, 
	unpack('C4',pack('N1',96)),
	unpack('C4',pack('N1',72)),
        0x00, 0x00, 0x00, 0x01, 0x00, 24, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	reverse unpack("C" . 96 * 72 * 3,$thumbnail));

  return pack('C' . scalar(@bmp) , @bmp); 
}

sub picture
{
# Returns the data for the requested picture
  my $this = shift;
  my $number = shift;

  my $picture = '';

  print STDERR "DC210::picture\n" if $debug;
  if (($number < 1) || ($number > $this->status->numberOfPictures))
  {
    carp "DC210::getPicture - invalid picture number";
  }
  else
  {
    # initialize local vars
    my $dcHandle = $this->dcHandle;
    my $pictureInfo = $this->pictureInfo($number);

    my $fileSize = $pictureInfo->fileSize;
    my $blockSize = 1024;
    my $read = 0;

    my $cancel = 0;
    my $lsb;
    my $msb;

    ($msb,$lsb) = unpack('C2',pack('n1',$number-1));

    $dcHandle->sendCommand(
		$DC210_PICTURE_DOWNLOAD,
		$msb,
		$lsb,
		0x00,
		0x00);

    while ($read < $fileSize)
    {
      my $block = $dcHandle->readPacket($blockSize);

      if ($read + $blockSize > $fileSize)
      {
        $blockSize = $fileSize - $read;
        $block = substr($block,0,$blockSize);
      }

      $picture .= $block;
      $read += $blockSize;

      if (defined($this->{'downloadNotify'}))
      {
	$cancel = &{$this->{'downloadNotify'}}($pictureInfo,$read);
      }
    }

    $this->dcHandle->commandComplete;
  }

  return $picture;
}

sub DESTROY
{
  my $this = shift;

  print STDERR "DC210::DESTROY\n" if $debug;

  # set the port speed back to 9600 since that's what the default
  # speed of the camera is.
  if (defined($this->dcHandle))
  {
    $this->dcHandle->setPortSpeed(9600);

    # close the dcHandle
    $this->dcHandle->close;

    # break reference to dcHandle for proper garbage collection
    $this->setDcHandle(undef);
  }
}


1;
