
/* --------------------------------------------------------------------
   Project: PAL: Palmtop Application Library
   Module:  palhelp.c
   Author:  Ron Crain
            Based on DFLAT help system by Al Stevens, published in
            Dr. Dobb's Journal, Nov, 1991 - Nov, 1992
   Started: 02/01/95 12:22 pm
   Subject: Routines for managing help file compression and
            decompression, searching and reading.

   -------------------------------------------------------------------- */
/* --------------------------------------------------------------------
                       standard includes
   -------------------------------------------------------------------- */
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* --------------------------------------------------------------------
                        type definitions
   -------------------------------------------------------------------- */

/* --------------------------------------------------------------------
                         local includes
   -------------------------------------------------------------------- */
#include "palhelp.h"

/* --------------------------------------------------------------------
                        global variables
   -------------------------------------------------------------------- */
/* Public */
struct htr *HelpTree;
struct helps *HelpTable;
int HelpCount;

/* Private */
static int in8;
static int ct8 = 8;
static FILE *fi;
static int root;

/* --------------------------------------------------------------------
                           functions
   -------------------------------------------------------------------- */
/* --------------------------------------------------------------------
   Build the help filename from the base help text filename passed in
   fname. Any file extension is removed, and the extension specified
   in ext is added.

   -------------------------------------------------------------------- */
void BuildFileName( char *fn, const char *fname, const char *ext )
{
   strcpy( fn, fname );
   fn [strchr( fn, '.' ) - fn] = '\0';
   strcat( fn, ext );
}

/* --------------------------------------------------------------------
   Open the help database file.
   The help file name is constructed from a base name and specified
   extension.
   -------------------------------------------------------------------- */
FILE *OpenHelpFile( const char *fn, const char *md )
{
   char *cp;
   int treect, i;
   char helpname[65];
   static BYTECOUNTER bytectr;

   BuildFileName( helpname, fn, HELP_EXT );                /* get the name of the help file */
   if( ( fi = fopen( helpname, md ) ) == NULL )
      return NULL;
   if( HelpTree == NULL )
   {
      fread( &bytectr, sizeof bytectr, 1, fi );            /* read the byte count */
      fread( &treect, sizeof treect, 1, fi );              /* read the frequency count */
      fread( &root, sizeof root, 1, fi );                  /* read the root offset */
      HelpTree = calloc( treect - 256, sizeof ( struct htr ) );
      if( HelpTree != NULL )
      {
         for( i = 0; i < treect - 256; i++ )               /* read in the tree */
         {
            fread( &HelpTree [i].left, sizeof ( int ), 1, fi );
            fread( &HelpTree [i].right, sizeof ( int ), 1, fi );
         }
      }
   }
   return fi;
}

/* --------------------------------------------------------------------
   Read the next line of text from the help database
   -------------------------------------------------------------------- */
void *GetHelpLine( char *line )
{
   int h;

   *line = '\0';
   while( TRUE )
   {
      h = root;                                            /* decompress a line from the file */
      while( h > 255 )                                     /* walk the Huffman tree */
      {                                                    /* h is a node pointer */
         if( ct8 == 8 )
         {
            if( ( in8 = fgetc( fi ) ) == EOF )             /* read 8 bits of compressed data */
            {
               *line = '\0';
               return NULL;
            }
            ct8 = 0;
         }
         if( in8 & 0x80 )                                  /* point to left or right node based on msb */
            h = HelpTree [h - 256].left;
         else
            h = HelpTree [h - 256].right;
         in8 <<= 1;                                        /* shift the next bit in */
         ct8++;
      }
      if( h == '\r' )                                      /* h < 255 = decompressed character */
         continue;                                         /* skip the '\r' character */
      *line++ = h;                                         /* put the character in the buffer */
      if( h == '\n' )                                      /* if '\n', end of line */
         break;
   }
   *line = '\0';                                           /* null-terminate the line */
   return line;
}

/* --------------------------------------------------------------------
   Compute the database file byte and bit position
   -------------------------------------------------------------------- */
void HelpFilePosition( long *offset, int *bit )
{
   *offset = ftell( fi );
   if( ct8 < 8 )
      --*offset;
   *bit = ct8;
}

/* --------------------------------------------------------------------
   Position the database to the specified byte and bit
   -------------------------------------------------------------------- */
void SeekHelpLine( long offset, int bit )
{
   int fs = fseek( fi, offset, 0 );

//   assert(fs == 0);
   ct8 = bit;
   if( ct8 < 8 )
   {
      in8 = fgetc( fi );
      in8 <<= bit;
   }
}

#ifdef WINSIZE
/* --------------------------------------------------------------------
   compute the displayed length of a help text line
   -------------------------------------------------------------------- */
int HelpLength( char *s )
{
   int len = strlen( s );
   char *cp = strchr( s, '[' );

   while( cp != NULL )
   {
      len -= 4;
      cp = strchr( cp + 1, '[' );
   }
   cp = strchr( s, '<' );
   while( cp != NULL )
   {
      char *cp1 = strchr( cp, '>' );
      if( cp1 != NULL )
         len -= (int) ( cp1 - cp ) + 1;
      cp = strchr( cp1, '<' );
   }
   return len;
}
#endif

/* --------------------------------------------------------------------
   compare help names with wild cards
   -------------------------------------------------------------------- */
static BOOL wildcmp( char *s1, char *s2 )
{
   while( *s1 || *s2 )
   {
      if( tolower( *s1 ) != tolower( *s2 ) )
         if( *s1 != '?' && *s2 != '?' )
            return TRUE;
      s1++, s2++;
   }
   return FALSE;
}

/* --------------------------------------------------------------------
   free the memory used by the help file table
   -------------------------------------------------------------------- */
void UnLoadIndexTable( void )
{
   int i;

   for( i = 0; i < HelpCount; i++ )
   {
      free( ( HelpTable + i )->comment );
      free( ( HelpTable + i )->hname );
//      free( ( HelpTable + i )->NextName );
//      free( ( HelpTable + i )->PrevName );
   }
   free( HelpTable );
   HelpTable = NULL;
   free( HelpTree );
   HelpTree = NULL;
}


/* --------------------------------------------------------------------
   load the the index table from the end of the compressed help file
   -------------------------------------------------------------------- */
void LoadIndexTable( char *fname )
{
   long where;
   int i;
   FILE *helpfp;

   UnLoadIndexTable();
   if( ( helpfp = OpenHelpFile( fname, "rb" ) ) == NULL )
      return;
//   strcpy( HelpFileName, fname );
   fseek( helpfp, -(long) sizeof ( long ), SEEK_END );
   fread( &where, sizeof ( long ), 1, helpfp );
   fseek( helpfp, where, SEEK_SET );
   fread( &HelpCount, sizeof ( int ), 1, helpfp );
   HelpTable = calloc( sizeof ( struct helps ) * HelpCount, 1 );    /* Reserve memory for the help structure */
   for( i = 0; i < HelpCount; i++ )                        /* Read the index table from the help file */
   {
      int len;

      fread( &len, sizeof ( int ), 1, helpfp );
      if( len )
      {
         ( HelpTable + i )->hname = calloc( len + 1, 1 );  /* memory for variable length name */
         fread( ( HelpTable + i )->hname, len + 1, 1, helpfp );
      }
      fread( &len, sizeof ( int ), 1, helpfp );
      if( len )
      {
         ( HelpTable + i )->comment = calloc( len + 1, 1 );    /* memory for variable length comment */
         fread( ( HelpTable + i )->comment, len + 1, 1, helpfp );
      }
#ifdef WINSIZE
      fread( &( HelpTable + i )->hptr, sizeof ( long ) + sizeof ( int ) * 5, 1, helpfp );
#else
      fread( &( HelpTable + i )->hptr, sizeof ( long ) + sizeof ( int ) * 3, 1, helpfp );
#endif

      fread( &len, sizeof( int ), 1, helpfp );
      if( len )
      {
         ( HelpTable + i )->PrevName = calloc( len + 1, 1 );
         fread( ( HelpTable + i )->PrevName, len + 1, 1, helpfp );
      }
      fread( &len, sizeof( int ), 1, helpfp );
      if( len )
      {
         ( HelpTable + i )->NextName = calloc( len + 1, 1 );
         fread( ( HelpTable + i )->NextName, len + 1, 1, helpfp );
      }

   }
   fclose( helpfp );
   helpfp = NULL;
}

/* --------------------------------------------------------------------
   Given a pointer to a Help ID string, this function searches the help
   index and returns a pointer to help table record containing the string.
   -------------------------------------------------------------------- */
struct helps *FindHelp( char *HelpID )
{
   int hlpnum = 0;
//static struct helps *ThisHelp;
   struct helps *thishelp = NULL;

   for( hlpnum = 0; hlpnum < HelpCount; hlpnum++ )
   {
//      if(  strcmp( HelpID,               thishelp->hname ) == FALSE )
      if( wildcmp( HelpID, ( HelpTable + hlpnum )->hname ) == FALSE )
      {
         thishelp = HelpTable + hlpnum;
         break;
      }
   }
   return thishelp;
}
