
/* --------------------------------------------------------------------
   Project: PALHELP
   Module:  makehelp.c Help File compiler
   Author:  Ron Crain
            Based on DFLAT help system by Al Stevens, published in
            Dr. Dobb's Journal, Nov, 1991 - Nov, 1992
   Started: 02/01/95 10:23 am
   Subject: Compiles a help text file using huffmann tree compression.
            Invoke with
                        makehelp HELP.TXT

            and makehelp will produce a helpfile, HELP.PH

            To build makehelp.exe, use:
               cl /c makehelp.c palhelp.c htree.c indxhelp.c
               link makehelp+palhelp+htree+indxhelp;
            in MSC, or equivalent commands for your compiler.

            If window size information is required, define the
               WINSIZE variable (/DWINSIZE) in the cl command
               line.

   -------------------------------------------------------------------- */
/* --------------------------------------------------------------------
                       standard includes
   -------------------------------------------------------------------- */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* --------------------------------------------------------------------
                         local includes
   -------------------------------------------------------------------- */
#include "palhelp.h"

/* --------------------------------------------------------------------
                      constant definitions
   -------------------------------------------------------------------- */

/* --------------------------------------------------------------------
                        global variables
   -------------------------------------------------------------------- */
extern struct htree *ht;
extern int root;
extern int treect;

/* --------------------------------------------------------------------
                           prototypes
   -------------------------------------------------------------------- */
static void compress( FILE *, int, int );
static void outbit( FILE *fo, int bit );
static int lastchar = '\n';

/* --------------------------------------------------------------------
                           functions
   -------------------------------------------------------------------- */
/* --------------------------------------------------------------------
   Get a character from the input file, bypassing all lines that begin
   with ";"
   -- Modified to fix bug of inserting a new line for every comment line.
   -------------------------------------------------------------------- */
static int fgetcx( FILE *fi )
{
   int c;

   if( ( c = fgetc( fi ) ) == ';' && lastchar == '\n' )
      do
      {
         while( c != '\n' && c != EOF )
            c = fgetc( fi );                               /* c contains the newline ending a comment line */
         c = fgetc( fi );                                  /* get the first character of the next line */
      }
      while( c == ';' );
   lastchar = c;
   return c;
}

/* --------------------------------------------------------------------
   compress a character value into a bit stream
   -------------------------------------------------------------------- */
static void compress( FILE *fo, int h, int child )
{
   if( ht [h].parent != -1 )
      compress( fo, ht [h].parent, h );

   if( child )
   {
      if( child == ht [h].right )
         outbit( fo, 0 );
      else if( child == ht [h].left )
         outbit( fo, 1 );
   }
}


/* --------------------------------------------------------------------
   collect and write bits to the compressed output file
   -------------------------------------------------------------------- */
static char out8;
static int ct8;

static void outbit( FILE *fo, int bit )
{
   if( ct8 == 8 || bit == -1 )
   {
      while( ct8 < 8 )
      {
         out8 <<= 1;
         ct8++;
      }
      fputc( out8, fo );
      ct8 = 0;
   }
   out8 = ( out8 << 1 ) | bit;
   ct8++;
}

/* --------------------------------------------------------------------
   -- Re-written to serve our needs.
   -- Added HelpIndex
   -------------------------------------------------------------------- */
void main( int argc, char *argv[] )
{
   FILE *fi, *fo;
   int c;
   BYTECOUNTER bytectr = 0;
   char *helpfilename;

   if( argc < 2 )
   {
      printf( "\nusage: palhelp infile" );
      exit( 1 );
   }
   if( ( fi = fopen( argv [1], "rb" ) ) == NULL )
   {
      printf( "\nCannot open %s", argv [1] );
      exit( 1 );
   }
   BuildFileName( helpfilename, argv [1], HELP_EXT );
   if( ( fo = fopen( helpfilename, "wb" ) ) == NULL )
   {
      printf( "\nCannot open %s", helpfilename );
      fclose( fi );
      exit( 1 );
   }
   ht = calloc( 256, sizeof ( struct htree ) );
   puts( "Analyzing Help text" );
   while( ( c = fgetcx( fi ) ) != EOF )                    /* read the input file and count character frequency */
   {
      c &= 255;
      ht [c].cnt++;
      bytectr++;
   }
   buildtree();                                            /* build the huffman tree */
   fwrite( &bytectr, sizeof bytectr, 1, fo );              /* write the byte count to the output file */
   fwrite( &treect, sizeof treect, 1, fo );                /* write the tree count to the output file */
   fwrite( &root, sizeof root, 1, fo );                    /* write the root offset to the output file */
   for( c = 256; c < treect; c++ )                         /* write the tree to the output file */
   {
      int lf = ht [c].left;
      int rt = ht [c].right;

      fwrite( &lf, sizeof lf, 1, fo );
      fwrite( &rt, sizeof rt, 1, fo );
   }
   puts( "Compressing help database" );
   fseek( fi, 0L, 0 );                                     /* compress the file */
   while( ( c = fgetcx( fi ) ) != EOF )
      compress( fo, ( c & 255 ), 0 );
   outbit( fo, -1 );
   fclose( fi );
   fclose( fo );
   free( ht );
   puts( "Building Index Table" );
   if( HelpIndex( argv [1] ) )
      puts( "Bad help file" );
   exit( 0 );
}
