#i linuxdoc.tm

#d slang \bf{S-lang}
#d kw#1 \tt{$1}
#d exmp#1 \tt{$1}
#d var#1 \tt{$1}
#d ldots ...
#d chapter#1 <chapt>$1<p>
#d preface <preface>
#d tag#1 <tag>$1</tag>

#d function#1 \section{<bf>$1</bf>\label{$1}}<descrip>
#d variable#1 \section{<bf>$1</bf>\label{$1}}<descrip>
#cd function#1 <p><bf>$1</bf>\label{$1}<p><descrip>
#d synopsis#1 <tag> Synopsis </tag> $1
#d keywords#1 <tag> Keywords </tag> $1
#d usage#1 <tag> Usage </tag> <tt>$1</tt>
#d description <tag> Description </tag>
#d example <tag> Example </tag>
#d notes <tag> Notes </tag>
#d seealso#1 <tag> See Also </tag> <tt>$1</tt>
#d documentstyle article
#d r#1 \ref{$1}{$1}
#d done </descrip><p>

#d user-manual \bf{A Guide to the S-Lang Language}

\linuxdoc
\begin{\documentstyle}

\title S-Lang Run-Time Library Reference
\author John E. Davis, \tt{davis@space.mit.edu}
\date \__today__

\toc

\variable{_traceback}
\synopsis{Generate a traceback upon error}
\usage{Integer_Type _traceback}
\description
  \var{_traceback} is an intrinsic integer variable whose value
  controls whether or not a traceback of the call stack is to be
  generated upon error.  If \var{_traceback} is greater than zero, a
  full traceback will be generated, which includes the values of local
  variables.  If the value is less than zero, a traceback will be
  generated without local variable information, and if
  \var{_traceback} is zero the traceback will not be generated.

  Local variables are represented in the form \var{$n} where \var{n} is an
  integer numbered from zero.  More explicitly, \var{$0} represents the
  first local variable, \var{$1} represents the second, and so on.
  Please note that function parameters are local variables and that the
  first parameter corresponds to \var{$0}.
\seealso{_slangtrace, error}
\done

\variable{_slangtrace}
\synopsis{Turn function tracing on or off.}
\usage{Integer_Type _slangtrace}
\description
  The \var{_slangtrace} variable is a debugging aid that when set to a
  non-zero value enables tracing when function declared by
  \var{_trace_function} is entered.  If the value is greater than
  zero, both intrinsic and user defined functions will get traced.
  However, if set to a value less than zero, intrinsic functions will
  not get traced.
\seealso{_trace_function, _traceback, _print_stack}
\done

\variable{errno}
\synopsis{Error code set by system functions.}
\usage{Integer_Type errno}
\description
  A system function can fail for a variety of reasons.  For example, a
  file operation may fail because lack of disk space, or the process
  does not have permission to perform the operation.  Such functions
  will return \var{-1} and set the variable \var{errno} to an error
  code describing the reason for failure.

  Particular values of \var{errno} may be specified by the following
  symbolic constants (read-only variables) and the corresponding
  \var{errno_string} value:
#v+
     EPERM            "Not owner"
     ENOENT           "No such file or directory"
     ESRCH            "No such process"
     ENXIO            "No such device or address"
     ENOEXEC          "Exec format error"
     EBADF            "Bad file number"
     ECHILD           "No children"
     ENOMEM           "Not enough core"
     EACCES           "Permission denied"
     EFAULT           "Bad address"
     ENOTBLK          "Block device required"
     EBUSY            "Mount device busy"
     EEXIST           "File exists"
     EXDEV            "Cross-device link"
     ENODEV           "No such device"
     ENOTDIR          "Not a directory"
     EISDIR           "Is a directory"
     EINVAL           "Invalid argument"
     ENFILE           "File table overflow"
     EMFILE           "Too many open files"
     ENOTTY           "Not a typewriter"
     ETXTBSY          "Text file busy"
     EFBIG            "File too large"
     ENOSPC           "No space left on device"
     ESPIPE           "Illegal seek"
     EROFS            "Read-only file system"
     EMLINK           "Too many links"
     EPIPE            "Broken pipe"
     ELOOP            "Too many levels of symbolic links"
     ENAMETOOLONG     "File name too long"
#v-
\example
  The \var{mkdir} function will attempt to create a directory.  If
  that directory already exists, the function will fail and set
  \var{errno} to \var{EEXIST}.
#v+
    define create_dir (dir)
    {
       if (0 == mkdir (dir)) return;
       if (errno != EEXIST)
         error ("mkdir %s failied: %s", dir, errno_string);
    }
#v-
\seealso{errno_string, error, mkdir}
\done

\function{where}
\synopsis{Get indices where an integer array is non-zero}
\usage{Array_Type where (Array_Type a)}
\description
  The \var{where} function examines an integer array \var{a} and
  returns a 2-d integer array whose rows are the indices of \var{a}
  where the corresponding element of \var{a} is non-zero.
\example
  Consider the following:
#v+
    variable X = [0.0:10.0:0.01];
    variable A = sin (X);
    variable I = where (A < 0.0);
    A[I] = cos (X) [I];
#v-
  Here the variable \var{X} has been assigned an array of doubles
  whose elements range from \exmp{0.0} through \exmp{10.0} in
  increments of \var{0.01}.  The second statement assigns \var{A} to
  an array whose elements are the \var{sin} of the elements of \var{X}.
  The third statement uses the where function to get the indices of
  the elements of \var{A} that are less than \var{0.0}.  Finally, the
  last statement substitutes into \var{A} the \var{cos} of the
  elements of \var{X} at the positions of \var{A} where the
  corresponding \var{sin} is less than \var{0}.  The end result is
  that the elements of \var{A} are a mixture of sines and cosines.
\seealso{array_info, sin, cos}
\done

\function{vmessage}
\synopsis{Print a formatted string onto the message device}
\usage{vmessage (String_Type fmt, ...)}
\description
  The \var{vmessage} function formats a sprintf style argument list
  and displays the resulting string onto the message device.
\notes
  In the current implementation, strictly speaking, the \var{vmessage}
  function is not an intrinsic function.  Rather it is a predefined
  \slang function using a combination of \var{Sprintf} and
  \var{message}.
\seealso{message, Sprintf, verror}
\done

\function{_apropos}
\synopsis{Generate a list of functions and variable}
\usage{Integer_Type _apropos (String_Type s, Integer_Type flags)}
\description
  The \var{_apropos} function may be used to get a list of all defined
  objects whose name consists of the substring \var{s} and whose type
  matches those specified by \var{flags}.  It returns the number of
  matches.  If the number returned is non-zero, that number of strings
  which represent the names of the matched objects will also be
  present on the stack.
  
  The second parameter \var{flags} is a bit mapped value whose bits
  are defined according to the following table
#v+
     1          Intrinsic Function
     2          User-defined Function
     4          Intrinsic Variable
     8          User-defined Variable
#v-
\example
#v+
    define apropos (s)
    {
      variable n, name;
      n = _apropos (s, 0xF);
      if (n) vmessage ("Found %d matches:", n);
      else message ("No matches.");
      loop (n)
        {
           name = ();
           message (name);
        }
    }
#v-
  prints a list of all matches.
\notes
  Since the function returns the matches to the stack, it is possible
  that a stack overfow error could result if there are two many
  matches.  If this happens, the interpreter should be recompiled to
  use a larger stack size.
\seealso{is_defined, sprintf}
\done

\function{_clear_error}
\synopsis{Clear an error condition}
\usage{_clear_error ()}
\description
  This function may be used in error-blocks to clear the error that
  triggered execution of the error block.  Execution resumes following
  the statement, in the scope of the error-block, that triggered the
  error.
\example
  Consider the following wrapper around the \var{putenv} function:
#v+
    define try_putenv (name, value)
    {
       variable status;
       ERROR_BLOCK
        {
          _clear_error ();
          status = -1;
        }
       status = 0;
       putenv (sprintf ("%s=%s", name, value);
       return status;
    }
#v-
  If \var{putenv} fails, it generates an error condition, which the
  \var{try_putenv} function catches and clears.  Thus \var{try_putenv}
  is a function that returns \exmp{-1} upon failure and \var{0} upon
  success.
\seealso{_trace_function, _slangtrace, _traceback}
\done

\function{_function_name}
\synopsis{Returns the name of the currently executing function}
\usage{String _function_name ();}
\description
  This function returns the name of the currently executing function.
  If called from top-level, it returns the empty string.
\seealso{_trace_function, is_defined}
\done

\function{_pop_n}
\synopsis{Remove objects from the stack}
\usage{_pop_n (Integer_Type n);}
\description
  The \var{_pop_n} function pops \var{n} objects from the top of the
  stack.
\example
#v+
    define add3 ()
    {
       variable x, y, z;
       if (_NARGS != 3)
         {
            _pop_n (_NARGS);
            error ("add3: Expecting 3 arguments");
         }
       (x, y, z) = ();
       return x + y + z;
    }
#v-
\seealso{_stkdepth, pop}
\done

\function{_print_stack}
\synopsis{print the values on the stack.}
\usage{_print_stack ()}
\description
  This function dumps out what is currently on the \slang.  It does not
  alter the stack and it is usually used for debugging purposes.
\seealso{_stkdepth, string}
\done

\function{_slang_guess_type}
\synopsis{Guess the data type that a string represents.}
\usage{DataType_Type _slang_guess_type (String_Type s)}
\description
  This function tries to determine whether its argument \var{s} represents
  an integer or a floating point number.  If it appears to be neither,
  then a string is assumed.  It returns one of three values depending on
  the format of the string \var{s}:
#v+
    Integer_Type     :   If it appears to be an integer
    Double_Type      :   If it appears to be a double
    String_Type      :   Anything else.
#v-
  For example, \exmp{_slang_guess_type("1e2")} returns
  \var{Double_Type} but \exmp{_slang_guess_type("e12")} returns
  \var{String_Type}.
\seealso{integer, string, double}
\done

\function{_stk_reverse}
\synopsis{Reverse the order of the objects on the stack.}
\usage{_stk_reverse (Integer_Type n)}
\description
   The \var{_stk_reverse} function reverses the order of the top
   \var{n} items on the stack.
\seealso{_stkdepth, _stk_roll}
\done

\function{_stk_roll}
\synopsis{Roll items on the stack}
\usage{_stk_roll (Integer_Type n);}
\description
  This function may be used to alter the arrangement of objects on the
  stack.  Specifically, if the integer \var{n} is positive, the top
  \var{n} items on the stack are rotated up.  If
  \var{n} is negative, the top \var{abs(n)} items on the stack are
  rotated down.
\example
  If the stack looks like:
#v+
    item-0
    item-1
    item-2
    item-3
#v-
  where \exmp{item-0} is at the top of the stack, then
  \exmp{_stk_roll(-3)} will change the stack to:
#v+
    item-2
    item-0
    item-1
    item-3
#v-
\notes
  This function only has an effect for \exmp{abs(n) > 1}.
\seealso{_stkdepth, _stk_reverse, _pop_n, _print_stack}
\done

\function{_stkdepth}
\usage{Get the number of objects currently on the stack.}
\synopsis{Integer_Type _stkdepth ()}
\description
  The \var{_stkdepth} function returns number of items on stack prior
  to the call of \var{_stkdepth}.
\seealso{_print_stack, _stk_reverse, _stk_roll}
\done

\function{_trace_function}
\synopsis{Set the function to trace}
\usage{_trace_function (String_Type f)}
\description
  \var{_trace_function} declares that the \slang function with name
  \var{f} is to be traced when it is called.  Calling
  \var{_trace_function} does not in itself turn tracing on.  Tracing
  is turned on only when the variable \var{_slangtrace} is non-zero.
\seealso{_slangtrace, _traceback}
\done

\function{array_info}
\synopsis{Returns information about an array}
\usage{(Array_Type, Integer_Type, DataType_Type) array_info (Array_Type a)}
\description
  The \var{array_info} function returns information about the array \var{a}.
  It returns three values: an 1-d integer array array specifying the
  size of each dimension of \var{a}, the number of dimensions of
  \var{a}, and the data type of \var{a}.
\example
  The \var{array_info} function may be used to find the number of rows
  of an array:
#v+
    define num_rows (a)
    {
       variable dims, num_dims, data_type;

       (dims, num_dims, data_type) = array_info (a);
       return dims [0];
    }
#v-
\seealso{typeof, reshape}
\done

\function{array_sort}
\synopsis{Sort an array}
\usage{Array_Type array_sort (Array_Type a, String_Type f)}
\description
  \var{array_sort} sorts the array \var{a} into ascending order
  according to the function specified by the name \var{f} and returns
  an integer array that represents the result of the sort.

  The sort function represented by \var{f} must be a \slang
  user-defined function that takes two arguments.  The function must
  return an integer that is less than zero if the first parameter is
  considered to be less than the second, zero if they are equal, and a
  value greater than zero if the first is greater than the second.

  The integer array returned by this function is simply an index that
  indicates the order of the sorted array.  The input array \var{a} is
  not changed.
\example
  An array of strings may be sorted using the \var{strcmp} function
  since it fits the specification for the sorting function described
  above:
#v+
     variable A = String_Type [3];
     A[0] = "gamma"; A[1] = "alpha"; A[2] = "beta";

     variable I = array_sort (A, "strcmp");
#v-
  After the \var{array_sort} has executed, the variable \var{I} will
  have the values \exmp{[2, 0, 1]}.  This array can be used to
  re-shuffle the elements of \var{A} into the sorted order via the
  array index expression \exmp{A = A[I]}.
\notes
  The current sorting algorithm is a heap-sort.
\seealso{strcmp}
\done

\function{atof}
\synopsis{Convert a string to a double precision number}
\usage{Double_Type atof (String_Type s)}
\description
  This function converts a string \var{s} to a double precision value
  and returns the result.  It performs no error checking on the format
  of the string.  The function \var{_slang_guess_type} may be used to
  check the syntax of the string.
\example
#v+
     define error_checked_atof (s)
     {
        switch (_slang_guess_type (s))
        {
           case Double_Type:
             return atof (s);
        }
        {
           case Integer_Type:
             return double (integer (s));
        }

        verror ("%s is is not a double", s);
    }
#v-
\seealso{typecast, double, _slang_guess_type}
\done

\function{autoload}
\synopsis{Load a function from a file}
\usage{autoload (String_Type funct, String_Type file)}
\description
    The \var{autoload} function is used to declare \var{funct} to the
    interpreter and indicate that it should be loaded from \var{file} when
    it is actually used.
\example
    Suppose \var{bessel_j0} is a function defined in the file
    \var{bessel.sl}.  Then the statement
#v+
      autoload ("bessel_j0", "bessel.sl");
#v-
    will cause \var{bessel.sl} to be loaded prior to the execution of
    \var{bessel_j0}
\seealso{evalfile}
\keywords{file function}
\done

\function{byte_compile_file}
\synopsis{Compile a file to byte-code for faster loading.}
\usage{byte_compile_file (String_Type file, Integer_Type method)}
\description
  The \var{byte_compile_file} function byte-compiles \var{file}
  producing a new file with the same name except a \var{'c'} is added
  to the output file name.  For example, \var{file} is
  \exmp{"site.sl"}, then the function produces a new file named
  \exmp{site.slc}.
\notes
  The \var{method} parameter is not used in the current
  implementation.  Its use is reserved for the future.  For now, set
  it to \exmp{0}.
\seealso{evalfile}
\done

\function{char}
\synopsis{Convert an ascii value into a string}
\usage{String_Type char (Integer_Type c)}
\description
  The \var{char} function converts an integer ascii value \var{c} to a string
  of unit length such that the first character of the string is \var{c}.
  For example, \exmp{char('a')} returns the string \exmp{"a"}.
\seealso{integer, string, typedef}
\done

\function{chdir}
\synopsis{Change the current working directory.}
\usage{Integer_Type chdir (String_Type dir)}
\description
  The \var{chdir} function may be used to changed the current working
  directory to the directory specified by \var{dir}.  Upon sucess it
  returns zero; however, upon failure it returns \exmp{-1} and sets
  \var{errno} accordingly.
\seealso{mkdir, stat_file}
\done

\function{chmod}
\synopsis{Change the mode of a file}
\usage{Integer_Type chmod (String_Type file, Integer_Type mode)}
\description
  The \var{chmod} function changes the permissions of \var{file} to those
  specified by \var{mode}.  It returns \exmp{0} upon success, or
  \exmp{-1} upon failure setting \var{errno} accordingly.

  See the system specific documentation for the C library
  function \var{chmod} for a discussion of the \var{mode} parameter.
\seealso{chown, stat_file}
\done

\function{chown}
\synopsis{Change the owner of a file}
\usage{Integer_Type chown (String_Type file, Integer_Type uid, Integer_Type gid)}
\description
  The \var{chown} function is used to change the user-id and group-id of
  \var{file} to \var{uid} and \var{gid}, respectively.  It returns
  \var{zero} upon success and \exmp{-1} upon failure, with \var{errno}
  set accordingly.
\notes
  On most systems, only the super user can change the ownership of a
  file.

  Some systems do not support this function.
\seealso{chmod, stat_file}
\done

\function{create_delimited_string}
\synopsis{Concatenate strings using a delimiter}
\usage{String_Type create_delimited_string (delim, s_1, s_2, ..., s_n, n)}
#v+
    String_Type delim, s_1, ..., s_n
    Integer_Type n
#v-
\description
  \var{create_delimited_string} performs a concatenation operation on
  the \var{n} strings \var{s_1}, ...,\var{s_n}, using the string
  \var{delim} as a delimiter.  The resulting string is equivalent to
  one obtained via
#v+
      s_1 + delim + s_2 + delim + ... + s_n
#v-
\example
  One use for this function is to construct path names, e.g.,
#v+
    create_delimited_string ("/", "user", "local", "bin", 3);
#v-
  will produce \exmp{"usr/local/bin"}.
\notes
  The expression \exmp{strcat(a,b)} is equivalent to
  \exmp{create_delimited_string("", a, b, 2)}.
\seealso{is_list_element, extract_element, strchop, strcat}
\done
  

\function{define_case}
\synopsis{Define upper-lower case conversion.}
\usage{define_case (Integer_Type ch_up, Integer_Type ch_low);}
\description
  This function defines an upper and lowercase relationship between two
  characters specified by the arguments.  This relationship is used by
  routines which perform uppercase and lowercase conversions.
  The first integer \var{ch_up} is the ascii value of the uppercase character
  and the second parameter \var{ch_low} is the ascii value of its
  lowercase counterpart.
\seealso{strlow, strup}
\done

\function{double}
\synopsis{Convert an object to double precision}
\usage{result = double (x)}
\description
  The \var{double} function typecasts an object \var{x} to double
  precision.  For example, if \var{x} is an array of integers, an
  array of double types will be returned.  If an object cannot be
  converted to \var{Double_Type}, a type-mismatch error will result.
\notes
  The \var{double} function is equivalent to the typecast operation
#v+
     typecast (x, Double_Type)
#v-
  To convert a string to a double precision number, use \var{atoi}
  function.
\seealso{typecast, atoi, int}
\done

\function{dup}
\synopsis{Duplicate the value at the top of the stack}
\usage{dup ()}
\description
  This function returns an exact duplicate of the object on top of the
  stack.  For some objects such as arrays or structures, it creates a
  new reference to the array.  However, for simple scalar S-Lang types such
  as strings, integers, and doubles, it creates a new copy of the
  object.
\seealso{pop, typeof}
\done

\function{errno_string}
\synopsis{Return a string describing an errno.}
\usage{String_Type errno_string (Integer_Type err)}
\description
  The \var{errno_string} function returns a string describing the
  integer error code \var{err}.  The variable \var{err} usually
  corresponds to the \var{errno} intrinsic function.  See the
  description for \var{errno} for more information.
\example
  The \var{errno_string} function may be used as follows:
#v+
    define sizeof_file (file)
    {
       variable st = stat (file);
       if (st == NULL)
         verror ("%s: %s", file, errno_string (errno);
       return st.st_size;
    }
#v-
\seealso{errno, stat, verror}
\done

\function{error}
\synopsis{Generate an error condition}
\usage{error (String_Type msg}
\description
  The \var{error} function generates a \slang error condition causing
  the interpreter to start unwinding to top-level.  It takes a single
  string parameter which is displayed on the stderr output device.
  The error condition may be cleared via an \var{ERROR_BLOCK} with the
  \var{_clear_error} function.  Consult \user-manual for more
  information.
\example
#v+
    define add_txt_extension (file)
    {
       if (typeof (file) != String_Type)
         error ("add_extension: parameter must be a string");
       file += ".txt";
       return file;
    }
#v-
\seealso{verror, _clear_error, message}
\done

\function{evalfile}
\synopsis{Interpret a file containing \slang code.}
\usage{Integer_Type evalfile (String_Type file)}
\description
  The \var{evalfile} function loads \var{file} into the interpreter.
  If no errors were encountered, \exmp{1} will be returned; otherwise,
  a \slang error will be generated and the function will return zero.
\example
#v+
    define load_file (file)
    {
       ERROR_BLOCK { _clear_error (); }
       () = evalfile (file);
    }
#v-
\seealso{eval, autoload}
\done

\function{eval}
\synopsis{Interpret a string as \slang code}
\usage{eval (String_Type expression)}
\description
  The \var{eval} function parses a string as S-Lang code and executes the
  result.  This is a useful function in many contexts such as dynamically
  generating function definitions where there is no way to generate
  them otherwise.
\example
#v+
    if (0 == is_defined ("my_function"))
      eval ("define my_function () { message (\"my_function\"); }");
#v-
\seealso{is_defined, autoload, evalfile}
\done

\function{extract_element}
\synopsis{Extract the nth element of a string with delimiters}
\usage{String_Type extract_element (String_Type list, Integer_Type nth, Integer_Type delim);}
\description
  The \var{extract_element} function may be used to extract the
  \var{nth} element of the \var{delim} delimited list of strings
  \var{list}.  The function will return the \var{nth} element of the
  list, unless \var{nth} specifies more elements than the list
  contains, in which case \var{NULL} will be returned.
  Elements in the list are numbered from \var{0}.
\example
  The expression
#v+
     extract_element ("element 0, element 1, element 2", 1, ',')
#v-
  returns the string \exmp{" element 1"}, whereas
#v+
     extract_element ("element 0, element 1, element 2", 1, ' ')
#v-
  returns \exmp{"0,"}.

  The following function may be used to compute the number of elements
  in the list:
#v+
     define num_elements (list, delim)
     {
        variable nth = 0;
        while (NULL != extract_element (list, nth, delim))
          nth++;
        return nth;
     }
#v-
\seealso{is_list_element, is_substr, strchop, create_delimited_string}
\done

\function{fclose}
\synopsis{Close a file}
\usage{Integer_Type fclose (File_Type fp)}
\description
  The \var{fclose} function may be used to close an open file pointer
  \var{fp}.  Upon success it returns zero, and upon failure it sets
  \var{errno} and returns \exmp{-1}.  Failure usually indicates a that
  the file system is full or that \var{fp} does not refer to an open file.
\notes
  Many C programmers call \var{fclose} without checking the return
  value.  The \slang language requires the programmer to explicitly
  handle any value returned by a \slang function.  The simplest way to
  handle the return value from \var{fclose} is to use it as:
#v+
     () = fclose (fp);
#v-
\seealso{fopen, fgets, fflush, errno}
\done

\function{fflush}
\synopsis{Flush an output stream}
\usage{Integer_Type fflush (File_Type fp)}
\description
  The \var{fflush} function may be used to update the \em{output}
  stream specified by \var{fp}.  It returns \var{0} upon success, or
  \var{-1} upon failure and sets \var{errno} accordingly.  In
  particular, this function will fail if \var{fp} does not represent
  an output stream, or if \var{fp} is associated with a disk file and
  there is insufficient disk space.
\example
  This example illustrates how to use the \var{fflush} function
  without regard to the return value:
#v+
    () = fputs ("Enter value> ", stdout);
    () = fflush (stdout);
#v-
\notes
  Many C programmers disregard the return value from the \var{fflush}
  function.  The above example illustrates how to properly do this in
  the \slang langauge.
\seealso{fopen, fclose}
\done

\function{fgets}
\synopsis{Read a line from a file.}
\usage{Integer_Type fgets (SLang_Ref_Type ref, File_Type fp)}
\description
  \var{fgets} reads a line from the open file specified by \var{fp}
  and places the characters in the variable whose reference is
  specified by \var{ref}.
  It returns \exmp{-1} if \var{fp} is not associated with an open file
  or an attempt was made to read at the end the file; otherwise, it
  returns the number of characters read.
\example
  The following example returns the lines of a file via a linked list:
#v+
    define read_file (file)
    {
       variable buf, fp, root, tail;
       variable list_type = struct { text, next };

       root = NULL;

       fp = fopen(file, "r");
       if (fp == NULL)
         error("fopen %s failed." file);
       while (-1 != fgets (&buf, fp))
         {
            if (root == NULL)
              {
                 root = @list_type;
                 tail = root;
              }
            else
              {
                 tail.next = @list_type;
                 tail = tail.next;
              }
            tail.text = buf;
            tail.next = NULL;
         }
       () = fclose (fp);
       return root;
    }
#v-
\seealso{fopen, fclose, fputs, error}
\done

\function{fopen}
\synopsis{Open a file}
\usage{File_Type fopen (String_Type f, String_Type m)}
\description
  The \var{fopen} function opens a file \var{f} according to the mode
  string \var{m}.  Allowed values for \var{m} are:
#v+
     "r"    Read only
     "w"    Write only
     "a"    Append
     "r+"   Reading and writing at the beginning of the file.
     "w+"   Reading and writing.  The file is created if it does not
              exist; otherwise, it is truncated.
     "a+"   Reading and writing at the end of the file.  The file is created
              if it does not already exist.
#v+
  In addition, the mode string can also include the letter \var{'b'}
  as the last character to indicate that the file is to be opened in
  binary mode.

  Upon success, \var{fopen} a \var{File_Type} object which is meant to
  be used in other operations that require an open file.  Upon
  failure, the function returns \var{NULL}.
\example
  The following function opens a file in append mode and writes a
  string to it:
#v+
    define append_string_to_file (file, str)
    {
       variable fp = fopen (file, "a");
       if (fp == NULL) verror ("%s could not be opened", file);
       () = fputs (string, fp);
       () = fclose (fp);
    }
#v-
  Note that the return values from \var{fputs} and \var{fclose} are
  ignored.
\notes
  There is no need to explicitly close a file opened with \var{fopen}.
  If the returned \var{File_Type} object goes out of scope, \slang
  will automatically close the file.  However, explicitly closing a
  file after use is recommended.
\seealso{fclose, fgets, fputs}
\done

\function{fputs}
\synopsis{Write a string to an open stream}
\usage{Integer_Type fputs (String_Type s, File_Type fp);}
\description
  The \var{fputs} function writes the string \var{s} to the open file
  pointer \var{fp}. It returns -1 upon failure and sets \var{errno},
  otherwise it returns the length of the string.
\example
  The following function opens a file in append mode and uses the
  \var{fputs} function to write to it.
#v+
    define append_string_to_file (str, file)
    {
       variable fp;
       fp = fopen (file, "a");
       if (fp == NULL) verror ("Unable to open %s", file);
       if ((-1 == fputs (s, fp))
           or (-1 == fclose (fp)))
         verror ("Error writing to %s", file);
    }
#v-
\notes
  One must not disregard the return value from the \var{fputs}
  function, as many C programmers do.  Doing so may lead to a stack
  overflow error.
\seealso{fclose, fopen, fgets}
\done

\function{getcwd}
\synopsis{Get the current working directory}
\usage{String_Type getcwd ()}
\description
  The \var{getcwd} function returns the absolute pathname of the
  current working directory.  If an error occurs or it cannot
  determine the working directory, it returns \var{NULL} and sets
  \var{errno} accordingly.
\notes
  Under Unix, OS/2, and MSDOS, the pathname returned by this function
  includes the trailing slash character.  Some versions also include
  the drive specifier.
\seealso{mkdir, chdir, errno}
\done

\function{get_doc_string_from_file}
\synopsis{Read documentation from a file}
\usage{String_Type get_doc_string_from_file (String_Type f, String_Type t)}
\description
  \var{get_doc_string_from_file} opens the documentation file \var{f}
  and searches it for topic \var{t}.  It returns the documentation for
  \var{t} upon success, otherwise it returns \var{NULL} upon error.
  It will fail if \var{f} could not be opened or does not contain
  documentation for the topic.
\seealso{stat_file}
\done

\function{getenv}
\synopsis{Get the value of an environment variable}
\usage{String_Type getenv(String_Type var)}
\description
   The \var{getenv} function returns a string that represents the
   value of an environment variable \var{var}.  It will return
   \var{NULL} if there is no environment variable whose name is given
   by \var{var}.
\example
#v+
    if (NULL != getenv ("USE_COLOR"))
      {
        set_color ("normal", "white", "blue");
        set_color ("status", "black", "gray");
        USE_ANSI_COLORS = 1;
      }
#v-
\seealso{putenv, strlen, is_defined}
\done

\function{init_char_array}
\synopsis{Initialize an array of characters}
\usage{init_char_array (Array_Type a, String_Type s)}
\description
  The \var{init_char_array} function may be used to initialize a
  character array \var{a} by settting the elements of the array
  \var{a} to the corresponding characters of the string \var{s}.
\example
  The statements
#v+
     variable a = Char_Type [10];
     init_char_array (a, "HelloWorld");
#v-
   creates an character array and initializes its elements to the
   characters in the string \exmp{"HelloWorld"}.
\notes
   The character array must be large enough to hold all the characters
   of the initialization string.
\seealso{strlen, strcat}
\done

\function{integer}
\synopsis{Convert a string to an integer}
\usage{Integer_Type integer (String_Type s)}
\description
  The \var{integer} function converts a string representation of an
  integer back to an integer.  If the string does not form a valid
  integer, a type-mismatch error will be generated.
\example
  \exmp{integer ("1234")} returns the integer value \exmp{1234}.
\notes
  This function operates only on strings and is not the same as the
  more general \var{typecast} operator.
\seealso{typecast, _slang_guess_type, string, sprintf, char}
\done

\function{int}
\synopsis{Typecast an object to an integer}
\usage{int (s)}
\description
  This function performs a typecast of \var{s} from its data type to
  an object of \var{Integer_Type}.  If \var{s} is a string, it returns
  returns the ascii value value of the first character of the string
  \var{s}.  If \var{s} is \var{Double_Type}, \var{int} truncates the
  number to an integer and returns it.
\example
  \var{int} can be used to convert single character strings to
  integers.  As an example, the intrinsic function \var{isdigit} may
  be defined as
#v+
    define isdigit (s)
    {
      if ((int (s) >= '0') and (int (s) <= '9')) return 1;
      return 0;
    }
#v-
\notes
  This function is equalent to \exmp{typecast (s, Integer_Type)};
\seealso{typecast, double, integer, char, isdigit}
\done

\function{is_defined}
\synopsis{Indicate whether a variable or function defined.}
\usage{Integer_Type is_defined (String_Type obj)}
\description
   This function is used to determine whether or not a function or
   variable whose name is \var{obj} has been defined.  If \var{obj} is not
   defined, the function returns 0.  Otherwise, it returns a non-zero
   value that defpends on the type of object \var{obj} represents.
   Specifically, it returns one of the following values:
#v+
     +1 if an intrinsic function
     +2 if user defined function
     -1 if intrinsic variable
     -2 if user defined variable
      0 if undefined
#v-
\example
    For example, consider the function:
#v+
    define runhooks (hook)
    {
       if (2 == is_defined(hook)) eval(hook);
    }
#v-
    This function could be called from another \slang function to
    allow customization of that function, e.g., if the function
    represents a mode, the hook could be called to setup keybindings
    for the mode.
\seealso{typeof, eval, autoload}
\done

\function{is_list_element}
\synopsis{Test whether a delimited string contains a specific element}
\usage{Integer_Type is_list_element (String_Type list, String_Type elem, Integer_Type delim)}
\description
  The \var{is_list_element} function may be used to determine whether
  or not a delimited list of strings, \var{list}, contains the element
  \var{elem}.  If \var{elem} is not an element of \var{list}, the function
  will return zero, otherwise, it returns 1 plus the matching element
  number.
\example
  The expression
#v+
     is_list_element ("element 0, element 1, element 2", "0,", ' ');
#v-
  returns \exmp{2} since \exmp{"0,"} is element number one of the list
  (numbered from zero).
\seealso{extract_element, is_substr, create_delimited_string}
\done

\function{is_substr}
\synopsis{Test for a specified substring within a string.}
\usage{Integer_Type is_substr (String_Type a, String_Type b)}
\description
  This function may be used to determine if \var{a} contains the
  string \var{b}.  If it does not, the function returns 0; otherwise it
  returns the position of the first occurance of \var{b} in \var{a}.
\notes
  It is important to remember that the first character of a string
  corresponds to a position value of \exmp{1}.
\seealso{substr, string_match, str_replace}
\done

\function{isdigit}
\synopsis{Tests for a decimal digit character}
\usage{Integer_Type isdigit (String_Type s)}
\description
  This function returns a non-zero value if the first character in the
  string \var{s} is a digit; otherwise, it returns zero.
\example
  A simple, user defined implementation of \var{isdigit} is
#v+
    define isdigit (s)
    {
       return ((s[0] <= '9') and (s[0]  >= '0'));
    }
#v-
  However, the intrinsic function \var{isdigit} executes many times faster
  than the equivalent representation defined above.
\notes
  Unlike the C function with the same name, the \slang function takes
  a string argument.
\seealso{int, integer}
\done

\function{lstat_file}
\synopsis{Get information about a symbolic link}
\usage{Struct_Type lstat_file (String_Type file)}
\description
  The \var{lstat_file} function behaves identically to \var{stat_file}
  but if \var{file} is a symbolic link, \var{lstat_file} returns
  information about the link itself, and not the file that it
  references.

  See the documentation for \var{stat_file} for more information.
\notes
  On systems that do not support symbolic links, there is no
  difference between this function and the \var{stat_file} function.
\seealso{stat_file}
\done

\function{make_printable_string}
\synopsis{Format a string suitable for parsing}
\usage{String_Type make_printable_string(String_Type str)}
\description
  This function formats a string in such a way that it may be used as
  an argument to the \var{eval} function.  The resulting string is
  identical to \var{str} except that it is enclosed in double quotes and the
  backslash, newline, and double quote characters are expanded.
\seealso{eval, str_quote_string}
\done

\function{message}
\synopsis{Print a string onto the message device}
\usage{message (String_Type s}
\description
  The \var{message} function will print the string specified by
  \var{s} onto the message device.
\example
#v+
     define print_current_time ()
     {
       message (time ());
     }
#v-
\notes
  The message device will depend upon the application.  For example,
  the output message device for the \var{jed} editor correspond to the
  line at the bottom of the display window.  The default message
  device is the standard output device.
\seealso{vmessage, sprintf, error}
\done

\function{mkdir}
\synopsis{Create a new directory}
\usage{Integer_Type mkdir (String_Type dir, Integer_Type mode)}
\description
  The \var{mkdir} function creates a directory whose name is specified
  by the \var{dir} parameter with permissions specified by \var{mode}.
  Upon success \var{mkdir} returns zero, or it returns \exmp{-1} and
  sets \var{errno} accordingly.  In particular, if the directory
  already exists, the function will fail and set errno to
  \var{EEXIST}.
\example
     define my_mkdir (dir)
     {
        if (0 == mkdir (dir, 0777)) return;
        if (errno == EEXIST) return;
        verror ("mkdir %s failed: %s", dir, errno_string (errno));
     }
\notes
  The \var{mode} parameter may not be meaningful on all systems.  On
  systems where it is meaningful, the actual permissions on the newly
  created directory are modified by the process's umask.
\seealso{rmdir, getcwd, chdir, fopen, errno}
\done

\function{polynom}
\synopsis{Evaluate a polynomial}
\usage{Double_Type polynom(Double_Type a, b, ...c, Integer_Type n, Double_Type x)}
\description
  The \var{polynom} function returns the value of the polynomial expression:
#v+
     ax^n + bx^(n - 1) + ... c
#v-
\notes
  The \var{polynom} function should be extended to work with complex
  and array data types.  The current implementation is limited to
  \var{Double_Type} quantities.
\seealso{exp}
\done

\function{putenv}
\synopsis{Add or change an environment variable}
\usage{putenv (String_Type s)}
\description
    This functions adds string \var{s} to the environment.  Typically,
    \var{s} should of the form \var{"name=value"}.  The function
    signals a \slang error upon failure.
\notes
    This function is not available on all systems.
\seealso{getenv, sprintf}
\done

\function{reshape}
\synopsis{Reshape an array}
\usage{reshape (Array_Type A, Array_Type I}
\description
  The \var{reshape} function changes the size of a to have the size
  specified by the 1-d integer array \var{I}.  The elements of \var{I}
  specify the new dimensions of \var{A} and must be consistent with
  the number of elements \var{A}.
\example
  If \var{A} is a \var{100} element 1-d array, it can be changed to a
  2-d \var{20} by \var{5} array via
#v+
      reshape (A, [20, 5]);
#v-
  However, \exmp{reshape(A, [11,5])} will result in an error because
  the the \exmp{[11,5]} array specifies \exmp{55} elements.
\seealso{array_info}
\done

\function{set_float_format}
\synopsis{Set the format for printing floating point values.}
\usage{set_float_format (String_Type fmt)}
\description
  The \var{set_float_format} function is used to set the floating
  point format to be used when floating point numbers are printed.
  The routines that use this are the traceback routines and the
  \var{string} function. The default value is \exmp{"%f"}
\example
#v+
     s = string (PI);                %  --> s = "3.14159"
     set_float_format ("%16.10f");
     s = string (PI);                %  --> s = "3.1415926536"
     set_float_format ("%10.6e");
     s = string (PI);                %  --> s = "3.141593e+00"
#v-
\seealso{string, sprintf, double}
\done

\function{Sprintf}
\synopsis{Format objects into a string}
\usage{String_Type Sprintf (String_Type format, ..., Integer_Type n)}
\description
  \var{Sprintf} formats a string from \var{n} objects according to
  \var{format}.  Unlike \var{sprintf}, the \var{Sprintf} function
  requires the number of items to format.
\example
#v+
     s = Sprintf("%f is greater than %f but %s is better than %s\n",
                 PI, E, "Cake" "Pie", 4);
#v-
  The final argument to \var{Sprintf} is the number of items to format; in
  this case, there are 4 items.
\seealso{sprintf, string}
\done

\function{sprintf}
\synopsis{Format objects into a string}
\usage{String sprintf (String format, ...);}
\description
  This function performs a similar task as the C function with the same
  name.  It differs from the \slang function \var{Sprintf} in that it
  does not require the number of items to format.
  See the documentation for \var{Sprintf} for more information.
\seealso{Sprintf, string, vmessage}
\done

\function{stat_file}
\synopsis{Get information about a file}
\usage{Struct_Type stat_file (String_Type file)}
\description
  The \var{stat_file} function returns information about \var{file}
  through the use of the system \var{stat} call.  If the stat call
  fails, the function returns \var{NULL} and sets errno accordingly.
  If it is successful, it returns a stat structure with the following
  integer fields:
#v+
    st_dev
    st_ino
    st_mode
    st_nlink
    st_uid
    st_gid
    st_rdev
    st_size
    st_atime
    st_mtime
    st_ctime
#v-
  See the man page for \var{stat} for a discussion of these fields.
\example
  The following example shows how the \var{stat_file} function may be
  used to get the size of a file:
#v+
     define file_size (file)
     {
        variable st;
        st = stat_file(file);
        if (st == NULL) verror ("Unable to stat %s", file);
        return st.st_size;
     }
#v-
\seealso{lstat_file, stat_is}
\done

\function{stat_is}
\synopsis{Parse the \var{st_mode} field of a stat structure}
\usage{Integer_Type stat_is (String_Type type, Integer_Type st_mode}}
\description
  The \var{stat_is} function returns an integer value about the type of file
  specified by \var{st_mode}.  Specifically, \var{type} must be one of the
  strings:
#v+
     "sock"     (socket)
     "fifo"     (fifo)
     "blk"      (block device)
     "chr"      (character device)
     "reg"      (regular file)
     "lnk"      (link)
     "dir"      (dir)
#v-
  It returns a non-zero value if \var{st_mode} corresponds to
  \var{type}.
\example
  The following example illustrates how to use the \var{stat_is}
  function to determine whether or not a file is a directory:
#v+
     define is_directory (file)
     {
        variable st;

        st = stat_file (file);
        if (st == NULL) return 0;
        return stat_is ("dir", st.st_mode);
     }
#v-
\seealso{stat_file, lstat_file}
\done

\function{str_quote_string}
\synopsis{Escape characters in a string.}
\usage{String_Type str_quote_string(String_Type str, String_Type qlis, Integer_Type quote)}
\description
  The \var{str_quote_string} returns a string identical to \var{str}
  except that all characters in the set specified by the string
  \var{qlis} are escaped with the \var{quote} character, including the
  quote character itself.   This function is useful for making a
  string that can be used in a regular expression.
\example
  Execution of the statements
#v+
   node = "Is it [the coat] really worth $100?";
   tag = str_quote_string (node, "\\^$[]*.+?", '\\');
#v-
  will result in \var{tag} having the value:
#v+
    Is it \[the coat\] really worth \$100\?
#v-
\seealso{str_uncomment_string, make_printable_string}
\done

\function{str_replace}
\synopsis{Replace a substring of a string}
\usage{Integer_Type str_replace (String_Type a, String_Type b, String_Type c)}
\description
  The \var{str_replace} function replaces the first occurance of \var{b} in
  \var{a} with \var{c} and returns an integer that indicates whether a
  replacement was made or not. If \var{b} does not occur in \var{a}, zero is
  returned.  However, if \var{b} occurs in \var{a}, a non-zero integer is
  returned as well as the new string resulting from the replacement.
\example
#v+
  define str_replace_all (orig, match, replacement)
  {
     while (str_replace (orig, match, replacement))
        orig = ();
     return orig;
  }
#v-
  is a function that replaces all occurances in a string.
\seealso{is_substr, strsub, strtrim}
\done

\function{str_uncomment_string}
\synopsis{Remove comments from a string}
\usage{String_Type str_uncomment_string(String_Type s, String_Type beg, String_Type end)}
\description
  This function may be used to remove comments from a string \var{s}.
  The parameters, \var{beg} and \var{end}, are strings of equal length
  whose corresponding characters specify the begin and end comment
  characters, respectively.  It returns the uncommented string.
\example
  The expression
#v+
     str_uncomment_string ("Hello (testing) 'example' World", "'(", "')")
#v-
  returns the string \exmp{"Hello   World"}.
\notes
  This routine does not handle multicharacter comment delimiters and it
  assumes that comments are not nested.
\seealso{str_quote_string}
\done

\function{strcat}
\synopsis{Concatenate two strings}
\usage{String_Type strcat (String_Type a, String_Type b)}
\description
   The \var{strcat} function takes two string valued quantities, \var{a} and
   \var{b}, concatenates them together and returns the result.
\example
#v+
    strcat ("Hello ", "World");
#v-
   produces the string \exmp{"Hello World"}.
\notes
   This function is equivalent to the binary operation \exmp{a+b}.
\seealso{sprintf, create_delimited_string}
\done

\function{strchopr}
\synopsis{Chop or split a string into substrings.}
\usage{Integer_Type strchopr (String_Type str, String_Type delim, String_Type quote)}
\description
  This routine performs exactly the same function as \var{strchop} except
  that it returns the substrings in the reverse order.  See the
  documentation for \var{strchop} for more information.
\seealso{strchop, extract_element}
\done

\function{strchop}
\synopsis{Chop or split a string into substrings.}
\usage{Integer_Type strchop (String_Type str, Integer_Type delim, Integer_Type quote)}
\description
     The \var{strchop} function may be used to split-up a string
     \var{str} that consists of substrings delimited by the character
     specified by \var{delim}.  If the integer \var{quote} is
     non-zero, it will be taken as a quote character for the
     delimiter.  The function returns to the stack a variable number
     of objects.  The top stack object represents the number of
     substrings that are on the stack.  For example,
#v+
          strchop ("apples,oranges,pears", ',', 0);
#v-
     will return the integer \exmp{3} to the top of the stack followed
     by the three strings:  \exmp{"apples"}, \var{"oranges"}, and
     \var{"pears"}.
\example
#v+
       define list_to_array (list)
       {
          variable n, i, s, a;
          n = strchop (list, ',', '\\');
          a = String_Type [n];
          for (i = 0; i < n; i++)
            a[i] = ();             % substring is on stack
          return a;
       }
#v-
     This defines a function that converts a comma-separated list of
     strings to string array containing the substrings.
\notes
   Since this function returns a variable number of values to the
   stack, it cannot chop up arbitrarily long strings that consist of
   many substrings.

   The function \var{strchopr} should be used if it is desired to have
   the string chopped-up in the reverse order.
\seealso{strchopr, extract_element, create_delimited_string}
\done

\function{strcmp}
\synopsis{Compare two strings}
\usage{Interpret strcmp (String_Type a, String_Type b)}
\description
   The \var{strcmp} function may be used to perform a case-sensitive
   string comparison, in the lexicongraphic sense, on strings \var{a} and
   \var{b}.  It returns 0 if the strings are identical, a negative integer
   if \var{a} is less than \var{b}, or a positive integer if \var{a} is greater
   than \var{b}.
\example
   The \var{strup} function may be used to perform a case-insensitive
   string comparison:
#v+
    define case_insensitive_strcmp (a, b)
    {
      return strcmp (strup(a), strup(b));
    }
#v-
\notes
   One may also use one of the binary comparison operators, e.g.,
   \exmp{a > b}.
\seealso{strup, strncmp}
\done

\function{strcompress}
\synopsis{Remove excess whitespace characters from a string}
\usage{String_Type strtrim (String_Type s, String_Type white)}
\description
  The \var{strcompress} function compresses the string \var{s} by
  removing all repeated characters specified by by the characters of
  \var{white} from the interior of \var{s}.  In addition, it also
  removes all leading and trailing characters from \var{s} that are
  part of \var{white}.
\example
  The expression
#v+
    strcompress (",;apple,,cherry;,banana", ",;");
#v-
  returns the string @"apple,cherry;banana"@.
\seealso{strtrim}
\done

\function{string_match_nth}
\synopsis{Get the result of the last call to \var{string_match}}
\usage{(Integer_Type, Integer_Type)  string_match_nth(Integer_Type nth)}
\description
  The \var{string_match_nth} function returns two integers describing
  the result of the last call to \var{string_match}.  It returns both
  the offset into the string and the length of characters matches by
  the \var{nth} submatch.

  By convention, \var{nth} equal to zero means the entire match.
  Otherwise, \var{nth} must be an integer with a value 1 through 9,
  and refers to the set of characters matched by the \var{nth} regular
  expression enclosed by the pairs \exmp{\\(, \\)}.
\example
  Consider:
#v+
     variable matched, pos, len;
     matched = string_match("hello world", "\\([a-z]+\\) \\([a-z]+\\)", 1);
     if (matched) (pos, len) = string_match_nth(2);
#v-
  This will set \var{matched} to 1 since a match will be found at the
  first position, \var{pos} to 6 since \var{w} is offset 6 characters
  from the beginning of the string, and \var{len} to 5 since
  \exmp{"world"} is 5 characters long.
\notes
  The position offset is \em{not} affected by the value of the offset
  parameter to the \var{string_match} function. For example, if the
  value of the last parameter to the \var{string_match} function had
  been 3, \var{pos} would still have been set to 6.

  Note also that \var{string_match_nth} returns the \em{offset} from
  the beginning of the string and not the position of the match.
\seealso{string_match}
\done

\function{string_match}
\synopsis{Match a string against a regular expression}
\usage{Integer_Type string_match(String_Type str, String_Type pat, Integer_Type pos)}
\description
  The \var{string_match} function returns zero if \var{str} does not
  match regular expression specified by \var{pat}.  This function
  performs the match starting at position \var{pos} (numbered from 1) in
  \var{str}.  This function returns the position of the start of the
  match.  To find the exact substring actually matched, use
  \var{string_match_nth}.
\seealso{string_match_nth, strcmp, strncmp}
\done

\function{string}
\synopsis{Convert an object to a string representation.}
\usage{Integer_Type string (obj)}
\description
   The \var{string} function may be used to convert an object
   \var{obj} of any type to a string representation.
   For example, \exmp{string(12.34)} returns \exmp{"12.34"}.
\example
#v+
     define print_anything (anything)
     {
        message (string (anything));
     }
#v-
\notes
   This function is \em{not} the same as typecasting to a \var{String_Type}
   using the \var{typecast} function.
\seealso{typecast, sprintf, integer, char}
\done

\function{strlen}
\synopsis{Compute the length of a string}
\usage{Integer_Type strlen (String_Type a)}
\description
   The \var{strlen} function may be used to compute the length of a string.
\example
   After execution of
#v+
   variable len = strlen ("hello");
#v-
   \var{len} will have a value of \exmp{5}.
\seealso{substr}
\done

\function{strlow}
\synopsis{Convert a string to lowercase}
\usage{String_Type strlow (String_Type s)}
\description
  The \var{strlow} function takes a string \var{s} and returns another
  string identical to \var{s} except that all upper case characters
  that comprise \var{s} will be converted to lower case.
\example
  The function
#v+
    define Strcmp (a, b)
    {
      return strcmp (strlow (a), strlow (b));
    }
#v-
  performs a case-insensitive comparison operation of two strings by
  converting them to lower case first.
\seealso{strup, tolower, strcmp, strtrim, define_case}
\done

\function{strncmp}
\synopsis{Compare the first few characters of two strings}
\usage{Integer_Type strncmp (String_Type a, String_Type b, Integer_Type n)}
\description
  This function behaves like \var{strcmp} except that it compares only the
  first \var{n} characters in the strings \var{a} and \var{b}.  See
  the documentation for \var{strcmp} for information about the return
  value.
\example
  The expression
#v+
     strcmp ("apple", "appliance", 3);
#v-
  will return zero since the first three characters match.
\seealso{strcmp, strlen}
\done

\function{strsub}
\synopsis{Replace a character with another in a string.}
\usage{String_Type strsub (String_Type s, Integer_Type pos, Integer_Type ch)}
\description
  The \var{strsub} character may be used to substitute the character
  \var{ch} for the character at position \var{pos} of the string
  \var{s}.  The resulting string is returned.
\example
#v+
    define replace_spaces_with_comma (s)
    {
      variable n;
      while (n = is_substr (s, " "), n) s = strsub (s, n, ',');
      return s;
    }
#v-
\notes
  The first character in the string \var{s} is specified by \var{pos}
  equal to 1.
\seealso{is_substr, str_replace, strlen}
\done

\function{strtrim}
\synopsis{Remove whitespace from the ends of a string}
\usage{String_Type strtrim (String_Type s)}
\description
  The \var{strtrim} function removes all leading and trailing whitespace
  characters from the string \var{s} and returns the result.  Whitespace is
  defined to be any combination of spaces, tabs, and newline characters.
\seealso{int, strlow, strup, strcompress}
\done

\function{strup}
\synopsis{Convert a string to uppercase}
\usage{String_Type strup (String_Type s)}
\description
  The \var{strup} function takes a string \var{s} and returns another
  string identical to \var{s} except that all lower case characters
  that comprise \var{s} will be converted to upper case.
\example
  The function
#v+
    define Strcmp (a, b)
    {
      return strcmp (strup (a), strup (b));
    }
#v-
  performs a case-insensitive comparison operation of two strings by
  converting them to upper case first.
\seealso{strlow, toupper, strcmp, strtrim, define_case}
\done

\function{substr}
\synopsis{Extract a substring from a string}
\usage{String_Type substr (String_Type s, Integer_Type n, Integer_Type len)}
\description
  The \var{substr} function returns a substring with length \var{len}
  of the string \var{s} beginning at position \var{n}.  If \var{len} is
  \exmp{-1}, the entire length of the string \var{s} will be used for
  \var{len}.  The first character of \var{s} is given by \var{n} equal
  to 1.
\example
#v+
     substr ("To be or not to be", 7, 5);
#v-
  returns \exmp{"or no"}
\notes
    The expression \exmp{substr(s, n, 1)} is equivalent to the array
    index expression \exmp{s[n-1]}.
\seealso{is_substr, strlen}
\done

\function{system}
\synopsis{Execute a shell command}
\usage{Integer_Type system (String_Type cmd)}
\description
  The \var{system} function may be used to execute the string
  expression \var{cmd} in an inferior shell.  This function is an
  interface to the C \var{system} function which returns an
  implementation-defined result.   On Linux, it returns 127 if the
  inferior shell could not be invoked, -1 if there was some other
  error, otherwise it returns the return code for \var{cmd}.
\example
#v+
    define dir ()
    {
       () = system ("DIR");
    }
#v-
  displays a directory listing of the current directory under MSDOS or
  VMS.
\done

\function{time}
\synopsis{Return the current data and time as a string}
\usage{String time ();}
\description
  This function returns the current time as a string of the form:
#v+
    Sun Apr 21 13:34:17 1996
#v-
\seealso{message, substr}
\done

\function{tolower}
\synopsis{Convert a character to lowercase.}
\usage{Integer_Type lower (Integer_Type ch)}
\description
  This function takes an integer \var{ch} and returns its lowercase
  equivalent.
\seealso{toupper, strup, strlow, int, char, define_case}
\done

\function{toupper}
\synopsis{Convert a character to uppercase.}
\usage{Integer_Type toupper (Integer_Type ch)}
\description
  This function takes an integer \var{ch} and returns its uppercase
  equivalent.
\seealso{tolower, strup, strlow, int, char, define_case}
\done

\function{typecast}
\synopsis{Convert an object from one data type to another.}
\usage{typecast (x, new_type)}
\description
  The \var{typecast} function performs a generic typecast operation on
  \var{x} to convert it to \var{new_type}.  If \var{x} represents an
  array, the function will attempt to convert all elements of \var{x}
  to \var{new_type}.  Not all objects can be converted and a
  type-mismatch error will result upon failure.
\example
#v+
    define to_complex (x)
    {
       return typecast (x, Complex_Type);
    }
#v-
  defines a function that converts its argument, \var{x} to a complex
  number.
\seealso{int, double, typeof}
\done

\function{typeof}
\synopsis{Get the data type of an object.}
\usage{DataType_Type typeof (x)}
\description
  This function returns the data type of \var{x}.
\example
#v+
  if (Integer_Type == typeof (x)) print ("x is an integer");
#v-
\seealso{array_info, _slang_guess_type, typecast}
\done

\function{unix_ctime}
\synopsis{Convert a calendar time to a string}
\usage{String_Type unix_ctime(Integer_Type secs)}
\description
  This function returns a string representation of the time as given
  by \var{secs} seconds since 1970.
\seealso{time}
\done

\function{unix_kill}
\synopsis{Send a signal to a process}
\usage{Integer_Type unix_kill (Integer_Type pid, Integer_Type sig)}
\description
  This function may be used to send a signal given by the integer \var{sig}
  to the process specified by \var{pid}.  The function returns zero upon
  sucess and \exmp{-1} upon failure setting errno accordingly.
\example
  The \var{unix_kill} function may be used to determine whether or not
  a specific process exists:
#v+
    define process_exists (pid)
    {
       if (-1 == unix_kill (pid, 0))
         return 0;     % Process does not exist
       return 1;
    }
#v-
\notes
  The \var{unix_kill} function is not available on all systems.
\seealso{getpid}
\done

\function{verror}
\synopsis{Generate an error condition}
\usage{verror (String_Type fmt, ...)}
\description
  The \var{verror} function performs the same role as the \var{error}
  function.  The only difference is that instead of a single string
  argument, \var{verror} takes a sprintf style argument list.
\example
#v+
    define open_file (file)
    {
       variable fp;

       fp = fopen (file, "r");
       if (fp == NULL) verror ("Unable to open %s", file);
       return fp;
    }
#v-
\notes
  In the current implementation, strictly speaking, the \var{verror}
  function is not an intrinsic function.  Rather it is a predefined
  \slang function using a combination of \var{Sprintf} and
  \var{error}.
\seealso{error, Sprintf, vmessage}
\done

\end{\documentstyle}
