/*

Noptec Vector
Copyright (C) 2000 Erik Agsjo (erik.agsjo@noptec.com)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include <stdio.h>

#include <allegro.h>
#include <assert.h>
#include <set>
#include "vector.h"


// Vectorizer implementation:

//
// Vectorizer constructor
// 	source - the bitmap to convert
//  fill - the default fill color for this scan
//         this will typically be 15 (white) for the picture plane
//		   and 4 (red) for the prio plane
//
Vectorizer::Vectorizer(BITMAP *source, int fill) {
  this->source = source;
  this->map = new PixelMap(source->w, source->h);
  this->fill = fill;
  this->lol = new PixelAreas(fill);
}

Vectorizer::~Vectorizer(void) {
}

//
// Scans a row of the bitmap and builds areas of
// identically colored pixels.
//
void Vectorizer::scanRow(int row) {
  int firstCol = 0;
  int firstPixel = getpixel(source, firstCol, row);

  PixelStripeList *list = new PixelStripeList(map, source, fill);
  lol->add(list);
  PixelStripe *stripe = new PixelStripe(row, firstCol, firstPixel);
  list->add(stripe);

  map->setList(firstCol, row, list);

  if(row > 0 && firstPixel == getpixel(source, firstCol, row - 1)) {
    //stripe->increaseOverlap();
    PixelStripeList *mergeList=map->getList(firstCol, row - 1);
    if(mergeList != list) {
      list->merge(mergeList);
      lol->remove(mergeList);
      delete mergeList;
    }
  }

  for(int col = 1; col < source->w; col++){
    int pix = getpixel(source, col, row);
    if(pix == firstPixel) {
      if(row > 0 && pix == getpixel(source, col, row - 1)) {
	//stripe->increaseOverlap();
	PixelStripeList *mergeList=map->getList(col, row - 1);
	if(mergeList != list) {
	  list->merge(mergeList);
	  lol->remove(mergeList);
	  delete mergeList;
	}
      }
      map->setList(col, row, list);
      continue;
    }
    stripe->range(col - 1);

    firstCol = col;
    firstPixel = pix;

    list = new PixelStripeList(map, source, fill);
    lol->add(list);
    stripe = new PixelStripe(row, firstCol, firstPixel);
    list->add(stripe);
    map->setList(firstCol, row, list);
    if(row > 0 && pix == getpixel(source, firstCol, row - 1)) {
      stripe->increaseOverlap();
      PixelStripeList *mergeList=map->getList(firstCol, row - 1);
      if(mergeList != list) {
	list->merge(mergeList);
	lol->remove(mergeList);
	delete mergeList;
      }
    }
  }

  stripe->range(source->w - 1);
}

void Vectorizer::scan(void) {
  for(int row = 0; row < source->h; row++){
    scanRow(row);
  }
  lol->convert();
}

void Vectorizer::report(void) {
  lol->report();
}

list<PixelStripeList*> *Vectorizer::getAreas(void) {
  return lol->getAreas();
}

//
// Pixel implementation:
//

Pixel::Pixel(int x, int y, int pixel, PixelStripe *stripe) {
  this->x = x;
  this->y = y;
  this->pixel = pixel;
  this->stripe = stripe;
}

Pixel::~Pixel(void) {
}

int Pixel::getX(void) {
  return x;
}

int Pixel::getY(void) {
  return y;
}

int Pixel::getPixel(void) {
  return pixel;
}

double Pixel::distance(Pixel *pix) {
  double x2 = (this->x - pix->x);
  x2 *= x2;
  double y2 = (this->y - pix->y);
  y2 *= y2;
  return sqrt(x2 + y2);
}

struct Xless {
  bool operator()(Pixel *a, Pixel *b) const {
    return (a->getX() < b->getX());
  }
};

struct Yless {
  bool operator()(Pixel *a, Pixel *b) const {
    return (a->getY() < b->getY());
  }
};


//
// PixelStripe implementation:
//

PixelStripe::PixelStripe(int row, int startcol, int pixel) {
  this->row = row;
  this->startcol = startcol;
  this->endcol = startcol;
  this->pixel = pixel;
  this->overlap = 0;
}

PixelStripe::~PixelStripe(void) {
}

void PixelStripe::range(int endcol) {
  this->endcol = endcol;
}

int PixelStripe::getStartCol(void) {
  return startcol;
}

int PixelStripe::getEndCol(void) {
  return endcol;
}

int PixelStripe::getRow(void) {
  return row;
}

void PixelStripe::increaseOverlap(void) {
  overlap++;
}

int PixelStripe::getOverlap(void) {
  return overlap;
}

void PixelStripe::shrink(void) {
  if((endcol - startcol) > 1) {
    endcol--;
    startcol++;
  }else{
    endcol = startcol = -1;
  }
}

int PixelStripe::size(void) {
  if(endcol == -1)
    return 0;
  else
    return (endcol - startcol + 1);
}

bool PixelStripe::lessThanOrEqual(PixelStripe *a, PixelStripe *b) {
  if(a->getRow() == b->getRow())
    return(a->getStartCol() <= b->getStartCol());
  else
    return(a->getRow() < b->getRow());
}

int PixelStripe::getPixel(void) {
  return pixel;
}

// PixelStripeList implementation:

PixelStripeList::PixelStripeList(PixelMap *map, BITMAP *bimp, int fill) {
  this->map = map;
  this->bitmap = bimp;
  this->filled = false;
  this->fills = new list<Pixel*>;
  this->lines = new list<Pixel*>;
  this->color = -1;
  this->fill = fill;
}

PixelStripeList::~PixelStripeList(void) {
  delete fills;
  delete lines;
}

void PixelStripeList::add(PixelStripe *stripe) {
  if(color == -1) color = stripe->getPixel();
  stripes.push_back(stripe);
}

void PixelStripeList::merge(PixelStripeList *plist) {
  if(this == plist) return;
  assert(plist != NULL);
  if(plist->stripes.empty()) return; // do we need to do more?
  for(list<PixelStripe*>::iterator i = plist->stripes.begin();
      i != plist->stripes.end(); i++) {
    for(int x = (*i)->getStartCol(); x <= (*i)->getEndCol(); x++)
      map->setList(x, (*i)->getRow(), this);
    stripes.push_back(*i);
  }
}

void PixelStripeList::report(void) {
  if(stripes.empty()) return; // do we need to do more?
  for(list<PixelStripe*>::iterator i = stripes.begin();
      i != stripes.end(); i++) {
    printf("Strip at row %d from %d to %d\n",
	   (*i)->getRow(),
	   (*i)->getStartCol(),
	   (*i)->getEndCol());
  }
}

void PixelStripeList::setFilled(bool state) {
  filled = state;
}

bool PixelStripeList::getFilled(void) {
  return filled;
}

//
// General conversion idea:
//
// 1)  Scan convert all edges like this:
// 		for each row, add start and end positions to a list
//		(at the same time removing these pixels from the strip.
//		if the strip becomes empty remove it from its list.)
//		if the strip contains edge pixels, add these too.
//
// 2)  Create a bitmap the size of the screen, clear it and plot
//		all pixels in the PixelStripeList.
//
// 3)  Find a starting position, walk to the first pixel set,
//		testing	through all eight positions "S, SE, E, NE N, ...".
//		Clear the pixel just left behind (if its not the starting
//		pixel) and add the line to the resulting line list.
//
// 4)  Repeat step 3 until either the first pixel is reached, or
//		it is impossible to move. In the first case, a fill
//		sequence is initiated (4b), in the second case the shape is
//		not strictly closed and we can move on to the next shape.
//		Either way, the first pixel should be cleared.
//
// 4b) Filling sequence:
//      During scanning of all objects the filling consists of simply
//		tagging the area as filled. Later on, after all areas has been
//		successfully scanned, the filling sequence begins where we try
//		to find the minimum fill positions needed to make the whole
//		shape filled. This is done by first drawing all lines for all
//		filled shapes, then trying to fill them one by one. The success
//		of the fills are tested and if the whole shape is not filled,
//      the first not-filled pixel is found and filled and the test is
//		repeated.
//
// 5)  The result of all this should be that the image is drawn in four
//		phases. I: Outlines of filled shapes. II: Filling. III: Outlines
//		of not filled shapes. IV: All simple oneliners.
//

list<Pixel*> *PixelStripeList::getLines(void) {
  return lines;
}

list<Pixel*> *PixelStripeList::getFills(void) {
  return fills;
}

list<PixelStripe*> &PixelStripeList::getStripes(void) {
  return stripes;
}

void PixelStripeList::addFill(int x, int y, int color) {
  fills->push_back(new Pixel(x, y, color, NULL));
}

void PixelStripeList::convert(void) {
  if(stripes.empty()) return;

  // The one stripe case:

  if(stripes.size() == 1){
    PixelStripe *stripe = stripes.front();
    lines->push_back(new Pixel(stripe->getStartCol(),
			       stripe->getRow(),
			       stripe->getPixel(),
			       stripe));
    lines->push_back(new Pixel(stripe->getEndCol(),
			       stripe->getRow(),
			       stripe->getPixel(),
			       stripe));
    return;
  }


  BITMAP *workbmp = create_bitmap_ex(8, 160, 168);

  int startx = 0;
  int starty = 0;
  int pixel = -1;

  int minx = stripes.front()->getStartCol();
  int maxx = minx;
  int miny = stripes.front()->getRow();
  int maxy = miny;

  stripes.sort(PixelStripe::lessThanOrEqual);

  for(list<PixelStripe*>::iterator i = stripes.begin(); i != stripes.end(); i++) {
    PixelStripe *ii = (*i);
    int thisRow = ii->getRow();
    int thisStart = ii->getStartCol();
    int thisEnd = ii->getEndCol();
    int thisPixel = ii->getPixel();

    if(pixel == -1) {
      pixel = thisPixel;
      clear_to_color(workbmp, pixel + 1);
    }

    if(minx > thisStart) minx = thisStart;
    if(maxx < thisEnd) maxx = thisEnd;
    if(miny > thisRow) miny = thisRow;
    if(maxy < thisRow) miny = thisRow;

    putpixel(workbmp, thisStart, thisRow, thisPixel);

    for(int x = thisStart + 1; x < thisEnd; x++) {
      int debugpix;
      if( (thisRow == 0 || thisRow == 167) ||
	  ((debugpix = getpixel(bitmap, x, thisRow - 1)) != thisPixel) ||
	  ((debugpix = getpixel(bitmap, x, thisRow + 1)) != thisPixel) ) {

	putpixel(workbmp, x, thisRow, thisPixel);
      }
    }

    putpixel(workbmp, thisEnd, thisRow, thisPixel);
  }

  // scan baby, scan

  while(1){
    int searchx, searchy;
    for(searchx = minx; searchx <= maxx; searchx++)
      for(searchy = miny; searchy <= maxy; searchy++) {
	if(getpixel(workbmp, searchx, searchy) == pixel){
	  startx = searchx;
	  starty = searchy;
	  searchx = searchy = 200;
	}
      }
    if(searchx < 200) break;

    int x = startx, y = starty;
    int xdiff = 0, ydiff = 1;
    int count = 0;


    while(1){
      count++;
      lines->push_back(new Pixel(x, y, pixel, NULL));
      putpixel(workbmp, x, y, pixel + 1);
      if(count == 3) putpixel(workbmp, startx, starty, pixel);
      if(getpixel(workbmp, x + xdiff, y + ydiff) == pixel) {
	x+=xdiff;
	y+=ydiff;
      }else if(getpixel(workbmp, x, y + 1) == pixel) {
	y++;
	xdiff = 0;
	ydiff = 1;
      }else if(getpixel(workbmp, x + 1, y) == pixel) {
	x++;
	xdiff = 1;
	ydiff = 0;
      }else if(getpixel(workbmp, x, y - 1) == pixel) {
	y--;
	xdiff = 0;
	ydiff = -1;
      }else if(getpixel(workbmp, x - 1, y) == pixel) {
	x--;
	xdiff = -1;
	ydiff = 0;
      }else if(getpixel(workbmp, x + 1, y + 1) == pixel) {
	y++; x++;
	xdiff = 1;
	ydiff = 1;
      }else if(getpixel(workbmp, x + 1, y - 1) == pixel) {
	x++; y--;
	xdiff = 1;
	ydiff = -1;
      }else if(getpixel(workbmp, x - 1, y - 1) == pixel) {
	x--; y--;
	xdiff = -1;
	ydiff = -1;
      }else if(getpixel(workbmp, x - 1, y + 1) == pixel) {
	x--; y++;
	xdiff = -1;
	ydiff = 1;
      }else{
	int searchx, searchy;
	if(x != startx || y != starty) {
	  lines->push_back(new Pixel(-1, -1, -1, NULL));
	  for(searchx = minx; searchx <= maxx; searchx++)
	    for(searchy = miny; searchy <= maxy; searchy++) {
	      if(getpixel(workbmp, searchx, searchy) == pixel){
		if(searchx == startx && searchy == starty) continue;
		x = searchx;
		y = searchy;
		searchx = searchy = 200;
		count = 0;
	      }
	    }
	  if(searchx < 200) break;
	}
      }
      if(x == startx && y == starty) {
	putpixel(workbmp, startx, starty, pixel + 1);
	lines->push_back(new Pixel(startx, starty, pixel, NULL));
	lines->push_back(new Pixel(-1, -1, -1, NULL));
	setFilled(true);
	break;
      }
    }
  }
  destroy_bitmap(workbmp);
}

int removed = 0;

bool PixelStripeList::colorSort(PixelStripeList *a, PixelStripeList *b) {
  return a->color <= b->color;
}

//
// Reduce the number of lines generated
//
void PixelStripeList::optimize(void) {
  list<Pixel*> *lines=getLines();
  Pixel *p0, *p1, *p2;
  int xdiff = 0;
  int ydiff = 0;

  list<Pixel*>::iterator i = lines->begin();
  p0 = *i;
  i++;
  p1 = *i;
  if(p0->getPixel() == fill || p1->getPixel() == fill) {
    lines->clear();
    return;
  }
  do {
    if(p1->getX() == p0->getX()) {
      i++;
      if(i == lines->end()) break;
      p2 = *i;
      if(p0->getY() == p1->getY() || p2->getX() == p0->getX()) {
	lines->remove(p1);
	removed++;
      }else{
	p0 = p1;
      }
      p1 = p2;
    }else if(p1->getY() == p0->getY()) {
      i++;
      if(i == lines->end()) break;
      p2 = *i;
      if(p2->getY() == p0->getY() || p0->getX() == p1->getX()) {
	lines->remove(p1);
	removed++;
      }else{
	p0 = p1;
      }
      p1 = p2;
    }else if(abs(xdiff = p1->getX() - p0->getX()) == 1 &&
	     abs(ydiff = p1->getY() - p0->getY()) == 1) {
      i++;
      if(i == lines->end()) break;
      p2 = *i;
      if(p2->getX() - p1->getX() == xdiff &&
	 p2->getY() - p1->getY() == ydiff) {
	lines->remove(p1);
	removed++;
      }else{
	p0 = p1;
      }
      p1 = p2;
    }else{
      p0 = p1;
      i++;
      if(i == lines->end()) break;
      p1 = *i;
    }
  }while(i != lines->end());
}

PixelMap *PixelStripeList::getMap(void) {
  return map;
}

//
// PixelAreas implementation:
//

PixelAreas::PixelAreas(int fc) {
  fillcolor = fc;
  areas = new list<PixelStripeList*>;
}

PixelAreas::~PixelAreas(void) {
  delete areas;
}

void PixelAreas::add(PixelStripeList *list){
  areas->push_back(list);
}

void PixelAreas::remove(PixelStripeList *list) {
  areas->remove(list);
}

void PixelAreas::report(void) {
  printf("Number of areas found:%ld\n", areas->size());
  if(areas->empty()) return;
  int count = 0;
  for(list<PixelStripeList*>::iterator i = areas->begin();
      i != areas->end(); i++) {
    printf("Area %d:\n", count++);
    (*i)->report();
  }
}

#define FILL_DEPTH 16384
struct dot{
  int x;
  int y;
};

bool erkfill(BITMAP *bmp, int x, int y, int color, PixelMap *map, int border) {
  struct dot list[FILL_DEPTH];
  int fills = 1;

  list[0].x = x;
  list[0].y = y;
  if(color == border) return true;
  if(getpixel(bmp, x, y) != border) return(true);
  PixelStripeList *slist = map->getList(x, y);

  while(fills > 0) {
    fills--;
    x = list[fills].x;
    y = list[fills].y;
    if(map->getList(x, y) != slist) return false;
    putpixel(bmp, x, y, color);
    if(getpixel(bmp, x + 1, y) == border) {
      putpixel(bmp, x + 1, y, color);
      list[fills].x = x + 1;
      list[fills++].y = y;
    }
    if(getpixel(bmp, x - 1, y) == border) {
      putpixel(bmp, x - 1, y, color);
      list[fills].x = x - 1;
      list[fills++].y = y;
    }
    if(getpixel(bmp, x, y + 1) == border) {
      putpixel(bmp, x, y + 1, color);
      list[fills].x = x;
      list[fills++].y = y + 1;
    }
    if(getpixel(bmp, x, y - 1) == border) {
      putpixel(bmp, x, y - 1, color);
      list[fills].x = x;
      list[fills++].y = y - 1;
    }
    if(fills > FILL_DEPTH - 5){
      fprintf(stderr, "Fill failed c=%d\n", color);
      break;
    }
  }
  return true;
}

extern void sline(BITMAP*, int, int, int, int, int);

void PixelAreas::convert(void) {
  //
  // run "convert" on every pixel area, to get the boundaries
  // for the shapes and to find out which are filled.
  //
  for(list<PixelStripeList*>::iterator i = areas->begin();
      i != areas->end(); i++) {
    (*i)->convert();
    (*i)->optimize();
  }

  areas->sort(PixelStripeList::colorSort);

  //
  // create a bitmap and plot closed shape outlines onto it
  //
  BITMAP *fillbimp = create_bitmap_ex(8, 160, 168);
  BITMAP *tempbimp = create_bitmap_ex(8, 160, 168);
  clear_to_color(fillbimp, fillcolor);

  for(list<PixelStripeList*>::iterator i = areas->begin(); i != areas->end(); i++) {
    PixelMap *map = (*i)->getMap();
    list<PixelStripe*> stripes = (*i)->getStripes();
    PixelStripe *first = stripes.front();
    int color = first->getPixel();
    if(color == fillcolor) continue;
    bool fillOK = true;

    blit(fillbimp, tempbimp, 0, 0, 0, 0, 160, 168);
    for(list<PixelStripe*>::iterator si = stripes.begin(); si != stripes.end(); si++) {
      PixelStripe *ps = *si;
      int row = ps->getRow();
      for(int col = ps->getStartCol(); col <= ps->getEndCol(); col++) {
	if(col == -1) break;
	if(getpixel(fillbimp, col, row) == fillcolor) {
	  fillOK = erkfill(fillbimp, col, row, color, map, fillcolor);
	  if(!fillOK) break;
	  (*i)->addFill(col, row, color);
	}
      }
      if(!fillOK) break;
    }

    if(fillOK) {
      (*i)->getLines()->clear();
    }else{
      blit(tempbimp, fillbimp, 0, 0, 0, 0, 160, 168);
      (*i)->getFills()->clear();
      list<Pixel*> *lines = (*i)->getLines();
      Pixel *p0 = NULL;
      Pixel *p1 = NULL;
      for(list<Pixel*>::iterator pi = lines->begin(); pi != lines->end(); pi++) {
	if(p0 == NULL) {
	  p0 = *pi;
	  putpixel(fillbimp, p0->getX(), p0->getY(), p0->getPixel());
	  pi++;
	  if(pi == lines->end()) break;
	}

	p1 = *pi;

	if(p1->getX() == -1) {
	  p0 = NULL;
	  continue;
	}

	sline(fillbimp, p0->getX(), p0->getY(), p1->getX(), p1->getY(), p0->getPixel());

	p0 = p1;
      }

      if(p0 != NULL && p1 != NULL && p0->getX() != -1 && p1->getX() != -1)
	sline(fillbimp, p0->getX(), p0->getY(), p1->getX(), p1->getY(), p0->getPixel());

      for(list<PixelStripe*>::iterator si = stripes.begin(); si != stripes.end(); si++) {
	PixelStripe *ps = *si;
	int row = ps->getRow();
	for(int col = ps->getStartCol(); col <= ps->getEndCol(); col++) {
	  if(col == -1) break;
	  if(getpixel(fillbimp, col, row) == fillcolor) {
	    (*i)->addFill(col, row, color);
	    floodfill(fillbimp, col, row, color);
	  }
	}
      }

    }
    destroy_bitmap(fillbimp);
    destroy_bitmap(tempbimp);
  }

  //
  // find the minimum number of fills needed to completely fill the
  // closed shapes
  //
  for(list<PixelStripeList*>::iterator i = areas->begin(); i != areas->end(); i++) {
    list<PixelStripe*> stripes = (*i)->getStripes();
    PixelStripe *first = stripes.front();
    int color = first->getPixel();
    if(color != fillcolor)
      for(list<PixelStripe*>::iterator si = stripes.begin(); si != stripes.end(); si++) {
	PixelStripe *ps = *si;
	int row = ps->getRow();
	for(int col = ps->getStartCol(); col <= ps->getEndCol(); col++) {
	  if(col == -1) break;
	  if(getpixel(fillbimp, col, row) == fillcolor) {
	    (*i)->addFill(col, row, color);
	    floodfill(fillbimp, col, row, color);
	  }
	}
      }
  }
}

list<PixelStripeList*> *PixelAreas::getAreas(void) {
  return areas;
}

//
// PixelMap implementation:
//
PixelMap::PixelMap(int w, int h) {
  width=w;
  map=new PixelStripeList*[w*h];
  for(int i = 0; i < w * h; i++){
    map[i] = NULL;
  }
}

PixelMap::~PixelMap(void) {
  delete map;
}

void PixelMap::setList(int x, int y, PixelStripeList *list) {
  map[y * width + x] = list;
}

PixelStripeList *PixelMap::getList(int x, int y) {
  return map[y * width + x];
}

