/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

#pragma once

/*
 * The sound resource class is basically a list of sound events along with
 * header information for each MIDI channel. Events are maintained in
 * chronological order and are accessible by their index in the event list.
 * Inserting or deleting events will shift the indices of events that follow
 * them in the list.
 */

class DataStream;

struct sound_event {
	// useful symbolic constants for the status field
	enum sound_status {TEMPO_CHANGE = 0x10, NOTE_OFF = 0x80, NOTE_ON = 0x90,
		KEY_AFTERTOUCH = 0xA0, CONTROL_CHANGE = 0xB0, PATCH_CHANGE = 0xC0,
		CHANNEL_AFTERTOUCH = 0xD0, PITCH_WHEEL = 0xE0, STOP = 0xFC};

	unsigned int absoluteTime;   // absolute time of the event in ticks (60 ticks to the second)
	unsigned char status;        // MIDI status (does not include channel)
	unsigned char channel;       // channel for status, if appropriate
	unsigned char param1;        // first message parameter, if appropriate
	unsigned char param2;        // second message parameter, if appropriate
};


struct sound_channel {
	unsigned char playFlags;       // specify which devices should play the channel
	unsigned char initialVoices;   // initial voices count on the channel
};


class SoundEventList {
public:
	SoundEventList(int capacity = 100);
	~SoundEventList();
	int GetLength() const;
	sound_event GetEvent(int i) const;
	int InsertEvent(const sound_event &event);
	void AppendEvent(const sound_event &event);
	void DeleteEvent(int i);
	void Clear();
	void operator=(const SoundEventList &other);
private:
	void GrowCapacity();
	sound_event *mEvents;
	int mCapacity;
	int mCount;
};


class SoundResource
{
public:
	SoundResource(void);
	~SoundResource(void);

	// Loading and saving
	void Load(DataStream &in);
	void ImportMidi(DataStream &in);
	void Save(DataStream &out) const;
	
	// Accessors and mutators
	int GetTotalTime() const;
	int GetNumEvents() const;
	unsigned char GetChannelPlayFlag(int channel) const;
	void SetChannelPlayFlag(int channel, unsigned char flag);
	unsigned char GetChannelVoices(int channel) const;
	void SetChannelVoices(int channel, unsigned char voices);
	sound_event GetEvent(int i) const;
	int InsertEvent(const sound_event &event);
	void DeleteEvent(int i);

private:
	sound_channel mChannels[16];
	SoundEventList mEvents;
};
