/***************************************************************************
 MtGmPlayer.cpp (C) 1999,2001 Christoph Reichenbach, TU Darmstadt
                (C) 1999-2000 Rickard Lind
				(C) 2002      Ravi Iyengar

 This program may be modified and copied freely according to the terms of
 the GNU general public license (GPL), as long as the above copyright
 notice and the licensing information contained herein are preserved.

 Please refer to www.gnu.org for licensing details.

 This work is provided AS IS, without warranty of any kind, expressed or
 implied, including but not limited to the warranties of merchantibility,
 noninfringement, and fitness for a specific purpose. The author will not
 be held liable for any damage caused by this work or derivatives of it.

 By using this source code, you agree to the licensing terms as stated
 above.


 Please contact the maintainer for bug reports or inquiries.

 Current Maintainer:

    Ravi Iyengar (RNI) [ravi.i@softhome.net]

 Roland MT-32 to General MIDI conversion:

    Rickard Lind [rpl@dd.chalmers.se]

***************************************************************************/


/*
 * The vast majority of the code in this file comes from FreeSCI and
 * was written by Rickard Lind. See http://freesci.linuxgames.com for
 * information about FreeSCI.
 *
 * I removed a few small pieces of the original code that were unnecessary
 * for this application. I also made small changes necessary to compile with
 * this application.
 *
 * I adapted the original event handling code for use with this playback
 * framework.
 *
 *   -- Ravi Iyengar [ravi.i@softhome.net]
 */

#include "StdAfx.h"
#include "SoundResource.h"
#include "MtGmPlayer.h"
#include "Exception.h"
#include "DataStream.h"


//-----------------------------------------------------------------------------


#define MAP_NOT_FOUND -1
#define NOMAP -2
#define RHYTHM -3

#define RHYTHM_CHANNEL 9

typedef char gint8;
typedef unsigned char guint8;
typedef unsigned long guint32;

typedef struct {
	gint8 gm_instr;
	int keyshift;
	int finetune;
	int bender_range;
	gint8 gm_rhythmkey;
	int volume;
} MIDI_map_t;

typedef struct {
	unsigned int size;
	unsigned char *data;
} resource_t;

MIDI_map_t MIDI_mapping[128];
int MIDI_mappings_nr = 128;

int map_MIDI_instruments(resource_t *patch1);


//-----------------------------------------------------------------------------


MtGmPlayer::MtGmPlayer(HWND listener)
: MidiPlayer(listener)
{
}

MtGmPlayer::~MtGmPlayer()
{
}

void MtGmPlayer::OpenDevice(DataStream *patch)
{
	MidiPlayer::OpenDevice(0);

	resource_t patch1;
	patch->Seekg(0, DataStream::END);
	patch1.size = patch->Tellg() - 2;
	patch->Seekg(0, DataStream::BEG);

	if (patch->ReadShort() != 0x89) throw DeviceException("Invalid SCI0 patch resource");

	patch1.data = new unsigned char[patch1.size];
	patch->ReadBuffer(patch1.data, patch1.size);

	map_MIDI_instruments(&patch1);

	delete[] patch1.data;
}

void MtGmPlayer::HandleEvent(const sound_event &event)
{
	sound_event copy = event;
	int xparam;

	switch (event.status) {
		case sound_event::NOTE_OFF:
		case sound_event::NOTE_ON:
			if (MIDI_mapping[mMt32GmChannelMap[event.channel]].gm_instr == RHYTHM) {
				copy.channel = RHYTHM_CHANNEL;
			} else if (MIDI_mapping[mMt32GmChannelMap[event.channel]].gm_instr == NOMAP) {
				break;
			}
			copy.param2 = (copy.param2 * MIDI_mapping[mMt32GmChannelMap[event.channel]].volume) >> 7;
			MidiPlayer::HandleEvent(copy);
			break;

		case sound_event::PATCH_CHANGE:
			mMt32GmChannelMap[event.channel] = event.param1;
			if (event.channel == RHYTHM_CHANNEL) {
				xparam = MIDI_mapping[event.param1].gm_rhythmkey;
			} else {
				xparam = MIDI_mapping[event.param1].gm_instr;

				if (event.channel != RHYTHM_CHANNEL && xparam >= 0) {
					sound_event setupEvent;
					setupEvent.absoluteTime = event.absoluteTime;
					setupEvent.channel = event.channel;

					setupEvent.status = sound_event::CONTROL_CHANGE;
					setupEvent.param1 = 0x65;
					setupEvent.param2 = 0;
					MidiPlayer::HandleEvent(setupEvent);

					setupEvent.param1 = 0x64;
					setupEvent.param2 = 2;
					MidiPlayer::HandleEvent(setupEvent);

					setupEvent.param1 = 0x06;
					setupEvent.param2 = MIDI_mapping[event.param1].keyshift;
					MidiPlayer::HandleEvent(setupEvent);

					setupEvent.param1 = 0x65;
					setupEvent.param2 = 0;
					MidiPlayer::HandleEvent(setupEvent);

					setupEvent.param1 = 0x64;
					setupEvent.param2 = 0;
					MidiPlayer::HandleEvent(setupEvent);

					setupEvent.param1 = 0x06;
					setupEvent.param2 = MIDI_mapping[event.param1].bender_range;
					MidiPlayer::HandleEvent(setupEvent);

					setupEvent.param1 = 0x26;
					setupEvent.param2 = 0;
					MidiPlayer::HandleEvent(setupEvent);
				}
			}

			if (xparam >= 0) {
				copy.param1 = xparam;
				MidiPlayer::HandleEvent(copy);
			}
			break;

		default:
			MidiPlayer::HandleEvent(event);
	}
}


//-----------------------------------------------------------------------------


static struct {
  guint8 group; /* 0=GroupA, 1=GroupB, 2=Memory, 3=Rhythm */
  char *name;
} MT32_patch[128];


static struct {
  char *name;
  gint8 gm_instr;
  gint8 gm_rhythm_key;
} MT32_PresetTimbreMaps[] = {
/*000*/  {"AcouPiano1", 0, NOMAP},
/*001*/  {"AcouPiano2", 1, NOMAP},
/*002*/  {"AcouPiano3", 0, NOMAP},
/*003*/  {"ElecPiano1", 4, NOMAP},
/*004*/  {"ElecPiano2", 5, NOMAP},
/*005*/  {"ElecPiano3", 4, NOMAP},
/*006*/  {"ElecPiano4", 5, NOMAP},
/*007*/  {"Honkytonk ", 3, NOMAP},
/*008*/  {"Elec Org 1", 16, NOMAP},
/*009*/  {"Elec Org 2", 17, NOMAP},
/*010*/  {"Elec Org 3", 18, NOMAP},
/*011*/  {"Elec Org 4", 18, NOMAP},
/*012*/  {"Pipe Org 1", 19, NOMAP},
/*013*/  {"Pipe Org 2", 19, NOMAP},
/*014*/  {"Pipe Org 3", 20, NOMAP},
/*015*/  {"Accordion ", 21, NOMAP},
/*016*/  {"Harpsi 1  ", 6, NOMAP},
/*017*/  {"Harpsi 2  ", 6, NOMAP},
/*018*/  {"Harpsi 3  ", 6, NOMAP},
/*019*/  {"Clavi 1   ", 7, NOMAP},
/*020*/  {"Clavi 2   ", 7, NOMAP},
/*021*/  {"Clavi 3   ", 7, NOMAP},
/*022*/  {"Celesta 1 ", 8, NOMAP},
/*023*/  {"Celesta 2 ", 8, NOMAP},
/*024*/  {"Syn Brass1", 62, NOMAP},
/*025*/  {"Syn Brass2", 63, NOMAP},
/*026*/  {"Syn Brass3", 62, NOMAP},
/*027*/  {"Syn Brass4", 63, NOMAP},
/*028*/  {"Syn Bass 1", 38, NOMAP},
/*029*/  {"Syn Bass 2", 39, NOMAP},
/*030*/  {"Syn Bass 3", 38, NOMAP},
/*031*/  {"Syn Bass 4", 39, NOMAP},
/*032*/  {"Fantasy   ", 88, NOMAP},
/*033*/  {"Harmo Pan ", 89, NOMAP},
/*034*/  {"Chorale   ", 52, NOMAP},
/*035*/  {"Glasses   ", 98, NOMAP},
/*036*/  {"Soundtrack", 97, NOMAP},
/*037*/  {"Atmosphere", 99, NOMAP},
/*038*/  {"Warm Bell ", 89, NOMAP},
/*039*/  {"Funny Vox ", 85, NOMAP},
/*040*/  {"Echo Bell ", 39, NOMAP},
/*041*/  {"Ice Rain  ", 101, NOMAP},
/*042*/  {"Oboe 2001 ", 68, NOMAP},
/*043*/  {"Echo Pan  ", 87, NOMAP},
/*044*/  {"DoctorSolo", 86, NOMAP},
/*045*/  {"Schooldaze", 103, NOMAP},
/*046*/  {"BellSinger", 88, NOMAP},
/*047*/  {"SquareWave", 80, NOMAP},
/*048*/  {"Str Sect 1", 48, NOMAP},
/*049*/  {"Str Sect 2", 48, NOMAP},
/*050*/  {"Str Sect 3", 49, NOMAP},
/*051*/  {"Pizzicato ", 45, NOMAP},
/*052*/  {"Violin 1  ", 40, NOMAP},
/*053*/  {"Violin 2  ", 40, NOMAP},
/*054*/  {"Cello 1   ", 42, NOMAP},
/*055*/  {"Cello 2   ", 42, NOMAP},
/*056*/  {"Contrabass", 43, NOMAP},
/*057*/  {"Harp 1    ", 46, NOMAP},
/*058*/  {"Harp 2    ", 46, NOMAP},
/*059*/  {"Guitar 1  ", 24, NOMAP},
/*060*/  {"Guitar 2  ", 25, NOMAP},
/*061*/  {"Elec Gtr 1", 26, NOMAP},
/*062*/  {"Elec Gtr 2", 27, NOMAP},
/*063*/  {"Sitar     ", 104, NOMAP},
/*064*/  {"Acou Bass1", 32, NOMAP},
/*065*/  {"Acou Bass2", 33, NOMAP},
/*066*/  {"Elec Bass1", 34, NOMAP},
/*067*/  {"Elec Bass2", 39, NOMAP},
/*068*/  {"Slap Bass1", 36, NOMAP},
/*069*/  {"Slap Bass2", 37, NOMAP},
/*070*/  {"Fretless 1", 35, NOMAP},
/*071*/  {"Fretless 2", 35, NOMAP},
/*072*/  {"Flute 1   ", 73, NOMAP},
/*073*/  {"Flute 2   ", 73, NOMAP},
/*074*/  {"Piccolo 1 ", 72, NOMAP},
/*075*/  {"Piccolo 2 ", 72, NOMAP},
/*076*/  {"Recorder  ", 74, NOMAP},
/*077*/  {"Panpipes  ", 75, NOMAP},
/*078*/  {"Sax 1     ", 64, NOMAP},
/*079*/  {"Sax 2     ", 65, NOMAP},
/*080*/  {"Sax 3     ", 66, NOMAP},
/*081*/  {"Sax 4     ", 67, NOMAP},
/*082*/  {"Clarinet 1", 71, NOMAP},
/*083*/  {"Clarinet 2", 71, NOMAP},
/*084*/  {"Oboe      ", 68, NOMAP},
/*085*/  {"Engl Horn ", 69, NOMAP},
/*086*/  {"Bassoon   ", 70, NOMAP},
/*087*/  {"Harmonica ", 22, NOMAP},
/*088*/  {"Trumpet 1 ", 56, NOMAP},
/*089*/  {"Trumpet 2 ", 56, NOMAP},
/*090*/  {"Trombone 1", 57, NOMAP},
/*091*/  {"Trombone 2", 57, NOMAP},
/*092*/  {"Fr Horn 1 ", 60, NOMAP},
/*093*/  {"Fr Horn 2 ", 60, NOMAP},
/*094*/  {"Tuba      ", 58, NOMAP},
/*095*/  {"Brs Sect 1", 61, NOMAP},
/*096*/  {"Brs Sect 2", 61, NOMAP},
/*097*/  {"Vibe 1    ", 11, NOMAP},
/*098*/  {"Vibe 2    ", 11, NOMAP},
/*099*/  {"Syn Mallet", 15, NOMAP},
/*100*/  {"Wind Bell ", 88, NOMAP},
/*101*/  {"Glock     ", 9, NOMAP},
/*102*/  {"Tube Bell ", 14, NOMAP},
/*103*/  {"Xylophone ", 13, NOMAP},
/*104*/  {"Marimba   ", 12, NOMAP},
/*105*/  {"Koto      ", 107, NOMAP},
/*106*/  {"Sho       ", 111, NOMAP},
/*107*/  {"Shakuhachi", 77, NOMAP},
/*108*/  {"Whistle 1 ", 78, NOMAP},
/*109*/  {"Whistle 2 ", 78, NOMAP},
/*110*/  {"BottleBlow", 76, NOMAP},
/*111*/  {"BreathPipe", 121, NOMAP},
/*112*/  {"Timpani   ", 47, NOMAP},
/*113*/  {"MelodicTom", 117, NOMAP},
/*114*/  {"Deep Snare", RHYTHM, 37},
/*115*/  {"Elec Perc1", 115, NOMAP}, /* ? */
/*116*/  {"Elec Perc2", 118, NOMAP}, /* ? */
/*117*/  {"Taiko     ", 116, NOMAP},
/*118*/  {"Taiko Rim ", 118, NOMAP},
/*119*/  {"Cymbal    ", RHYTHM, 50},
/*120*/  {"Castanets ", RHYTHM, NOMAP},
/*121*/  {"Triangle  ", 112, NOMAP},
/*122*/  {"Orche Hit ", 55, NOMAP},
/*123*/  {"Telephone ", 124, NOMAP},
/*124*/  {"Bird Tweet", 123, NOMAP},
/*125*/  {"OneNoteJam", NOMAP, NOMAP}, /* ? */
/*126*/  {"WaterBells", 98, NOMAP},
/*127*/  {"JungleTune", NOMAP, NOMAP} /* ? */
};

static struct {
char *name;
gint8 gm_instr;
gint8 gm_rhythmkey;
} MT32_RhythmTimbreMaps[] = {
/*00*/  {"Acou BD   ", RHYTHM, 34},
/*01*/  {"Acou SD   ", RHYTHM, 37},
/*02*/  {"Acou HiTom", 117, 49},
/*03*/  {"AcouMidTom", 117, 46},
/*04*/  {"AcouLowTom", 117, 40},
/*05*/  {"Elec SD   ", RHYTHM, 39},
/*06*/  {"Clsd HiHat", RHYTHM, 41},
/*07*/  {"OpenHiHat1", RHYTHM, 45},
/*08*/  {"Crash Cym ", RHYTHM, 48},
/*09*/  {"Ride Cym  ", RHYTHM, 50},
/*10*/  {"Rim Shot  ", RHYTHM, 36},
/*11*/  {"Hand Clap ", RHYTHM, 38},
/*12*/  {"Cowbell   ", RHYTHM, 55},
/*13*/  {"Mt HiConga", RHYTHM, 61},
/*14*/  {"High Conga", RHYTHM, 62},
/*15*/  {"Low Conga ", RHYTHM, 63},
/*16*/  {"Hi Timbale", RHYTHM, 64},
/*17*/  {"LowTimbale", RHYTHM, 65},
/*18*/  {"High Bongo", RHYTHM, 59},
/*19*/  {"Low Bongo ", RHYTHM, 60},
/*20*/  {"High Agogo", 113, 66},
/*21*/  {"Low Agogo ", 113, 67},
/*22*/  {"Tambourine", RHYTHM, 53},
/*23*/  {"Claves    ", RHYTHM, 74},
/*24*/  {"Maracas   ", RHYTHM, 69},
/*25*/  {"SmbaWhis L", 78, 71},
/*26*/  {"SmbaWhis S", 78, 70},
/*27*/  {"Cabasa    ", RHYTHM, 68},
/*28*/  {"Quijada   ", RHYTHM, 72},
/*29*/  {"OpenHiHat2", RHYTHM, 43}
};

static gint8 MT32_PresetRhythmKeymap[] = {
NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP,
NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP,
NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP,
NOMAP, NOMAP, NOMAP, NOMAP, 34, 34, 36, 37, 38, 39,
40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
50, NOMAP, NOMAP, 53, NOMAP, 55, NOMAP, NOMAP, NOMAP, 59,
60, 61, 62, 63, 64, 65, 66, 67, 68, 69,
70, 71, 72, NOMAP, 74, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP,
NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP,
NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP,
NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP,
NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP,
NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP, NOMAP
};

/* +++ - Don't change unless you've got a good reason
   ++  - Looks good, sounds ok
   +   - Not too bad, but is it right?
   ?   - Where do I map this one?
   ??  - Any good ideas?
   ??? - I'm clueless?
   R   - Rhythm... */
static struct {
  char *name;
  gint8 gm_instr;
  gint8 gm_rhythm_key;
} MT32_MemoryTimbreMaps[] = {
{"AccPnoKA2 ", 1, NOMAP},     /* ++ (KQ1) */
{"Acou BD   ", RHYTHM, 34},   /* R (PQ2) */
{"Acou SD   ", RHYTHM, 37},   /* R (PQ2) */
{"AcouPnoKA ", 0, NOMAP},     /* ++ (KQ1) */
{"BASS      ", 32, NOMAP},    /* + (LSL3) */
{"BASSOONPCM", 70, NOMAP},    /* + (CB) */
{"BEACH WAVE", 122, NOMAP},   /* + (LSL3) */
{"BagPipes  ", 109, NOMAP},
{"BassPizzMS", 45, NOMAP},    /* ++ (HQ) */
{"BassoonKA ", 70, NOMAP},    /* ++ (KQ1) */
{"Bell    MS", 112, NOMAP},   /* ++ (iceMan) */
{"Bells   MS", 112, NOMAP},   /* + (HQ) */
{"Big Bell  ", 14, NOMAP},    /* + (CB) */
{"Bird Tweet", 123, NOMAP},
{"BrsSect MS", 61, NOMAP},    /* +++ (iceMan) */
{"CLAPPING  ", 126, NOMAP},   /* ++ (LSL3) */
{"Cabasa    ", RHYTHM, 68},   /* R (HBoG) */
{"Calliope  ", 82, NOMAP},    /* +++ (HQ) */
{"CelticHarp", 46, NOMAP},    /* ++ (CoC) */
{"Chicago MS", 1, NOMAP},     /* ++ (iceMan) */
{"Chop      ", 117, NOMAP},
{"Chorale MS", 52, NOMAP},    /* + (CoC) */
{"ClarinetMS", 71, NOMAP},
{"Claves    ", RHYTHM, 74},   /* R (PQ2) */
{"Claw    MS", 118, NOMAP},    /* + (HQ) */
{"ClockBell ", 14, NOMAP},    /* + (CB) */
{"ConcertCym", RHYTHM, 54},   /* R ? (KQ1) */
{"Conga   MS", RHYTHM, 63},   /* R (HQ) */
{"CoolPhone ", 124, NOMAP},   /* ++ (LSL3) */
{"CracklesMS", 115, NOMAP}, /* ? (CoC, HQ) */
{"CreakyD MS", NOMAP, NOMAP}, /* ??? (KQ1) */
{"Cricket   ", 120, NOMAP}, /* ? (CB) */
{"CrshCymbMS", RHYTHM, 56},   /* R +++ (iceMan) */
{"CstlGateMS", NOMAP, NOMAP}, /* ? (HQ) */
{"CymSwellMS", RHYTHM, 54},   /* R ? (CoC, HQ) */
{"CymbRollKA", RHYTHM, 56},   /* R ? (KQ1) */
{"Cymbal Lo ", NOMAP, NOMAP}, /* R ? (LSL3) */
{"card      ", NOMAP, NOMAP}, /* ? (HBoG) */
{"DirtGtr MS", 30, NOMAP},    /* + (iceMan) */
{"DirtGtr2MS", 29, NOMAP},    /* + (iceMan) */
{"E Bass  MS", 33, NOMAP},    /* + (SQ3) */
{"ElecBassMS", 33, NOMAP},
{"ElecGtr MS", 27, NOMAP},    /* ++ (iceMan) */
{"EnglHornMS", 69, NOMAP},
{"FantasiaKA", 88, NOMAP},
{"Fantasy   ", 99, NOMAP},    /* + (PQ2) */
{"Fantasy2MS", 99, NOMAP},    /* ++ (CoC, HQ) */
{"Filter  MS", 95, NOMAP},    /* +++ (iceMan) */
{"Filter2 MS", 95, NOMAP},    /* ++ (iceMan) */
{"Flame2  MS", 121, NOMAP},   /* ? (HQ) */
{"Flames  MS", 121, NOMAP},   /* ? (HQ) */
{"Flute   MS", 73, NOMAP},    /* +++ (HQ) */
{"FogHorn MS", 58, NOMAP},
{"FrHorn1 MS", 60, NOMAP},    /* +++ (HQ) */
{"FunnyTrmp ", 56, NOMAP},    /* ++ (CB) */
{"GameSnd MS", 80, NOMAP},
{"Glock   MS", 9, NOMAP},     /* +++ (HQ) */
{"Gunshot   ", 127, NOMAP},   /* +++ (CB) */
{"Hammer  MS", NOMAP, NOMAP}, /* ? (HQ) */
{"Harmonica2", 22, NOMAP},    /* +++ (CB) */
{"Harpsi 1  ", 6, NOMAP},     /* + (HBoG) */
{"Harpsi 2  ", 6, NOMAP},     /* +++ (CB) */
{"Heart   MS", 116, NOMAP},   /* ? (iceMan) */
{"Horse1  MS", 115, NOMAP},   /* ? (CoC, HQ) */
{"Horse2  MS", 115, NOMAP},   /* ? (CoC, HQ) */
{"InHale  MS", 121, NOMAP},   /* ++ (iceMan) */
{"KNIFE     ", 120, NOMAP},   /* ? (LSL3) */
{"KenBanjo  ", 105, NOMAP},   /* +++ (CB) */
{"Kiss    MS", 25, NOMAP},    /* ++ (HQ) */
{"KongHit   ", NOMAP, NOMAP}, /* ??? (KQ1) */
{"Koto      ", 107, NOMAP},   /* +++ (PQ2) */
{"Laser   MS", 81, NOMAP},    /* ?? (HQ) */
{"Meeps   MS", 62, NOMAP},    /* ? (HQ) */
{"MTrak   MS", 62, NOMAP},    /* ?? (iceMan) */
{"MachGun MS", 127, NOMAP},   /* ? (iceMan) */
{"OCEANSOUND", 122, NOMAP},   /* + (LSL3) */
{"Oboe 2001 ", 68, NOMAP},    /* + (PQ2) */
{"Ocean   MS", 122, NOMAP},   /* + (iceMan) */
{"PPG 2.3 MS", 75, NOMAP},    /* ? (iceMan) */
{"PianoCrank", NOMAP, NOMAP}, /* ? (CB) */
{"PicSnareMS", RHYTHM, 39},   /* R ? (iceMan) */
{"PiccoloKA ", 72, NOMAP},    /* +++ (KQ1) */
{"PinkBassMS", 39, NOMAP},
{"Pizz2     ", 45, NOMAP},    /* ++ (CB) */
{"Portcullis", NOMAP, NOMAP}, /* ? (KQ1) */
{"Raspbry MS", 81, NOMAP},    /* ? (HQ) */
{"RatSqueek ", 72, NOMAP},    /* ? (CB, CoC) */
{"Record78  ", NOMAP, NOMAP}, /* +++ (CB) */
{"RecorderMS", 74, NOMAP},    /* +++ (CoC) */
{"Red Baron ", 125, NOMAP},   /* ? (CB) */
{"ReedPipMS ", 20, NOMAP},    /* +++ (Coc) */
{"RevCymb MS", 119, NOMAP},
{"RifleShot ", 127, NOMAP},   /* + (CB) */
{"RimShot MS", RHYTHM, 36},   /* R */
{"SHOWER    ", 52, NOMAP},    /* ? (LSL3) */
{"SQ Bass MS", 32, NOMAP},    /* + (SQ3) */
{"ShakuVibMS", 79, NOMAP},    /* + (iceMan) */
{"SlapBassMS", 36, NOMAP},    /* +++ (iceMan) */
{"Snare   MS", RHYTHM, 37},   /* R (HQ) */
{"Some Birds", 123, NOMAP},   /* + (CB) */
{"Sonar   MS", 78, NOMAP},    /* ? (iceMan) */
{"Soundtrk2 ", 97, NOMAP},    /* +++ (CB) */
{"Soundtrack", 97, NOMAP},    /* ++ (CoC) */
{"SqurWaveMS", 80, NOMAP},
{"StabBassMS", 34, NOMAP},    /* + (iceMan) */
{"SteelDrmMS", 114, NOMAP},   /* +++ (iceMan) */
{"StrSect1MS", 48, NOMAP},    /* ++ (HQ) */
{"String  MS", 45, NOMAP},    /* + (CoC) */
{"Syn-Choir ", 91, NOMAP},
{"Syn Brass4", 63, NOMAP},    /* ++ (PQ2) */
{"SynBass MS", 38, NOMAP},
{"SwmpBackgr", 120, NOMAP},    /* ?? (CB,HQ) */
{"T-Bone2 MS", 57, NOMAP},    /* +++ (HQ) */
{"Taiko     ", 116, 34},      /* +++ (Coc) */
{"Taiko Rim ", 118, 36},      /* +++ (LSL3) */
{"Timpani1  ", 47, NOMAP},    /* +++ (CB) */
{"Tom     MS", 117, 47},      /* +++ (iceMan) */
{"Toms    MS", 117, 47},      /* +++ (CoC, HQ) */
{"Tpt1prtl  ", 56, NOMAP},    /* +++ (KQ1) */
{"TriangleMS", 112, 80},      /* R (CoC) */
{"Trumpet 1 ", 56, NOMAP},    /* +++ (CoC) */
{"Type    MS", 114, NOMAP},   /* ? (iceMan) */
{"WaterBells", 98, NOMAP},    /* + (PQ2) */
{"WaterFallK", NOMAP, NOMAP}, /* ? (KQ1) */
{"Whiporill ", 123, NOMAP},   /* + (CB) */
{"Wind      ", NOMAP, NOMAP}, /* ? (CB) */
{"Wind    MS", NOMAP, NOMAP}, /* ? (HQ, iceMan) */
{"Wind2   MS", NOMAP, NOMAP}, /* ? (CoC) */
{"Woodpecker", 115, NOMAP},   /* ? (CB) */
{"WtrFall MS", NOMAP, NOMAP}, /* ? (CoC, HQ, iceMan) */
{0, 0}
};

gint8
_lookup_instrument(char *iname)
{
  int i = 0;

  while (MT32_MemoryTimbreMaps[i].name) {
    if (strnicmp(iname, MT32_MemoryTimbreMaps[i].name, 10) == 0)
      return MT32_MemoryTimbreMaps[i].gm_instr;
    i++;
  }
  return MAP_NOT_FOUND;
}

gint8
_lookup_rhythm_key(char *iname)
{
  int i = 0;

  while (MT32_MemoryTimbreMaps[i].name) {
    if (strnicmp(iname, MT32_MemoryTimbreMaps[i].name, 10) == 0)
      return MT32_MemoryTimbreMaps[i].gm_rhythm_key;
    i++;
  }
  return MAP_NOT_FOUND;
}

int
map_MIDI_instruments(resource_t *patch1)
{
  int memtimbres;
  int patches;
  int i;
  guint8 group, number, keyshift, finetune, bender_range;
  guint8 *patchpointer;
  guint32 pos;

  for (i = 0; i < 128; i++) {
    MIDI_mapping[i].gm_instr = MT32_PresetTimbreMaps[i].gm_instr;
    MIDI_mapping[i].keyshift = 0x40;
    MIDI_mapping[i].finetune = 0x2000;
    MIDI_mapping[i].bender_range = 0x0c;
    MT32_patch[i].name = MT32_PresetTimbreMaps[i].name;
  }

  for (i = 0; i < 128; i++) {
    MIDI_mapping[i].gm_rhythmkey = MT32_PresetRhythmKeymap[i];
    MIDI_mapping[i].volume = 128;
  }

  memtimbres = *(patch1->data + 0x1EB);
  pos = 0x1EC + memtimbres * 0xF6;

  if (patch1->size > pos && \
      ((0x100 * *(patch1->data + pos) + *(patch1->data +pos + 1)) == 0xABCD)) {
    patches = 96;
    pos += 2 + 8 * 48;
  } else
    patches = 48;

  for (i = 0; i < patches; i++) {
    if (i < 48) {
      group = *(patch1->data + 0x6B + 8 * i);
      number = *(patch1->data + 0x6B + 8 * i + 1);
      keyshift = *(patch1->data + 0x6B + 8 * i + 2);
      finetune = *(patch1->data + 0x6B + 8 * i + 3);
      bender_range = *(patch1->data + 0x6B + 8 * i + 4);
      patchpointer = patch1->data + 0x6B + 8 * i;

    } else {
      group = *(patch1->data + 0x1EC + 8 * (i - 48) + memtimbres * 0xF6 + 2);
      number = *(patch1->data + 0x1EC + 8 * (i - 48) + memtimbres * 0xF6 + 2 + 1);
      keyshift = *(patch1->data + 0x1EC + 8 * (i - 48) + memtimbres * 0xF6 + 2 + 2);
      finetune = *(patch1->data + 0x1EC + 8 * (i - 48) + memtimbres * 0xF6 + 2 + 3);
      bender_range = *(patch1->data + 0x1EC + 8 * (i - 48) + memtimbres * 0xF6 + 2 + 4);
      patchpointer = patch1->data + 0x1EC + 8 * (i - 48) + memtimbres * 0xF6 + 2;
    }

    MT32_patch[i].group = group;

    /*    if ((patchpointer[0] == 0) &&
	  (patchpointer[1] == 0) &&
	  (patchpointer[2] == 0) &&
	  (patchpointer[3] == 0) &&
	  (patchpointer[4] == 0) &&
	  (patchpointer[5] == 0) &&
	  (patchpointer[6] == 0) &&
	  (patchpointer[7] == 0)) {
	  MIDI_mapping[i].gm_instr = NOMAP; */
      /* SCIsdebug("Patch %d will be unmapped!\n", i); */
    /*    } else { */
      switch (group) {
        case 0:
	  MT32_patch[i].name = MT32_PresetTimbreMaps[number].name;
	  MIDI_mapping[i].gm_instr = MT32_PresetTimbreMaps[number].gm_instr;
	  break;
        case 1:
	  MT32_patch[i].name = MT32_PresetTimbreMaps[number + 64].name;
          MIDI_mapping[i].gm_instr = MT32_PresetTimbreMaps[number + 64].gm_instr;
	  break;
        case 2:
	  MT32_patch[i].name = (char *) patch1->data + 0x1EC + number * 0xF6;
	  MIDI_mapping[i].gm_instr = _lookup_instrument((char *) patch1->data + 0x1EC + number * 0xF6);
	  /* SCIsdebug("Patch %d => %d\n",i, MIDI_mapping[i].gm_instr); */
	  break;
        case 3:
	  MT32_patch[i].name = MT32_RhythmTimbreMaps[number].name;
	  MIDI_mapping[i].gm_instr = MT32_RhythmTimbreMaps[number].gm_instr;
	  break;
        default:
	  break;
	  /*      } */
      MIDI_mapping[i].keyshift = 0x40 + ((int) (keyshift & 0x3F) - 24);
      if (finetune != 50)
      /* MIDI_mapping[i].finetune = 0x2000 + (((gint32) (finetune & 0x7F) - 50) << 11) / 25; */
      MIDI_mapping[i].finetune = 0x2000 + ((int) (finetune & 0x7F) - 50);
      MIDI_mapping[i].bender_range = (int) (bender_range & 0x1F);
    }
  }

  /*
  for (i = patches; i < 128; i++)
    MIDI_mapping[i].gm_instr = NOMAP;
  */

  if (patch1->size > pos && \
      ((0x100 * *(patch1->data + pos) + *(patch1->data + pos + 1)) == 0xDCBA)) {
    for (i = 0; i < 64 ; i++) {
      number = *(patch1->data + pos + 4 * i + 2);
      if (number < 64)
	MIDI_mapping[i + 23].gm_rhythmkey = _lookup_rhythm_key((char *) patch1->data + 0x1EC + number * 0xF6);
      else if (number < 94)
        MIDI_mapping[i + 23].gm_rhythmkey = MT32_RhythmTimbreMaps[number - 64].gm_rhythmkey;
      else
	MIDI_mapping[i + 23].gm_rhythmkey = NOMAP;
      MIDI_mapping[i + 23].volume = (((long) *(patch1->data + pos + 4 * i + 3))*128) /100;
      /* SCIsdebug("%d => %d\n", i + 23, MIDI_mapping[i + 23].gm_rhythmkey); */
    }
  }
/*
  if (logfile)
    fclose(logfile);
	*/
  return 0;
}
