/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

// CuesDlg.cpp : implementation file
//

#include "stdafx.h"
#include "Soundbox.h"
#include "CuesDlg.h"
#include "SoundResource.h"


// CCuesDlg dialog

IMPLEMENT_DYNAMIC(CCuesDlg, CDialog)
CCuesDlg::CCuesDlg(SoundResource *sound, CWnd* pParent /*=NULL*/)
	: CDialog(CCuesDlg::IDD, pParent), mSound(sound)
{
}

CCuesDlg::~CCuesDlg()
{
}

void CCuesDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
}

void CCuesDlg::FixIndices(int startSoundIndex, int delta)
{
	int localIndex, i, n;

	localIndex = 0;
	n = (int)mDirectIndex.GetCount();
	while ((localIndex < n) && (mDirectIndex.GetAt(localIndex) < startSoundIndex)) ++localIndex;
	for (i = localIndex; i < n; ++i) {
		mDirectIndex.SetAt(i, mDirectIndex.GetAt(i) + delta);
	}

	localIndex = 0;
	n = (int)mCumulativeIndex.GetCount();
	while ((localIndex < n) && (mCumulativeIndex.GetAt(localIndex) < startSoundIndex)) ++localIndex;
	for (i = localIndex; i < n; ++i) {
		mCumulativeIndex.SetAt(i, mCumulativeIndex.GetAt(i) + delta);
	}
}


BEGIN_MESSAGE_MAP(CCuesDlg, CDialog)
	ON_LBN_SELCHANGE(IDC_DIRECTLIST, OnDirectListSelChange)
	ON_LBN_SELCHANGE(IDC_CUMULATIVELIST, OnCumulativeListSelChange)
	ON_BN_CLICKED(IDC_DIRECTDELBTN, OnDirectDelBtnClicked)
	ON_BN_CLICKED(IDC_CUMULATIVEDELBTN, OnCumulativeDelBtnClicked)
	ON_BN_CLICKED(IDC_ADDBTN, OnAddBtnClicked)
END_MESSAGE_MAP()


// CCuesDlg message handlers

BOOL CCuesDlg::OnInitDialog()
{
	CDialog::OnInitDialog();

	CListBox *directList = (CListBox*)GetDlgItem(IDC_DIRECTLIST);
	CListBox *cumulativeList = (CListBox*)GetDlgItem(IDC_CUMULATIVELIST);
	CButton *btn;
	CSpinButtonCtrl *spin;
	CEdit *edit;

	directList->SetFont(&CSoundboxApp::mListFont);
	cumulativeList->SetFont(&CSoundboxApp::mListFont);

	btn = (CButton*)GetDlgItem(IDC_DIRECTDELBTN);
	btn->EnableWindow(false);
	btn = (CButton*)GetDlgItem(IDC_CUMULATIVEDELBTN);
	btn->EnableWindow(false);

	spin = (CSpinButtonCtrl*)GetDlgItem(IDC_TIMESPIN);
	spin->SetRange(0, mSound->GetTotalTime());
	spin = (CSpinButtonCtrl*)GetDlgItem(IDC_VALUESPIN);
	spin->SetRange(0, 127);

	edit = (CEdit*)GetDlgItem(IDC_VALUEEDIT);
	edit->SetWindowText("1");

	char txt[32];
	int n = mSound->GetNumEvents();
	for (int i = 0; i < n; ++i) {
		sound_event event = mSound->GetEvent(i);
		if ((event.channel == 15) &&
			(event.status == sound_event::PATCH_CHANGE) &&
			(event.param1 != 127)) {

			// direct cue
			sprintf(txt, "%-6d %d", event.absoluteTime, event.param1);
			directList->AddString(txt);
			mDirectIndex.Add(i);
		} else if ((event.status == sound_event::CONTROL_CHANGE) &&
			(event.param1 == 0x60)) {

			// cumulative cue
			sprintf(txt, "%-6d %d", event.absoluteTime, event.param2);
			cumulativeList->AddString(txt);
			mCumulativeIndex.Add(i);
		}
	}

	return TRUE;
}

void CCuesDlg::OnDirectListSelChange()
{
	// enable delete button
	CListBox *listBox = (CListBox*)GetDlgItem(IDC_DIRECTLIST);
	CButton *btn = (CButton*)GetDlgItem(IDC_DIRECTDELBTN);
	btn->EnableWindow(listBox->GetCurSel() != -1);
}

void CCuesDlg::OnCumulativeListSelChange()
{
	// enable delete button
	CListBox *listBox = (CListBox*)GetDlgItem(IDC_CUMULATIVELIST);
	CButton *btn = (CButton*)GetDlgItem(IDC_CUMULATIVEDELBTN);
	btn->EnableWindow(listBox->GetCurSel() != -1);
}

void CCuesDlg::OnDirectDelBtnClicked()
{
	// remove the event and its entry from the list
	CListBox *listBox = (CListBox*)GetDlgItem(IDC_DIRECTLIST);
	int localIndex = listBox->GetCurSel();
	int soundIndex = mDirectIndex.GetAt(localIndex);
	mSound->DeleteEvent(soundIndex);
	listBox->DeleteString(localIndex);

	// fix indices
	mDirectIndex.RemoveAt(localIndex);
	FixIndices(soundIndex, -1);

	// select another item from the list
	if (mDirectIndex.GetCount() > localIndex) {
		listBox->SetCurSel(localIndex);
	} else if (localIndex > 0) {
		listBox->SetCurSel(localIndex - 1);
	} else {
		CButton *delBtn = (CButton*)GetDlgItem(IDC_DIRECTDELBTN);
 		delBtn->EnableWindow(false);
	}
}

void CCuesDlg::OnCumulativeDelBtnClicked()
{
	// remove the event and its entry from the list
	CListBox *listBox = (CListBox*)GetDlgItem(IDC_CUMULATIVELIST);
	int localIndex = listBox->GetCurSel();
	int soundIndex = mCumulativeIndex.GetAt(localIndex);
	mSound->DeleteEvent(soundIndex);
	listBox->DeleteString(localIndex);

	// fix indices
	mCumulativeIndex.RemoveAt(localIndex);
	FixIndices(soundIndex, -1);

	// select another item from the list
	if (mCumulativeIndex.GetCount() > localIndex) {
		listBox->SetCurSel(localIndex);
	} else if (localIndex > 0) {
		listBox->SetCurSel(localIndex - 1);
	} else {
		CButton *delBtn = (CButton*)GetDlgItem(IDC_CUMULATIVEDELBTN);
 		delBtn->EnableWindow(false);
	}
}

void CCuesDlg::OnAddBtnClicked()
{
	CEdit *edit = (CEdit*)GetDlgItem(IDC_TIMEEDIT);
	CString str;
	edit->GetWindowText(str);
	int time = atoi(str);

	edit = (CEdit*)GetDlgItem(IDC_VALUEEDIT);
	edit->GetWindowText(str);
	int value = atoi(str);

	CButton *btn = (CButton*)GetDlgItem(IDC_CUMULATIVECHECK);
	bool cumulative = (btn->GetCheck() == BST_CHECKED);

	// verify a valid time
	if (time < 0) {
		MessageBox("Time must be greater than or equal to 0");
		return;
	}
	if (time > mSound->GetTotalTime()) {
		MessageBox("Time cannot extend past end of soung");
		return;
	}

	// verify a valid value
	if (value < 0) {
		MessageBox("Value must be greater than or equal to 0");
		return;
	}
	if (value > 127) {
		MessageBox("Value must be less than 128");
		return;
	}

	char txt[16];
	sprintf(txt, "%-6d %d", time, value);

	sound_event event;
	if (cumulative) {
		event.absoluteTime = time;
		event.status = sound_event::CONTROL_CHANGE;
		event.channel = 15;
		event.param1 = 0x60;
		event.param2 = value;

		int soundIndex = mSound->InsertEvent(event);
		FixIndices(soundIndex, 1);
		int localIndex = 0;
		int n = (int)mCumulativeIndex.GetCount();
		while ((localIndex < n) && (mCumulativeIndex.GetAt(localIndex) < soundIndex)) ++localIndex;
		mCumulativeIndex.InsertAt(localIndex, soundIndex);

		CListBox *listBox = (CListBox*)GetDlgItem(IDC_CUMULATIVELIST);
		listBox->InsertString(localIndex, txt);
	} else {
		event.absoluteTime = time;
		event.status = sound_event::PATCH_CHANGE;
		event.channel = 15;
		event.param1 = value;
		event.param2 = 0xFF;

		int soundIndex = mSound->InsertEvent(event);
		FixIndices(soundIndex, 1);
		int localIndex = 0;
		int n = (int)mDirectIndex.GetCount();
		while ((localIndex < n) && (mDirectIndex.GetAt(localIndex) < soundIndex)) ++localIndex;
		mDirectIndex.InsertAt(localIndex, soundIndex);

		CListBox *listBox = (CListBox*)GetDlgItem(IDC_DIRECTLIST);
		listBox->InsertString(localIndex, txt);
	}
}
